#include "chatview_webkit.h"

#include "msgmle.h"
#include "psioptions.h"
#include "textutil.h"

#include <QWidget>
#include <QWebFrame>
#include <QFile>
#include <QDebug>
#include <QLayout>
#include <QPalette>

#include "webview.h"
#include "psiapplication.h"

//----------------------------------------------------------------------------
// ChatView
//----------------------------------------------------------------------------
ChatView::ChatView(QWidget *parent)
	: QFrame(parent)
{
	webView = new WebView(this);
	webView->setFocusPolicy(Qt::NoFocus);

	QVBoxLayout *layout = new QVBoxLayout;
	layout->setContentsMargins(0,0,0,0);
	layout->addWidget(webView);
	setLayout(layout);
	setFrameStyle(QFrame::StyledPanel);

	webView->page()->mainFrame()->setHtml(QString(
"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">"
"<html>"
"<head>"
"  <style id=\"baseStyle\" type=\"text/css\" media=\"screen,print\">"
"     *{ word-wrap:break-word; word-break:break-all;}"
"	  html, body {padding:0; margin:0;width:100%}"
"     body {color:" + palette().color(QPalette::WindowText).name() + "}"
#if QT_VERSION < 0x040600
"body::selection {color:"+palette().color(QPalette::Active, QPalette::HighlightedText).name()+";"
"background-color:"+palette().color(QPalette::Active, QPalette::Highlight).name()+"}"
#endif
"	  a {text-decoration:underline}"
"  </style>"
"</head><body></body></html>"));

	importJSChatFunctions();

#ifndef Q_WS_X11	// linux has this feature built-in
	connect(webView->page(), SIGNAL(selectionChanged()), this, SLOT(autoCopy()));
#endif
}

ChatView::~ChatView()
{
}

void ChatView::markReceived(QString id)
{
	//stub
}

QSize ChatView::sizeHint() const
{
	return minimumSizeHint();
}

void ChatView::setFont(const QFont &f)
{
	QWebSettings *s = webView->settings();

	QString weight = "normal";
	switch (f.weight()) {
		case QFont::Light: weight = "lighter"; break;
		case QFont::DemiBold: weight = "bold"; break;
		case QFont::Bold: weight = "bolder"; break;
		case QFont::Black: weight = "900"; break;
	}
	QString script = QString("psi.modifyBaseStyle({fontFamily:'%1',fontSize:'%2pt',fontStyle:'%3',fontVariant:'%4',fontWeight:'%5'});")
					 .arg(f.family())
					 .arg(f.pointSize())
					 .arg(f.style()==QFont::StyleNormal?"normal":(f.style()==QFont::StyleItalic?"italic":"oblique"))
					 .arg(f.capitalization() == QFont::SmallCaps?"small-caps":"normal")
					 .arg(weight);
	webView->evaluateJS(script);
}

void ChatView::contextMenuEvent(QContextMenuEvent *e)
{
	QWebHitTestResult r = webView->page()->mainFrame()->hitTestContent(e->pos());
	if ( r.linkUrl().scheme() == "addnick" ) {
		showNM(r.linkUrl().path().mid(1));
		e->accept();
	}
}

void ChatView::importJSChatFunctions()
{
	QFile file(":/chatViewWebKit.js");
	if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
		qDebug() << "WARNING: HTMLChaTView::importJSChatFunction() - loading Qt Resource failed";
		//NOTE: I (senu) assume that reading from Qt Resource always succeed - there's no error handling
	}

	QString jsCode = file.readAll();
	webView->evaluateJS(jsCode);
}

bool ChatView::focusNextPrevChild(bool next)
{
	return QWidget::focusNextPrevChild(next);
}

/**
 * Copies any selected text to the clipboard
 * if autoCopy is enabled and ChatView is in read-only mode.
 */
void ChatView::autoCopy()
{
	if (PsiOptions::instance()->getOption("options.ui.automatically-copy-selected-text").toBool()) {
		webView->copySelected();
	}
}

void ChatView::appendText(QString text)
{
	webView->escapeString(text);
	qDebug()<<text;
	webView->evaluateJS(QString("psi.appendText(\"%1\")").arg(text));
}

bool ChatView::handleCopyEvent(QObject *object, QEvent *event, ChatEdit *chatEdit){
	if (object == chatEdit && event->type() == QEvent::KeyRelease) { //was KeyPress but no this type events here..
		QKeyEvent *e = (QKeyEvent *)event;
		if ((e->key() == Qt::Key_C && (e->modifiers() & Qt::ControlModifier)) ||
			(e->key() == Qt::Key_Insert && (e->modifiers() & Qt::ControlModifier)))
		{
			if (!chatEdit->textCursor().hasSelection() &&
				 !(webView->page()->selectedText().isEmpty()))
			{
				webView->copySelected();
				return true;
			}
		}
	}

	return false;
}

void ChatView::appendMucMessage(bool local, const QString &text, bool emote, const QString &nick, const QDateTime &time, const QString &nickcolor, bool alert)
{
	updateLastMsgTime(time);
	const QString timestr = formatTimeStamp(time);
	QString alerttagso, alerttagsc;
	QString textcolor = palette().color(QPalette::WindowText).name();
	if(alert) {
		textcolor = PsiOptions::instance()->getOption("options.ui.look.colors.messages.highlighting").value<QColor>().name();
		alerttagso = "<b>";
		alerttagsc = "</b>";
	}

	QString ewho = Qt::escape(nick);
	QString qewho = QUrl::toPercentEncoding(nick);
	if(emote) {
		appendText(QString("<font color=\"%1\">").arg(nickcolor) + QString("[%1]").arg(timestr) + "<a href=\"addnick://psi/"+qewho+"\" style=\"color: "+nickcolor+"; text-decoration: none; \"> *" + ewho + " </a>" + alerttagso + text + alerttagsc + "</font>");
	}
	else {
		if(PsiOptions::instance()->getOption("options.ui.chat.use-chat-says-style").toBool()) {
			appendText(QString("<font color=\"%1\">").arg(nickcolor) + QString("[%1] ").arg(timestr) + "<a href=\"addnick://psi/"+qewho+"\" style=\"color: "+nickcolor+"; text-decoration: none; \">"+tr("%1 says:").arg(ewho)+"</a></font><br>" + QString("<font color=\"%1\">").arg(textcolor) + alerttagso + text + alerttagsc + "</font>");
		}
		else {
			appendText(QString("<font color=\"%1\">").arg(nickcolor) + QString("[%1] &lt;").arg(timestr) + "<a href=\"addnick://psi/"+qewho+"\" style=\"color: "+nickcolor+"; text-decoration: none; \">"+ewho+"</a>&gt;</font> " + QString("<font color=\"%1\">").arg(textcolor) + alerttagso + text + alerttagsc +"</font>");
		}
	}
	if (local) {
		webView->evaluateJS("psi.scrollToBottom(250)");
	}
}

void ChatView::appendMessage(const QString &id, bool local, const QString &text, bool emote, const QString &nick, const QDateTime &time, const QString &color, bool remember,const QString &subject)
{
	//TODO receipts
	updateLastMsgTime(time);
	QString timestr = formatTimeStamp(time);
	if (emote) {
		appendText(QString("<span style=\"color: %1\">").arg(color) + QString("[%1]").arg(timestr) + QString(" *%1 ").arg(nick) + subject + text + "</span>");
	} else {
		if (PsiOptions::instance()->getOption("options.ui.chat.use-chat-says-style").toBool()) {
			appendText(QString("<p style=\"color: %1\">").arg(color) + QString("[%1] ").arg(timestr) + tr("%1 says:").arg(nick) + "</p>"+ (subject==""?"":QString("%1<br>").arg(subject)) + text);
		}
		else {
			appendText(QString("<span style=\"color: %1\">").arg(color) + QString("[%1] &lt;").arg(timestr) + nick + QString("&gt;</span> ") + (subject==""?"":QString("<br>%1<br>").arg(subject)) + text);
		}
	}
	if (local) {
		webView->evaluateJS("psi.scrollToBottom(250)");
	}
}

void ChatView::appendLastMsgTime(const QDate &date)
{
	QString color = PsiOptions::instance()->getOption("options.ui.look.colors.messages.informational").toString();
	appendText(QString("<font color=\"%1\">*** %2</font>").arg(color).arg(date.toString(Qt::ISODate)));
}

void ChatView::appendSysMessage(const QString &text, const QString &userText, const QDateTime &time)
{
	updateLastMsgTime(time);
	QString timestr = formatTimeStamp(time);
	QString ut;
	if (!userText.isEmpty()) {
		ut = TextUtil::plain2rich(Qt::escape(userText));
		ut = TextUtil::linkify(ut);
		if(PsiOptions::instance()->getOption("options.ui.emoticons.use-emoticons").toBool())
			ut = TextUtil::emoticonify(ut);
	}
	QString color = PsiOptions::instance()->getOption("options.ui.look.colors.messages.informational").toString();
	QString userTextColor = PsiOptions::instance()->getOption("options.ui.look.colors.messages.usertext").toString();
	appendText(QString("<font color=\"%1\">[%2] *** ").arg(color, timestr) + text +
						(userText.isEmpty()?"":":") + "</font>" +
						(userText.isEmpty()?"":QString(" <span style=\"color:%1;\">%2</span>").arg(userTextColor, ut)));
}

void ChatView::appendSubject(const QString &subject)
{
	appendText(QString("<b>") + tr("Subject:") + "</b> " + QString("%1").arg(Qt::escape(subject)));
}

void ChatView::appendMucSubject(const QString &sysMsg, const QString &subject, const QDateTime &time)
{
	updateLastMsgTime(time);
	QString timestr = formatTimeStamp(time);
	QString ut;
	if (!subject.isEmpty()) {
		ut = TextUtil::plain2rich(Qt::escape(subject));
		ut = TextUtil::linkify(ut);
		if(PsiOptions::instance()->getOption("options.ui.emoticons.use-emoticons").toBool())
			ut = TextUtil::emoticonify(ut);
	}
	QString color = PsiOptions::instance()->getOption("options.ui.look.colors.messages.informational").toString();
	QString userTextColor = PsiOptions::instance()->getOption("options.ui.look.colors.messages.usertext").toString();
	appendText(QString("<font color=\"%1\">[%2] *** ").arg(color, timestr) + sysMsg +
						(ut.isEmpty()?"":":<br>") + "</font>" +
						(ut.isEmpty()?"":QString(" <span style=\"color:%1;font-weight:bold\">%2</span>").arg(userTextColor, ut)));
}

void ChatView::startUrlsBlock()
{
	appendText(QString("<i>") + tr("-- Attached URL(s) --") + "</i>");
}

void ChatView::appendUrl(const QString &url, const QString &desc)
{
	appendText(QString("<b>") + tr("URL:") + "</b> " + QString("%1").arg(TextUtil::linkify(Qt::escape(url))));
	appendText(QString("<b>") + tr("Desc:") + "</b> " + QString("%1").arg(desc));
}

void ChatView::scrollUp()
{
	QWebFrame *f = webView->page()->mainFrame();
	f->setScrollBarValue(Qt::Vertical, f->scrollBarValue(Qt::Vertical) - 50);
}

void ChatView::scrollDown()
{
	QWebFrame *f = webView->page()->mainFrame();
	f->setScrollBarValue(Qt::Vertical, f->scrollBarValue(Qt::Vertical) + 50);
}

void ChatView::scrollToBottom()
{
	//done in js
}

void ChatView::clear()
{
	webView->evaluateJS("psi.clear();");
}

void ChatView::doTrackBar()
{
	webView->evaluateJS("psi.doTrackbar()");
}

bool ChatView::internalFind(QString str, bool startFromBeginning)
{
	bool found = webView->page()->findText(str, startFromBeginning ?
								  QWebPage::FindWrapsAroundDocument : (QWebPage::FindFlag)0);

	if (!found && !startFromBeginning) {
		return internalFind(str, true);
	}

	return found;
}

WebView * ChatView::textWidget()
{
	return webView;
}