#include "chatview_te.h"

#include "msgmle.h"
#include "psioptions.h"
#include "textutil.h"
#include "psirichtext.h"
#include "chatviewbase.h"
#include "iconset.h"

#include <QWidget>
#include <QTextOption>
#include <QScrollBar>
#include <QUrl>

//----------------------------------------------------------------------------
// ChatView
//----------------------------------------------------------------------------
ChatView::ChatView(QWidget *parent)
	: PsiTextView(parent)
	, oldTrackBarPosition(0)
{
	setWordWrapMode(QTextOption::WrapAtWordBoundaryOrAnywhere);

	setReadOnly(true);
	setUndoRedoEnabled(false);
	setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

#ifndef Q_WS_X11	// linux has this feature built-in
	connect(this, SIGNAL(selectionChanged()), SLOT(autoCopy()));
	connect(this, SIGNAL(cursorPositionChanged()), SLOT(autoCopy()));
#endif

	useMessageIcons_ = PsiOptions::instance()->getOption("options.ui.chat.use-message-icons").toBool();
	if (useMessageIcons_) {
		int logIconsSize = fontInfo().pixelSize()*0.93;
		if (PsiOptions::instance()->getOption("options.ui.chat.scaled-message-icons").toBool()) {
			logIconReceive = IconsetFactory::iconPixmap("psiplus/notification_chat_receive").scaledToHeight(logIconsSize, Qt::SmoothTransformation);
			logIconSend = IconsetFactory::iconPixmap("psiplus/notification_chat_send").scaledToHeight(logIconsSize, Qt::SmoothTransformation);
			logIconDelivered = IconsetFactory::iconPixmap("psiplus/notification_chat_delivery_ok").scaledToHeight(logIconsSize, Qt::SmoothTransformation);
			logIconTime = IconsetFactory::iconPixmap("psiplus/notification_chat_time").scaledToHeight(logIconsSize, Qt::SmoothTransformation);
			logIconInfo = IconsetFactory::iconPixmap("psiplus/notification_chat_info").scaledToHeight(logIconsSize, Qt::SmoothTransformation);
		} else {
			logIconReceive = IconsetFactory::iconPixmap("psiplus/notification_chat_receive");
			logIconSend = IconsetFactory::iconPixmap("psiplus/notification_chat_send");
			logIconDelivered = IconsetFactory::iconPixmap("psiplus/notification_chat_delivery_ok");
			logIconTime = IconsetFactory::iconPixmap("psiplus/notification_chat_time");
			logIconInfo = IconsetFactory::iconPixmap("psiplus/notification_chat_info");
		}
		addLogIconsResources();
	}

	const QString css = PsiOptions::instance()->getOption("options.ui.chat.css").toString();
	if (!css.isEmpty()) {
		setStyleSheet(css);
	}

}

ChatView::~ChatView()
{
}

QSize ChatView::sizeHint() const
{
	return minimumSizeHint();
}

void ChatView::clear()
{
	PsiTextView::clear();
	addLogIconsResources();
}

void ChatView::contextMenuEvent(QContextMenuEvent *e)
{
	const QUrl anc = QUrl::fromEncoded(anchorAt(e->pos()).toAscii());

	if ( anc.scheme() == "addnick" ) {
		showNM(anc.path().mid(1));
		e->accept();
	} else {
		PsiTextView::contextMenuEvent(e);
	}
}

void ChatView::addLogIconsResources()
{
	document()->addResource(QTextDocument::ImageResource, QUrl("log_icon_receive"), logIconReceive);
	document()->addResource(QTextDocument::ImageResource, QUrl("log_icon_send"), logIconSend);
	document()->addResource(QTextDocument::ImageResource, QUrl("log_icon_time"), logIconTime);
	document()->addResource(QTextDocument::ImageResource, QUrl("log_icon_info"), logIconInfo);
	document()->addResource(QTextDocument::ImageResource, QUrl("log_icon_delivered"), logIconDelivered);
}

void ChatView::markReceived(QString id)
{
	if (useMessageIcons_) {
		document()->addResource(QTextDocument::ImageResource, QUrl(QString("delivery") + id), logIconDelivered);
		setLineWrapColumnOrWidth(lineWrapColumnOrWidth());
	}
}

bool ChatView::focusNextPrevChild(bool next)
{
	return QWidget::focusNextPrevChild(next);
}

void ChatView::keyPressEvent(QKeyEvent *e)
{
/*	if(e->key() == Qt::Key_Escape)
		e->ignore();
#ifdef Q_WS_MAC
	else if(e->key() == Qt::Key_W && e->modifiers() & Qt::ControlModifier)
		e->ignore();
	else
#endif
	else if(e->key() == Qt::Key_Return && ((e->modifiers() & Qt::ControlModifier) || (e->modifiers() & Qt::AltModifier)) )
		e->ignore();
	else if(e->key() == Qt::Key_H && (e->modifiers() & Qt::ControlModifier))
		e->ignore();
	else if(e->key() == Qt::Key_I && (e->modifiers() & Qt::ControlModifier))
		e->ignore(); */
	/*else*/ if(e->key() == Qt::Key_M && (e->modifiers() & Qt::ControlModifier) && !isReadOnly()) // newline
		append("\n");
/*	else if(e->key() == Qt::Key_U && (e->modifiers() & Qt::ControlModifier) && !isReadOnly())
		setText(""); */
	else if ((e->key() == Qt::Key_Return || e->key() == Qt::Key_Enter) && ((e->modifiers() & Qt::ControlModifier) || (e->modifiers() & Qt::AltModifier))) {
		e->ignore();
	}
	else {
		PsiTextView::keyPressEvent(e);
	}
}

/**
 * Copies any selected text to the clipboard
 * if autoCopy is enabled and ChatView is in read-only mode.
 */
void ChatView::autoCopy()
{
	if (isReadOnly() && PsiOptions::instance()->getOption("options.ui.automatically-copy-selected-text").toBool()) {
		copy();
	}
}

/**
 * Handle KeyPress events that happen in ChatEdit widget. This is used
 * to 'fix' the copy shortcut.
 * \param object object that should receive the event
 * \param event received event
 * \param chatEdit pointer to the dialog's ChatEdit widget that receives user input
 */
bool ChatView::handleCopyEvent(QObject *object, QEvent *event, ChatEdit *chatEdit)
{
	if (object == chatEdit && event->type() == QEvent::KeyPress) {
		QKeyEvent *e = (QKeyEvent *)event;
		if ((e->key() == Qt::Key_C && (e->modifiers() & Qt::ControlModifier)) ||
			(e->key() == Qt::Key_Insert && (e->modifiers() & Qt::ControlModifier)))
		{
			if (!chatEdit->textCursor().hasSelection() &&
				 this->textCursor().hasSelection())
			{
				this->copy();
				return true;
			}
		}
	}

	return false;
}

void ChatView::appendText(const QString &text)
{
	bool doScrollToBottom = atBottom();

	// prevent scrolling back to selected text when
	// restoring selection
	int scrollbarValue = verticalScrollBar()->value();

	PsiTextView::appendText(text);

	if (doScrollToBottom)
		scrollToBottom();
	else
		verticalScrollBar()->setValue(scrollbarValue);
}

void ChatView::appendMucMessage(bool local, const QString &text, bool emote, const QString &nick, const QDateTime &time, const QString &nickcolor, bool alert)
{
	updateLastMsgTime(time);
	const QString timestr = formatTimeStamp(time);
	QString alerttagso, alerttagsc;
	QString textcolor = palette().active().text().name();
	if(alert) {
		textcolor = PsiOptions::instance()->getOption("options.ui.look.colors.messages.highlighting").value<QColor>().name();
		alerttagso = "<b>";
		alerttagsc = "</b>";
	}
	QString icon = useMessageIcons_?
				   (QString("<img src=\"%1\" />").arg(local?"log_icon_delivered":"log_icon_receive")):"";
	QString ewho = Qt::escape(nick);
	QString qewho = QUrl::toPercentEncoding(nick);
	if(emote) {
		appendText(icon + QString("<font color=\"%1\">").arg(nickcolor) + QString("[%1]").arg(timestr) + "<a href=\"addnick://psi/"+qewho+"\" style=\"color: "+nickcolor+"; text-decoration: none; \"> *"+ewho+" </a>" + alerttagso + text + alerttagsc + "</font>");
	}
	else {
		if(PsiOptions::instance()->getOption("options.ui.chat.use-chat-says-style").toBool()) {
			appendText(icon + QString("<font color=\"%1\">").arg(nickcolor) + QString("[%1] ").arg(timestr) + "<a href=\"addnick://psi/"+qewho+"\" style=\"color: "+nickcolor+"; text-decoration: none; \">"+tr("%1 says:").arg(ewho)+"</a></font><br>" + QString("<font color=\"%1\">").arg(textcolor) + alerttagso + text + alerttagsc + "</font>");
		}
		else {
			appendText(icon + QString("<font color=\"%1\">").arg(nickcolor) + QString("[%1] &lt;").arg(timestr) + "<a href=\"addnick://psi/"+qewho+"\" style=\"color: "+nickcolor+"; text-decoration: none; \">"+ewho+"</a>" + QString("&gt;</font> ") + QString("<font color=\"%1\">").arg(textcolor) + alerttagso + text + alerttagsc +"</font>");
		}
	}
}

void ChatView::appendMessage(const QString &id, bool local, const QString &text, bool emote, const QString &nick, const QDateTime &time, const QString &color, bool remember,const QString &subject)
{
	if (remember) {
		document()->addResource(QTextDocument::ImageResource, QUrl(QString("delivery") + id), logIconSend);
	}
	QString icon = useMessageIcons_?
				   (QString("<img src=\"%1\" />").arg(local?(remember?QString("delivery") + id:"log_icon_send"):"log_icon_receive")):"";
	updateLastMsgTime(time);
	QString timestr = formatTimeStamp(time);

	if (emote) {
		appendText(icon + QString("<span style=\"color: %1\">").arg(color) + QString("[%1]").arg(timestr) + QString(" *%1 ").arg(nick) + subject + text + "</span>");
	} else {
		if (PsiOptions::instance()->getOption("options.ui.chat.use-chat-says-style").toBool()) {
			appendText(icon + QString("<span style=\"color: %1\">").arg(color) + QString("[%1] ").arg(timestr) + tr("%1 says:").arg(nick) + "</span>" + (subject==""?"":QString("<br>%1").arg(subject)) + "<br>" +text);
		}
		else {
			appendText(icon + QString("<span style=\"color: %1\">").arg(color) + QString("[%1] &lt;").arg(timestr) + nick + QString("&gt;</span> ") + (subject==""?"":QString("<br>%1<br>").arg(subject))+ text);
		}
	}
}

void ChatView::appendLastMsgTime(const QDate &date)
{
	QString color = PsiOptions::instance()->getOption("options.ui.look.colors.messages.informational").toString();
	appendText(QString(useMessageIcons_?"<img src=\"log_icon_time\" />":"") +
			   QString("<font color=\"%1\">*** %2</font>").arg(color).arg(date.toString(Qt::ISODate)));
}

void ChatView::appendSysMessage(const QString &text, const QString &userText, const QDateTime &time)
{
	updateLastMsgTime(time);
	QString timestr = formatTimeStamp(time);
	QString ut;
	if (!userText.isEmpty()) {
		ut = TextUtil::plain2rich(Qt::escape(userText));
		ut = TextUtil::linkify(ut);
		if(PsiOptions::instance()->getOption("options.ui.emoticons.use-emoticons").toBool())
			ut = TextUtil::emoticonify(ut);
	}
	QString color = PsiOptions::instance()->getOption("options.ui.look.colors.messages.informational").toString();
	QString userTextColor = PsiOptions::instance()->getOption("options.ui.look.colors.messages.usertext").toString();
	appendText(QString(useMessageIcons_?"<img src=\"log_icon_info\" />":"") +
			   QString("<font color=\"%1\">[%2] *** ").arg(color, timestr) + text +
						(userText.isEmpty()?"":":") + "</font>" +
						(userText.isEmpty()?"":QString(" <span style=\"color:%1;\">%2</span>").arg(userTextColor, ut)));
}

void ChatView::appendSubject(const QString &subject)
{
	appendText(QString(useMessageIcons_?"<img src=\"log_icon_info\" />":"") + "<b>" + tr("Subject:") + "</b> " + QString("%1").arg(Qt::escape(subject)));
}

void ChatView::appendMucSubject(const QString &sysMsg, const QString &subject, const QDateTime &time)
{
	updateLastMsgTime(time);
	QString timestr = formatTimeStamp(time);
	QString ut;
	if (!subject.isEmpty()) {
		ut = TextUtil::plain2rich(Qt::escape(subject));
		ut = TextUtil::linkify(ut);
		if(PsiOptions::instance()->getOption("options.ui.emoticons.use-emoticons").toBool())
			ut = TextUtil::emoticonify(ut);
	}
	QString color = PsiOptions::instance()->getOption("options.ui.look.colors.messages.informational").toString();
	QString userTextColor = PsiOptions::instance()->getOption("options.ui.look.colors.messages.usertext").toString();
	appendText(QString(useMessageIcons_?"<img src=\"log_icon_info\" />":"") +
			   QString("<font color=\"%1\">[%2] *** ").arg(color, timestr) + sysMsg +
			   (ut.isEmpty()?"":":<br>") + "</font>" +
			   (ut.isEmpty()?"":QString(" <span style=\"color:%1;font-weight:bold\">%2</span>").arg(userTextColor, ut)));
}

void ChatView::startUrlsBlock()
{
	appendText(QString("<i>") + tr("-- Attached URL(s) --") + "</i>");
}

void ChatView::appendUrl(const QString &url, const QString &desc)
{
	appendText(QString("<b>") + tr("URL:") + "</b> " + QString("%1").arg(TextUtil::linkify(Qt::escape(url))));
	appendText(QString("<b>") + tr("Desc:") + "</b> " + QString("%1").arg(desc));
}

void ChatView::scrollUp()
{
	verticalScrollBar()->setValue(verticalScrollBar()->value() - verticalScrollBar()->pageStep() / 2);
}

void ChatView::scrollDown()
{
	verticalScrollBar()->setValue(verticalScrollBar()->value() + verticalScrollBar()->pageStep() / 2);
}

void ChatView::doTrackBar()
{
	// save position, because our manipulations could change it
	int scrollbarValue = verticalScrollBar()->value();

	QTextCursor cursor = textCursor();
	cursor.beginEditBlock();
	PsiRichText::Selection selection = PsiRichText::saveSelection(this, cursor);

	//removeTrackBar(cursor);
	if (oldTrackBarPosition) {
		cursor.setPosition(oldTrackBarPosition, QTextCursor::KeepAnchor);
		QTextBlockFormat blockFormat = cursor.blockFormat();
		blockFormat.clearProperty(QTextFormat::BlockTrailingHorizontalRulerWidth);
		cursor.clearSelection();
		cursor.setBlockFormat(blockFormat);
	}

	//addTrackBar(cursor);
	cursor.movePosition(QTextCursor::End, QTextCursor::KeepAnchor);
	oldTrackBarPosition = cursor.position();
	QTextBlockFormat blockFormat = cursor.blockFormat();
	blockFormat.setProperty(QTextFormat::BlockTrailingHorizontalRulerWidth, QVariant(true));
	cursor.clearSelection();
	cursor.setBlockFormat(blockFormat);

	PsiRichText::restoreSelection(this, cursor, selection);
	cursor.endEditBlock();
	setTextCursor(cursor);

	verticalScrollBar()->setValue(scrollbarValue);
}

bool ChatView::internalFind(QString str, bool startFromBeginning)
{
	if (startFromBeginning) {
		QTextCursor cursor = textCursor();
		cursor.movePosition(QTextCursor::Start, QTextCursor::KeepAnchor);
		cursor.clearSelection();
		setTextCursor(cursor);
	}

	bool found = find(str);
	if(!found) {
		if (!startFromBeginning)
			return internalFind(str, true);

		return false;
	}

	return true;
}

ChatView * ChatView::textWidget()
{
	return this;
}