#!/bin/sh
#######################################################################
#                                                                     #
#       Universal build script of Psi+ under Linux                    #
#       Универсальный скрипт сборки Psi+ под Linux                    #
#                                                                     #
#######################################################################

# REQUIREMENTS / ТРЕБОВАНИЯ

# In order to build Psi+ you must have next packages in your system
# Для сборки Psi+ вам понадобятся следующие пакеты
# git - vcs system / система контроля версий
# gcc - compiler / компилятор
# qt4 tools libraries and headers (most probably "dev" packages)
#     qt4 тулзы либы и хидеры (наверное "dev" пакеты)
# qca/QtCrypto - encryption libs / криптовальные либы

# OPTIONS / НАСТРОЙКИ

# build and store directory / каталог для сорсов и сборки
DEBS_DIR="${HOME}/workspace/deb"
PSI_DIR="${HOME}/workspace/build/psi"

# result package name
PKG_NAME="psi-plus"
# You  full  name,  eg  "John  Doe"  which  will  appear in the control and changelog entries.
#DEBFULLNAME="Stefan Zerkalica"
DEBFULLNAME="Ivan Borzenkov"
# Email address to use in control and changelog entries
DEBEMAIL="zerkalica@gmail.com"

INSTALL_PREFIX="/usr"

PSIPLUS_SVN="http://psi-dev.googlecode.com/svn/trunk"
# official repository / репозиторий официальной Psi
GIT_REPO_PSI="git://git.psi-im.org/psi.git"
#
INJECT_FROM_WEB="${PSIPLUS_SVN} certs/rootcert.xml
http://psi-ru.googlecode.com/svn/branches/psi-plus psi_ru.qm
${PSIPLUS_SVN} iconsets/clients/fingerprint.jisp
${PSIPLUS_SVN} iconsets/moods/silk.jisp
${PSIPLUS_SVN} iconsets/psiplus/psiplus.jisp
${PSIPLUS_SVN} iconsets/emoticons/tasha_18x18x93.jisp"
#${PSIPLUS_SVN} iconsets/emoticons/bombus2psi.jisp

PLUGINS_LIST="${PSIPLUS_SVN}/plugins generic psiplusplugins
https://delta.affinix.com/svn/trunk psimedia
* antievil"

QCONF_TRUNK="https://delta.affinix.com/svn/trunk/qconf"

# do git pull on psi git working copy on start / обновляться с репозитория перед сборкой
FORCE_REPO_UPDATE=1

# configure options / опции скрипта configure
CONF_OPTS="--prefix=${INSTALL_PREFIX}"

GIT_DIR="${PSI_DIR}/git_sources"
BUILD_ROOT="${PSI_DIR}/build"
PATCHES_DIR="${PSI_DIR}/patches"
RESOURCES_DIR="${PSI_DIR}/resources"

GIT_ADDONS="iris
src/libpsi
src/libpsi/tools
src/libpsi/tools/zip
src/libpsi/tools/idle
src/libpsi/tools/atomicxmlfile
"
GIT_PSI_DIR="${GIT_DIR}/psi"


LOG_FILE="${HOME}/workspace/psibuild.log"

QCONF=qconf
QMAKE=qmake-qt4
MAKE=make
MAKEOPT="-j4"

#SKIP_INVALID_PATCH=1
# FUNCTIONS /ФУНКЦИИ

putlog() {
	MSG="${1}"
	LEVEL="${2}"
	[ "${LEVEL}" != "1" ] && echo "${MSG}"
	[ "${LEVEL}" != "2" ] && echo "${MSG}" >> ${LOG_FILE}
}

# Exit with error message
die() {
	putlog " !!!ERROR: ${1}"
	exit 1
}

_mkdir() {
	[ -d "${1}" ] || mkdir -p "${1}" || die "can't create directory ${1}"
}

check()	{
	v=$(${1} --version 2>/dev/null)
	result=0
	[ ! "${v}" ] && result=1
	putlog "check ${1}: ${result}" 1
	return ${result}
}

check_die()	{
	check "${1}" || die "You should install ${1} first. / Сначала установите ${1}"
}

remake_dir()	{
	if [ -d "${1}" ] ; then
		rm -rf "${1}" || die "can't delete directory ${1}"
	fi
	_mkdir "${1}"
}

copy_stuff() {
	INSTALL_TO="$1"
	_mkdir "${INSTALL_TO}"
	echo "${INJECT_FROM_WEB}" | \
	while read url_prefix path_name; do
		out_file="${INSTALL_TO}/${path_name}"
		out_dir=$(dirname "${out_file}")
		if [ ! -d "${out_dir}" ] ; then
			mkdir -p "${out_dir}" || continue
		fi
		cd $(dirname "${out_file}") && wget -N "${url_prefix}/${path_name}"
		if [ "$?" != "0" ] ; then
			putlog "can't fetching ${url_prefix}/${path_name}"
		fi
	done
}

spatch() {
	PATCH_TARGET="${1}"
	for count in $(seq 1 5) ; do
		OUTPUT=$(patch -p${count} -f --no-backup-if-mismatch < ${PATCH_TARGET} 2>/dev/null)
		RETVAL="$?"
		[ "${RETVAL}" = "0" ] && break
	done
	putlog $(pwd) 1
	putlog "patch -p1 -f --no-backup-if-mismatch < ${PATCH_TARGET}" 1
	putlog "${OUTPUT}" 1
	HUNK=$(echo "${OUTPUT}" | grep "^Hunk #")
	[ "${HUNK}" ] && return 2
	[ "${RETVAL}" != "0" ] && return 1
	return 0
}

patch_all()	{
	putlog "patching"
	cd "${BUILD_DIR}"
	for p in ${PATCHES_DIR}/*.diff ; do
		spatch "${p}"
		RETVAL="$?"
		pn=$(basename ${p})
		if [ "${RETVAL}" = "1" ] ; then
			putlog "${pn} ... failed"
			[ ! "${SKIP_INVALID_PATCH}" ] && die "can't continue due to patch failed"
		elif [ "${RETVAL}" = "2" ] ; then
			putlog "${pn} ... hunks"
		fi
	done
	sed "s/\(.*define PROG_VERSION .*\)xxx/\1${REVISION}/" -i src/applicationinfo.cpp
	sed 's/<!--\(.*plugins.*\)-->/\1/' -i psi.qc
	sed '/.*debug.*/,/<\/dep>/d;' -i psi.qc
}



check_env() {
	putlog "check install enviroment"
	check "gmake" && MAKE="gmake"
	check_die "git"
	check_die "$MAKE"
	check $QCONF || QCONF=
	_mkdir "${PSI_DIR}"
	_mkdir "${GIT_DIR}"
	QC_OUT_DIR="${PSI_DIR}/qconf_bin"
	QCONF=${QC_OUT_DIR}/bin/qconf
	check $QCONF || QCONF=
}

check_env_deb() {
	if check "apt-get" ; then
		putlog "apt-get found, resolve depends"
		sudo apt-get install \
			git-core subversion \
			build-essential devscripts dpkg-dev debhelper cdbs fakeroot make automake autotools-dev qt4-qmake \
			libqt4-dev libqca2-dev libxss-dev libaspell-dev zlib1g-dev libsm-dev \
			libgstreamer-plugins-base0.10-dev liboil0.3-dev libspeexdsp-dev
	fi
}

download_qconf() {
	cd "${PSI_DIR}" && svn co ${QCONF_TRUNK} qconf
}

download_psi() {
	cd "${GIT_DIR}" || die "can't cd to ${GIT_DIR}"
	if [ -d "${GIT_PSI_DIR}/.git" ] ; then
		putlog "updating from git"
		cd "${GIT_PSI_DIR}"
		if [ "${FORCE_REPO_UPDATE}" != "0" ]; then
			git pull || die "git update failed"
			git submodule update || die "git submodule update failed"
		else
			putlog "Update disabled in options"
		fi
	else
		putlog "creating new fresh repo from git.."
		git clone "${GIT_REPO_PSI}" || die "git clone failed"
		cd "${GIT_PSI_DIR}"
		git submodule init || die "git submodule init failed"
		git submodule update || die "git submodule update failed"
	fi
}

download_patches() {
	putlog "downloading patches into ${PATCHES_DIR}"
	cd "${PSI_DIR}"
	if [ -d "${PATCHES_DIR}" ] ; then
	  [ "${FORCE_PATCH_UPDATE}" != "0" ] && ( svn up patches || die "patches update failed" )
	else
	  svn co ${PSIPLUS_SVN}/patches/ patches || die "patches checkout failed"
	#	rm ${PATCHES_DIR}/*win32* 2>/dev/null
	fi
}

download_resources() {
	cd "${PSI_DIR}"
	copy_stuff "${PSI_DIR}/resources"
}

download_plugins() {
	_mkdir "${PSI_DIR}/plugins"
	cd "${PSI_DIR}/plugins"
	echo "${PLUGINS_LIST}" | \
	while read TRUNK PLUG_NAME PLUG_NAME_OUT; do
		[ "${PLUG_NAME_OUT}" ] || PLUG_NAME_OUT="${PLUG_NAME}"
		putlog "Downloading plugin: svn co ${TRUNK}/${PLUG_NAME} ${PLUG_NAME_OUT}"
		[ "${TRUNK}" = "*" ] || svn co ${TRUNK}/${PLUG_NAME} ${PLUG_NAME_OUT}
	done
}

export_psi_sources() {
	DST="${1}"
	putlog "exporting sources to ${DST}"
	cd "${GIT_PSI_DIR}"
	git archive --format=tar master | ( cd "${DST}" ; tar xf - )
	for i in ${GIT_ADDONS} ; do
		cd "${GIT_PSI_DIR}/${i}"
		git archive --format=tar master | ( cd "${DST}/${i}" ; tar xf - )
	done
}

export_resources() {
	DST="${1}"
	_mkdir "${DST}"
	putlog "exporting resources to ${DST}"
	cp -ra ${PSI_DIR}/resources/* ${DST}/
	mkdir -p "${DST}/../lang/ru"
	cp "${PSI_DIR}/resources/psi_ru.qm" "${DST}/../lang/ru/"
}

export_plugins() {
	PLUG_OUT="${1}"
	_mkdir "${PLUG_OUT}" && cd "${PLUG_OUT}"
	echo "${PLUGINS_LIST}" | \
	while read TRUNK PLUG_NAME PLUG_NAME_OUT; do
		[ "${TRUNK}" = "*" ] && continue
		putlog "Export plugin: ${PLUG_NAME}"
		PLUG_NAME_OUT_TMP="${PLUG_NAME_OUT}"
		[ "${PLUG_NAME_OUT}" ] || PLUG_NAME_OUT="${PLUG_NAME}"
		svn export "${PSI_DIR}/plugins/${PLUG_NAME_OUT}" "${PLUG_NAME_OUT}"
		if [ "${PLUG_NAME_OUT_TMP}" ] ; then
			mv ${PLUG_NAME_OUT}/* ./
			rmdir "${PLUG_NAME_OUT}"
		fi
	done
}

delete_plugins() {
	PLUG_OUT="${1}"
	for i in $PLUGINS_DIR/* ; do
		bn=`echo ${PLUGINS_LIST} | grep $(basename "${i}")`
		if [ ! "${bn}" ] ; then
			rm -rf "${i}"
		fi
	done
}

export_qconf() {
	rm -rf "${PSI_DIR}/qconf_tmp" && svn export ${PSI_DIR}/qconf "${PSI_DIR}/qconf_tmp"
}

build_qconf() {
	cd "${PSI_DIR}/qconf_tmp"
	if [ ! -e "$QCONF" ] ; then
		./configure --prefix=${QC_OUT_DIR} --qtdir=/usr/share/qt4/ && $MAKE $MAKEOPT && strip qconf && $MAKE install
	fi
}

build_plugins() {
	PLUG_OUT="${1}"
	rm -rf "${PLUG_OUT}/*"
	echo "${PLUGINS_LIST}" | \
	while read TRUNK PLUG_NAME; do
		putlog "Building plugin: ${PSI_DIR}/plugins/${PLUG_NAME}"

		cd ${PLUG_OUT}/${PLUG_NAME}

		case "${PLUG_NAME}" in
		psimedia)
			$QCONF && ./configure && $MAKE $MAKEOPT && cp ./gstprovider/*.so ./
		;;

		screenshot)
			$QMAKE screenshotplugin.pro && $MAKE $MAKEOPT
		;;

		*)
			$QMAKE && $MAKE $MAKEOPT
		;;
		esac
		strip ./*.so
	done
}

build_psi() {
	putlog "using raw make install method"
	./configure "${CONF_OPTS}" || die "configure failed"
	cd "${BUILD_DIR}"
	$MAKE $MAKEOPT || die "make failed"
	strip "${BUILD_DIR}/src/psi"
	$MAKE install || die "make install failed"
	#cp -r ${BUILD_DIR}/share/* ${INSTALL_PREFIX}/share/psi/
	_mkdir "${INSTALL_PREFIX}/lib/psi"
	cp ${BUILD_DIR}/src/plugins/generic/*/*.so ${INSTALL_PREFIX}/lib/psi/
}

build_psi_deb() {
	putlog "using deb-based system"

	cd ${BUILD_DIR}/.. && tar czf "${TAR_NAME}.orig.tar.gz" "${BUILD_NAME}"

	cd "${BUILD_DIR}"
	spatch "${PSI_DIR}/600-psi-debianize.diff"

	BUILDDEB_DIR="${BUILD_DIR}/debian"

	_QCONF=$(echo ${QCONF} | sed 's/\//\\\//g')
	sed "s/QCONF\=.*/QCONF\=${_QCONF}/" -i "${BUILDDEB_DIR}/rules"
	sed "s/Ivan Borzenkov <ivan1986\@list\.ru>/${DEBFULLNAME} <${DEBEMAIL}>/g" -i "${BUILDDEB_DIR}/control"
	DATE=$(date -R)
	mv "${BUILDDEB_DIR}/changelog" "${BUILDDEB_DIR}/changelog.old"
	cat > "${BUILDDEB_DIR}/changelog" << EOF
${PKG_NAME} (${VERSION}.${REVISION}.${DATE_SHORT}-1) unstable; urgency=low

  * New upstream release

 -- ${DEBFULLNAME} <${DEBEMAIL}>  ${DATE}

EOF
	cat "${BUILDDEB_DIR}/changelog.old" >> "${BUILDDEB_DIR}/changelog"
	rm "${BUILDDEB_DIR}/changelog.old"

	chmod +x "${BUILDDEB_DIR}/rules"
	putlog "yeeah, building"
	cd "${BUILD_DIR}"
	debuild
}

#main()
PWD_DIR=$(pwd)
date -R > "${LOG_FILE}"

putlog "part 1. check"
check_env
check_env_deb

putlog "part 2. download"
[ -z "${QCONF}" ] && download_qconf
download_psi
download_patches
download_resources
download_plugins

VERSION=$(cat "${GIT_PSI_DIR}/src/applicationinfo.cpp" | sed -n 's/^\#define PROG_VERSION\s*\"\([[:digit:].]*\).*/\1/p')
REVISION=$(svnversion "${PATCHES_DIR}" | sed s/M//)
DATE_SHORT=$(date +"%Y%m%d")
TAR_NAME="${PKG_NAME}_${VERSION}.${REVISION}.${DATE_SHORT}"
BUILD_NAME="${PKG_NAME}-${VERSION}.${REVISION}.${DATE_SHORT}"
BUILD_DIR="${BUILD_ROOT}/${BUILD_NAME}"
PLUGINS_DIR="${BUILD_DIR}/src/plugins/generic"

remake_dir "${BUILD_ROOT}"
_mkdir "${BUILD_DIR}"

putlog "part 3. export"
[ -z "${QCONF}" ] && export_qconf
export_psi_sources "${BUILD_DIR}"
export_resources "${BUILD_DIR}/share"

mv ${PATCHES_DIR}/600-psi-debianize.diff ${PSI_DIR}/
cd "${BUILD_DIR}"
patch_all
delete_plugins "${PLUGINS_DIR}"
export_plugins "${PLUGINS_DIR}"

putlog "part 4. building"
[ -z "${QCONF}" ] && build_qconf
check_die "$QCONF"
export QCONF

cd "${BUILD_DIR}" && $QCONF

if check "dpkg" ; then
	build_psi_deb
else
	build_plugins "${PLUGINS_DIR}"
	build_psi
fi
