/* $Cambridge: hermes/src/prayer/lib/os_bsd.c,v 1.3 2008/09/16 09:59:57 dpc22 Exp $ */
/************************************************
 *    Prayer - a Webmail Interface              *
 ************************************************/

/* Copyright (c) University of Cambridge 2000 - 2008 */
/* See the file NOTICE for conditions of use and distribution. */

#include "lib.h"

#include <sys/resource.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <netdb.h>

#include <libutil.h>

/* ====================================================================== */

BOOL os_socketpair(int *sockfd)
{
    int rc;

    do {
        rc = socketpair(AF_UNIX, SOCK_STREAM, 0, sockfd);
    }
    while ((rc < 0) && (errno == EINTR));

    return ((rc == 0) ? T : NIL);
}

/* ====================================================================== */

int os_connect_unix_socket(char *name)
{
    struct sockaddr_un serv_addr;
    int sockfd, servlen;

    /* Open the socket */

    if ((sockfd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
        return (-1);

    bzero((char *) &serv_addr, sizeof(serv_addr));
    serv_addr.sun_family = AF_UNIX;
    strncpy(serv_addr.sun_path, name, sizeof(serv_addr.sun_path)-1);
    servlen = sizeof(serv_addr);

    if (connect(sockfd, (struct sockaddr *) &serv_addr, servlen) < 0) {
        close(sockfd);
        return (-1);
    }

    return (sockfd);
}

int os_connect_inet_socket(char *host, unsigned long port)
{
    struct addrinfo *first_ai, *ai;
    char port_str[NI_MAXSERV];
    int sockfd;
    
    snprintf(port_str, sizeof(port_str), "%lu", port);
    if (getaddrinfo(host, port_str, NULL, &first_ai)) {
        return (-1);
    }
    for (ai = first_ai; ai->ai_next; ai = ai->ai_next) {
        /* Open the socket */
        if ((sockfd = socket(ai->ai_family, SOCK_STREAM, 0)) < 0) {
            break;
        }
        if (connect(sockfd, (struct sockaddr *) ai->ai_addr, 
                    ai->ai_addrlen) < 0) {
            close(sockfd);
            break;
        }
        freeaddrinfo(first_ai);
        return (sockfd);
    }
    freeaddrinfo(first_ai);
    return (-1);
}

/* ====================================================================== */

int os_bind_unix_socket(char *name)
{
    struct sockaddr_un serv_addr;
    int sockfd, servlen;
    int i;

    /* Generate well known connect address for frontend servers */
    bzero((char *) &serv_addr, sizeof(serv_addr));
    serv_addr.sun_family = AF_UNIX;

    strncpy(serv_addr.sun_path, name, sizeof(serv_addr.sun_path)-1);
    servlen = strlen(serv_addr.sun_path) + sizeof(serv_addr.sun_family);
    unlink(serv_addr.sun_path);

    /* Open the socket */
    if ((sockfd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
        log_panic("[os_bind_unix_socket()] socket() failed, %s",
                  strerror(errno));
        return (-1);
    }

    /* Set socket reuseaddr, otherwise bind will fail after fast stop/start */
    i = 1;
    if (setsockopt
        (sockfd, SOL_SOCKET, SO_REUSEADDR, (void *) &i, sizeof(int))) {
        log_panic(("[os_bind_unix_socket()] setsockopt() failed: "
                   "couldn't set SO_REUSEADDR on init port, %s"),
                  strerror(errno));
        close(sockfd);
        return (-1);
    }

    /* bind() as UNIX domain socket */
    if (bind(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) <
        0) {
        log_panic("[os_bind_unix_socket()] bind() failed: %s",
                  strerror(errno));
        close(sockfd);
        return (-1);
    }

    /* Requests should queue on sockfd until we are ready to serve them */
    if (listen(sockfd, 10) < 0) {
        log_panic("[os_bind_unix_socket()] listen() failed, %s",
                  strerror(errno));
        close(sockfd);
        return (-1);
    }
    return (sockfd);
}

/* ====================================================================== */

int os_bind_inet_socket(unsigned long port, char *interface)
{
    int i, sockfd;
    struct sockaddr_storage serv_addr;
    struct addrinfo *ai, ai_hints;
    char port_str[NI_MAXSERV];
    socklen_t serv_addrlen;

    snprintf(port_str, sizeof(port_str), "%lu", port);
    bzero((void *) &ai_hints, sizeof(ai_hints));
    ai_hints.ai_flags = AI_PASSIVE;
    ai_hints.ai_family = AF_UNSPEC;
    ai_hints.ai_socktype = SOCK_STREAM;

    if (getaddrinfo(interface, port_str, &ai_hints, &ai)) {
        log_panic
            ("[os_bind_inet_socket()] Failed to lookup hostname: %s",
             interface);
        return -1;
    }
    if ((sockfd = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol)) 
        < 0) {
        log_panic("[os_bind_inet_socket()] socket() failed, %s",
                  strerror(errno));
        return (-1);
    }

    /* Set socket reuseaddr, otherwise bind will fail after fast stop/start */
    i = 1;
    if (setsockopt
        (sockfd, SOL_SOCKET, SO_REUSEADDR, (void *) &i, sizeof(int))) {
        log_panic(("[os_bind_inet_socket()] setsockopt() failed: "
                   "couldn't set SO_REUSEADDR: %s"), strerror(errno));
        close(sockfd);
        return (-1);
    }
    memcpy(&serv_addr, ai->ai_addr, ai->ai_addrlen);
    serv_addrlen = ai->ai_addrlen;
    freeaddrinfo(ai);

    /* bind() as Internet domain socket */
    if (bind(sockfd, (struct sockaddr *) &serv_addr, serv_addrlen) < 0) {
        log_panic("[os_bind_inet_socket()] bind() failed: %s",
                  strerror(errno));
        close(sockfd);
        return (-1);
    }

    if (listen(sockfd, 10) < 0) {
        log_panic("[os_bind_inet_socket()] listen() failed, %s",
                  strerror(errno));
        close(sockfd);
        return (-1);
    }

    return (sockfd);
}

/* ====================================================================== */

int os_accept_unix(int sockfd)
{
    struct sockaddr_un addr;
    socklen_t len = (socklen_t) sizeof(struct sockaddr_un);
    int newsockfd;

    do {
        newsockfd = accept(sockfd, (struct sockaddr *) &addr, &len);
    }
    while ((newsockfd < 0) && (errno == EINTR));

    if (newsockfd < 0) {
        log_panic("[os_accept_unix()] accept() failed: %s",
                  strerror(errno));
        close(newsockfd);
        return (-1);
    }

    /* Set close on exec so subprocesses can't interfere */
    if (fcntl(newsockfd, F_SETFD, FD_CLOEXEC) < 0) {
        log_panic("[os_accept_unix()] fcntl() (close-on-exec) failed:  %s",
                  strerror(errno));
        close(newsockfd);
        return (-1);
    }

    return (newsockfd);
}

int os_accept_inet(int sockfd, struct ipaddr *ipaddr)
{
    struct sockaddr_storage addr;
    socklen_t len = (socklen_t) sizeof(addr);
    int newsockfd;

    do {
        newsockfd = accept(sockfd, (struct sockaddr *) &addr, &len);
    }
    while ((newsockfd < 0) && (errno == EINTR));

    if (newsockfd < 0) {
        log_panic("[os_accept_inet()] accept() failed: %s",
                  strerror(errno));
        close(newsockfd);
        return (-1);
    }

    if (ipaddr) {
        if (addr.ss_family == AF_INET6)
            ipaddr_set(ipaddr, 6, (unsigned char *)
                                  &((struct sockaddr_in6 *) &addr)->sin6_addr);
        else
            ipaddr_set(ipaddr, 4, (unsigned char *)
                                  &((struct sockaddr_in *) &addr)->sin_addr);
    }

    /* Set close on exec so subprocesses can't interfere */
    if (fcntl(newsockfd, F_SETFD, FD_CLOEXEC) < 0) {
        log_panic("[os_accept_inet()] fcntl() (close-on-exec) failed:  %s",
                  strerror(errno));
        close(newsockfd);
        return (-1);
    }

    return (newsockfd);
}

/* ====================================================================== */

int os_socket_blocking(int sockfd)
{
    int mode;

    mode = fcntl(sockfd, F_GETFL, 0);
    mode &= ~O_NDELAY;

    if (fcntl(sockfd, F_SETFL, mode) != 0) {
        log_panic("[os_socket_nonblocking()] fcntl() failed: %s",
                  strerror(errno));
        return (NIL);
    }
    return (T);
}


int os_socket_nonblocking(int sockfd)
{
    int mode;

    mode = fcntl(sockfd, F_GETFL, 0);
    mode |= O_NDELAY;

    if (fcntl(sockfd, F_SETFL, mode) != 0) {
        log_panic("[os_socket_nonblocking()] fcntl() failed: %s",
                  strerror(errno));
        return (NIL);
    }
    return (T);
}

/* ====================================================================== */

/* Convert address into text form */

char *os_gethostbyaddr(void *opaque, unsigned int version)
{
    struct hostent *hostent;

    if (version == 6)
        hostent = gethostbyaddr(opaque, 16, AF_INET6);
    else
        hostent = gethostbyaddr(opaque, 4, AF_INET);

    if (hostent && hostent->h_name && hostent->h_name[0])
        return (hostent->h_name);

    return (NIL);
}

int os_inet_ntop(void *addr, unsigned long version, char *buf, 
                 unsigned long buflen) {
    if (version == 6) {
        if (inet_ntop(AF_INET6, addr, buf, buflen))
            return (T);
    }
    else {
        if (inet_ntop(AF_INET, addr, buf, buflen))
            return (T);
    }
    return (NIL);
}

int os_inet_pton(char *str, struct ipaddr *addr) {
    unsigned char buf[16];

    if (inet_pton(AF_INET6, str, buf)) {
        ipaddr_set(addr, 6, buf);
        return (T);
    }
    else if (inet_pton(AF_INET, str, buf)) {
        ipaddr_set(addr, 4, buf);
        return (T);
    }
    return (NIL);
}

/* ====================================================================== */

/* Trivial SIG_CLD handler to prevent zombies from hanging around */

void os_child_reaper()
{
    int status;
    pid_t child;

    do {
        child = waitpid(0, &status, WNOHANG);
    }
    while (child > 0);
}

pid_t os_waitpid_nohang()
{
    int status;

    return (waitpid(0, &status, WNOHANG));
}

BOOL os_signal_child_init(void (*fn) ())
{
    struct sigaction act, oact;

    sigemptyset(&act.sa_mask);
    act.sa_handler = fn;
    act.sa_flags = 0;

    if (sigaction(SIGCHLD, &act, &oact) == 0)
        return (T);

    log_panic("[os_signal_child_init()] sigaction() failed: %s",
              strerror(errno));
    return (NIL);
}

BOOL os_signal_child_clear(void)
{
    struct sigaction act, oact;

    sigemptyset(&act.sa_mask);
    act.sa_handler = SIG_DFL;
    act.sa_flags = 0;

    if (sigaction(SIGCHLD, &act, &oact) == 0)
        return (T);

    log_panic("[os_signal_child_clear()] sigaction() failed: %s",
              strerror(errno));
    return (NIL);
}

/* ====================================================================== */

BOOL os_signal_alarm_init(void (*fn) ())
{
    struct sigaction act, oact;

    sigemptyset(&act.sa_mask);
    act.sa_handler = fn;
    act.sa_flags = 0;

    if (sigaction(SIGALRM, &act, &oact) == 0)
        return (T);

    log_panic("[os_signal_alarm_init()] sigaction() failed: %s",
              strerror(errno));
    return (NIL);
}

BOOL os_signal_alarm_clear(void)
{
    struct sigaction act, oact;

    sigemptyset(&act.sa_mask);
    act.sa_handler = SIG_DFL;
    act.sa_flags = 0;

    if (sigaction(SIGALRM, &act, &oact) == 0)
        return (T);

    log_panic("[os_signal_alarm_clear()] sigaction() failed: %s",
              strerror(errno));
    return (NIL);
}

/* ====================================================================== */

/* Force core dump for nominated signals (just to help debugging) */

#if 0
static void os_signal_handler_abort()
{
    abort();
}
#endif

BOOL os_signal_init()
{
    struct sigaction act, oact;

    sigemptyset(&act.sa_mask);
    act.sa_handler = SIG_IGN;   /* IGNORE => write() will return EPIPE */
    act.sa_flags = 0;

    if (sigaction(SIGPIPE, &act, &oact) == 0)
        return (T);

    log_panic("[os_signal_init()] sigaction() failed: %s",
              strerror(errno));
    return (NIL);
}

/* ====================================================================== */

BOOL os_lock_exclusive(int fd)
{
    int rc;

    do {
        rc = flock(fd, LOCK_EX);
    }
    while ((rc < 0) && (errno == EINTR));

    return ((rc >= 0) ? T : NIL);
}

BOOL os_lock_shared(int fd)
{
    int rc;

    do {
        rc = flock(fd, LOCK_SH);
    }
    while ((rc < 0) && (errno == EINTR));

    return ((rc >= 0) ? T : NIL);
}

BOOL os_lock_release(int fd)
{
    int rc;

    do {
        rc = flock(fd, LOCK_UN);
    }
    while ((rc < 0) && (errno == EINTR));

    return ((rc >= 0) ? T : NIL);
}

BOOL os_lock_exclusive_allow_break(int fd)
{
    return ((flock(fd, LOCK_EX) >= 0) ? T : NIL);
}

BOOL os_lock_shared_allow_break(int fd)
{
    return ((flock(fd, LOCK_SH) >= 0) ? T : NIL);
}

BOOL os_lock_release_allow_break(int fd)
{
    return ((flock(fd, LOCK_UN) >= 0) ? T : NIL);
}

/* ====================================================================== */

static int os_read(int fd, char *buf, unsigned long count)
{
    int rc;

    while (count > 0) {
        rc = read(fd, buf, count);

        if (rc > 0) {
            buf += rc;
            count -= rc;
        } else if (rc == 0)     /* EOF */
            break;
        else if (errno != EINTR)        /* read() failed */
            break;
    }

    return ((count == 0) ? T : NIL);
}

BOOL os_random(struct ssl_config * ssl_config, void *buffer, unsigned long count)
{
    int fd;

    if (ssl_config->egd_socket) {
        if ((fd = os_connect_unix_socket(ssl_config->egd_socket)) < 0)
            return (NIL);

        while (count > 0) {
            unsigned char buf[2];
            unsigned long bytes = (count > 255) ? (count % 256) : count;

            buf[0] = 0x02;
            buf[1] = (unsigned char) bytes;

            if (!((write(fd, buf, 2) == 2) && os_read(fd, buffer, bytes))) {
                close(fd);
                return (NIL);
            }
            count -= bytes;
        }

        close(fd);
        return (T);
    }

    if ((fd = open("/dev/urandom", O_RDONLY)) < 0)
        return (NIL);

    if (!os_read(fd, buffer, count)) {
        close(fd);
        return (NIL);
    }

    close(fd);
    return (T);
}

/* ====================================================================== */

void
os_limit_vm(unsigned long x)
{
    /* Linux specific */
}

void
os_prctl_set_dumpable()
{
  /* Linux specific */
}

/* ====================================================================== */

/* Run child outside PTY */
BOOL os_run(char *cmdline, int *fdp, int *childpidp)
{
    int fd[2];
    int pid;

    if (!os_socketpair(fd))
        return(NIL);

    if ((pid = fork()) < 0)
        return (NIL);

    if (pid == 0) {
        dup2(fd[1], 0);
        dup2(fd[1], 1);
        dup2(fd[1], 2);

        close(fd[0]);
        close(fd[1]);

        process_exec(cmdline);
        /* NOTREACHED */
        exit(1);
    }

    /* Parent process */
    close(fd[1]);
    *fdp = fd[0];
    *childpidp = pid;
    return (T);
}


/* BSD Psuedo-Terminal support */
BOOL os_run_pty(char *cmdline, int *fdp, int *childpidp)
{
    int fd;
    int pid;

    if ((pid = forkpty(&fd, NIL, NIL, NIL)) < 0)
        return (NIL);

    if (pid == 0) {
        /* Child process */
        dup2(fd, 0);
        dup2(fd, 1);
        dup2(fd, 2);
        close(fd);

        process_exec(cmdline);
        /* NOTREACHED */
        exit(1);
    }

    /* Parent process */
    *fdp = fd;
    *childpidp = pid;
    return (T);
}

