/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <regex.h>
#include <glib.h>
#include "buffer.h"
#include "format.h"
#include "process.h"
#include "input.h"
#include "input-audiofile.h"
#include "input-avcodec.h"
#include "input-flac.h"
#include "input-mad.h"
#include "input-modplug.h"
#include "input-vorbis.h"
#include "output.h"
#include "player.h"
#include "main.h"

// --- Common code ---

typedef enum {
	PLAYER_START,
	PLAYER_PAUSE,
	PLAYER_UNPAUSE,
	PLAYER_STOP,
	PLAYER_RELEASE,
	PLAYER_SEEK,
} PlayerCommand;
typedef struct {
	PlayerCommand cmd;
	union {
		gchar *s;
		gdouble d;
	} u;
} player_message;

// The logic behind this is that the player thread only needs to look at its
// queue after every block, but the interface thread must read its queue from
// the GTK main loop. We thus use a real queue for interface-to-player
// commands, and a pipe for player-to-interface commands.
static GAsyncQueue *player_queue;
static int player_pipe[2];

static void status_reset(player_status *status) {
	status->seekable = 0;
	status->pos = 0.0;
	status->len = 1.0;
	status->bitrate = -1.0;
	status->converting = 0;
}

void status_set_error(player_status *status, const char *error) {
	if (status->error != NULL)
		free(status->error);
	status->error = (error == NULL) ? NULL : strdup(error);
}

// --- Player thread code ---

typedef struct {
	const char *ext;
	input_constructor cons;
} ext_mapping;
static ext_mapping extensions[] = {
	{"\\.(ogg|vorbis)$", input_new_vorbis},
	{"\\.(mp[123a]|mpeg[^.]*)$", input_new_mad},
	{"\\.flac$", input_new_flac},
	{"\\.(wav|au|aiff)$", input_new_audiofile},
	{"^m[oe]d\\.", input_new_modplug},
	{"\\.(m[oe]d|s3m|xm|it)$", input_new_modplug},
	{NULL, NULL}
};

static input *get_input_for_file(const char *fn) {
	const char *bn = strrchr(fn, '/');
	if (bn == NULL)
		bn = fn;
	else
		++bn;

	for (ext_mapping *p = extensions; p->ext != NULL; p++) {
		regex_t reg;
		if (regcomp(&reg, p->ext,
		            REG_EXTENDED | REG_ICASE | REG_NOSUB) != 0)
			g_error("Bad regular expression: %s\n", p->ext);
		int found = (regexec(&reg, bn, 0, NULL, 0) == 0);
		regfree(&reg);

		if (found)
			return p->cons();
	}
	// Give unrecognised formats to avcodec, since it autodetects lots of
	// them.
	return input_new_avcodec();
}

static gpointer player_thread_func(gpointer data) {
	player_status *status = (player_status *) data;

	// NULL means not currently playing.
	input *p = NULL;
	gboolean finished = FALSE;

	buffer *read_buf = buffer_new();
	if (read_buf == NULL)
		g_error("out of memory");

	buffer *play_buf = buffer_new();
	if (play_buf == NULL)
		g_error("out of memory");

	sample_format *force_fmt = NULL;
	output *out = get_output(&force_fmt);
	if (out == NULL) {
		status_set_error(status, "No output method available");
		goto finished;
	}

	while (1) {
		player_message *msg;

		if (p == NULL || finished) {
			msg = g_async_queue_pop(player_queue);
		} else {
			msg = g_async_queue_try_pop(player_queue);
		}

		if (msg != NULL) {
			switch (msg->cmd) {
			case PLAYER_START:
				if (p != NULL)
					g_error("PLAY while playing");
				p = get_input_for_file(msg->u.s);
				if (p == NULL) {
					status_set_error(status, "Unknown file type");
					goto finished;
				}
				if (p->open(p, msg->u.s) < 0) {
					status_set_error(status, "Cannot open file");
					goto finished;
				}
				free(msg->u.s);
				finished = FALSE;
				break;
			case PLAYER_PAUSE:
				if (p == NULL)
					g_error("PAUSE while not playing");
				out->release(out);
				msg = g_async_queue_pop(player_queue);
				if (msg->cmd == PLAYER_UNPAUSE)
					break;
				else if (msg->cmd != PLAYER_STOP)
					g_error("expected UNPAUSE or STOP");
				// fall through
			case PLAYER_STOP:
				if (p == NULL && !finished)
					g_error("STOP while not playing");
				if (p != NULL && p->close(p) < 0)
					g_error("player close failed");
				p = NULL;
				status_reset(status);
				break;
			case PLAYER_UNPAUSE:
				g_error("UNPAUSE while not paused");
				break;
			case PLAYER_RELEASE:
				out->release(out);
				break;
			case PLAYER_SEEK:
				if (p == NULL)
					g_error("SEEK while not playing");
				if (p->set_pos(p, msg->u.d) < 0) {
					status_set_error(status, "Cannot seek in file");
					goto finished;
				}
				break;
			}
		}
		free(msg);

		if (p == NULL)
			continue;

		int n = p->get_audio(p, read_buf);
		if (n == 0 && read_buf->used == 0 && play_buf->used == 0) {
			// Once the file's finished *and* the buffers are empty, we're done.
			goto finished;
		} else if (n < 0) {
			status_set_error(status, "Cannot read audio from file");
			goto finished;
		}

		status->seekable = p->get_seekable(p);
		if (p->get_pos(p, &status->pos) != 0) {
			status->pos = -1.0;
		}
		if (p->get_len(p, &status->len) != 0) {
			status->len = -1.0;
		}
		status->bitrate = p->bitrate;

		buffer *output_buf = read_buf;
		sample_format *output_fmt = &p->fmt;
		if (force_fmt != NULL)
			output_fmt = force_fmt;
		if (status->processing != PROCESS_NONE || !format_same(&p->fmt, output_fmt)) {
			const char *error;
			if (convert_format(read_buf, &p->fmt, play_buf, output_fmt, status->processing, &error) < 0) {
				status_set_error(status, error);
				goto finished;
			}
			output_buf = play_buf;
			status->converting = 1;
		} else {
			status->converting = 0;
		}
		copy_format(&status->in_fmt, &p->fmt);
		copy_format(&status->out_fmt, output_fmt);

		if (out->play(out, output_buf, output_fmt) < 0) {
			status_set_error(status, "Cannot output audio");
			goto finished;
		}

		continue;

	finished:
		status_reset(status);
		if (write(player_pipe[1], "f", 1) != 1)
			g_error("error writing to pipe");
		finished = TRUE;
		continue;
	}

	buffer_free(read_buf);
	buffer_free(play_buf);
	out->close(out);

	return NULL;
}

// --- Main thread code ---

static GThread *player_thread;

static player_message *new_message(PlayerCommand cmd) {
	player_message *m = malloc(sizeof *m);
	if (m == NULL)
		g_error("out of memory");

	m->cmd = cmd;

	return m;
}

static gboolean handle_read(GIOChannel *s, GIOCondition cond, gpointer data) {
	char c;
	if (read(player_pipe[0], &c, 1) != 1)
		g_error("read from pipe failed");

	switch (c) {
	case 'f':
		handle_player_finished();
		break;
	}

	return TRUE;
}

static void send_message(player_message *m) {
	g_async_queue_push(player_queue, m);
}

void player_init(int *argc, char ***argv, player_status *status) {
	if (!g_thread_supported())
		g_thread_init(NULL);

	player_queue = g_async_queue_new();
	if (pipe(player_pipe) < 0)
		g_error("pipe failed");

	GIOChannel *read_ioc = g_io_channel_unix_new(player_pipe[0]);
	g_io_add_watch(read_ioc, G_IO_IN, handle_read, NULL);

	status->error = NULL;
	status_reset(status);

	player_thread = g_thread_create(player_thread_func, status, FALSE, NULL);
	if (player_thread == NULL)
		g_error("g_thread_create failed");
}

gboolean player_start(const gchar *path) {
	player_message *m = new_message(PLAYER_START);
	m->u.s = strdup(path);
	send_message(m);
	return TRUE;
}

void player_stop() {
	send_message(new_message(PLAYER_STOP));
}

void player_release() {
	send_message(new_message(PLAYER_RELEASE));
}

void player_pause() {
	send_message(new_message(PLAYER_PAUSE));
}

void player_unpause() {
	send_message(new_message(PLAYER_UNPAUSE));
}

void player_seek(gdouble pos) {
	player_message *m = new_message(PLAYER_SEEK);
	m->u.d = pos;
	send_message(m);
}
