/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

namespace Postler {

    [DBus (name = "org.elementary.Postler")]
    interface PostlerClient : Object {
        public signal void progress (string account, string text, double fraction);
        public abstract void receive (string account) throws IOError;
        public signal void received (string account, string error_message);
        public abstract bool fetch (string account) throws IOError;
        public abstract void send (string account, string filename) throws IOError;
        public signal void sent (string account, string filename, string error_message);
        public abstract void quit () throws IOError;
    }

    public class Client : Object {
        PostlerClient client;

        public Client () {
            try {
                client = Bus.get_proxy_sync (BusType.SESSION,
                                             "org.elementary.Postler",
                                             "/org/elementary/postler");
                client.progress.connect ((account, text, fraction) => {
                    progress (account, text != "" ? text : null, fraction);
                });
                client.received.connect ((account, error_message) => {
                    received (account, error_message != "" ? error_message : null);
                });
                client.sent.connect ((account, filename, error_message) => {
                    sent (account, filename, error_message != "" ? error_message : null);
                });
                /* Ensure Postler is running, ignore errors */
                Process.spawn_async (null,
                    { Postler.App.argv0, "-m", "service" }, null,
                    SpawnFlags.SEARCH_PATH,
                    null, null);
            } catch (GLib.Error error) { }
        }

        public signal void progress (string account, string? text, double fraction);

        public void receive (string account="") {
            try {
                client.receive (account);
            } catch (GLib.Error error) {
                received (account, error.message);
            }
        }

        public signal void received (string account, string? error_message);

        public bool fetch (string account="") {
            try {
                return client.fetch (account);
            } catch (GLib.Error error) {
                return false;
            }
        }

        public void send (string account, string filename) {
            try {
                client.send (account, filename);
            } catch (GLib.Error error) {
                sent (account, filename, error.message);
            }
        }

        public signal void sent (string account, string filename, string? error_message);

        public void quit () {
            try {
                client.quit ();
            } catch (GLib.Error error) {
            }
        }
    }
}

