/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "polyxedit-globals.h"
#include "polyxmass-window-mngmt.h"
#include "polyxmass-ui-window-mngmt.h"
#include "polyxmass-ui-seqed-widget.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS, COPYING FUNCTIONS ...
 */
PxmWinMngmt*
polyxmass_winmngmt_new (void)
{
  PxmWinMngmt* winmngmt = g_malloc0 (sizeof (PxmWinMngmt));
  
  return winmngmt;
}


PxmWinMngmt*
polyxmass_winmngmt_copy (PxmWinMngmt* dest, PxmWinMngmt* src)
{
  g_assert (dest != NULL);
  g_assert (src != NULL);
  
  g_assert (src->wnd != NULL);
  dest->wnd = src->wnd;
  
  dest->sequence_editor_wnd = src->sequence_editor_wnd;
  dest->parent_wnd = src->parent_wnd;
  dest->seqed_widget = src->seqed_widget;
  dest->can_report = src->can_report;

  /* 'module' is compulsory. Go with assertions.
   */
  g_assert (src->module != NULL);
  g_assert (dest->module != NULL);
  g_free (dest->module);
  dest->module = g_strdup (src->module);

  /* 'desc' is compulsory. Go with assertions.
   */
  g_assert (src->desc != NULL);
  g_assert (dest->desc != NULL);
  g_free (dest->desc);
  dest->desc = g_strdup (src->desc);

  /* 'comment' is optional, go prudently.
   */
  if (src->comment != NULL)
    {
      if (dest->comment != NULL)
	g_free (dest->comment);
      
      dest->comment = g_strdup (src->comment);
    }
  
  dest->sequence_name = src->sequence_name;
  
  return dest;
}


/* DATA-MODIFYING FUNCTIONS
 */
gchar *
polyxmass_winmngmt_set_module (PxmWinMngmt* winmngmt, gchar *module)
{
  
  g_assert (winmngmt != NULL);
  g_assert (module != NULL);
  
  if (winmngmt->module != NULL)
    g_free (winmngmt->module);
  
  winmngmt->module = g_strdup (module);
  
  return winmngmt->module;
}
  

gchar *
polyxmass_winmngmt_set_desc (PxmWinMngmt* winmngmt, gchar *desc)
{
  
  g_assert (winmngmt != NULL);
  g_assert (desc != NULL);
  
  if (winmngmt->desc != NULL)
    g_free (winmngmt->desc);
  
  winmngmt->desc = g_strdup (desc);
  
  return winmngmt->desc;
}
  


gchar *
polyxmass_winmngmt_set_comment (PxmWinMngmt* winmngmt, gchar *comment)
{
  
  g_assert (winmngmt != NULL);
  g_assert (comment != NULL);
  
  if (winmngmt->comment != NULL)
    g_free (winmngmt->comment);
  
  winmngmt->comment = g_strdup (comment);
  
  return winmngmt->comment;
}



/*  LOCATING FUNCTIONS
 */
PxmWinMngmt*
polyxmass_winmngmt_find_registered_window (GtkWidget *wnd,
					   gchar *module)
{
  gint iter = 0;
  gint kter = 0;
  
  PxmWinMngmt* winmngmt = NULL;

  GPtrArray *currentGPA = NULL;
  
  g_assert (polyxmass_winmngmtGPA_GPA != NULL);
  g_assert (module != NULL);
  

  for (iter = 0; iter < polyxmass_winmngmtGPA_GPA->len; iter++)
    {
      currentGPA = g_ptr_array_index (polyxmass_winmngmtGPA_GPA, iter);
      g_assert (currentGPA != NULL);

      /* Since all the instances in the currently iterated array have
	 necessarily the same 'module' gchar string, we can just ask
	 for the first one to see if we are in the proper array for
	 dealing with the current 'wnd' and 'module' parameters.

	 Further, if the array currentGPA is there, that means that
	 there must be at least one winmngmt instance in it, so we
	 have not to check if the array is empty or not.
      */
      winmngmt = g_ptr_array_index (currentGPA, 0);
      g_assert (winmngmt != NULL);
      
      if (0 != strcmp (winmngmt->module, module))
	continue;
      
      /* OK, apparently, we are iterating in the correct array of
	 PxmWinMngmt instances, so we can continue looking for the
	 proper instance there.
       */
      for (kter = 0; kter < currentGPA->len; kter++)
	{
	  winmngmt = g_ptr_array_index (currentGPA, kter);
	  g_assert (winmngmt != NULL);
	  
	  if (winmngmt->wnd == wnd)
	    return winmngmt;
	}
    }
  
  return NULL;
}


GPtrArray*
polyxmass_winmngmt_find_module_GPA (gchar *module)
{
  gint iter = 0;
  
  PxmWinMngmt* winmngmt = NULL;

  GPtrArray *currentGPA = NULL;
  
  g_assert (polyxmass_winmngmtGPA_GPA != NULL);
  g_assert (module != NULL);
  

  for (iter = 0; iter < polyxmass_winmngmtGPA_GPA->len; iter++)
    {
      currentGPA = g_ptr_array_index (polyxmass_winmngmtGPA_GPA, iter);
      g_assert (currentGPA != NULL);

      /* Since all the instances in the currently iterated array have
	 necessarily the same 'module' gchar string, we can just ask
	 for the first one to see if we are in the proper array for
	 dealing with the current 'wnd' and 'module' parameters.

	 Further, if the array currentGPA is there, that means that
	 there must be at least one winmngmt instance in it, so we
	 have not to check if the array is empty or not.
      */
      winmngmt = g_ptr_array_index (currentGPA, 0);
      g_assert (winmngmt != NULL);
      
      if (0 != strcmp (winmngmt->module, module))
	continue;
      
      return currentGPA;
    }
  
  return NULL;
}


PxmWinMngmt*
polyxmass_winmngmt_register_window (gchar *module,
				    GtkWidget *wnd,
				    GtkWidget *parent_wnd,
				    GtkWidget *seqed_widget,
				    gchar *desc,
				    gchar *comment,
				    gboolean can_report,
				    gchar* (*make_report) ())
{
  GtkWidget *window = NULL;

  PxmWinMngmt* winmngmt = NULL;
  PxmWinMngmt* winmngmt_old = NULL;

  GPtrArray *moduleGPA = NULL;

  /* Only the actual window pointer is compulsory, along with both the
     module and the description strings. All the other parameters are
     not obligarory.
  */
  g_assert (module != NULL);
  g_assert (wnd != NULL);
  g_assert (desc != NULL);
  
  winmngmt = polyxmass_winmngmt_new ();
  
  winmngmt->wnd = wnd;
  
  if (seqed_widget != NULL)
    {
      winmngmt->sequence_name = 
	PXM_SEQED_WIDGET (seqed_widget)->polymer->plminfo->name;
    }
  
  winmngmt->module = g_strdup (module);
  winmngmt->parent_wnd = parent_wnd;
  winmngmt->seqed_widget = seqed_widget;
  winmngmt->desc = g_strdup (desc);

  /* Note that the comment is not compulsory.
   */
  if (comment != NULL)
    winmngmt->comment = g_strdup (comment);

  winmngmt->can_report = can_report;

  if (make_report != NULL)
    winmngmt->make_report = make_report;
  
 
  /* At this point we should make sure the pointer to the array of
     arrays that contain a list of all the windows opened in the
     program space is non-NULL!
  */
  g_assert (polyxmass_winmngmtGPA_GPA != NULL);
  
  /* We first should check is the window has been registered already.
     If so, we update the found instance with the new one. Finally we
     free the new one. This allows us to make the economy of shuffling
     array'ed items around.
  */

  winmngmt_old = polyxmass_winmngmt_find_registered_window (wnd,
							    module);

  if (winmngmt_old != NULL)
    {
      /* The window was registered already. So we update it.
       */
      polyxmass_winmngmt_copy (winmngmt_old, winmngmt);
      
      polyxmass_winmngmt_free (winmngmt);

      return winmngmt_old;
    }
  else
    {
      /* OK, we really have to do the work. But, is there a GPtrArray
	 already for the 'module', or should we allocate one ?
       */
      moduleGPA = polyxmass_winmngmt_find_module_GPA (module);
      
      if (moduleGPA == NULL)
	{
	  /* OK, we are certainly registering the first window ever
	     for the 'module'. We thus have to create the 'module's
	     GPtrArray of PxmWinMngmt instances. And append that
	     GPtrArray to the great array of arrays.
	   */
	  moduleGPA = g_ptr_array_new ();
	  
	  /* Add the newly allocated and characterized array to the array
	     of arrays.
	  */
	  g_ptr_array_add (polyxmass_winmngmtGPA_GPA, moduleGPA);
	}
      
      /* And now add to that module array the winmngmt instance.
       */
      g_ptr_array_add (moduleGPA, winmngmt);

      /* And now make sure we have this item added to the treeview of
	 available windows. Of course, only if the "window-management"
	 window is opened already !
      */
      window = g_object_get_data (G_OBJECT (polyxmass_main_wnd),
				  "window_mngmt_wnd");
      if (window != NULL)
	polyxmass_window_mngmt_wnd_winmngmt_treeview_add_winmngmt (window,
								   winmngmt,
								   moduleGPA);
      return winmngmt;
    }
    
  return NULL ;
}


gint
polyxmass_winmngmt_un_register_window (GtkWidget *wnd, gchar *module)
{
  GtkWidget *window = NULL;
  
  gint iter = 0;
  gint kter = 0;
  
  PxmWinMngmt* winmngmt = NULL;

  GPtrArray *currentGPA = NULL;
  
  g_assert (polyxmass_winmngmtGPA_GPA != NULL);
  g_assert (module != NULL);
  

  for (iter = 0; iter < polyxmass_winmngmtGPA_GPA->len; iter++)
    {
      currentGPA = g_ptr_array_index (polyxmass_winmngmtGPA_GPA, iter);
      g_assert (currentGPA != NULL);

      /* Since all the instances in the currently iterated array have
	 necessarily the same 'module' gchar string, we can just ask
	 for the first one to see if we are in the proper array for
	 dealing with the current 'wnd' and 'module' parameters.

	 Further, if the array currentGPA is there, that means that
	 there must be at least one winmngmt instance in it, so we
	 have not to check if the array is empty or not.
      */
      winmngmt = g_ptr_array_index (currentGPA, 0);
      g_assert (winmngmt != NULL);
      
      if (0 != strcmp (winmngmt->module, module))
	continue;
      
      /* OK, apparently, we are iterating in the correct array of
	 PxmWinMngmt instances, so we can continue looking for the
	 proper instance there.
       */
      for (kter = 0; kter < currentGPA->len; kter++)
	{
	  winmngmt = g_ptr_array_index (currentGPA, kter);
	  g_assert (winmngmt != NULL);
	  
	  if (winmngmt->wnd == wnd)
	    {
	      g_assert (TRUE == g_ptr_array_remove (currentGPA, 
						    winmngmt));

	      /*
		We call a function to refresh the contents of the
		window management treeview, so that it becomes
		impossible to perform actions on non-existent windows.

		Of course, we do this only if the "window-management"
		window is actually opened !
	      */
	      window = g_object_get_data (G_OBJECT (polyxmass_main_wnd),
					  "window_mngmt_wnd");
	      if (window != NULL)
		polyxmass_window_mngmt_wnd_winmngmt_treeview_remove_winmngmt
		(window, winmngmt);
	      
	      /* Now that we have successfully remove the item from
		 the treeview, we can free the corresponding object.
	       */	     
	      polyxmass_winmngmt_free (winmngmt);

	      /* Note that if we have just removed the last item of
		 the array, that array should now be empty. In this
		 case, we just remove it from the array of arrays and
		 free it. This way we are always certain that we are
		 not having empty arrays in the array of arrays, thus
		 simplifying considerably the handling of these
		 arrays.
	      */
	      if (currentGPA->len == 0)
		{
		  g_assert (FALSE != 
			    g_ptr_array_remove (polyxmass_winmngmtGPA_GPA,
						currentGPA));
		
		  g_ptr_array_free (currentGPA, TRUE);
		}
	      
	      return 1;
	    }
	}
    }
  
  return 0 ;  
}




/* FREE'ING FUNCTIONS
 */
gint
polyxmass_winmngmt_free (PxmWinMngmt* winmngmt)
{
  g_assert (winmngmt != NULL);

  if (winmngmt->module != NULL)
    g_free (winmngmt->module);
  
  if (winmngmt->desc != NULL)
    g_free (winmngmt->desc);
  
  if (winmngmt->comment != NULL)
    g_free (winmngmt->comment);
  
  
  g_free (winmngmt);
  
  return 1;
}



/* GPtrArray-RELATED FUNCTIONS
 */
gint
polyxmass_winmngmt_GPA_free (GPtrArray *GPA)
{
  PxmWinMngmt* winmngmt = NULL;
  gint count = 0;

  while (GPA->len > 0)
    {
      winmngmt = g_ptr_array_remove_index (GPA, 0);
      g_assert (winmngmt != NULL);
      
      polyxmass_winmngmt_free (winmngmt);
      
      count++;
    }

  g_ptr_array_free (GPA, TRUE);
  
  return count;
}


gint
polyxmass_winmngmt_GPA_of_GPA_free (GPtrArray *GPA)
{
  /* The GPA that we get is a pointer array of arrays that contain 
     PxmWinMngmt instances. So we have to iterate in the different
     arrays and iterate in  each one and free all the allocated stuff.
  */
  gint count = 0;

  GPtrArray *currentGPA = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      currentGPA = g_ptr_array_remove_index (GPA, 0);
      g_assert (currentGPA != NULL);

      count += polyxmass_winmngmt_GPA_free (currentGPA);
    }
  
  g_ptr_array_free (GPA, TRUE);

  return count;
}


