/*
 * The strace2tsv program converts strace output into tab separated values.
 * Copyright (C) 2004 The MITRE Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <unistd.h>
#include "scanner.h"

extern int yyparse(void);

#ifdef PACKAGE
const char package[] = PACKAGE;
#else
const char *package = (const char *)0;
#endif

#ifdef VERSION
const char version[] = VERSION;
#else
const char version[] = "Version information not available";
#endif

static void
print_version(const char *program)
{
  if (package)
    program = package;
  fprintf(stderr, "Package: %s %s\n", program, version);
}

static void
usage(const char *prog)
{
  fprintf(stderr,
	  "Usage: %s [options] [input]\n"
	  "Options:\n"
	  "  -o file -- output to file (default is standard output)\n"
	  "  -v      -- print version information\n"
	  "  -h      -- print this message\n",
	  prog);
  print_version(prog);
}

int
filter(void)
{
  int status = 0;
  while (yyparse()) {		/* skip the rest of a line with an error */
    status = 1;
    for (;;) {			/* look for end of line */
      int token = yylex();
      if (!token)		/* if eof return */
	return status;
      if (token == '\n')	/* eol found so parse again */
	break;
    }
  }
  return status;
}

int
main(int argc, char **argv)
{
  extern char *optarg;
  extern int optind;

  char *output = 0;

  for (;;) {
    int c = getopt(argc, argv, "o:vh");
    if (c == -1)
      break;
    switch (c) {
    case 'o':
      output = optarg;
      break;
    case 'v':
      print_version(argv[0]);
      return 0;
    case 'h':
      usage(argv[0]);
      return 0;
    default:
      usage(argv[0]);
      return 1;
    }
  }

  switch (argc - optind) {
  case 0:			/* Use stdin */
    break;
  case 1:
    if (!freopen(argv[optind], "r", stdin)) {
      perror(argv[optind]);
      return 1;
    }
    setfile(argv[optind]);
    break;
  default:
    fprintf(stderr, "Bad arg count\n");
    usage(argv[0]);
    return 1;
  }

  if (output && !freopen(output, "w", stdout)) {
    perror(output);
    return 1;
  }

  return filter();
}
