//
// LogStream.h
//
// $Id: //poco/1.2/Foundation/include/Poco/LogStream.h#1 $
//
// Library: Foundation
// Package: Logging
// Module:  LogStream
//
// Definition of the LogStream class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Foundation_LogStream_INCLUDED
#define Foundation_LogStream_INCLUDED


#include "Poco/Foundation.h"
#include "Poco/Logger.h"
#include "Poco/UnbufferedStreamBuf.h"
#include <istream>


namespace Poco {


class Foundation_API LogStreamBuf: public UnbufferedStreamBuf
	/// This class implements a streambuf interface
	/// to a Logger.
	///
	/// The streambuf appends all characters written to it
	/// to a string. As soon as a CR or LF (std::endl) is written,
	/// the string is sent to the Logger, with the set
	/// priority.
{
public:
	LogStreamBuf(Logger& logger, Message::Priority priority);
		/// Creates the LogStream.

	~LogStreamBuf();
		/// Destroys the LogStream.
		
	void setPriority(Message::Priority priority);
		/// Sets the priority for log messages.
		
	Message::Priority getPriority() const;
		/// Returns the priority for log messages.

private:
	int writeToDevice(char c);

private:
	Logger&           _logger;
	Message::Priority _priority;
	std::string       _message;
};


class Foundation_API LogIOS: public virtual std::ios
	/// The base class for LogStream.
	///
	/// This class is needed to ensure the correct initialization
	/// order of the stream buffer and base classes.
{
public:
	LogIOS(Logger& logger, Message::Priority priority);
	~LogIOS();
	LogStreamBuf* rdbuf();

protected:
	LogStreamBuf _buf;
};


class Foundation_API LogStream: public LogIOS, public std::ostream
	/// This class implements an ostream interface
	/// to a Logger.
	///
	/// The stream's buffer appends all characters written to it
	/// to a string. As soon as a CR or LF (std::endl) is written,
	/// the string is sent to the Logger, with the current
	/// priority.
	///
	/// Usage example:
	///     LogStream ls(someLogger);
	///     ls << "Some informational message" << std::endl;
	///     ls.error() << "Some error message" << std::endl;
{
public:
	LogStream(Logger& logger, Message::Priority priority = Message::PRIO_INFORMATION);
		/// Creates the LogStream, using the given logger and priority.

	LogStream(const std::string& loggerName, Message::Priority priority = Message::PRIO_INFORMATION);
		/// Creates the LogStream, using the logger identified
		/// by loggerName, and sets the priority.
		
	~LogStream();
		/// Destroys the LogStream.
		
	LogStream& fatal();
		/// Sets the priority for log messages to Message::PRIO_FATAL.
		
	LogStream& critical();
		/// Sets the priority for log messages to Message::PRIO_CRITICAL.

	LogStream& error();
		/// Sets the priority for log messages to Message::PRIO_ERROR.

	LogStream& warning();
		/// Sets the priority for log messages to Message::PRIO_WARNING.

	LogStream& notice();
		/// Sets the priority for log messages to Message::PRIO_NOTICE.

	LogStream& information();
		/// Sets the priority for log messages to Message::PRIO_INFORMATION.

	LogStream& debug();
		/// Sets the priority for log messages to Message::PRIO_DEBUG.

	LogStream& trace();
		/// Sets the priority for log messages to Message::PRIO_TRACE.

	LogStream& priority(Message::Priority priority);
		/// Sets the priority for log messages.
};


//
// inlines
//
inline Message::Priority LogStreamBuf::getPriority() const
{
	return _priority;
}


} // namespace Poco


#endif // Foundation_LogStream_INCLUDED
