/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades;

import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.log.Logger.Level;
import jp.sourceforge.mergedoc.pleiades.resource.TranslationNotFoundProperties;

import org.apache.commons.lang.StringUtils;

/**
 * Pleiades 起動オプションです。
 * <p>
 * @author cypher256
 */
public class PleiadesOption {
	
	/** Eclipse の -clean オプション */
	private boolean isClean;
	
	/** ニーモニック表示オプション */
	private boolean isNoMnemonic;
	
	/** デフォルト・スプラッシュ画面表示オプション */
	private final boolean isDefaultSplash;
	
	/** ログ・レベル */
	private Level logLevel = Level.INFO;
	
	/**
	 * 起動オプションを構築します。
	 * <p>
	 * @param agentArg エージェント起動引数
	 */
	PleiadesOption(String agentArg) {
		
		String arg = (agentArg == null) ? "" : "," + agentArg.replace(" ", "") + ",";
		
		isDefaultSplash = arg.contains(",default.splash,");
		isNoMnemonic = arg.contains(",no.mnemonic,");
		
		// 辞書に訳が無い場合のログ出力をするか
		boolean isEnabledNotFoundLog = arg.contains(",enabled.not.found.log,");
		if (isEnabledNotFoundLog) {
			System.setProperty(TranslationNotFoundProperties.NOT_FOUND_PROPERTIES_ENABELD, "true");
		}
		
		// ログ・レベルの取得
		if (arg.contains("log.level=")) {
			String s = arg.replaceFirst("^.*?log\\.level=", "").replaceFirst(",.*$", "");
			try {
				logLevel = Level.valueOf(s.toUpperCase());
			} catch (IllegalArgumentException e) {
				throw new IllegalArgumentException(
					"起動オプション log.level の値 [" + logLevel + "] が不正です。" +
					StringUtils.join(Logger.Level.values(), ",") +
					" のいずれかである必要があります。");
			}
		}
	}
	
	/**
	 * Eclipse の -clean オプションをセットします。
	 * <p>
	 * @param isClean 起動時に -clean が指定されている場合は true
	 */
	public void setClean(boolean isClean) {
		this.isClean = isClean;
	}
	
	/**
	 * Eclipse の -clean オプションを取得します。
	 * <p>
	 * @return 起動時に -clean が指定されている場合は true
	 */
	public boolean isClean() {
		return isClean;
	}
	
	/**
	 * ニーモニックの表示設定をセットします。
	 * <p>
	 * @param isNoMnemonic ニーモニックを使用しない場合は true
	 */
	public void setNoMnemonic(boolean isNoMnemonic) {
		this.isNoMnemonic = isNoMnemonic;
	}
	
	/**
	 * ニーモニックを表示しないか判定します。
	 * <p>
	 * @return ニーモニックを使用しない場合は true
	 */
	public boolean isNoMnemonic() {
		return isNoMnemonic;
	}
	
	/**
	 * Eclipse のデフォルト・スプラッシュ画面を使用するか判定します。
	 * <p>
	 * @return 使用する場合は true
	 */
	public boolean isDefaultSplash() {
		return isDefaultSplash;
	}
	
	/**
	 * ログ・レベルを取得します。
	 * <p>
	 * @return ログ・レベル
	 */
	public Level getLogLevel() {
		return logLevel;
	}

	/**
	 * プリロード起動か判定します。
	 * @return プリロード起動の場合は true
	 */
	public boolean isPreload() {
		return System.getProperty("pleiades.preload") != null;
	}
}
