/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file with_input_reader.cpp
 * \brief Implementation of the ptb::item_with_player_control_reader class.
 * \author Julien Jorge
 */

#include "ptb/controller_config.hpp"
#include "ptb/game_variables.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to start the action associated with a keyboard key.
 * \param key The code of the key.
 */
template<class Base>
bool ptb::item_with_player_control_reader<Base>::key_pressed
( const bear::input::key_info& key )
{
  bool result = false;
  const unsigned int m( game_variables::get_players_count() );
  controller_config cfg;

  for ( unsigned int i=1; i<=m; ++i )
    {
      player_action::value_type action = cfg.get_layout(i)(key.get_code());

      if ( action != player_action::action_null )
        {
          result = true;
          start_action( i, action );
        }
    }

  return result;
} // item_with_player_control_reader::key_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to stop the action associated with a keyboard key.
 * \param key The code of the key.
 */
template<class Base>
bool ptb::item_with_player_control_reader<Base>::key_released
( const bear::input::key_info& key )
{
  bool result = false;
  const unsigned int m( game_variables::get_players_count() );
  controller_config cfg;

  for ( unsigned int i=1; i<=m; ++i )
    {
      player_action::value_type action = cfg.get_layout(i)(key.get_code());

      if ( action != player_action::action_null )
        {
          result = true;
          stop_action( i, action );
        }
    }

  return result;
} // item_with_player_control_reader::key_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to continue the action associated with a keyboard key.
 * \param elapsed_time How long the action is done.
 * \param key The code of the key.
 */
template<class Base>
bool ptb::item_with_player_control_reader<Base>::key_maintained
( bear::universe::time_type elapsed_time,
  const bear::input::key_info& key )
{
  bool result = false;
  const unsigned int m( game_variables::get_players_count() );
  controller_config cfg;

  for ( unsigned int i=1; i<=m; ++i )
    {
      player_action::value_type action = cfg.get_layout(i)(key.get_code());

      if ( action != player_action::action_null )
        {
          result = true;
          do_action( elapsed_time, i, action );
        }
    }

  return result;
} // item_with_player_control_reader::key_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to start the action associated with a joystick button.
 * \param button The code of the button.
 * \param joy_index The index of the joystick.
 */
template<class Base>
bool ptb::item_with_player_control_reader<Base>::button_pressed
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = false;
  const unsigned int m( game_variables::get_players_count() );
  controller_config cfg;

  for ( unsigned int i=1; i<=m; ++i )
    {
      player_action::value_type action = cfg.get_layout(i)
        (joy_index, button);

      if ( action != player_action::action_null )
        {
          result = true;
          start_action( i, action );
        }
    }

  return result;
} // item_with_player_control_reader::button_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to stop the action associated with a joystick button.
 * \param button The code of the button.
 * \param joy_index The index of the joystick.
 */
template<class Base>
bool ptb::item_with_player_control_reader<Base>::button_released
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = false;
  const unsigned int m( game_variables::get_players_count() );
  controller_config cfg;

  for ( unsigned int i=1; i<=m; ++i )
    {
      player_action::value_type action =
        cfg.get_layout(i)(joy_index, button);

      if ( action != player_action::action_null )
        {
          result = true;
          stop_action( i, action );
        }
    }

  return result;
} // item_with_player_control_reader::button_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to continue the action associated with a joystick
 *        button.
 * \param elapsed_time How long the action is done.
 * \param button The code of the button.
 * \param joy_index The index of the joystick.
 */
template<class Base>
bool ptb::item_with_player_control_reader<Base>::button_maintained
( bear::universe::time_type elapsed_time,
 bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = false;
  const unsigned int m( game_variables::get_players_count() );
  controller_config cfg;

  for ( unsigned int i=1; i<=m; ++i )
    {
      player_action::value_type action =
        cfg.get_layout(i)(joy_index, button);

      if ( action != player_action::action_null )
        {
          result = true;
          do_action( elapsed_time, i, action );
        }
    }

  return result;
} // item_with_player_control_reader::button_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to start the action associated with a mouse button.
 * \param button The code of the button.
 * \param pos The position of the cursor on the screen.
 */
template<class Base>
bool ptb::item_with_player_control_reader<Base>::mouse_pressed
( bear::input::mouse::mouse_code button,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;
  const unsigned int m( game_variables::get_players_count() );
  controller_config cfg;

  for ( unsigned int i=1; i<=m; ++i )
    {
      player_action::value_type action = cfg.get_layout(i)(button);

      if ( action != player_action::action_null )
        {
          result = true;
          start_action( i, action );
        }
    }

  return result;
} // item_with_player_control_reader::mouse_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to stop the action associated with a mouse button.
 * \param button The code of the button.
 * \param pos The position of the cursor on the screen.
 */
template<class Base>
bool ptb::item_with_player_control_reader<Base>::mouse_released
( bear::input::mouse::mouse_code button,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;
  const unsigned int m( game_variables::get_players_count() );
  controller_config cfg;

  for ( unsigned int i=1; i<=m; ++i )
    {
      player_action::value_type action = cfg.get_layout(i)(button);

      if ( action != player_action::action_null )
        {
          result = true;
          stop_action( i, action );
        }
    }

  return result;
} // item_with_player_control_reader::mouse_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the player to continue the action associated with a mouse button.
 * \param elapsed_time How long the action is done.
 * \param button The code of the button.
 * \param pos The position of the cursor on the screen.
 */
template<class Base>
bool ptb::item_with_player_control_reader<Base>::mouse_maintained
( bear::universe::time_type elapsed_time,
  bear::input::mouse::mouse_code button,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;
  const unsigned int m( game_variables::get_players_count() );
  controller_config cfg;

  for ( unsigned int i=1; i<=m; ++i )
    {
      player_action::value_type action = cfg.get_layout(i)(button);

      if ( action != player_action::action_null )
        {
          result = true;
          do_action( elapsed_time, i, action );
        }
    }

  return result;
} // item_with_player_control_reader::mouse_maintained()
