/*
    Bear Engine - Model editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/mark_list_frame.cpp
 * \brief Implementation of the bf::mark_list_frame class.
 * \author Sbastien Angibaud
 */
#include "bf/mark_list_frame.hpp"

#include "bf/gui_model.hpp"
#include "bf/model_frame.hpp"
#include "bf/mark_properties_frame.hpp"
#include "bf/wx_facilities.hpp"

#include "bf/history/action_add_mark.hpp"
#include "bf/history/action_remove_mark.hpp"
#include "bf/history/action_modify_mark.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The frame owning this one.
 */
bf::mark_list_frame::mark_list_frame( wxWindow* parent )
  : wxPanel( parent ), m_model_frame(NULL), m_mark_list(NULL),
    m_new_mark_button(NULL)
{
  create_controls();
} // mark_list_frame::mark_list_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the model on which we work.
 * \param f The model frame in which we take the actions.
 */
void bf::mark_list_frame::set_model_frame( model_frame* f )
{
  m_model_frame = f;
  fill();
} // mark_list_frame::set_model_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update when the action is modified.
 */
void bf::mark_list_frame::update_action()
{
  fill();
} // mark_list_frame::update_action()


/*----------------------------------------------------------------------------*/
/**
 * \brief Update when the current selected mark is modified.
 */
void bf::mark_list_frame::update_mark()
{
  fill();
} // mark_list_frame::update_mark()

/*----------------------------------------------------------------------------*/
/**
 * \brief Select an mark of the model.
 */
void bf::mark_list_frame::select_mark( const std::string& mark_name )
{
  m_model_frame->set_mark_selection(mark_name);
} // mark_list_frame::select()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 */
void bf::mark_list_frame::create_controls()
{
  create_member_controls();
  create_sizer_controls();
} // mark_list_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls for which we keep a reference.
 */
void bf::mark_list_frame::create_member_controls()
{
  m_mark_list = new wxListBox( this, IDC_MARK_LIST );
  m_new_mark_button = new wxButton(this, wxID_NEW);
  
  m_new_mark_button->SetLabel(wxT(""));
} // mark_list_frame::create_member_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and add them in sizers.
 */
void bf::mark_list_frame::create_sizer_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer( wxHORIZONTAL );

  // the sizer for the buttons
  wxBoxSizer* s_sizer = new wxBoxSizer( wxVERTICAL );

  s_sizer->Add( m_new_mark_button, 0, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_DELETE), 0, wxEXPAND | wxALL, 0 );
  s_sizer->Add( new wxButton(this, wxID_PROPERTIES), 0, wxEXPAND | wxALL, 0 );

  sizer->Add( m_mark_list, 1, wxEXPAND );
  sizer->Add( s_sizer, 0, wxEXPAND );

  SetSizer(sizer);
} // mark_list_frame::create_sizer_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Clear the list of marks.
 */
void bf::mark_list_frame::clear()
{
  m_mark_list->Clear();
} // mark_list_frame::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the list with the marks of the model.
 */
void bf::mark_list_frame::fill()
{
  clear();
  
  if ( m_model_frame != NULL )
    if ( m_model_frame->get_model().has_active_action() )
      {
        const action& a = m_model_frame->get_model().get_active_action();
        
        action::const_mark_iterator it;
            
        for ( it = a.mark_begin(); it != a.mark_end(); ++it)
          m_mark_list->Append( std_to_wx_string( it->get_label() ) );

        if ( m_model_frame->get_model().has_mark_selection() ) 
          {
            const mark* m
              ( m_model_frame->get_model().get_main_mark_selection() );

            m_mark_list->SetStringSelection( std_to_wx_string(m->get_label()) );
          }
      }
} // mark_list_frame::fill()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on a mark menu entry.
 * \param event This event occured.
 */
void bf::mark_list_frame::on_new_mark(wxCommandEvent& event)
{
  if ( m_model_frame != NULL )
    if ( m_model_frame->get_model().has_active_action() )
      {
        const action& a = m_model_frame->get_model().get_active_action();
        mark_properties_frame dlg(this,&a);
          
        if ( dlg.ShowModal() == wxID_OK )
          {
            mark* m = new mark();
            m->set_label(dlg.get_mark_name());
            m->set_animation_file(dlg.get_mark_animation());
            m->apply_angle_to_animation(dlg.apply_angle_to_animation());
            m_model_frame->do_action
              ( new action_add_mark(a.get_name(), m) );
            fill();
            
            select_mark(dlg.get_mark_name());
          }
      }
} // mark_list_frame::on_new_mark()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the button "delete".
 * \param event This event occured.
 */
void bf::mark_list_frame::on_delete(wxCommandEvent& event)
{
  if ( m_model_frame != NULL )
    if ( m_model_frame->get_model().has_active_action() )
      {
	int index = m_mark_list->GetSelection();
	if ( index != wxNOT_FOUND ) 
	  {
	    action& a = m_model_frame->get_model().get_active_action();
	    std::string selection = 
	      wx_to_std_string(m_mark_list->GetStringSelection());
	    m_model_frame->clear_mark_selection();
	    	    
	    m_model_frame->do_action
	      ( new action_remove_mark
          ( m_model_frame->get_model().get_active_action().get_name(), 
            a.get_mark(selection)));
	    
	    fill();
	    
	    if ( a.mark_begin() != a.mark_end() )
	      {
          if ( index < (int)m_mark_list->GetCount() ) 
            m_mark_list->Select(index);
          else 
            m_mark_list->Select(index-1);
          
          select_mark
            (wx_to_std_string(m_mark_list->GetStringSelection()));
	      }
	  }
      }
} // mark_list_frame::on_delete()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the button "Properties".
 * \param event This event occured.
 */
void bf::mark_list_frame::on_show_properties(wxCommandEvent& event)
{
  if ( m_model_frame != NULL )
    if ( m_model_frame->get_model().has_active_action() )
      {
        int index = m_mark_list->GetSelection();
  
        if (index != wxNOT_FOUND)
          {
            action& a = m_model_frame->get_model().get_active_action();
            mark* m = a.get_mark
              (wx_to_std_string(m_mark_list->GetStringSelection()));
            
            mark_properties_frame dlg(this, &a );
            dlg.fill_from( *m );
            
            if ( dlg.ShowModal() == wxID_OK )
              m_model_frame->do_action
                ( new action_modify_mark
                  (m,dlg.get_mark_name(),dlg.get_mark_animation(),
                   dlg.apply_angle_to_animation()) );
            
            fill();
            m_mark_list->Select(index);
          }
      }
} // model_list_frame::on_show_properties()


/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a mark selection.
 * \param event This event occured.
 */
void bf::mark_list_frame::on_select_mark(wxCommandEvent& event)
{
  std::string s(wx_to_std_string(m_mark_list->GetStringSelection()));
  
  if ( !s.empty() ) 
    select_mark( s );
} // mark_list_frame::on_select_mark()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when closing the window.
 * \param event This event occured.
 */
void bf::mark_list_frame::on_close(wxCloseEvent& event)
{
  if ( event.CanVeto() )
    {
      Hide();
      event.Veto();
    }
} // mark_list_frame::on_close()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::mark_list_frame, wxPanel)
  EVT_BUTTON( wxID_NEW, bf::mark_list_frame::on_new_mark )
  EVT_BUTTON( wxID_DELETE, bf::mark_list_frame::on_delete )
  EVT_BUTTON( wxID_PROPERTIES, bf::mark_list_frame::on_show_properties )
  EVT_LISTBOX_DCLICK( bf::mark_list_frame::IDC_MARK_LIST,
                      bf::mark_list_frame::on_show_properties )
  EVT_LISTBOX(bf::mark_list_frame::IDC_MARK_LIST,
	      bf::mark_list_frame::on_select_mark )
  EVT_CLOSE( bf::mark_list_frame::on_close )
END_EVENT_TABLE()
