/*
  Bear Engine - Level editor

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/item_field_edit.cpp
 * \brief Implementation of the bf::item_field_edit class.
 * \author Julien Jorge
 */
#include "bf/item_field_edit.hpp"

#include "bf/bool_edit.hpp"
#include "bf/custom_type.hpp"
#include "bf/free_edit.hpp"
#include "bf/item_class_pool.hpp"
#include "bf/set_edit.hpp"
#include "bf/wx_facilities.hpp"
#include "bf/animation_file_edit.hpp"
#include "bf/font_file_edit.hpp"
#include "bf/sample_file_edit.hpp"
#include "bf/sprite_edit.hpp"

#include <list>
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
const wxColour bf::item_field_edit::s_field_prefix_colour(*wxLIGHT_GREY);
const std::string bf::item_field_edit::s_no_prefix("- no class -");

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param p An object that will do the modifications on the item.
 * \param parent Pointer to the owner.
 * \param id The identifier of this window.
 */
bf::item_field_edit::item_field_edit
( proxy& p, wxWindow* parent, wxWindowID id )
  : wxListView( parent, id, wxDefaultPosition, wxDefaultSize,
                wxLC_REPORT | wxLC_VRULES | wxLC_SINGLE_SEL ), m_proxy(p),
    m_item(NULL)
{
  InsertColumn(0, _("Property"));
  InsertColumn(1, _("Value"));

  Connect( wxEVT_SIZE,
           wxSizeEventHandler(item_field_edit::on_size) );
  Connect( wxEVT_COMMAND_LIST_COL_BEGIN_DRAG,
           wxListEventHandler(item_field_edit::on_column_begin_drag) );
  Connect( wxEVT_COMMAND_LIST_COL_END_DRAG,
           wxListEventHandler(item_field_edit::on_column_end_drag) );
  Connect( wxEVT_COMMAND_LIST_ITEM_ACTIVATED,
           wxListEventHandler(item_field_edit::on_item_activated) );
  Connect( wxEVT_KEY_UP,
           wxKeyEventHandler(item_field_edit::on_key_up) );

} // item_field_edit::item_field_edit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the item for which we want the properties.
 * \param item The item instance concerned by this window.
 */
void bf::item_field_edit::set_item( item_instance* item )
{
  int index = GetFirstSelected();
  m_item = item;
  DeleteAllItems();

  if ( m_item != NULL )
    {
      m_item_class_name = m_item->get_class().get_class_name();
      enumerate_properties();

      if (index != wxNOT_FOUND )
        {
          Select(index);
          EnsureVisible(index);
        }

      update_values();
    }
} // item_field_edit::set_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is an item edited.
 */
bool bf::item_field_edit::has_item() const
{
  return m_item != NULL;
} // item_field_edit::has_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the item currently edited.
 */
bf::item_instance& bf::item_field_edit::get_item()
{
  CLAW_PRECOND( has_item() );
  return *m_item;
} // item_field_edit::get_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the item currently edited.
 */
const bf::item_instance& bf::item_field_edit::get_item() const
{
  CLAW_PRECOND( has_item() );
  return *m_item;
} // item_field_edit::get_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of a field from the properties list.
 * \param i The index of the row in the properties list.
 * \param name (out) The name of the field.
 * \return false if \a i does not corresponds to a field line (ie. this is a
 *         prefix).
 */
bool
bf::item_field_edit::get_field_name( unsigned int i, std::string& name ) const
{
  bool result = false;

  name = wx_to_std_string(GetItemText(i));

  if ( GetItemBackgroundColour(i) != s_field_prefix_colour )
    {
      result = true;
      std::string field_prefix;

      while ( (i!=0) && field_prefix.empty() )
        {
          --i;
          if ( GetItemBackgroundColour(i) == s_field_prefix_colour )
            field_prefix = wx_to_std_string(GetItemText(i));
        }

      if ( !field_prefix.empty() && (field_prefix != s_no_prefix) )
        name = field_prefix + '.' + name;
    }

  CLAW_POSTCOND( !result || m_item->get_class().has_field(name) );

  return result;
} // item_field_edit::get_field_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Enumerate all the properties of the current item.
 */
void bf::item_field_edit::enumerate_properties()
{
  std::list<item_class const*> hierarchy;
  std::list<item_class const*>::const_iterator it;

  m_item->get_class().find_hierarchy( hierarchy );

  std::vector<std::string> fields;

  for ( it=hierarchy.begin(); it!=hierarchy.end(); ++it )
    get_fields_of(fields, **it);

  std::sort( fields.begin(), fields.end() );
  const std::vector<std::string>::iterator eit =
    std::unique(fields.begin(), fields.end());

  fields.resize( eit - fields.begin() );

  show_fields(fields);
} // item_field_edit::enumerate_properties()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the fields of a given class.
 * \param fields (out) The fields of \a classname are pushed back to this table.
 * \param item The class for which we want the fields.
 */
void bf::item_field_edit::get_fields_of
( std::vector<std::string>& fields, const item_class& item ) const
{
  item_class::field_iterator it;

  for ( it=item.field_begin(); it!=item.field_end(); ++it )
    fields.push_back(it->get_name());
} // item_field_edit::get_fields_of()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add the fields in the view of the properties.
 * \param fields The names of the fields to add.
 */
void bf::item_field_edit::show_fields( const std::vector<std::string>& fields )
{
  wxString prefix;
  wxString last_prefix;
  std::size_t index = 0;

  for ( std::size_t i=0; i!=fields.size(); ++i )
    {
      wxString f(std_to_wx_string(fields[i]));
      prefix = f.BeforeFirst(wxT('.'));

      if (prefix == f)
        prefix = std_to_wx_string(s_no_prefix);
      else
        f = f.AfterFirst(wxT('.'));

      if (prefix != last_prefix)
        {
          wxFont font( GetFont() );

          if (m_hidden.find(wx_to_std_string(prefix)) != m_hidden.end())
            font.SetStyle( wxFONTSTYLE_ITALIC );

          wxListItem head;
          head.SetFont( font );
          head.SetText(prefix);
          head.SetBackgroundColour(s_field_prefix_colour);
          head.SetId(index);
          ++index;

          InsertItem(head);
          last_prefix = prefix;
        }

      if ( m_hidden.find(wx_to_std_string(prefix)) == m_hidden.end() )
        {
          wxListItem prop;
          prop.SetText(f);
          prop.SetId(index);
          ++index;
          InsertItem(prop);
        }
    }

  SetColumnWidth(0, wxLIST_AUTOSIZE);
  adjust_last_column_size();
} // item_field_edit::show_fields()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the value of all fields of the current item.
 */
void bf::item_field_edit::update_values()
{
  for ( long i=0; i!=GetItemCount(); ++i )
    {
      wxListItem prop;
      prop.SetId(i);
      GetItem(prop);

      std::string name;

      if( get_field_name(i, name) )
        {
          const item_class& item(m_item->get_class());
          const type_field& f = item.get_field(name);

          if ( m_item->has_value(f) )
            prop.SetText( convert_value_to_text(f) );
          else
            prop.SetText
              ( std_to_wx_string(item.get_default_value(f.get_name())) );

          prop.SetColumn(1);
          SetItem(prop);

          if ( f.get_required() )
            set_required_color( i, m_item->has_value(f) );
          else
            set_default_value_color(i, m_item->has_value(f));
        }
    }
} // item_field_edit::update_values()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the correct font for a required field.
 * \param i The line to change.
 * \param b Tell if the field has a value or not.
 */
void bf::item_field_edit::set_required_color( unsigned int i, bool b )
{
  wxListItem prop;
  prop.SetId(i);
  GetItem(prop);

  wxFont font( GetFont() );
  font.SetWeight( wxFONTWEIGHT_BOLD );
  prop.SetFont( font );

  if (b)
    prop.SetTextColour( *wxBLACK );
  else
    prop.SetTextColour( *wxRED );

  SetItem(prop);
} // item_field_edit::set_required_color()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the correct font for a non valued field.
 * \param i The line to change.
 * \param b Tell if the field has a value or not.
 */
void bf::item_field_edit::set_default_value_color( unsigned int i, bool b )
{
  wxListItem prop;
  prop.SetId(i);

  GetItem(prop);

  wxFont font( GetFont() );

  if (b)
    {
      font.SetStyle( wxFONTSTYLE_NORMAL );
      prop.SetTextColour( *wxBLACK );
    }
  else
    {
      font.SetStyle( wxFONTSTYLE_ITALIC );
      prop.SetTextColour( wxColour(wxT("DARK GREY")) );
    }

  prop.SetFont( font );
  SetItem(prop);
} // item_field_edit::set_default_value_color()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the value of a field in a string.
 * \param f The field for which we want the value.
 */
wxString bf::item_field_edit::convert_value_to_text( const type_field& f ) const
{
  wxString result;

  if ( f.is_list() )
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        result = convert_value_to_text< std::list<integer_type> >(f.get_name());
        break;
      case type_field::u_integer_field_type:
        result =
          convert_value_to_text< std::list<u_integer_type> >(f.get_name());
        break;
      case type_field::real_field_type:
        result = convert_value_to_text< std::list<real_type> >(f.get_name());
        break;
      case type_field::boolean_field_type:
        result = convert_value_to_text< std::list<bool_type> >(f.get_name());
        break;
      case type_field::string_field_type:
        result = convert_value_to_text< std::list<string_type> >(f.get_name());
        break;
      case type_field::sprite_field_type:
        result = convert_value_to_text< std::list<sprite> >(f.get_name());
        break;
      case type_field::animation_field_type:
        result =
          convert_value_to_text< std::list<animation_file_type> >(f.get_name());
        break;
      case type_field::item_reference_field_type:
        result =
          convert_value_to_text< std::list<item_reference_type> >(f.get_name());
        break;
      case type_field::font_field_type:
        result =
          convert_value_to_text< std::list<font_file_type> >(f.get_name());
        break;
      case type_field::sample_field_type:
	result =
          convert_value_to_text< std::list<sample_file_type> >(f.get_name());
        break;
      }
  else
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        result = convert_value_to_text<integer_type>(f.get_name());
        break;
      case type_field::u_integer_field_type:
        result = convert_value_to_text<u_integer_type>(f.get_name());
        break;
      case type_field::real_field_type:
        result = convert_value_to_text<real_type>(f.get_name());
        break;
      case type_field::boolean_field_type:
        result = convert_value_to_text<bool_type>(f.get_name());
        break;
      case type_field::string_field_type:
        result = convert_value_to_text<string_type>(f.get_name());
        break;
      case type_field::sprite_field_type:
        result = convert_value_to_text<sprite>(f.get_name());
        break;
      case type_field::animation_field_type:
        result = convert_value_to_text<animation_file_type>(f.get_name());
        break;
      case type_field::item_reference_field_type:
        result = convert_value_to_text<item_reference_type>(f.get_name());
        break;
      case type_field::font_field_type:
        result = convert_value_to_text<font_file_type>(f.get_name());
        break;
      case type_field::sample_field_type:
        result = convert_value_to_text<sample_file_type>(f.get_name());
        break;
      }

  return result;
} // item_field_edit::convert_value_to_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Ajust the size of the last column so there is no empty space on the
 *        right.
 */
void bf::item_field_edit::adjust_last_column_size()
{
  SetColumnWidth( 1, GetSize().x - GetColumnWidth(0) );
} // item_field_edit::adjust_last_column_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove the value in the selected field.
 */
void bf::item_field_edit::delete_selected_field()
{
  long index = GetFocusedItem();

  if ( index != wxNOT_FOUND )
    {
      std::string name;

      if ( get_field_name(index, name) )
        {
          m_proxy.delete_field(*m_item, name);
          update_values();
        }
    }
} // item_field_edit::delete_selected_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create and show the edit frame for a field.
 * \param name The name of the field.
 */
void bf::item_field_edit::create_field_editor( const std::string& name )
{
  const item_class& item(m_item->get_class());
  const type_field& f = item.get_field(name);

  switch ( f.get_field_type() )
    {
    case type_field::integer_field_type:
      show_simple_property_dialog<integer_type>(f, _("integer"));
      break;
    case type_field::u_integer_field_type:
      show_simple_property_dialog<u_integer_type>(f, _("unsigned integer"));
      break;
    case type_field::real_field_type:
      show_simple_property_dialog<real_type>(f, _("real"));
      break;
    case type_field::boolean_field_type:
      show_property_dialog<bool_edit>(f, _("boolean"));
      break;
    case type_field::string_field_type:
      show_string_property_dialog(f);
      break;
    case type_field::sprite_field_type:
      show_property_dialog<sprite_edit>(f, _("sprite"));
      break;
    case type_field::animation_field_type:
      show_property_dialog<animation_file_edit>(f, _("animation"));
      break;
    case type_field::item_reference_field_type:
      show_item_reference_property_dialog(f);
      break;
    case type_field::font_field_type:
      show_property_dialog<font_file_edit>(f, _("font"));
      break;
    case type_field::sample_field_type:
      show_property_dialog<sample_file_edit>(f, _("sample"));
      break;
    }
} // item_field_edit::create_field_editor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the adequate dialog for editing a given string field.
 * \param f The type of the field we are editing.
 */
void bf::item_field_edit::show_string_property_dialog( const type_field& f )
{
  switch ( f.get_range_type() )
    {
    case type_field::field_range_free:
      show_property_dialog< free_edit<string_type> >(f, _("string"));
      break;
    case type_field::field_range_set:
      show_property_dialog< set_edit<string_type> >(f, _("string"));
      break;
    default:
      {
        CLAW_ASSERT(false, "range type is not valid.");
      }
    }
} // item_field_edit::show_string_property_dialog()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the adequate dialog for editing a given item reference field.
 * \param f The type of the field we are editing.
 */
void
bf::item_field_edit::show_item_reference_property_dialog( const type_field& f )
{
  wxArrayString values;

  m_proxy.get_item_identifiers(values, f);
  values.Sort();

  if ( f.is_list() )
    edit_item_reference_field< std::list<item_reference_type> >(f, values);
  else
    edit_item_reference_field<item_reference_type>(f, values);
} // item_field_edit::show_item_reference_property_dialog()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent to a resized window.
 * \param event The event.
 */
void bf::item_field_edit::on_size(wxSizeEvent& event)
{
  adjust_last_column_size();
  event.Skip();
} // item_field_edit::on_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user starts to resize a column.
 * \param event The event.
 */
void bf::item_field_edit::on_column_begin_drag(wxListEvent& event)
{
  if ( event.GetColumn() + 1 == GetColumnCount() )
    event.Veto();
  else
    event.Skip();
} // item_field_edit::on_column_begin_drag()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user ends resizing a column.
 * \param event The event.
 */
void bf::item_field_edit::on_column_end_drag(wxListEvent& event)
{
  adjust_last_column_size();
} // item_field_edit::on_column_begin_drag()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user activates an item.
 * \param event The event.
 */
void bf::item_field_edit::on_item_activated(wxListEvent& event)
{
  std::string name;

  if ( get_field_name(event.GetIndex(), name) )
    create_field_editor( name );
  else
    {
      if ( m_hidden.find(name) == m_hidden.end() )
        m_hidden.insert(name);
      else
        m_hidden.erase(name);

      DeleteAllItems();
      enumerate_properties();
      update_values();

      if ( event.GetIndex() < GetItemCount() )
        Select( event.GetIndex() );
    }
} // item_field_edit::on_item_activated()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user presses a key.
 * \param event The keyboard event that occured.
 */
void bf::item_field_edit::on_key_up(wxKeyEvent& event)
{
  switch( event.GetKeyCode() )
    {
    case WXK_DELETE:
      delete_selected_field();
      break;
    default:
      event.Skip();
    }
} // item_field_edit::on_key_up()
