/*
    Bear Engine - Animation editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/main_frame.cpp
 * \brief Implementation of the bf::main_frame class.
 * \author Julien Jorge
 */
#include "bf/main_frame.hpp"

#include "bf/animation_editor.hpp"
#include "bf/about_dialog.hpp"
#include "bf/image_pool.hpp"
#include "bf/animation_file_xml_reader.hpp"
#include "bf/animation_file_xml_writer.hpp"
#include "bf/wx_facilities.hpp"

#include "bf/icon/compile.xpm"

#include <fstream>
#include <wx/filename.h>
#include <wx/toolbar.h>
#include <wx/artprov.h>

DECLARE_APP(bf::animation_editor)

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
bf::main_frame::main_frame()
  : wxFrame(NULL, wxID_ANY, wxT("Bear Factory - Animation editor"))
{
  create_menu();
  create_toolbar();
  create_controls();
  Fit();

  //turn_animation_menu_entries(false);
} // main_frame::main_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load an animation.
 * \param path The path to the animation to load.
 */
void bf::main_frame::load_animation( const wxString& path )
{
  try
    {
      wxXmlDocument doc;
      animation_file_xml_reader reader;

      if ( !doc.Load(path) )
        throw std::ios_base::failure
          ( "Cannot load the XML file '" + wx_to_std_string(path) + "'" );

      animation anim = reader.load( doc.GetRoot() );
      m_animation_edit->set_value(anim);
      m_last_saved_animation = m_animation_edit->get_value();
      m_animation_file = path;
      make_title();
    }
  catch( std::exception& e )
    {
      wxMessageDialog msg
        ( this, std_to_wx_string( e.what() ),
          _("Error when loading the animation."), wxOK );

      msg.ShowModal();
    }
} // main_frame::load_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the title of the window.
 */
void bf::main_frame::make_title()
{
  if ( m_animation_file.empty() )
    SetTitle( wxT("Bear Factory - Animation editor") );
  else
    SetTitle( wxFileName(m_animation_file).GetName() );

  if ( is_changed() )
    SetTitle( GetTitle() + wxT("*") );
} // main_frame::make_title()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the animation has changes that are not saved.
 */
bool bf::main_frame::is_changed() const
{
  return m_last_saved_animation != m_animation_edit->get_value();
} // main_frame::is_changed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the animation.
 */
bool bf::main_frame::save()
{
  bool result = false;

  if ( m_animation_file.empty() )
    result = save_as();
  else
    result = effective_save();

  return result;
} // main_frame::save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the animation and ask the file to the user.
 */
bool bf::main_frame::save_as()
{
  bool result = false;
  wxFileDialog dlg( this, _("Choose a file"), wxT(""), m_animation_file,
                    _("Animation files (*.anim)|*.anim"),
                    wxFD_SAVE | wxFD_OVERWRITE_PROMPT );

  if ( dlg.ShowModal() == wxID_OK )
    {
      m_animation_file = dlg.GetPath();
      result = effective_save();
    }

  return result;
} // main_frame::save_as()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the animation without asking anything.
 */
bool bf::main_frame::effective_save()
{
  bool result = m_animation_edit->validate();

  if (result)
    {
      std::string std_path( wx_to_std_string(m_animation_file) );
      std::ofstream f( std_path.c_str() );

      if (f)
        {
          animation_file_xml_writer writer;

          writer.save( f, m_animation_edit->get_value() );
          m_last_saved_animation = m_animation_edit->get_value();
          result = true;
          make_title();
        }
      else
        {
          wxMessageDialog dlg
            ( this, _("Error"), _("Can't open the animation file."), wxOK );

          dlg.ShowModal();
        }
    }

  return result;
} // main_frame::effective_save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Enable/disable the controls relative to the animation.
 * \param b Tell if the controls are enabled.
 */
void bf::main_frame::turn_animation_menu_entries( bool b )
{
  GetMenuBar()->Enable( wxID_SAVE, b );
  GetMenuBar()->Enable( wxID_SAVEAS, b );
} // main_frame::turn_animation_menu_entries()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the menu bar.
 */
void bf::main_frame::create_menu()
{
  wxMenuBar* menu_bar = new wxMenuBar();

  menu_bar->Append(create_animation_menu(), _("&Animation"));
  menu_bar->Append(create_help_menu(), _("&Help"));

  SetMenuBar(menu_bar);
} // main_frame::create_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the toolbar.
 */
void bf::main_frame::create_toolbar()
{
  wxToolBar* bar = CreateToolBar();

  bar->AddTool( wxID_NEW, _("&New"), wxArtProvider::GetBitmap(wxART_NEW) );
  bar->AddTool
    ( wxID_OPEN, _("&Open"), wxArtProvider::GetBitmap(wxART_FILE_OPEN) );
  bar->AddTool
    ( wxID_SAVE, _("&Save"), wxArtProvider::GetBitmap(wxART_FILE_SAVE) );
  bar->AddSeparator();
  bar->AddTool( ID_COMPILE, _("&Compile"), wxBitmap(compile_xpm) );

  bar->Realize();
} // main_frame::create_toolbar()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls.
 */
void bf::main_frame::create_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer(wxHORIZONTAL);
  m_animation_edit = new animation_edit(*this);

  sizer->Add(m_animation_edit, 1, wxEXPAND | wxALL, 5 );

  SetSizer(sizer);

  SetStatusBar( new wxStatusBar(this, wxID_ANY) );
} // main_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the "Animation" menu of the menu bar.
 */
wxMenu* bf::main_frame::create_animation_menu() const
{
  wxMenu* menu = new wxMenu();

  menu->Append( wxID_NEW, _("&New..."), _("Create a new animation.") );
  menu->Append( wxID_OPEN, _("&Open..."), _("Open an existing animation.") );
  menu->Append( wxID_SAVE, _("&Save"), _("Save the animation.") );
  menu->Append( wxID_SAVEAS, _("Save &as..."),
                _("Save the animation under a new name.") );
  menu->AppendSeparator();
  menu->Append( ID_COMPILE, _("&Compile\tF9"), _("Compile the animation.") );
  menu->AppendSeparator();
  menu->Append( wxID_PREFERENCES, _("&Configuration..."),
                _("Configure the paths of the datas.") );
  menu->Append( ID_UPDATE_IMAGE_POOL, _("&Update image pool"),
                _("Scan the directory of images for new images.") );
  menu->AppendSeparator();
  menu->Append( wxID_EXIT, _("&Exit"), _("Close the window.") );

  return menu;
} // main_frame::create_animation_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the "Help" menu of the menu bar.
 */
wxMenu* bf::main_frame::create_help_menu() const
{
  wxMenu* menu = new wxMenu();

  menu->Append( wxID_HELP, _("&Online documentation"),
                _("Open the online documentation in your web browser.") );

  menu->AppendSeparator();

  menu->Append( wxID_ABOUT, _("&About"),
                _("Display some informations about the program.") );

  return menu;
} // main_frame::create_help_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the configuration of the program.
 */
void bf::main_frame::save_config()
{
  wxGetApp().set_main_rect(GetScreenRect());
} // main_frame::save_config()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the animation.
 */
void bf::main_frame::compile_animation()
{
  Disable();
  Refresh();

  if ( m_animation_edit->validate() )
    {
      bool do_it = true;

      if ( m_animation_file.IsEmpty() )
        do_it = save_as();

      if ( do_it )
        compile_animation_no_check();
    }

  Enable();
  GetStatusBar()->SetStatusText( _("Compilation done."), 0 );
} // main_frame::compile_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the animation without checking anything.
 */
void bf::main_frame::compile_animation_no_check()
{
  std::string std_path( wx_to_std_string(m_animation_file) );
  std::size_t pos = std_path.rfind(".anim");

  if ( pos != std::string::npos )
    std_path = std_path.substr(0, pos);

  std_path += ".canim";

  std::ofstream f( std_path.c_str() );

  if (f)
    {
      compiled_file cf(f);
      m_animation_edit->get_value().compile(cf);
    }
  else
    {
      wxMessageDialog dlg
        ( this, _("Error"), _("Can't open the animation file."), wxOK );

      dlg.ShowModal();
    }
} // main_frame::compile_animation_no_check()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Configuration" menu.
 * \param event This event occured.
 */
void bf::main_frame::on_configuration_menu(wxCommandEvent& event)
{
  wxGetApp().configure();
} // main_frame::on_configuration_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Update image pool" menu.
 * \param event This event occured.
 */
void bf::main_frame::on_update_image_pool_menu(wxCommandEvent& event)
{
  wxGetApp().update_image_pool();
} // main_frame::on_update_image_pool_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "New animation".
 * \param event This event occured.
 */
void bf::main_frame::on_new_animation(wxCommandEvent& event)
{
  main_frame* frame = new main_frame;
  frame->Show();
} // main_frame::on_new_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Open animation".
 * \param event This event occured.
 */
void bf::main_frame::on_open_animation(wxCommandEvent& event)
{
  wxFileDialog dlg( this, _("Choose an animation"), wxT(""), wxT(""),
                    _("Animation files (*.anim)|*.anim"),
                    wxFD_OPEN | wxFD_FILE_MUST_EXIST );

  if ( dlg.ShowModal() == wxID_OK )
    {
      if ( is_changed() || !m_animation_file.empty() )
        {
          main_frame* frm = new main_frame;
          frm->load_animation( dlg.GetPath() );
          frm->Show();
        }
      else
        load_animation( dlg.GetPath() );
    }
} // main_frame::on_open_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Save".
 * \param event This event occured.
 */
void bf::main_frame::on_save(wxCommandEvent& event)
{
  save();
} // main_frame::on_save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Save as".
 * \param event This event occured.
 */
void bf::main_frame::on_save_as(wxCommandEvent& event)
{
  save_as();
} // main_frame::on_save_as()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Compile".
 * \param event This event occured.
 */
void bf::main_frame::on_compile(wxCommandEvent& event)
{
  compile_animation();
} // main_frame::on_compile()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on "Exit".
 * \param event This event occured.
 */
void bf::main_frame::on_exit(wxCommandEvent& event)
{
  Close();
} // main_frame::on_exit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to an activation of the "online doc" menu.
 * \param event This event occured.
 */
void bf::main_frame::on_online_doc(wxCommandEvent& event)
{
  wxLaunchDefaultBrowser
    ( wxT("http://plee-the-bear.sourceforge.net/" )
      wxT("wiki/index.php/%C9diteur_d'animations") );
} // main_frame::on_online_doc()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to an activation of the "about" menu.
 * \param event This event occured.
 */
void bf::main_frame::on_about(wxCommandEvent& event)
{
  about_dialog dlg(*this);

  dlg.ShowModal();
} // main_frame::on_about()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when closing the window.
 * \param event This event occured.
 */
void bf::main_frame::on_close(wxCloseEvent& event)
{
  save_config();

  bool quit = !event.CanVeto();

  if ( !quit )
    {
      quit = true;
      m_animation_edit->validate();

      if ( is_changed() )
        {
          wxMessageDialog dlg
            ( this,
              _("The animation is not saved."
                 " Do you want to save it now?"),
              _("Animation is not saved."), wxYES_NO | wxCANCEL );

          int answer = dlg.ShowModal();

          if ( answer == wxID_CANCEL )
            quit = false;
          else if ( answer == wxID_YES )
            quit = save();
        }
    }

  if ( quit )
    event.Skip();
  else
    event.Veto();
} // main_frame::on_close()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::main_frame, wxFrame)
  EVT_MENU( wxID_PREFERENCES, bf::main_frame::on_configuration_menu )
  EVT_MENU( bf::main_frame::ID_UPDATE_IMAGE_POOL,
            bf::main_frame::on_update_image_pool_menu )

  EVT_MENU( wxID_NEW, bf::main_frame::on_new_animation )
  EVT_MENU( wxID_OPEN, bf::main_frame::on_open_animation )
  EVT_MENU( wxID_SAVE, bf::main_frame::on_save )
  EVT_MENU( wxID_SAVEAS, bf::main_frame::on_save_as )

  EVT_MENU( bf::main_frame::ID_COMPILE, bf::main_frame::on_compile )

  EVT_MENU( wxID_EXIT, bf::main_frame::on_exit )

  EVT_MENU( wxID_HELP, bf::main_frame::on_online_doc )
  EVT_MENU( wxID_ABOUT, bf::main_frame::on_about )

  EVT_CLOSE( bf::main_frame::on_close )
END_EVENT_TABLE()
