/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file straight_slope.cpp
 * \brief Implementation of the bear::straight_slope class.
 * \author Julien Jorge
 */
#include "generic_items/straight_slope.hpp"

#include "universe/collision_info.hpp"
#include "engine/export.hpp"

BASE_ITEM_EXPORT( straight_slope, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::straight_slope::straight_slope()
  : m_margin(5), m_top_side_is_active(false),
    m_bottom_side_is_active(false)
{

} // straight_slope::straight_slope()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type bool.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::straight_slope::set_bool_field
( const std::string& name, bool value )
{
  bool ok(true);

  if (name == "straight_slope.top_side_is_active")
    m_top_side_is_active = value;
  else if (name == "straight_slope.bottom_side_is_active")
    m_bottom_side_is_active = value;
  else
    ok = super::set_bool_field(name, value);

  return ok;
} // straight_slope::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c <real>.
 * \param name The name of the field to set.
 * \param value The new value of the field.
 */
bool bear::straight_slope::set_real_field
( const std::string& name, double value )
{
  bool result = true;

  if ( name == "straight_slope.margin" )
    m_margin = value;
  else
    result = super::set_real_field(name, value);

  return result;
} // straight_slope::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void bear::straight_slope::build()
{
  super::build();

  if (m_top_side_is_active)
    {
      set_height( get_height() + m_margin );
      claw::math::coordinate_2d<int> gap(get_gap());
      gap.y -= m_margin;
      set_gap(gap);
    }

  if (m_bottom_side_is_active)
    {
      set_height( get_height() + m_margin );
      set_bottom( get_bottom() - m_margin );
    }
} // straight_slope::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if that the center of the other item is included in the bounds
 * of \a this before processing the collision.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void bear::straight_slope::collision_check_center_included
( engine::base_item& that, universe::collision_info& info )
{
  double f;

  const universe::coordinate_type pos_x
    ( info.get_bottom_left_on_contact().x + that.get_width() / 2 );

  if ( (pos_x >= get_left()) && (pos_x <= get_right()) )
    {
      bool align = false;

      if ( (that.get_bottom() <= get_top())
           && (info.other_previous_state().get_bottom()
               >= get_top() - 2 * m_margin) )
        {
          if (m_top_side_is_active)
            {
              const universe::position_type pos
                (info.get_bottom_left_on_contact().x, get_top() - m_margin);

              align = collision_align_top(info, pos);
              f = get_top_friction();
            }
        }
      else if ( (that.get_top() >= get_bottom())
                && (info.other_previous_state().get_top()
                    <= get_bottom() + 2 * m_margin) )
        {
          if (m_bottom_side_is_active)
            {
              const universe::position_type pos
                ( info.get_bottom_left_on_contact().x,
                  get_bottom() + m_margin - that.get_height() );

              align = collision_align_bottom(info, pos);
              f = get_bottom_friction();
            }
        }

      if ( align )
        {
          that.set_contact_friction(f);
          that.set_system_angle(0);
          z_shift(that);
        }
    }
} // straight_slope::collision_check_center_include()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call collision_check_center_included().
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void bear::straight_slope::collision
( engine::base_item& that, universe::collision_info& info )
{
  if ( satisfy_collision_condition(that) )
    collision_check_center_included(that, info);
} // straight_slope::collision()
