/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file ambient_sound.cpp
 * \brief Implementation of the bear::ambient_sound class.
 * \author Sebastie Angibaud
 */
#include "generic_items/ambient_sound.hpp"

#include "engine/export.hpp"
#include "engine/level_globals.hpp"

#include <limits>

BASE_ITEM_EXPORT( ambient_sound, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::ambient_sound::ambient_sound()
  : m_loops(-1), m_volume(1),
    m_time_limit(std::numeric_limits<universe::time_type>::infinity()),
    m_delay(0), m_fadeout(0), m_elapsed_time(0), m_sample(NULL)
{
  set_can_move_items(false);
} // ambient_sound::ambient_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::ambient_sound::~ambient_sound()
{
  if (m_sample != NULL)
    delete m_sample;
} // ambient_sound::~ambient_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c <string>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::ambient_sound::set_string_field
( const std::string& name, const std::string& value )
{
  bool ok = true;

  if (name == "ambient_sound.sample")
    m_sound_name = value;
  else
    ok = super::set_string_field(name,value);

  return ok;
} // ambient_sound::set_string_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c <real>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::ambient_sound::set_real_field
( const std::string& name, double value )
{
  bool ok = true;

  if (name == "ambient_sound.time_limit")
    m_time_limit = value;
  else if (name == "ambient_sound.delay")
    m_delay = value;
  else if (name == "ambient_sound.fadeout")
    m_fadeout = value;
  else if (name == "ambient_sound.volume")
    m_volume = value;
  else
    ok = super::set_real_field(name,value);

  return ok;
} // ambient_sound::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c int.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
bear::ambient_sound::set_integer_field( const std::string& name, int value )
{
  bool ok = true;

  if (name == "ambient_sound.loops")
    m_loops = std::max(-1, value);
  else
    ok = super::set_integer_field(name, value);

  return ok;
} // ambient_sound::set_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void bear::ambient_sound::build()
{
  m_sample = get_level_globals().new_sample(m_sound_name);
} // ambient_sound::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 */
void bear::ambient_sound::progress( universe::time_type elapsed_time )
{
  if ( m_elapsed_time + elapsed_time >= m_delay )
    if ( m_elapsed_time <= m_delay )
      play_sample();

  if (!is_global())
    {
      audio::sound_effect effect(m_sample->get_effect());
      effect.set_position( get_center_of_mass() );
      m_sample->set_effect(effect);
    }

  if ( m_elapsed_time + elapsed_time >= m_time_limit + m_delay )
    if ( m_elapsed_time <= m_time_limit + m_delay )
      m_sample->stop(m_fadeout);

  m_elapsed_time += elapsed_time;
} // ambient_sound::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is well initialized.
 */
bool bear::ambient_sound::is_valid() const
{
  return !m_sound_name.empty() && super::is_valid();
} // ambient_sound::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Play the sample.
 */
void bear::ambient_sound::play_sample() const
{
  audio::sound_effect effect;
  effect.set_loops(m_loops);
  effect.set_volume(m_volume);

  if (!is_global())
    effect.set_position( get_center_of_mass() );

  m_sample->play(effect);
} // ambient_sound::play_sample()

