/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file static_ambient_sound.hpp
 * \brief A sound in the level.
 * \author Julien Jorge
 */
#ifndef __BEAR_AMBIENT_SOUND_HPP__
#define __BEAR_AMBIENT_SOUND_HPP__

#include "audio/sample.hpp"
#include "engine/base_item.hpp"
#include "generic_items/class_export.hpp"

#include "engine/export.hpp"

namespace bear
{
  /**
   * \brief A sound in the level.
   *
   * The following fields are supported by this item:
   *  - sample (string) [required], the path of the sound to play,
   *  - loops (integer), tell how many times the sound sample loops
   *    (default = infinity),
   *  - volume (real), in [0,1], the volume of the sound sample (default = 1),
   *  - time_limit (real), stop playing when the sample has been playing this
   *    duration (default = infinity),
   *  - delay (real), start playing the sample after this duration
   *    (default = 0),
   *  - fadeout (real), fade the sample out during this duration, in seconds,
   *    after the time limit (default = 0).
   *
   * \author Julien Jorge
   */
  class GENERIC_ITEMS_EXPORT ambient_sound:
    public engine::base_item
  {
    DECLARE_BASE_ITEM(ambient_sound, bear);

    /** \brief The type of the parent class. */
    typedef engine::base_item super;

  public:
    ambient_sound();
    ~ambient_sound();

    bool set_string_field( const std::string& name, const std::string& value );
    bool set_real_field( const std::string& name, double value );
    bool set_integer_field( const std::string& name, int value );

    void build();
    void progress( universe::time_type elapsed_time );
    bool is_valid() const;

  private:
    void play_sample() const;

  private:
    /** \brief The path of the sound to play. */
    std::string m_sound_name;

    /** \brief Tell how many times the sound sample loops. */
    int m_loops;

    /** \brief The volume of the sound. */
    double m_volume;

    /** \brief Stop playing when the sample has been playing this duration. */
    universe::time_type m_time_limit;

    /** \brief Start playing the sample after this duration. */
    universe::time_type m_delay;

    /** \brief Fade out the sample during this duration when stopping. */
    universe::time_type m_fadeout;

    /** \brief Time elapsed since the creation of the item. */
    universe::time_type m_elapsed_time;

    /** \brief The sample played. */
    audio::sample* m_sample;

  }; // class ambient_sound
} // namespace bear

#endif // __BEAR_AMBIENT_SOUND_HPP__
