/*************************************************************************\
*   Copyright (C) 2009 by Ulf Kreissig                                    *
*   udev@gmx.net                                                          *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
\*************************************************************************/

//--- LOCAL ---
#include "ion_google.h"
#include "logger/streamlogger.h"

//--- QT4 ---
#include <QHash>
#include <QMutex>
#include <QMutexLocker>

//--- KDE4 ---
#include <plasma/weather/weatherutils.h>

const QString GoogleIon::IonName("google");
const QString GoogleIon::ActionValidate("validate");
const QString GoogleIon::ActionWeather("weather");


struct GoogleIon::Private
{
	QHash<KJob *, XmlJobData *>     vJobData;
	QMap<QString, ConditionIcons>   vConditionList;
};

GoogleIon::GoogleIon( QObject * parent, const QVariantList & args )
	: IonInterface(parent, args),
	  d( new Private )
{
	d->vConditionList["thunderstorm"]           = Thunderstorm;
	d->vConditionList["chance_of_thunderstorm"] = ChanceThunderstormDay;
	d->vConditionList["rain"]                   = Rain;
	d->vConditionList["chance_of_rain"]         = LightRain;
	d->vConditionList["sunny"]                  = ClearDay;
	d->vConditionList["clear"]                  = ClearDay;
	d->vConditionList["mostly_sunny"]           = FewCloudsDay;
	d->vConditionList["partly_sunny"]           = PartlyCloudyDay;
	d->vConditionList["fog"]                    = Mist;
	d->vConditionList["chance_of_fog"]          = Mist;
	d->vConditionList["cloudy"]                 = PartlyCloudyDay;
	d->vConditionList["mostly_cloudy"]          = Overcast;
	d->vConditionList["partly_cloudy"]          = PartlyCloudyDay;
	d->vConditionList["snow"]                   = Snow;
	d->vConditionList["chance_of_snow"]         = ChanceSnowDay;
	d->vConditionList["snow-showers"]           = RainSnow;
	d->vConditionList["chance_of_snow_showers"] = Mist;
	d->vConditionList["showers"]                = Showers;
	d->vConditionList["chance_of_showers"]      = ChanceShowersDay;
	d->vConditionList["storm"]                  = Thunderstorm;
	d->vConditionList["chance_of_storm"]        = Thunderstorm;
	d->vConditionList["icy"]                    = Snow;
	d->vConditionList["haze"]                   = Haze;
	d->vConditionList["windy"]                  = ClearDay;
	d->vConditionList["hot"]                    = ClearDay;
	d->vConditionList["mist"]                   = Mist;
	d->vConditionList["sandstorm"]              = Thunderstorm;
}

GoogleIon::~GoogleIon()
{
	reset();
	delete d;
}

void
GoogleIon::init()
{
	setInitialized(true);
}

bool
GoogleIon::updateIonSource( const QString & source )
{
	// We expect the applet to send the source in the following tokenization:
	// ionname|validate|place_name - Triggers validation of place
	// ionname|weather|place_name|extra - Triggers receiving weather of place

	QStringList vTokens = source.split('|');

    if( vTokens.size() < 3 )
	{
        setData(source, "validate", QString("%1|timeout").arg(IonName));
        return true;
    }

	if( vTokens.at(1) == ActionValidate )
	{
		// Look for places to match
		findPlace( vTokens.at(2).simplified(), source );
		return true;
	}
	else if( vTokens.at(1) == ActionWeather )
	{
		getWeatherXmlData( vTokens.at(2).simplified(), source );
		return true;
	}
	return false;
}

void
GoogleIon::reset()
{
 	QHash<KJob *, XmlJobData *>::iterator it;
	for( it = d->vJobData.begin(); it != d->vJobData.end(); ++it )
	{
		it.key()->kill( KJob::Quietly );
		delete it.value();
	}
	d->vJobData.clear();

#if KDE_VERSION_MINOR >= 3
	/**  Triggered when we get initial setup data for ions that provide a list of places */
    emit(resetCompleted(this, true));
#endif
}


/*   Since google does not supports search requests, we are requesting the weather information of
 *   the place the user is looking for, and extract the city.
 *   To avoid to download the entire xml, we are parsing the data, as soon we get them
 *   and stop the job, when we got the city-tag.
 */
void
GoogleIon::findPlace( const QString & place, const QString & source )
{
	KUrl url( QByteArray("http://www.google.com/ig/api?weather=")
		+ QUrl::toPercentEncoding(place) );
	KIO::TransferJob * pJob = KIO::get( url, KIO::Reload, KIO::HideProgressInfo );
	if( pJob )
	{
		/*   we are using the name to descide wheather we have to look in d->vSearchJobs
		 *   or in d->vObservationJobs to find this certain job.
		 */
		pJob->setObjectName( ActionValidate );
		pJob->addMetaData("cookies", "none"); // Disable displaying cookies

		struct XmlJobData * pData = new XmlJobData;
		pData->sPlace = place;
		pData->sSource = source;

		d->vJobData.insert( pJob, pData );

		connect( pJob, SIGNAL(data(KIO::Job *, const QByteArray &)), this,
			SLOT(setup_slotDataArrived(KIO::Job *, const QByteArray &)) );
		connect( pJob, SIGNAL(result(KJob *)), this, SLOT(setup_slotJobFinished(KJob *)));
	}
}

//---return true when we extracted the city informations, so we can kill this job ---
bool
GoogleIon::readSearchXmlData( const QString & place, const QString & source, QXmlStreamReader & xml )
{
	dStartFunct();
	int bReturn = false;
	while( !xml.atEnd() )
	{
		xml.readNext();
		if( xml.error() == QXmlStreamReader::PrematureEndOfDocumentError )
		{
			break;
		}
		else if( xml.isStartElement() && xml.name().compare("city") == 0 )
		{
			setData( source, ActionValidate, QString("%1|valid|single|place|%2")
				.arg(IonName).arg(getNodeValue(xml)) );
			bReturn = true;
			break;
		}
		else if( xml.isStartElement() && xml.name().compare("problem_cause") == 0 )
		{
			setData( source, ActionValidate, QString("%1|invalid|single|%2").arg(IonName).arg(place) );
			bReturn = true;
			break;
		}
	}
	dEndFunct();
	return bReturn;
}

void
GoogleIon::setup_slotDataArrived( KIO::Job * job, const QByteArray & data )
{
	if( data.isEmpty() || !d->vJobData.contains(job) )
		return;

	struct XmlJobData * pXmlData = d->vJobData[job];
	pXmlData->xmlReader.addData( data );
	if( readSearchXmlData( pXmlData->sPlace, pXmlData->sSource, pXmlData->xmlReader ) )
	{
		// we are using EmitResult, otherwise the job will NOT send the signal
		// KJob::result(...) to remove the job.
		job->setObjectName( "completed" );
		job->kill( KJob::EmitResult );
	}
}

void
GoogleIon::setup_slotJobFinished( KJob * job )
{
	if( !d->vJobData.contains(job) )
		return;
	dStartFunct();
	struct XmlJobData * pXmlData = d->vJobData[job];

	if( job->error() != 0 && job->objectName().compare("completed") != 0 )
	{
		setData( pXmlData->sSource, ActionValidate, QString("%1|timeout").arg(IonName) );
		disconnectSource( pXmlData->sSource, this );
		dWarning() << job->errorString();
	}
	d->vJobData.remove( job );
	job->deleteLater();
	delete pXmlData;

	dDebug() << "Running Search/Observation Jobs: " << d->vJobData.count();
	dEndFunct();
}

void
GoogleIon::getWeatherXmlData( const QString & place, const QString & source )
{
	KUrl url( QByteArray("http://www.google.com/ig/api?weather=")
		+ QUrl::toPercentEncoding(place) );
	KIO::TransferJob * pJob = KIO::get( url, KIO::NoReload, KIO::HideProgressInfo );
	if( pJob )
	{
		/*   we are using the name to descide wheather we have to look in d->vSearchJobs
		 *   or in d->vObservationJobs to find this certain job.
		 */
		pJob->setObjectName( ActionWeather );
		pJob->addMetaData("cookies", "none"); // Disable displaying cookies

		struct XmlJobData * pXmlData = new XmlJobData;
		pXmlData->sPlace = place;
		pXmlData->sSource = source;
		pXmlData->iTempSystem = 0;
		pXmlData->iSpeedSystem = 0;

		d->vJobData.insert( pJob, pXmlData );

		connect( pJob, SIGNAL(data(KIO::Job *, const QByteArray &)), this,
			SLOT(slotDataArrived(KIO::Job *, const QByteArray &)) );
		connect( pJob, SIGNAL(result(KJob *)), this, SLOT(slotJobFinished(KJob *)));
	}
}

void
GoogleIon::slotDataArrived( KIO::Job * job, const QByteArray & data )
{
	if( data.isEmpty() || !d->vJobData.contains(job) )
		return;
	d->vJobData[job]->xmlReader.addData( data );
}

void
GoogleIon::slotJobFinished( KJob * job )
{
	if( !d->vJobData.contains(job) )
		return;
	dStartFunct();
	struct XmlJobData * pXmlData = d->vJobData[job];

	if( job->error() != 0 )
	{
		dWarning() << "XML error: " << job->errorString() << pXmlData->sSource << pXmlData->sSource;
	}
	else
	{
		readWeatherXmlData( pXmlData->sSource, pXmlData->xmlReader, pXmlData );
	}
	d->vJobData.remove( job );
	job->deleteLater();
	delete pXmlData;

	dDebug() << "Running Search/Observation Jobs: " << d->vJobData.count();
	dEndFunct();
}

void
GoogleIon::readWeatherXmlData( const QString & source, QXmlStreamReader & xml, XmlJobData * pXmlData )
{
	dStartFunct();
	removeAllData( source );	// clear the old values
	setData( source, Data() );	// start the update timer

	short iState = 0;
	short iDays  = 0;
	while( !xml.atEnd() )
	{
		xml.readNext();
		if( xml.isStartElement() )
		{
			if( iState == 0 && xml.name().compare("xml_api_reply") == 0 )
			{
				if( xml.attributes().value("version").compare("1") == 0 )
					iState = 1;
				else
					break;	//-- invalid version !?
			}
			else if( iState == 1 && xml.name().compare("weather") == 0 )
			{
				iState = 2;
			}
			else if( iState == 2 )
			{
				if( xml.name().compare("forecast_information") == 0 )
					readForecastInformations( source, xml, pXmlData );
				else if( xml.name().compare("current_conditions") == 0 )
					readCurrentConditions( source, xml, pXmlData );
				else if( xml.name().compare("forecast_conditions") == 0 )
				{
					readForecastConditions( source, iDays, xml, pXmlData );
					iDays+=1;
				}
				else
					dDebug() << xml.name();
			}
		}
		else if( xml.isEndElement() && iState > 0 )
		{
			if( iState == 2 && xml.name().compare("weather") == 0 )
				iState = 1;
			else if( iState == 1 && xml.name().compare("xml_api_reply") == 0 )
				iState = 0;
		}
	}
	if( xml.error() != 0 )
		dWarning() << xml.errorString();

	if( pXmlData->iTempSystem == FAHRENHEIT )
		setData( source, "Temperature", pXmlData->sCurrentTempF );
	else
		setData( source, "Temperature", pXmlData->sCurrentTempC );
	setData( source, "Total Weather Days", QString::number(iDays) );
	setData( source, "Credit", i18n("Supported by Google Weather Service") );
	setData( source, "Credit Url", "http://www.google.com" );
	dEndFunct();
}

void
GoogleIon::readForecastInformations( const QString & source,  QXmlStreamReader & xml, XmlJobData * pXmlData )
{
//	dStartFunct();
	Q_ASSERT_X( xml.isStartElement() && xml.name().compare("forecast_information") == 0,
	            "Google Ion",
	            "GoogleIon::readForecastInformation - current tag is wrong!" );

	while( !xml.atEnd() )
	{
		xml.readNext();
		if( xml.isEndElement() && xml.name().compare("forecast_information") == 0 )
			break;
		else if( xml.isStartElement() )
		{
//			dDebug() << xml.name().toString();
			if( xml.name().compare("unit_system") == 0 )
			{
				QString sUnitString = getNodeValue(xml);

				if( sUnitString.compare("US") == 0 )
				{
					pXmlData->iSpeedSystem = MPH;
					pXmlData->iTempSystem  = FAHRENHEIT;
				}
				else
				{
					pXmlData->iSpeedSystem = MPH;
					pXmlData->iTempSystem = CELSIUS;
				}
				setData( source, "Temperature Unit", QString::number(pXmlData->iTempSystem) );
				setData( source, "Wind Speed Unit",  QString::number(pXmlData->iSpeedSystem) );
			}
			else if( xml.name().compare("forecast_date") == 0 )
				pXmlData->observationDate = QDate::fromString( getNodeValue(xml), "yyyy-MM-dd" );
			else if( xml.name().compare("city") == 0 )
				setData( source, "Place", getNodeValue(xml) );
		}
	}
	if( xml.error() != 0 )
		dWarning() << xml.errorString();
//	dEndFunct();
}

void
GoogleIon::readCurrentConditions( const QString & source, QXmlStreamReader & xml, XmlJobData * pXmlData )
{
//	dStartFunct();
	Q_ASSERT_X( xml.isStartElement() && xml.name().compare("current_conditions") == 0,
	            "Google Ion",
	            "GoogleIon::readCurrentConditions - current tag is wrong!" );

	while( !xml.atEnd() )
	{
		xml.readNext();
		if( xml.isEndElement() && xml.name().compare("current_conditions") == 0 )
			break;
		else if( xml.isStartElement() )
		{
			if( xml.name().compare("condition") == 0 )
				setData( source, "Current Conditions", getNodeValue(xml) );
			else if( xml.name().compare("temp_f") == 0 )
				pXmlData->sCurrentTempF = getNodeValue(xml);
			else if( xml.name().compare("temp_c") == 0 )
				pXmlData->sCurrentTempC = getNodeValue(xml);
			else if( xml.name().compare("humidity") == 0 )
			{
				QString sHumidity = getNodeValue(xml);
				int iPos = sHumidity.indexOf(" ")+1;
				if( iPos > 0 )
					setData( source, "Humidity", sHumidity.right(sHumidity.length()-iPos) );
			}
			else if( xml.name().compare("icon") == 0 )
				setData( source, "Condition Icon", getIconName( getNodeValue(xml) ) );
			else if( xml.name().compare("wind_condition") == 0 )
			{
				// we need to extract the wind condtions from  a string like this "Wind: E at 0 mph"
				//                                                                       ^    ^
				QString sWind = getNodeValue(xml);
				int iPos = sWind.indexOf(" ", 6 );
				setData( source, "Wind Direction", sWind.mid(6, iPos-6 ) );
				iPos+=4;
				int iSpeedPos = sWind.indexOf(" ", iPos);
				setData( source, "Wind Speed", sWind.mid(iPos, iSpeedPos-iPos) );
			}
		}
	}
	if( xml.error() != 0 )
		dWarning() << xml.errorString();
//	dEndFunct();
}

void
GoogleIon::readForecastConditions( const QString & source, int iDayIndex, QXmlStreamReader & xml, XmlJobData * pXmlData )
{
	Q_UNUSED( pXmlData );
	Q_ASSERT_X( xml.isStartElement() && xml.name().compare("forecast_conditions") == 0,
	            "Google Ion",
	            "GoogleIon::readForecastConditions - current tag is wrong!" );

//	dStartFunct();
	QString sDay, sLowTemp, sHighTemp, sIcon, sCondition;
	while( !xml.atEnd() )
	{
		xml.readNext();
		if( xml.isEndElement() && xml.name().compare("forecast_conditions") == 0 )
			break;
		else if( xml.isStartElement() )
		{
			if( xml.name().compare("day_of_week") == 0 )
				sDay = getNodeValue(xml);
			else if( xml.name().compare("low") == 0 )
				sLowTemp = getNodeValue(xml);
			else if( xml.name().compare("high") == 0 )
				sHighTemp = getNodeValue(xml);
			else if( xml.name().compare("icon") == 0 )
				sIcon = getIconName( getNodeValue(xml) );
			else if( xml.name().compare("condition") == 0 )
				sCondition = getNodeValue(xml);
		}
	}
	if( xml.error() == 0 )
		setData( source,
			QString("Short Forecast Day %1").arg(iDayIndex),
			QString("%1|%2|%3|%4|%5|N/U").arg(sDay).arg(sIcon).arg(sCondition).arg(sHighTemp).arg(sLowTemp) );
	else
		dWarning() << xml.errorString();
//	dEndFunct();
}

inline QString
GoogleIon::getNodeValue( const QXmlStreamReader & xml ) const
{
	return xml.attributes().value("data").toString();
}

inline QString
GoogleIon::getIconName( const QString & sNodeValue )
{
	int iPos = sNodeValue.lastIndexOf("/")+1;
	if( iPos > 0 )
		return getWeatherIcon( d->vConditionList, sNodeValue.mid(iPos, sNodeValue.length()-iPos-4) );
	return "weather-none-available";
}

#if KDE_VERSION_MINOR >= 3
	 K_EXPORT_PLASMA_DATAENGINE(google, GoogleIon);
#else
	 K_EXPORT_PLASMA_ION(google, GoogleIon);
#endif
