/*************************************************************************\
*   Copyright (C) 2009 by Ulf Kreissig                                    *
*   udev@gmx.net                                                          *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
\*************************************************************************/

//--- LOCAL ---
#include "weatherdataprocessor.h"
#include "utils.h"
#include "logger/streamlogger.h"

//--- QT4 ---
#include <QDir>
#include <QFile>
#include <QHash>

//--- KDE4 ---

#include <values.h>
#if KDE_IS_VERSION(4,3,70)
	#include <KUnitConversion/Value>
#endif

const QString WeatherDataProcessor::CacheDirectory = QDir::homePath() + QLatin1String("/.cache/yawp");


struct WeatherDataProcessor::Private
{
	QHash<QString, QString>         vConditionIcons;

	YAWP_DISTANCE_UNIT              distanceSystem;
	YAWP_PRESSURE_UNIT              pressureSystem;
	YAWP_SPEED_UNIT                 speedSystem;
	YAWP_TEMPERATURE_UNIT           temperatureSystem;

	QHash<Yawp::DetailsProperty, short> vDetailsPropertyMap;

	bool initIconMap(const QString & resource);

	/** Check if a string is empty or contains "N/A" or "N/U".
	 *  In this cases the function returns false, otherwise true.
	 */
	bool checkStringValue( const QString & value ) const;

	/** Parse a string that contains a float number.
	 *  Returns FLT_MAX when parsing fails.
	 */
	float parseFloat( const QString & value ) const;

	short  convertDistance( const QString & value, YAWP_DISTANCE_UNIT system ) const;
	short  convertPressure( const QString & value, YAWP_PRESSURE_UNIT system ) const;
	short  convertTemp( const QString & value, YAWP_TEMPERATURE_UNIT system ) const;
	short  convertSpeed( const QString & value, YAWP_SPEED_UNIT system ) const;

#if KDE_VERSION_MINOR <= 2
	WeatherUtils::Unit convertSpeed2Distance( WeatherUtils::Unit fromSpeed ) const;
#endif

	void setUVValues( YawpWeather & weather, const QString & sUVIndex, const QString & sUVRating ) const;

	bool isNightTime( QString & sShortDay ) const;
	bool findDateForWeekday( QDate & currDate, const QString & sLookingDayName ) const;

	QString getSourceCacheFileName( const CityWeather & pCity ) const;
};

WeatherDataProcessor::WeatherDataProcessor()
	: d( new WeatherDataProcessor::Private )
{
	d->initIconMap( ":/iconnames.conf" );

	QDir dir(CacheDirectory);
	if( !dir.exists() )
		dir.mkpath(CacheDirectory);
}

WeatherDataProcessor::~WeatherDataProcessor()
{
	delete d;
}

YAWP_DISTANCE_UNIT WeatherDataProcessor::distanceSystem() const {return d->distanceSystem;}
void WeatherDataProcessor::setDistanceSystem( YAWP_DISTANCE_UNIT unitsystem ) {d->distanceSystem = unitsystem;}

YAWP_PRESSURE_UNIT WeatherDataProcessor::pressureSystem() const {return d->pressureSystem;}
void WeatherDataProcessor::setPressureSystem( YAWP_PRESSURE_UNIT unitsystem ) {d->pressureSystem = unitsystem;}

YAWP_TEMPERATURE_UNIT WeatherDataProcessor::temperatureSystem() const {return d->temperatureSystem;}
void WeatherDataProcessor::setTemperatureSystem( YAWP_TEMPERATURE_UNIT unitsystem ) {d->temperatureSystem = unitsystem;}

YAWP_SPEED_UNIT WeatherDataProcessor::speedSystem() const {return d->speedSystem;}
void WeatherDataProcessor::setSpeedSystem( YAWP_SPEED_UNIT unitsystem ) {d->speedSystem = unitsystem;}

bool
WeatherDataProcessor::updateLocation( CityWeather & cityInfo,
                                      const Plasma::DataEngine::Data & data ) const
{
	/* When data is empty or data contains validate return.
	 * The data contains the key "validate" in case someone is searching for a city - we do not care about
	 * this data.
	 */
	if( data.isEmpty() || data.contains("validate") )
		return false;

	cityInfo.deleteAllDays();

/*** FOR DEBUG - PRINTS A SORTED LIST OF ALL KEYS THAT THIS PROVIDER/ION SUPPORTS ***/
	dDebug() << cityInfo.localizedCityString() << cityInfo.provider();
	QStringList sKeyList;
	QHash<QString, QVariant>::const_iterator itKeys;
	for( itKeys = data.begin(); itKeys != data.end(); ++itKeys )
		sKeyList.append( itKeys.key() );
	sKeyList.sort();
	foreach( QString sKey, sKeyList )
		qDebug() << "     " << sKey << "\t\t -> " << data.value(sKey);
/************************************************************************************/

	YAWP_TEMPERATURE_UNIT fromTempSystem
		= (YAWP_TEMPERATURE_UNIT)data.value("Temperature Unit").toInt();
	YAWP_TEMPERATURE_UNIT fromDewTempSystem
		= (YAWP_TEMPERATURE_UNIT)data.value("Dewpoint Unit").toInt();
	YAWP_SPEED_UNIT       fromSpeedSystem
		= (YAWP_SPEED_UNIT)data.value("Wind Speed Unit").toInt();
	YAWP_DISTANCE_UNIT    fromVisibilitySystem
		= (YAWP_DISTANCE_UNIT)data.value("Visibility Unit").toInt();
	YAWP_PRESSURE_UNIT    fromPressureSystem
		= (YAWP_PRESSURE_UNIT)data.value("Pressure Unit").toInt();

//--- Get weather forecast information ---
	cityInfo.setCredit( data.value("Credit").toString() );
	cityInfo.setCreditURL( data.value("Credit Url").toString() );
	int iForecastDays = data.value("Total Weather Days", "0").toInt();
	QDate currDate;	// date of current day
	int iTodayDays(0);
	YawpDay * pLastDay = NULL;
	QString sLastDayName;

	for( int iDay = 0; iDay < iForecastDays; ++iDay )
	{
		QVariant var;
		var = data.value( QString("Short Forecast Day %1").arg(iDay) );
		QStringList vTokens = var.toString().split("|", QString::SkipEmptyParts);

		YawpWeather * pWeather = NULL;

		//--- find the date to the dayname ---
		if( vTokens.at(0).compare("Today", Qt::CaseInsensitive) == 0 ||
		    vTokens.at(0).compare( i18n("Today"), Qt::CaseInsensitive) == 0 )
		{
			iTodayDays += 1;
			pLastDay = new YawpDay;
			cityInfo.days().append( pLastDay );
			pWeather = &pLastDay->weather();
		}
		else if( vTokens.at(0).compare("Tonight", Qt::CaseInsensitive) == 0 ||
		    vTokens.at(0).compare( i18n("Tonight"), Qt::CaseInsensitive) == 0 )
		{
			iTodayDays += 1;
			if( !pLastDay )
			{
				pLastDay = new YawpDay;
				cityInfo.days().append( pLastDay );
				pWeather = &pLastDay->weather();
			}
			else
			{
				pWeather = &pLastDay->nightWeather();
				pLastDay->setHasNightValues(true);
			}
			pWeather->setDayTime(false);
		}
		else if( !currDate.isValid() )
		{
			QString sShortDay( vTokens.at(0) );
			bool bNightTime = d->isNightTime( sShortDay );

			QDate date( QDate::currentDate().addDays( iTodayDays > 0 ? 2 : 1 ) );
			if( d->findDateForWeekday( date, sShortDay ) )
			{
				int iFirstDateIndexDay = cityInfo.days().count();
				sLastDayName = sShortDay;
				currDate = date;

				pLastDay = new YawpDay;
				pLastDay->setDate( currDate );
				cityInfo.days().append( pLastDay );
				pWeather = &pLastDay->weather();
				pWeather->setDayTime( !bNightTime );

				//--- set the date for all previous days according to the current date. ---
				date = date.addDays( -iFirstDateIndexDay );
				for( int iDay = 0; iDay < iFirstDateIndexDay; ++iDay )
				{
					cityInfo.days().at(iDay)->setDate( date );
					date = date.addDays(1);
				}
			}
		}
		else
		{
			QString sShortDay( vTokens.at(0) );
			bool bNightTime = d->isNightTime( sShortDay );
			if( bNightTime && pLastDay && sLastDayName.compare( sShortDay ) == 0 )
			{
				pWeather = &pLastDay->nightWeather();
				pWeather->setDayTime( false );
				pLastDay->setHasNightValues( true );
			}
			else
			{
				currDate = currDate.addDays(1);
				pLastDay = new YawpDay;
				pLastDay->setDate( currDate );
				cityInfo.days().append( pLastDay );
				sLastDayName = sShortDay;
				pWeather = &pLastDay->weather();
				pWeather->setDayTime( !bNightTime );
			}
		}

		if( pWeather )
		{
			setForecastValues( *pWeather, vTokens, fromTempSystem, fromSpeedSystem );
			setForecastExtraValues( *pWeather,
			                        data.value( QString("Forecast Extra Day %1").arg(iDay) ).toString(),
			                        fromTempSystem, fromSpeedSystem );
		}
	}

	//--- Set the sun for all forecast days - this is an accuweather specific key. ---
	for( int iDay = 1; iDay < cityInfo.days().count(); ++iDay )
		setForecastSun( *cityInfo.days().at(iDay),
					    data.value( QString("Forecast Sun %1").arg(iDay) ).toString() );

	/***  Add the current weather informations to the first day ***
	*/
	QString sTmp;
	YawpDay * pDay = NULL;
	if( cityInfo.days().count() == 0 )
	{
		pDay = new YawpDay;
		pDay->setDate( QDate::currentDate() );
		cityInfo.days().append( pDay );
	}
	else
		pDay = cityInfo.days().at(0);

	pDay->setSunrise( QTime::fromString(data.value("Sunrise At").toString(), "h:m") );
	pDay->setSunset(  QTime::fromString(data.value("Sunset At").toString(), "h:m") );

	YawpWeather & weather
		= (pDay->hasNightValues() && !Utils::IsDayTime(pDay->sunrise(), pDay->sunset()) ?
			pDay->nightWeather() : pDay->weather());

	weather.setCurrentIconName( mapConditionIcon( data.value("Condition Icon").toString() ) );
	weather.setCurrentTemperature( d->convertTemp(  data.value("Temperature").toString(), fromTempSystem ) );
	weather.setWindSpeed( d->convertSpeed( data.value("Wind Speed").toString(),  fromSpeedSystem ) );
	weather.setDewpoint( d->convertTemp( data.value("Dewpoint").toString(), fromDewTempSystem ) );
	weather.setVisibility( d->convertDistance( data.value("Visibility").toString(), fromVisibilitySystem ) );

	weather.setPressure( d->convertPressure( data.value("Pressure").toString(), fromPressureSystem ) );
	sTmp = data.value("Pressure Tendency").toString();
	if( d->checkStringValue( sTmp ) )
		weather.setPressureTendency( sTmp );

	sTmp = data.value("Wind Direction").toString();
	if( d->checkStringValue( sTmp) )
		weather.setWindDirection( sTmp );

	sTmp = data.value("Current Conditions").toString();
	if( d->checkStringValue( sTmp ) )
		weather.setDescription( sTmp );

	if( data.contains("Humidity") )
		weather.setHumidity( (short)(data.value("Humidity").toString().replace("%", " ").toDouble()) );

	d->setUVValues( weather, data.value("UV Index").toString(), data.value("UV Rating").toString() );


	//--- remove all old days ---
	//
	bool bRemovedOldDays( false );
	currDate = QDate::currentDate();
	QDate yesterday = currDate.addDays(-1);
	while( cityInfo.days().count() > 0 )
	{
		YawpDay * pDay = cityInfo.days().at(0);
		if( pDay->date() >= currDate )
		{
			break;
		}
		else if( pDay->date() == yesterday && pDay->sunrise().isValid() &&
		    QTime::currentTime() < pDay->sunrise() )
		{
			break;
		}
		else
		{
			bRemovedOldDays = true;
			delete cityInfo.days().takeFirst();
		}
	}

	//--- Add YawpWeather::SunriseSunset property and sort the properties ---
	QList<YawpDay *>::iterator itDay = cityInfo.days().begin();
	for( ; itDay != cityInfo.days().end(); ++itDay )
	{
		YawpDay * pDay = (*itDay);

		if( (*itDay)->sunrise().isValid() || (*itDay)->sunset().isValid() )
		{
			pDay->weather().addProperty( Yawp::SunriseSunset );
			pDay->nightWeather().addProperty( Yawp::SunriseSunset );
		}
		sortPropertyKeys( pDay->weather() );

		if( pDay->hasNightValues() )
			sortPropertyKeys( pDay->nightWeather() );
	}

	if( !bRemovedOldDays )
	{
		//--- Add the satellite map to the city ---
		if( data.contains("Satellite Map") )
		{
			QImage img( data.value("Satellite Map").value<QImage>() );
			if( !img.isNull() )
				cityInfo.setSatelliteImage( img );
		}
	}
	return true;
}

bool
WeatherDataProcessor::updateCountryInfo( CityWeather & cityInfo, const Plasma::DataEngine::Data & data ) const
{
	/*   Some weatherengines provides more informations for a certain location
	 *   doing a data request than searching for a city, so we update the locationinformations here.
	 */
	bool bUpdate(false);
	QString sCountry, sCountryCode;
	sCountry = data.value("Country").toString();
	if( !sCountry.isEmpty() &&
	    cityInfo.country().compare( sCountry ) != 0 &&
        Utils::GetCountryCode( sCountry, sCountryCode ) )
	{
		cityInfo.setCountry( sCountry );
		cityInfo.setCountryCode( sCountryCode );
		bUpdate = true;
	}
	else if( cityInfo.countryCode().isEmpty() || cityInfo.country().isEmpty() )
	{
		QString sCity, sDistrict;

		if( cityInfo.countryCode().isEmpty() && data.contains("Place") )
		{
			Utils::ExtractLocationInfo(data.value("Place").toString(), sCity, sDistrict, sCountry);
			if( cityInfo.country().isEmpty() && !sCountry.isEmpty() )
			{
				cityInfo.setCountry( sCountry );
				bUpdate = true;
			}
			if( cityInfo.countryCode().isEmpty() && Utils::GetCountryCode( sCountry, sCountryCode ) )
			{
				cityInfo.setCountryCode( sCountryCode );
				bUpdate = true;
			}
		}
	}
	return bUpdate;
}

bool
WeatherDataProcessor::saveData( const CityWeather & cityInfo, const Plasma::DataEngine::Data & data ) const
{
	if( !cityInfo.isValid() || cityInfo.days().count() == 0 )
		return false;

	bool bReturn = false;
	QFile file( d->getSourceCacheFileName(cityInfo) );
	if( file.open(QIODevice::WriteOnly|QIODevice::Truncate) )
	{
		QDataStream ds(&file);
		ds << cityInfo.days().at(0)->date();
		ds << (quint64)(cityInfo.days().count());
		Plasma::DataEngine::Data::const_iterator it = data.begin();
		for( ; it != data.end(); ++it )
			ds << it.key() << it.value();
		file.close();
		bReturn = true;
	}
	return bReturn;
}

bool
WeatherDataProcessor::loadData( CityWeather & cityInfo ) const
{
	if( !cityInfo.isValid() )
		return false;

	bool bReturn = false;
	QFile file( d->getSourceCacheFileName(cityInfo) );
	if( file.open(QIODevice::ReadOnly) )
	{
		QDataStream ds(&file);
		QDate date;
		quint64 iForecasts;
		ds >> date >> iForecasts;
		if( date.addDays(iForecasts-1) >= QDate::currentDate() )
		{
			Plasma::DataEngine::Data data;
			QString sKey;
			QVariant value;
			while( !ds.atEnd() )
			{
				ds >> sKey >> value;
				data.insert( sKey, value );
			}
			updateLocation( cityInfo, data );
			bReturn = true;
		}
		file.close();
	}
	return bReturn;
}

void
WeatherDataProcessor::createDetailsPropertyMap( const QList<Yawp::DetailsProperty> & vProperties )
{
	d->vDetailsPropertyMap.clear();
	QList<Yawp::DetailsProperty>::const_iterator itList = vProperties.begin();
	short iPos(0);
	for( ; itList != vProperties.end(); ++itList, ++iPos )
	{
		d->vDetailsPropertyMap[ *itList ] = iPos;
	}
}

bool
WeatherDataProcessor::sortPropertyKeys( YawpWeather & weather ) const
{
	if( d->vDetailsPropertyMap.isEmpty() )
		return false;
	if( weather.propertyKeys().count() <= 1 )	// nothing to sort
		return true;

	QMap<short, Yawp::DetailsProperty> vMap;
	QList<Yawp::DetailsProperty>::const_iterator itList = weather.propertyKeys().begin();
	for( ; itList != weather.propertyKeys().end(); ++itList )
	{
		if( d->vDetailsPropertyMap.contains( *itList ) )
		{
			short iIndex = d->vDetailsPropertyMap.value( *itList );
			vMap[ iIndex ] = *itList;
		}
	}

	weather.propertyKeys().clear();
	QMap<short, Yawp::DetailsProperty>::const_iterator itMap = vMap.begin();
	for( ; itMap != vMap.end(); ++itMap )
		weather.addProperty( itMap.value() );
	return true;
}


bool
WeatherDataProcessor::setForecastValues(
		YawpWeather                   & weather,
		const QStringList             & vForecast,
		YAWP_TEMPERATURE_UNIT           fromTempSystem,
		YAWP_SPEED_UNIT                 fromSpeedSystem ) const
{
	Q_UNUSED( fromSpeedSystem );

	bool bReturn(false);
	if( vForecast.count() >= 5 )
	{
		weather.setIconName( mapConditionIcon( vForecast.at(1) ) );
		weather.setDescription( vForecast.at(2) );
		weather.setHighTemperature( d->convertTemp(vForecast.at(3), fromTempSystem) );
		weather.setLowTemperature(  d->convertTemp(vForecast.at(4), fromTempSystem) );

		bReturn = true;
	}
	return bReturn;
}

bool
WeatherDataProcessor::setForecastExtraValues(
		YawpWeather                   & weather,
		const QString                 & sExtras,
		YAWP_TEMPERATURE_UNIT           fromTempSystem,
		YAWP_SPEED_UNIT                 fromSpeedSystem ) const
{
	if( sExtras.isEmpty() )
		return false;

	bool bReturn(false);
	QStringList vForecast = sExtras.split("|", QString::SkipEmptyParts);

	if( vForecast.count() >= 8 )
	{
		weather.setWindSpeed( d->convertSpeed(vForecast.at(1), fromSpeedSystem) );
		if( d->checkStringValue( vForecast.at(2) ) )
			weather.setWindDirection( vForecast.at(2) );
		d->setUVValues( weather, vForecast.at(4), vForecast.at(5) );
		weather.setTemperatureRealFeelHigh( d->convertTemp(vForecast.at(6), fromTempSystem) );
		weather.setTemperatureRealFeelLow(  d->convertTemp(vForecast.at(7), fromTempSystem) );

		bReturn = true;
	}
	return bReturn;
}

bool
WeatherDataProcessor::setForecastSun( YawpDay & day, const QString & sValue ) const
{
	QStringList vSun = sValue.split("|", QString::SkipEmptyParts);

	if( vSun.count() >= 3 )
	{
		day.setSunrise( QTime::fromString(vSun.at(1), "hh:mm") );
		day.setSunset( QTime::fromString(vSun.at(2), "hh:mm") );
		return true;
	}
	return false;
}

void
WeatherDataProcessor::Private::setUVValues( YawpWeather & weather,
                                            const QString & sUVIndex,
                                            const QString & sUVRating ) const
{
	float fValue = parseFloat( sUVIndex );
	if( fValue < 1.0 || fValue > 11.0 )
		return;
	short iIndex = (short)fValue;
	weather.setUVIndex( iIndex );
	if( checkStringValue( sUVRating ) )
		weather.setUVRating( i18n(sUVRating.toUtf8().constData()) );
	else
	{
		//--- according to http://www.epa.gov/sunwise/uviscale.html ---
		if( iIndex <= 2 )
			weather.setUVRating( i18nc("UV Index Low", "Low") );
		else if( iIndex >= 3 && iIndex <= 5 )
			weather.setUVRating( i18nc("UV Index Moderate", "Moderate") );
		else if( iIndex >=6 && iIndex <= 7 )
			weather.setUVRating( i18nc("UV Index High", "High") );
		else if( iIndex >= 8 && iIndex <= 10 )
			weather.setUVRating( i18nc("UV Index Very High", "Very High") );
		else if( iIndex >= 11 )
			weather.setUVRating( i18nc("UV Index Extreme", "Extreme") );
	}
}

bool
WeatherDataProcessor::Private::checkStringValue( const QString & value ) const
{
	if( value.isEmpty() ||
	    value.compare("N/A", Qt::CaseInsensitive) == 0 ||
	    value.compare("N/U", Qt::CaseInsensitive) == 0 )
	{
		return false;
	}
	return true;
}

float
WeatherDataProcessor::Private::parseFloat( const QString & value ) const
{
	if( !checkStringValue( value ) )
		return FLT_MAX;

	bool bOk(false);
	float fValue = value.toFloat( &bOk );
	if( bOk )
		return fValue;
	return FLT_MAX;
}

short
WeatherDataProcessor::Private::convertDistance( const QString & value,
												YAWP_DISTANCE_UNIT fromSystem ) const
{
	float fValue = parseFloat( value );
	if( fValue == FLT_MAX )
		return SHRT_MAX;

	if( fromSystem != distanceSystem )
	{
#if KDE_IS_VERSION(4,3,70)
		KUnitConversion::Value v( (double) fValue, fromSystem);
		fValue = v.convertTo(distanceSystem).number();
#elif KDE_VERSION_MINOR >= 3
		fValue = WeatherUtils::convertDistance( fValue, fromSystem, distanceSystem );
#else
		fValue = WeatherUtils::convert( fValue, fromSystem, distanceSystem );
#endif
	}
	return (short)qRound(fValue);
}

short
WeatherDataProcessor::Private::convertPressure( const QString & value,
												YAWP_PRESSURE_UNIT fromSystem ) const
{
	float fValue = parseFloat( value );
	if( fValue == FLT_MAX )
		return SHRT_MAX;
	if( fromSystem != pressureSystem )
	{
#if KDE_IS_VERSION(4,3,70)
		KUnitConversion::Value v( (double) fValue, fromSystem);
		fValue = v.convertTo(pressureSystem).number();
#elif KDE_VERSION_MINOR >= 3
		fValue = WeatherUtils::convertPressure( fValue, fromSystem, pressureSystem );
#else
		fValue = WeatherUtils::convert( fValue, fromSystem, pressureSystem );
#endif
	}
	return (short)qRound(fValue);
}


short
WeatherDataProcessor::Private::convertTemp( const QString & value,
											YAWP_TEMPERATURE_UNIT fromSystem ) const
{
	float fValue = parseFloat( value );
	if( fValue == FLT_MAX )
		return SHRT_MAX;

	if( fromSystem != temperatureSystem )
	{
#if KDE_IS_VERSION(4,3,70)
		KUnitConversion::Value v( (double) fValue, fromSystem);
		fValue = v.convertTo(temperatureSystem).number();
#elif KDE_VERSION_MINOR >= 3
		fValue = WeatherUtils::convertTemperature( fValue, fromSystem, temperatureSystem );
#else
		fValue = WeatherUtils::convert( fValue, fromSystem, temperatureSystem );
#endif
	}
	return (short)qRound(fValue);
}

short
WeatherDataProcessor::Private::convertSpeed( const QString & value,
											 YAWP_SPEED_UNIT fromSystem ) const
{
	float fValue = parseFloat( value );
	if( fValue == FLT_MAX )
		return SHRT_MAX;

	if( fromSystem != speedSystem )
	{
#if KDE_IS_VERSION(4,3,70)
		KUnitConversion::Value v( (double) fValue, fromSystem);
		fValue = v.convertTo(speedSystem).number();
#elif KDE_VERSION_MINOR >= 3
		fValue = WeatherUtils::convertSpeed( fValue, fromSystem, speedSystem );
#else
		/*   There is a bug - WeatherUtils can not convert speed values
		 *   from one to another unit system. Therefore i convert the units to distances.
		 *   I do not know if we need this in KDE 4.3 as well.
		 *   If this is the case we might want to convert the from and to speed unit systems in
		 *   function WeatherServiceModel::collectWeatherInformations() and pass them to this function...
		 */
		WeatherUtils::Unit from = convertSpeed2Distance( fromSystem );
		WeatherUtils::Unit to   = convertSpeed2Distance( speedSystem );
		fValue = WeatherUtils::convert( fValue, from, to );
#endif
	}
	return (short)qRound(fValue);
}

#if KDE_VERSION_MINOR <= 2
WeatherUtils::Unit
WeatherDataProcessor::Private::convertSpeed2Distance( WeatherUtils::Unit fromSpeed ) const
{
	WeatherUtils::Unit toDistance;
	switch( fromSpeed )
	{
	case WeatherUtils::MilesAnHour:      toDistance = WeatherUtils::Miles; break;
	case WeatherUtils::KilometersAnHour: toDistance = WeatherUtils::Kilometers; break;
	default:
		toDistance = fromSpeed;
	}
	return toDistance;
}
#endif

bool
WeatherDataProcessor::Private::isNightTime( QString & sShortDay ) const
{
	bool bNight(false);

	int iPos = sShortDay.lastIndexOf(QChar(' '));
	if( iPos > 0 )
	{
		QString sNightTag( sShortDay.right(sShortDay.length()-iPos-1) );
		sShortDay = sShortDay.left(iPos);

		if( sNightTag.compare("nt", Qt::CaseInsensitive) == 0 ||
			sNightTag.compare(i18n("nt"), Qt::CaseInsensitive) == 0 )
		{
			bNight = true;
		}
		else if( sNightTag.compare("night", Qt::CaseInsensitive) == 0 ||
			sNightTag.compare(i18n("night"), Qt::CaseInsensitive) == 0 )
		{
			bNight = true;
		}
	}
	return bNight;
}

bool
WeatherDataProcessor::Private::findDateForWeekday( QDate & currDate, const QString & sLookingDayName ) const
{
	for(int iOffset = 0; iOffset < 6; ++iOffset )
	{
		QString sCurrDateName( currDate.toString("ddd") );

		if( sLookingDayName.contains( sCurrDateName ) ||
		    sLookingDayName.contains( i18n(sCurrDateName.toUtf8().constData()) ) )
		{
			return true;
		}
		currDate = currDate.addDays(-1);
	}
	return false;
}

QString
WeatherDataProcessor::Private::getSourceCacheFileName( const CityWeather & cityInfo ) const
{
	QString sCityName( cityInfo.city() );
	for( int i=0; i < sCityName.length(); ++i )
	{
		QCharRef chr = sCityName[i];
		if( !chr.isLetterOrNumber() )
			chr = QChar(' ');
	}
	sCityName = sCityName.simplified().replace(QChar(' '), QChar('_'));
	QDir dir( CacheDirectory );
	return dir.absoluteFilePath( QString("%1_%2.dat").arg(cityInfo.provider()).arg(sCityName) );
}


bool
WeatherDataProcessor::Private::initIconMap(const QString & resource)
{
	vConditionIcons.clear();

	QFile file(resource);
	if( !file.open(QIODevice::ReadOnly) )
		return false;
	QTextStream stream(&file);
	QString line;
	QString icon;
	QString value;
	while(!stream.atEnd()) {
		line = stream.readLine().trimmed();
		if( line.isEmpty() || line.at(0) == '#' )
			continue;

		QStringList list( line.split( QLatin1String("="), QString::SkipEmptyParts ) );
		if( list.size() >= 2 )
		{
			QString sIcon( list.at(0).simplified() );
			QString sValue( list.at(1).simplified() );
			vConditionIcons[sIcon] = sValue;
		}
	}
	file.close();
	return true;
}

inline QString
WeatherDataProcessor::mapConditionIcon( const QString & iconName ) const
{
	return d->vConditionIcons.value( iconName, "weather-none-available" );
}
