/*************************************************************************\
*   Copyright (C) 2009 by Ulf Kreissig                                    *
*   udev@gmx.net                                                          *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
\*************************************************************************/

//--- LOCAL ---
#include "../config.h"
#include "countrymap.h"
#include "dlgaddcity.h"
#include "ionlistmodel.h"
#include "utils.h"
#include "logger/streamlogger.h"

//--- QT4 ---
#include <QPushButton>

//--- KDE4 ---
#include <kmessagebox.h>
#include <kprogressdialog.h>
#include <plasma/dataengine.h>

#define CityRole		Qt::UserRole
#define CountryRole		Qt::UserRole + 1
#define CountryCodeRole		Qt::UserRole + 2
#define ExtraDataRole		Qt::UserRole + 3
#define ProviderRole		Qt::UserRole + 4


DlgAddCity::DlgAddCity( QWidget * parent )
	: QDialog( parent )
{
	setupUi( this );

	bttnFind->setIcon( KIcon("edit-find") );
	buttonBox->button( QDialogButtonBox::Apply )->setIcon( KIcon("dialog-ok") );
	buttonBox->button( QDialogButtonBox::Cancel )->setIcon( KIcon("dialog-cancel") );

	//--- workarround for the not working accept signal ---
	connect(buttonBox,     SIGNAL(clicked(QAbstractButton *)),   this, SLOT(slotApplySelection(QAbstractButton *)));
	connect(bttnFind,      SIGNAL(released()),                   this, SLOT(slotFindLocations()));
	connect(editLocation,  SIGNAL(textChanged(const QString &)), this, SLOT(slotValidateTextInput(const QString &)));

	comboProvider->clear();
	comboProvider->setModel( IonListModel::instance() );
	bttnFind->setEnabled( false );
	slotValidateTextInput(editLocation->text());
	enableApply();
}

DlgAddCity::~DlgAddCity()
{
}

CityWeather
DlgAddCity::getSelectedCity() const
{
	dStartFunct();
	const QListWidgetItem * item = cityList->currentItem();
	CityWeather cityInfo;
	if( item )
	{
		cityInfo.setCity(        item->data(CityRole).toString() );
		cityInfo.setCountry(     item->data(CountryRole).toString() );
		cityInfo.setCountryCode( item->data(CountryCodeRole).toString() );
		cityInfo.setExtraData(   item->data(ExtraDataRole).toString() );
		cityInfo.setProvider(    item->data(ProviderRole).toString() );
	}
	dEndFunct();
	return cityInfo;
}

void
DlgAddCity::updateLocations( const QVariant & data )
{
	dStartFunct();
	QStringList vTokens = data.toString().split("|");
	if( vTokens.at(1).compare("valid") == 0 )
	{
		if( vTokens.at(2).compare("single") == 0 ||
		    vTokens.at(2).compare("multiple") == 0 )
		{
			int iPos = 3;
			QString sLocation, sExtra;

			dDebug() << vTokens;

			//--- go through all places and extract all informations ---
			while( iPos < vTokens.count() && vTokens.at(iPos).compare("place") == 0 )
			{
				sLocation = vTokens.at(iPos+1);
				sExtra.clear();

				//--- go through all attributes that belongs to the current place ---
				iPos+=2;
				while( iPos+1 < vTokens.count() && vTokens.at(iPos).compare("place") != 0 )
				{
					if( vTokens.at(iPos).compare("extra") == 0 )
						sExtra = vTokens.at(iPos+1);
					iPos+=2;
				}

				if( !sLocation.isEmpty() )
				{
					QListWidgetItem * item = new QListWidgetItem;
					item->setText(i18nc("City Country/State in the search dialog", "%1", sLocation));

					/*  We can not change the value sLocation, since some
					 *  providers or dataengines use this as a key to find the specific city!!!
					 */
					item->setData( CityRole,		sLocation );
					item->setData( ExtraDataRole,		sExtra );
					item->setData( ProviderRole,		vTokens.at(0) );

					QString sCity, sDistrict, sCountry, sCountryCode;
					Utils::ExtractLocationInfo(sLocation, sCity, sDistrict, sCountry);
					if( Utils::GetCountryCode(sCountry, sCountryCode) )
					{
						item->setData( CountryCodeRole, sCountryCode );
						item->setData( CountryRole,     sCountry );
						QPixmap pix = CountryMap::instance()->getPixmapForCountryCode(sCountryCode);
						item->setIcon(QIcon(pix));
					}
					cityList->addItem( item );
				}
			}
		}
	}
	else if( vTokens.at(1).compare("timeout") == 0 )
	{
		KMessageBox::error(this,
			i18n("The applet was not able to contact the server, please try again later."));
//		dDebug() << vTokens;
	}
	else
	{
		KMessageBox::error(this,
			i18n("The place '%1' is not valid. The weather-service is not able to find this place.",
			vTokens.at(3)));
//		dDebug() << vTokens;
	}
	dEndFunct();
}

void
DlgAddCity::enableApply()
{
	dStartFunct();
	QPushButton * bttnApply = buttonBox->button(QDialogButtonBox::Apply);
	bttnApply->setEnabled( cityList->count() > 0 );
	dEndFunct();
}

void
DlgAddCity::slotValidateTextInput(const QString & text)
{
	bttnFind->setEnabled( !text.isEmpty() );
}

void
DlgAddCity::slotFindLocations()
{
	dStartFunct();
	if( comboProvider->count() == 0 || editLocation->text().isEmpty() )
	{
		KMessageBox::sorry( this, i18n("You have to enter a city to search for.") );
	}
	else
	{
		cityList->clear();
		enableApply();
		bttnFind->setEnabled( false );

		m_pProgressDlg = new KProgressDialog(this, i18nc("Progress dialog caption", "Please wait..."),
			i18nc("Progress dialog description","%1 is retrieving the list of cities from the internet, "
			"please wait.",  YAWP_NAME));
		m_pProgressDlg->progressBar()->setMinimum(0);
		m_pProgressDlg->progressBar()->setMaximum(0);
		m_pProgressDlg->setAllowCancel(false);
		m_pProgressDlg->setModal(false);
		m_pProgressDlg->setAutoClose(false);
		m_pProgressDlg->setVisible(true);

		QString sProvider = comboProvider->itemData( comboProvider->currentIndex() ).toString();
        	QString sLocation = editLocation->text();//.toLower();
		QString sCommand  = QString("%1|validate|%2").arg(sProvider).arg(sLocation);
		IonListModel::instance()->engine()->connectSource(sCommand, this);
	}
	dEndFunct();
}

void
DlgAddCity::dataUpdated(const QString & sAction, const Plasma::DataEngine::Data & data)
{
	dStartFunct();
	IonListModel::instance()->engine()->disconnectSource(sAction, this);
	QStringList actionList = sAction.split("|");

	if( actionList.at(1).compare("validate") == 0 &&
	    data.value("validate").toBool() && data["validate"].type() == QVariant::String )
	{
		updateLocations(data["validate"]);

		bttnFind->setEnabled( true );

		m_pProgressDlg->setVisible(false);
		delete m_pProgressDlg;
		m_pProgressDlg = NULL;
	}
	enableApply();
	dEndFunct();
}

void
DlgAddCity::slotApplySelection( QAbstractButton * button )
{
	switch( buttonBox->buttonRole( button ) )
	{
	case QDialogButtonBox::ApplyRole:
		accept();
		break;
	case QDialogButtonBox::RejectRole:
		reject();
		break;
	default:
		break;
	}
}
