#!/usr/bin/python
#
# Copyright 2005 Lars Wirzenius (liw@iki.fi)
# Copyright 2009-2012 Holger Levsen (holger@layer-acht.org)
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA


"""Create HTML reports of piuparts log files

Lars Wirzenius <liw@iki.fi>
"""


import os
import sys
import time
import logging
import ConfigParser
import urllib
import shutil
import re
import string

# if python-rpy ain't installed, we don't draw fancy graphs
try:
  from rpy import *
except:
  pass

import piupartslib


CONFIG_FILE = "/etc/piuparts/piuparts.conf"


HTML_HEADER = """
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN">
 <html><head><meta http-equiv="Content-Type" content="text/html; charset=utf-8">
  <!-- Generated by piuparts-report __PIUPARTS_VERSION__ -->
  <title>
   $page_title
  </title>
  <link type="text/css" rel="stylesheet" href="$doc_root/style.css">
  <link rel="shortcut icon" href="$doc_root/favicon.ico">
 </head>

 <body>
 <div id="header">
   <h1 class="header">
    <a href="http://www.debian.org/">
     <img src="$doc_root/images/openlogo-nd-50.png" border="0" hspace="0" vspace="0" alt=""></a>
    <a href="http://www.debian.org/">
     <img src="$doc_root/images/debian.png" border="0" hspace="0" vspace="0" alt="Debian Project"></a>
    Quality Assurance
   </h1>
   <div id="obeytoyourfriend">Policy is your friend. Trust the Policy. Love the Policy. Obey the Policy.</div>
 </div>
 <hr>
<div id="main">
<table class="containertable">
 <tr class="containerrow" valign="top">
  <td class="containercell">
   <table class="lefttable">
    <tr class="titlerow">
     <td class="titlecell">
      General information
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="$doc_root/">About + News</a>
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="http://wiki.debian.org/piuparts" target="_blank">Overview</a>
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="http://wiki.debian.org/piuparts/FAQ" target="_blank">FAQ</a>
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="http://bugs.debian.org/src:piuparts" target="_blank">Bugs</a> / <a href="http://anonscm.debian.org/gitweb/?p=piuparts/piuparts.git;a=blob;f=TODO" target="_blank">ToDo</a>
     </td>
    </tr>
    <tr class="titlerow">
     <td class="titlecell">
      Documentation
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="$doc_root/doc/README.html" target="_blank">piuparts README</a>
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="$doc_root/doc/piuparts.1.html" target="_blank">piuparts manpage</a>
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      piuparts.d.o configuration:<br>
      <a href="http://anonscm.debian.org/gitweb/?p=piuparts/piuparts.git;hb=piatti;a=blob_plain;f=org/piuparts.debian.org/etc/piuparts.conf.piatti" target="_blank">piuparts.conf.piatti</a><br>
      <a href="http://anonscm.debian.org/gitweb/?p=piuparts/piuparts.git;hb=piatti;a=tree;f=org/piuparts.debian.org/etc/scripts" target="_blank">scripts</a>
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="$doc_root/bug_howto.html">How to file bugs</a>
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="http://www.debian.org/doc/debian-policy/" target="_blank">Debian policy</a>
     </td>
    </tr>
    <tr class="titlerow">
     <td class="alerttitlecell">
      Available reports
     </td>
    </tr>
    <tr>
     <td class="contentcell">
      <a href="http://bugs.debian.org/cgi-bin/pkgreport.cgi?tag=piuparts;users=debian-qa@lists.debian.org&amp;archive=both" target="_blank">Bugs filed</a>
     </td>
    </tr>
    $section_navigation
    <tr class="titlerow">
     <td class="titlecell">
      Other Debian QA efforts
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="http://edos.debian.net" target="_blank">EDOS tools</a>
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="http://lintian.debian.org" target="_blank">Lintian</a>
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="http://packages.qa.debian.org" target="_blank">Package Tracking System</a>
     </td>
    <tr class="normalrow">
     <td class="contentcell">
      <a href="http://udd.debian.org" target="_blank">Ultimate Debian Database</a>
     </td>
    </tr>
    <tr class="titlerow">
     <td class="titlecell">
      Last update
     </td>
    </tr>
    <tr class="normalrow">
     <td class="lastcell">
      $time
     </td>
    </tr>
   </table>
  </td>
  <td class="containercell">
"""


HTML_FOOTER = """
  </td>
 </tr>
</table>
</div>
 <hr>
 <div id="footer">
  <div>
   <a href="http://packages.qa.debian.org/piuparts" target="_blank">piuparts</a>
   is GPL2 <a href="http://packages.debian.org/changelogs/pool/main/p/piuparts/current/copyright" target="_blank">licenced</a>
   and was written by <a href="mailto:liw@iki.fi">Lars Wirzenius</a> and is now maintained by
   <a href="mailto:holger@debian.org">Holger Levsen</a> and
   <a href="mailto:piuparts-devel@lists.alioth.debian.org">others</a> using
   <a href="http://anonscm.debian.org/gitweb/?p=piuparts/piuparts.git" target="_blank">piuparts.git</a>.
   Weather icons are from the <a href="http://tango.freedesktop.org/Tango_Icon_Library" target="_blank">Tango Icon Library</a>.
   <a href="http://validator.w3.org/check?uri=referer">
    <img border="0" src="$doc_root/images/valid-html401.png" alt="Valid HTML 4.01!" height="15" width="80" align="middle">
   </a>
   <a href="http://jigsaw.w3.org/css-validator/check/referer">
    <img border="0" src="$doc_root/images/w3c-valid-css.png" alt="Valid CSS!"  height="15" width="80" align="middle">
   </a>
  </div>
 </div>
</body>
</html>
"""


LOG_LIST_BODY_TEMPLATE = """
   <table class="righttable">
    <tr class="titlerow">
     <td class="$title_style" colspan="2">
      $title in $section
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell2" colspan="2">
      $preface
      The list has $count packages, with $versioncount total versions.
     </td>
    </tr>
    $logrows
   </table>
"""


STATE_BODY_TEMPLATE = """
   <table class="righttable">
    <tr class="titlerow">
     <td class="alerttitlecell">
      Packages in state "$state" in $section $aside
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell2">
      <ul>
       $list
      </ul>
     </td>
    </tr>
   </table>
"""


SECTION_INDEX_BODY_TEMPLATE = """
   <table class="righttable">
    <tr class="titlerow">
     <td class="titlecell" colspan="3">
      $section statistics
     </td>
    </tr>
    <tr class="normalrow">
     <td class="contentcell2" colspan="3">
      $description
     </td>
    </tr>
    <tr class="titlerow">
     <td class="alerttitlecell" colspan="3">
      Binary packages per state
     </td>
    </tr>
    $tablerows
    <tr class="titlerow">
     <td class="titlecell" colspan="3">
      URL to Packages file
     </td>
    </tr>
     <tr class="normalrow">
     <td class="contentcell2" colspan="3">
      <code>$packagesurl</code>
     </td>
    </tr>
   </table>
"""

MAINTAINER_BODY_TEMPLATE = """
   <table class="righttable">
    <tr class="titlerow">
     <td class="titlecell" colspan="6">
      $maintainer
     </td>
    </tr>
    $distrolinks
    $rows
   </table>
"""


SOURCE_PACKAGE_BODY_TEMPLATE = """
   <table class="righttable">
    $rows
   </table>
"""

ANALYSIS_BODY_TEMPLATE = """
   <table class="righttable">
    $rows
   </table>
"""


title_by_dir = {
    "pass": "PASSED piuparts logs",
    "fail": "Failed UNREPORTED piuparts logs",
    "bugged": "Failed REPORTED piuparts logs",
    "affected": "Failed AFFECTED piuparts logs",
    "reserved": "RESERVED packages",
    "untestable": "UNTESTABLE packages",
}


desc_by_dir = {
    "pass": "Log files for packages that have PASSED testing.",
    "fail": "Log files for packages that have FAILED testing. " +
            "Bugs have not yet been reported.",
    "bugged": "Log files for packages that have FAILED testing. " +
              "Bugs have been reported, but not yet fixed.",
    "affected": "Log files for packages that have dependencies FAILED testing. " +
              "Bugs have been reported, but not yet fixed.",
    "reserved": "Packages that are RESERVED for testing on a node in a " +
                "distributed piuparts network.",
    "untestable": "Log files for packages that have are UNTESTABLE with " +
                  "piuparts at the current time.",
}

state_by_dir = {
    "pass": "successfully-tested",
    "fail": "failed-testing",
    "bugged": "failed-testing",
    "affected": "failed-testing",
    "reserved": "waiting-to-be-tested",
    "untestable": "cannot-be-tested",
}

# better use XX_name.tpl and get the linktarget from the template
# (its a substring of the <title> of the that template
# maintaining this list is errorprone and tiresome
linktarget_by_template = [
    ("initdscript_lsb_header_issue.tpl", "but logfile contains update-rc.d issues"),
    ("command_not_found_issue.tpl", "but logfile contains 'command not found'"),
    ("alternatives_after_purge_issue.tpl", "but logfile contains forgotten alternatives"),
    ("owned_files_after_purge_issue.tpl", "but logfile contains owned files existing after purge"),
    ("unowned_files_after_purge_issue.tpl", "but logfile contains unowned files after purge"),
    ("maintainer_script_issue.tpl", "but logfile contains maintainer script failures"),
    ("installs_over_symlink_issue.tpl", "but package installs something over existing symlinks"),
    ("broken_symlinks_issue.tpl", "but logfile contains 'broken symlinks'"),
    ("packages_have_been_kept_back_issue.tpl", "but logfile contains 'packages have been kept back'"),

    ("dependency_error.tpl", "due to unsatisfied dependencies"),
    ("packages_have_been_kept_back_error.tpl", "...and logfile also contains 'packages have been kept back'"),
    ("command_not_found_error.tpl", "due to a 'command not found' error"),
    ("files_in_usr_local_error.tpl", "due to files in /usr/local"),
    ("overwrite_other_packages_files_error.tpl", "due to overwriting other packages files"),
    ("debsums_mismatch_error.tpl", "due to modifying conffiles or other shipped files"),
    ("alternatives_after_purge_error.tpl", "due to forgotten alternatives after purge"),
    ("owned_files_by_many_packages_error.tpl", "due to owned files by many packages"),
    ("owned_files_after_purge_error.tpl", "due to owned files existing after purge"),
    ("unowned_files_after_purge_error.tpl", "due to unowned files after purge"),
    ("modified_files_after_purge_error.tpl", "due to files having been modified after purge"),
    ("disappeared_files_after_purge_error.tpl", "due to files having disappeared after purge"),
    ("diversion_error.tpl", "due to diversions being modified after purge"),
    ("processes_running_error.tpl", "due to leaving processes running behind"),
    ("excessive_output_error.tpl", "due to being terminated after excessive output"),
    ("conffile_prompt_error.tpl", "due to prompting due to modified conffiles"),
    ("db_setup_error.tpl", "due to failing to setup a database"),
    ("insserv_error.tpl", "due to a problem with insserv"),
    ("problems_and_no_force_error.tpl", "due to not enough force being used"),
    ("pre_depends_error.tpl", "due to a problem with pre-depends"),
    ("pre_installation_script_error.tpl", "due to pre-installation maintainer script failed"),
    ("post_installation_script_error.tpl", "due to post-installation maintainer script failed"),
    ("pre_removal_script_error.tpl", "due to pre-removal maintainer script failed"),
    ("post_removal_script_error.tpl", "due to post-removal maintainer script failed"),
    ("unknown_purge_error.tpl", "due to purge failed due to an unknown reason"),
    ("cron_error_after_removal_error.tpl", "due to errors from cronjob after removal"),
    ("logrotate_error_after_removal_error.tpl", "due to errors from logrotate after removal"),
    ("installs_over_symlink_error.tpl", "...and package installs something over existing symlinks"),
    ("broken_symlinks_error.tpl", "...and logfile also contains 'broken symlinks'"),
    ("unknown_failures.tpl", "due to unclassified failures"),
]


class Config(piupartslib.conf.Config):

    def __init__(self, section="report", defaults_section=None):
        self.section = section
        piupartslib.conf.Config.__init__(self, section,
            {
                "sections": "report",
                "output-directory": "html",
                "master-directory": ".",
                "description": "",
                "mirror": None,
                "distro": None,
                "area": None,
                "arch": None,
                "upgrade-test-distros": None,
                "max-reserved": 1,
                "doc-root": "/",
            },
            defaults_section=defaults_section)


def setup_logging(log_level, log_file_name):
    logger = logging.getLogger()
    logger.setLevel(log_level)

    formatter = logging.Formatter(fmt="%(asctime)s %(message)s",
                                  datefmt="%H:%M")

    handler = logging.StreamHandler(sys.stderr)
    handler.setFormatter(formatter)
    logger.addHandler(handler)

    if log_file_name:
        handler = logging.FileHandler(log_file_name)
        logger.addHandler(handler)


def html_protect(vstr):
    vstr = "&amp;".join(vstr.split("&"))
    vstr = "&lt;".join(vstr.split("<"))
    vstr = "&gt;".join(vstr.split(">"))
    vstr = "&#34;".join(vstr.split('"'))
    vstr = "&#39;".join(vstr.split("'"))
    return vstr


def emphasize_reason(reason):
    bad_states = [
        #"successfully-tested",
        "failed-testing",
        "cannot-be-tested",
        #"essential-required",
        #"waiting-to-be-tested",
        #"waiting-for-dependency-to-be-tested",
        "dependency-failed-testing",
        "dependency-cannot-be-tested",
        "dependency-does-not-exist",
        "circular-dependency",  # obsolete
        "unknown",
        "unknown-preferred-alternative",  # obsolete
        "no-dependency-from-alternatives-exists",  # obsolete
        "does-not-exist",
    ]
    if reason in bad_states:
      reason = "<em>"+reason+"</em>"
    return reason


def source_subdir(source):
    if source[:3] == "lib":
      return source[:4]
    else:
      return source[:1]


def maintainer_subdir(maintainer):
    return maintainer.lower()[:1]


def find_files_with_suffix(vdir, suffix):
    pairs = []  # (mtime, name)
    for name in os.listdir(vdir):
        if name.endswith(suffix):
            try:
                if os.path.isfile(os.path.join(vdir, name)):
                    mtime = os.path.getmtime(os.path.join(vdir, name))
                    pairs.append((mtime, name))
            except OSError:
                pass
    # sort by mtime
    return [x[1] for x in sorted(pairs)]

def update_file(source, target):
    if os.path.exists(target):
        try:
            aa = os.stat(source)
            bb = os.stat(target)
        except OSError:
            pass
        else:
            if aa.st_size == bb.st_size and aa.st_mtime < bb.st_mtime:
                return
    try:
        shutil.copyfile(source, target)
    except IOError as (errno, strerror):
        logging.error("failed to copy %s to %s: I/O error(%d): %s" \
                       % (source, target, errno, strerror))


def copy_logs(logs_by_dir, output_dir):
    for vdir in logs_by_dir:
        fulldir = os.path.join(output_dir, vdir)
        if not os.path.exists(fulldir):
            os.makedirs(fulldir)
        for basename in logs_by_dir[vdir]:
            source = os.path.join(vdir, basename)
            target = os.path.join(fulldir, basename)
            update_file(source, target)

def remove_old_logs(logs_by_dir, output_dir):
    for vdir in logs_by_dir:
        fulldir = os.path.join(output_dir, vdir)

        # convert logs_by_dir array to a dict to avoid linear search
        logs_dict = {}
        for log in logs_by_dir[vdir]:
            logs_dict[log] = 1

        if os.path.exists(fulldir):
            for basename in os.listdir(fulldir):
                if not basename in logs_dict:
                    os.remove(os.path.join(fulldir, basename))


def write_file(filename, contents):
    f = file(filename, "w")
    f.write(contents)
    f.close()


def append_file(filename, contents):
    f = file(filename, "a")
    f.write(contents)
    f.close()

def read_file(filename):
    f = file(filename, "r")
    l = f.readlines()
    f.close()
    return l

def create_section_navigation(section_names,current_section, doc_root):
    tablerows = ""
    for section in section_names:
        tablerows += ("<tr class=\"normalrow\"><td class=\"contentcell\"><a href='%s/%s'>%s</a></td></tr>\n") % \
                          (doc_root, html_protect(section), html_protect(section))
    tablerows += "<tr><td class=\"contentcell\"><a href=\"%s/%s/maintainer/\">by maintainer / uploader</a></td></tr>\n" \
                 % (doc_root, current_section)
    tablerows += "<tr><td class=\"contentcell\"><a href=\"%s/%s/source/\">by source package</a></td></tr>\n" \
                 % (doc_root, current_section)
    return tablerows;

def get_email_address(maintainer):
    email = "INVALID maintainer address: %s" % (maintainer)
    try:
      m = re.match(r"(.+)(<)(.+@.+)(>)", maintainer)
      email = m.group(3)
    except:
      pass
    return email


class Section:

    def __init__(self, section, master_directory, doc_root):
        self._config = Config(section=section, defaults_section="global")
        self._config.read(CONFIG_FILE)
        logging.debug("-------------------------------------------")
        logging.debug("Running section " + self._config.section)

        self._master_directory = os.path.abspath(os.path.join(master_directory, \
                                                         self._config.section))
        if not os.path.exists(self._master_directory):
            logging.debug("Warning: %s did not exist, now created. Did you ever let the slave work?"
                           % self._master_directory)
            os.makedirs(self._master_directory)

        self._doc_root = doc_root

        logging.debug("Loading and parsing Packages file")
        logging.info("Fetching %s" % self._config.get_packages_url())
        packages_file = piupartslib.open_packages_url(self._config.get_packages_url())
        self._binary_db = piupartslib.packagesdb.PackagesDB(prefix=self._master_directory)
        self._binary_db.read_packages_file(packages_file)
        self._binary_db.calc_rrdep_counts()

        packages_file.close()

        logging.info("Fetching %s" % self._config.get_sources_url())
        sources_file = piupartslib.open_packages_url(self._config.get_sources_url())
        self._source_db = piupartslib.packagesdb.PackagesDB()
        self._source_db.read_packages_file(sources_file)
        sources_file.close()

        self._log_name_cache = {}

    def write_log_list_page(self, filename, title, preface, logs):
        packages = {}
        for pathname, package, version in logs:
            packages[package] = packages.get(package, []) + [(pathname, version)]

        names = packages.keys()
        names.sort()
        lines = []
        version_count = 0
        for package in names:
            versions = []
            for pathname, version in packages[package]:
                version_count += 1
                versions.append("<a href=\"%s\">%s</a>" %
                                (html_protect(pathname),
                                 html_protect(version)))
            line = "<tr class=\"normalrow\"><td class=\"contentcell2\">%s</td><td class=\"contentcell2\">%s</td></tr>" % \
                                (html_protect(package),
                                 ", ".join(versions))
            lines.append(line)

        if "FAIL" in preface:
          title_style="alerttitlecell"
        else:
          title_style="titlecell"

        htmlpage = string.Template(HTML_HEADER + LOG_LIST_BODY_TEMPLATE + HTML_FOOTER)
        f = file(filename, "w")
        f.write(htmlpage.safe_substitute( {
                    "page_title": html_protect(title+" in "+self._config.section),
                    "section_navigation":
                       create_section_navigation(self._section_names,
                                                 self._config.section,
                                                 self._doc_root),
                    "time": time.strftime("%Y-%m-%d %H:%M %Z"),
                    "title": html_protect(title),
                    "section": html_protect(self._config.section),
                    "title_style": title_style,
                    "preface": preface,
                    "count": len(packages),
                    "versioncount": version_count,
                    "logrows": "".join(lines),
                    "doc_root": self._doc_root,
                }))
        f.close()


    def print_by_dir(self, output_directory, logs_by_dir):
        for vdir in logs_by_dir:
            vlist = []
            for basename in logs_by_dir[vdir]:
                assert basename.endswith(".log")
                assert "_" in basename
                package, version = basename[:-len(".log")].split("_")
                vlist.append((os.path.join(vdir, basename), package, version))
            self.write_log_list_page(os.path.join(output_directory, vdir + ".html"),
                                title_by_dir[vdir],
                                desc_by_dir[vdir], vlist)

    def find_links_to_logs(self, package_name, dirs, logs_by_dir):
        links = []
        for vdir in dirs:

          # avoid linear search against log file names by caching in a dict
          #
          # this cache was added to avoid a very expensive linear search
          # against the arrays in logs_by_dir. Note that the use of this cache
          # assumes that the contents of logs_by_dir is invarient across calls
          # to find_links_to_logs()
          #
          if vdir not in self._log_name_cache:
              self._log_name_cache[vdir] = {}

              for basename in logs_by_dir[vdir]:
                  if basename.endswith(".log"):
                      package, version = basename[:-len(".log")].split("_")

                      self._log_name_cache[vdir][package] = version

          if vdir == "fail":
            style = " class=\"needs-bugging\""
          else:
            style = ""

          if package_name in self._log_name_cache[vdir]:
              basename = package_name \
                       + "_" \
                       + self._log_name_cache[vdir][package_name] \
                       + ".log"

              links.append("<a href=\"%s/%s\"%s>%s</a>" % (
                      self._doc_root,
                      os.path.join(self._config.section, vdir, basename),
                      style,
                      html_protect(self._log_name_cache[vdir][package_name]),
                      ))

        return links

    def link_to_maintainer_summary(self, maintainer):
        email = get_email_address(maintainer)
        return "<a href=\"%s/%s/maintainer/%s/%s.html\">%s</a>" \
               % (self._doc_root,self._config.section,maintainer_subdir(email),
                  email,html_protect(maintainer))

    def link_to_uploaders(self, uploaders):
        link = ""
        for uploader in uploaders.split(","):
            link += self.link_to_maintainer_summary(uploader.strip()) + ", "
        return link[:-2]

    def link_to_source_summary(self, package_name):
        source_name = self._binary_db.get_control_header(package_name, "Source")
        link = "<a href=\"%s/%s/source/%s\">%s</a>" % (
                self._doc_root,
                self._config.section,
                source_subdir(source_name)+"/"+source_name+".html",
                html_protect(package_name))
        return link

    def link_to_state_page(self, section, package_name, link_target):
        if self._binary_db.has_package(package_name):
            state = self._binary_db.get_package_state(package_name)
            link = "<a href=\"%s/%s/%s\">%s</a>" % (
                self._doc_root,
                section,
                "state-"+state+".html"+"#"+package_name,
                link_target)
        else:
          if link_target == package_name:
            link = html_protect(package_name)
          else:
            link = "unknown-package"

        return link

    def links_to_logs(self, package_name, state, logs_by_dir):
        link = "N/A"
        dirs = ""

        if state == "successfully-tested":
          dirs = ["pass"]
        elif state == "failed-testing":
          dirs = ["fail", "bugged", "affected"]
        elif state == "cannot-be-tested":
          dirs = ["untestable"]

        if dirs != "":
          links = self.find_links_to_logs (package_name, dirs, logs_by_dir)
          link = ", ".join(links)

        if "/bugged/" in link or "/affected/" in link:
          link += " - <a href=\"http://bugs.debian.org/cgi-bin/pkgreport.cgi?package=" \
                  + package_name \
                  + "\" target=\"_blank\" class=\"bugged\">&nbsp;bug filed&nbsp;</a>\n"

        return link

    def write_counts_summary(self):
        logging.debug("Writing counts.txt")
        header = "date"
        current_day = "%s" % time.strftime("%Y%m%d")
        counts = current_day
        total = 0
        for state in self._binary_db.get_states():
            count = len(self._binary_db.get_pkg_names_in_state(state))
            header += ", %s" % state
            counts += ", %s" % count
            logging.debug("%s: %s" % (state, count))
            total += count
        logging.debug("total: %s" % total)
        logging.debug("source: %s" % len(self._source_db.get_all_packages()))
        header += "\n"
        counts += "\n"

        countsfile = os.path.join(self._output_directory, "counts.txt")
        if not os.path.isfile(countsfile):
          logging.debug("writing new file: %s" % countsfile)
          write_file(countsfile, header)
          last_line = ""
        else:
          last_line = read_file(countsfile)[-1]
        if not current_day in last_line:
            append_file(countsfile, counts)
            logging.debug("appending line: %s" % counts.strip())
        return total


    def create_maintainer_summaries(self, maintainers, source_data):
        logging.debug("Writing maintainer summaries in %s" % self._output_directory)
        maintainer_dir = os.path.join(self._output_directory, "maintainer")
        if not os.path.exists(maintainer_dir):
            os.mkdir(maintainer_dir)
        states = ["fail", "unknown", "pass"]
        for maintainer in maintainers.keys():
            sources = maintainers[maintainer]
            maintainer_subdir_path = os.path.join(maintainer_dir, maintainer_subdir(maintainer))
            if not os.path.exists(maintainer_subdir_path):
              os.mkdir(maintainer_subdir_path)
            rows = ""
            package_rows = ""
            packages = {}
            for state in states:
                packages[state] = []
            for source in sorted(sources):
                (state, sourcerows, binaryrows) = source_data[source]
                packages[state].append(source)
                package_rows += sourcerows + binaryrows

            for state in states:
                if len(packages[state]) > 0:
                    links = ""
                    for package in packages[state]:
                        links += "<a href=\"#%s\">%s</a> " % (package, package)
                else:
                    links = "&nbsp;"
                rows +=   "<tr class=\"normalrow\">" \
                        + "<td class=\"labelcell\">%s:</td>" % state \
                        + "<td class=\"contentcell2\">%s</td>" % len(packages[state]) \
                        + "<td class=\"contentcell2\" colspan=\"4\">%s</td>" % links \
                        + "</tr>\n"

            distrolinks = "<tr class=\"normalrow\">" \
                          + "<td class=\"labelcell\">other distributions: </td>" \
                          + "<td class=\"contentcell2\" colspan=\"5\">"

            for section in self._section_names:
              if section != self._config.section:
                distrolinks += "<a href=\"" \
                               + self._doc_root \
                               + "/" \
                               + section \
                               + "/maintainer/" \
                               + maintainer_subdir(maintainer) \
                               + "/" \
                               + maintainer \
                               + ".html\">" \
                               + html_protect(section) \
                               + "</a> "
            distrolinks += "</td></tr>"

            htmlpage = string.Template(HTML_HEADER + MAINTAINER_BODY_TEMPLATE + HTML_FOOTER)
            filename = os.path.join(maintainer_subdir_path, maintainer + ".html")
            f = file(filename, "w")
            f.write(htmlpage.safe_substitute( {
               "page_title": html_protect(  "Status of " \
                                          + maintainer \
                                          + " packages in " \
                                          + self._config.section),
               "maintainer": html_protect(maintainer+" in "+self._config.section),
               "distrolinks": distrolinks,
               "section_navigation": create_section_navigation(self._section_names,
                                                               self._config.section,
                                                               self._doc_root),
               "time": time.strftime("%Y-%m-%d %H:%M %Z"),
               "rows": rows + package_rows,
               "doc_root": self._doc_root,
             }))
            f.close()

    def create_source_summary (self, source, logs_by_dir):
        source_version = self._source_db.get_control_header(source, "Version")
        binaries = self._source_db.get_control_header(source, "Binary")
        maintainer = self._source_db.get_control_header(source, "Maintainer")
        uploaders = self._source_db.get_control_header(source, "Uploaders")
        current_version = self._source_db.get_control_header(source, "Version")

        success = True
        failed = False
        binaryrows = ""
        for binary in sorted([x.strip() for x in binaries.split(",") if x.strip()]):
          state = self._binary_db.get_package_state(binary)
          if state == "unknown":
            # Don't track udebs and binary packages on other archs.
            # The latter is a FIXME which needs parsing the Packages files from other archs too
            continue

          if not "waiting" in state and "dependency" in state:
            state_style="lightalertlabelcell"
          elif state == "failed-testing":
            state_style="lightlabelcell"
          else:
            state_style="labelcell"

          binaryrows +=   "<tr class=\"normalrow\">" \
                        + "<td class=\"labelcell\">Binary:</td>" \
                        + "<td class=\"contentcell2\">%s</td>" \
                          % binary\
                        + "<td class=\"%s\">piuparts-result:</td>" \
                          % state_style \
                        + "<td class=\"contentcell2\">%s %s</td>" \
                          % ( self.link_to_state_page(self._config.section,binary,state),
                              self.links_to_logs(binary, state, logs_by_dir) ) \
                        + "<td class=\"labelcell\">Version:</td>" \
                        + "<td class=\"contentcell2\">%s</td>" \
                          % html_protect(current_version) \
                        + "</tr>\n"

          if state not in ("successfully-tested", "essential-required"):
            success = False
          if state in ("failed-testing", "dependency-does-not-exist", "cannot-be-tested"):
            failed = True

        if binaryrows != "":
          source_state="unknown"

          if success: source_state="<img src=\"%s/images/sunny.png\">" % self._doc_root
          if failed:  source_state="<img src=\"%s/images/weather-severe-alert.png\">" % self._doc_root

          sourcerows =    "<tr class=\"titlerow\">" \
                        + "<td class=\"titlecell\" colspan=\"6\" id=\"%s\">%s in %s</td>" \
                          % (source, source, self._config.section) \
                        + "</tr>\n"

          sourcerows +=   "<tr class=\"normalrow\">" \
                        + "<td class=\"labelcell\">Source:</td>" \
                        + "<td class=\"contentcell2\">" \
                          + "<a href=\"http://packages.qa.debian.org/%s\" target=\"_blank\">%s</a>" \
                            % ( source, html_protect(source) ) \
                          + "</td>" \
                        + "<td class=\"labelcell\">piuparts summary:</td>" \
                        + "<td class=\"contentcell2\">%s</td>" \
                          % source_state \
                        + "<td class=\"labelcell\">Version:</td>" \
                        + "<td class=\"contentcell2\">%s</td>" \
                          % html_protect(source_version) \
                        + "</tr>\n"

          sourcerows += "<tr class=\"normalrow\">" \
                        + "<td class=\"labelcell\">Maintainer:</td>" \
                        + "<td class=\"contentcell2\" colspan=\"5\">%s</td>" \
                          % self.link_to_maintainer_summary(maintainer) \
                        + "</tr>\n"

          if uploaders:
            sourcerows += "<tr class=\"normalrow\">" \
                          + "<td class=\"labelcell\">Uploaders:</td>" \
                          + "<td class=\"contentcell2\" colspan=\"5\">%s</td>" \
                            % self.link_to_uploaders(uploaders) \
                          + "</tr>\n"

          source_summary_page_path = os.path.join( self._output_directory,
                                                  "source",
                                                   source_subdir(source))

          if not os.path.exists(source_summary_page_path):
             os.makedirs(source_summary_page_path)

          filename = os.path.join(source_summary_page_path, (source + ".html"))
          htmlpage = string.Template(HTML_HEADER + SOURCE_PACKAGE_BODY_TEMPLATE + HTML_FOOTER)

          f = file(filename, "w")
          f.write(htmlpage.safe_substitute( {
             "page_title": html_protect("Status of source package "+source+" in "+self._config.section),
             "section_navigation": create_section_navigation(self._section_names,self._config.section,self._doc_root),
             "time": time.strftime("%Y-%m-%d %H:%M %Z"),
             "rows": sourcerows+binaryrows,
             "doc_root": self._doc_root,
          }))
          f.close()

          # return parsable values
          if success: source_state = "pass"
          if failed:  source_state = "fail"
        else:
          source_state = "udeb"
          sourcerows = ""

        return sourcerows, binaryrows, source_state, maintainer, uploaders


    def create_package_summaries(self, logs_by_dir):
        logging.debug("Writing package templates in %s" % self._config.section)

        maintainers = {}
        source_binary_rows = {}
        sources = ""
        for source in self._source_db.get_all_packages():
            (sourcerows, binaryrows, source_state, maintainer, uploaders) = \
                               self.create_source_summary(source, logs_by_dir)

            if source_state != "udeb":
                sources += "%s: %s\n" % (source, source_state)
                source_binary_rows[source] = (source_state, sourcerows, binaryrows)
                for maint in [maintainer] + uploaders.split(","):
                    if maint.strip():
                        email = get_email_address(maint.strip())
                        if not "INVALID" in email:
                            if not email in maintainers:
                                maintainers[email] = []
                            maintainers[email].append(source)

        write_file(os.path.join(self._output_directory, "sources.txt"), sources)

        self.create_maintainer_summaries(maintainers, source_binary_rows)


    def make_stats_graph(self):
        countsfile = os.path.join(self._output_directory, "counts.txt")
        pngfile = os.path.join(self._output_directory, "states.png")
        r('t <- (read.table("'+countsfile+'",sep=",",header=1,row.names=1))')
        r('cname <- c("date",rep(colnames(t)))')
        # here we define how many days we wants stats for (163=half a year)
        #r('v <- t[(nrow(t)-163):nrow(t),0:12]')
        # make graph since day 1
        r('v <- t[0:nrow(t),0:12]')
        # thanks to http://tango.freedesktop.org/Generic_Icon_Theme_Guidelines for those nice colors
        r('palette(c("#4e9a06", "#ef2929", "#d3d7cf", "#5c3566", "#c4a000", \
                     "#fce94f", "#a40000", "#888a85", "#2e3436", "#729fcf", \
                     "#3465a4", "#204a87", "#555753"))')
        r('bitmap(file="'+pngfile+'",type="png16m",width=16,height=9,pointsize=10,res=100)')
        r('barplot(t(v),col = 1:13, \
          main="Binary packages per state in '+self._config.section+'", \
          xlab="", ylab="Number of binary packages", space=0.1, border=0)')
        r('legend(x="bottom",legend=colnames(t), ncol=2,fill=1:13,xjust=0.5,yjust=0,bty="n")')

        stats_html = "<tr class=\"normalrow\"> " \
                     + "<td class=\"contentcell2\" colspan=\"3\">" \
                     + "<a href=\"%s\">" \
                       % "states.png" \
                     + "<img src=\"/%s/%s\" height=\"450\" width=\"800\" alt=\"Binary package states in %s\">" \
                       % (self._config.section, "states.png", self._config.section) \
                     + "</a></td></tr>\n"

        return stats_html


    def create_and_link_to_analysises(self,state):
        link="<ul>"
        for template, linktarget in linktarget_by_template:
          # sucessful logs only have issues and failed logs only have errors
          if (state == "failed-testing" and template[-9:] != "issue.tpl") \
              or (state == "successfully-tested" and template[-9:] == "issue.tpl"):
            substats = ""

            tpl = os.path.join(self._output_directory, template)
            try:
              f = file(tpl, "r")
              rows = file.read(f)
              f.close()
              os.unlink(tpl)

              htmlpage = string.Template(HTML_HEADER + ANALYSIS_BODY_TEMPLATE + HTML_FOOTER)
              filename = os.path.join(self._output_directory, template[:-len(".tpl")]+".html")
              f = file(filename, "w")
              f.write(htmlpage.safe_substitute( {
                 "page_title": html_protect("Packages in state "+state+" "+linktarget),
                 "section_navigation": create_section_navigation(self._section_names,self._config.section,self._doc_root),
                 "time": time.strftime("%Y-%m-%d %H:%M %Z"),
                 "rows": rows,
                 "doc_root": self._doc_root,
               }))
              f.close()
              if state == "failed-testing":
                count_bugged = string.count(rows, '"bugged/')
                count_affected = string.count(rows, '"affected/')
                count_failed = string.count(rows, '"fail/')
                sep = ": "
                if count_bugged > 0:
                  substats += sep + "%s bugged" % count_bugged
                  sep = ", "
                if count_affected > 0:
                  substats += sep + "%s affected" % count_affected
                  sep = ", "
                if count_failed > 0:
                  substats += sep + "<span id=\"needs-bugging\">%s failed</span>" % count_failed
              else:
                  count_passed = string.count(rows, '"pass/')
                  if count_passed > 0:
                    substats += ": %s passed" % count_passed
              link += "<li><a href=%s>%s</a>%s</li>\n" % \
                       (
                           template[:-len(".tpl")]+".html",
                           linktarget,
                           substats,
                       )
            except:
              logging.debug("analysis template %s does not exist." % template)

        link += "</ul>"
        if link == "<ul></ul>":
          link = ""
        return link

    def write_section_index_page(self,dirs,total_packages):
        tablerows = ""
        for state in self._binary_db.get_active_states():
            dir_link = ""
            analysis = ""
            for vdir in dirs:
              if vdir in ("pass", "fail", "bugged", "affected", "untestable") and state_by_dir[vdir] == state:
                dir_link += "<a href='%s.html'>%s</a> logs<br>" % (vdir, html_protect(vdir))
            if state in ("successfully-tested", "failed-testing"):
              analysis = self.create_and_link_to_analysises(state)
            tablerows += ("<tr class=\"normalrow\"><td class=\"contentcell2\"><a href='state-%s.html'>%s</a>%s</td>" +
                          "<td class=\"contentcell2\">%d</td><td class=\"contentcell2\">%s</td></tr>\n") % \
                          (html_protect(state), html_protect(state), analysis, len(self._binary_db.get_pkg_names_in_state(state)),
                          dir_link)
        try:
          tablerows += self.make_stats_graph();
        except:
          logging.debug("Error generating the graph images, probably python-rpy is not installed, disabling graphs.")

        tablerows += "<tr class=\"normalrow\"> <td class=\"labelcell2\">Total</td> <td class=\"labelcell2\" colspan=\"2\">%d</td></tr>\n" % total_packages
        htmlpage = string.Template(HTML_HEADER + SECTION_INDEX_BODY_TEMPLATE + HTML_FOOTER)
        write_file(os.path.join(self._output_directory, "index.html"), htmlpage.safe_substitute( {
            "page_title": html_protect(self._config.section+" statistics"),
            "section_navigation": create_section_navigation(self._section_names,self._config.section,self._doc_root),
            "time": time.strftime("%Y-%m-%d %H:%M %Z"),
            "section": html_protect(self._config.section),
            "description": html_protect(self._config["description"]),
            "tablerows": tablerows,
            "packagesurl": html_protect(self._config.get_packages_url()),
            "doc_root": self._doc_root,
           }))

    def _show_providers(self, dep):
        providers = self._binary_db.get_providers(dep)
        vlist = ""
        if providers:
            vlist += "\n<ul>\n"
            for provider in providers:
                vlist += "<li>provider %s is %s</li>\n" % \
                          (self.link_to_state_page(self._config.section, provider, provider),
                          emphasize_reason(html_protect(self._binary_db.get_package_state(provider))))
            vlist += "</ul>\n"
        return vlist

    def write_state_pages(self):
        for state in self._binary_db.get_active_states():
            logging.debug("Writing page for %s" % state)
            with_counts = False
            aside = ""
            vlist = ""
            if state in self._binary_db.get_error_states():
                with_counts = True
                aside = " (reverse deps, blocked pkgs)"

            def cmp_func(a, b):
                """Sort by block count first"""
                rrdep_cmp = cmp( a.block_count(), b.block_count())
                if rrdep_cmp != 0:
                    return -rrdep_cmp
                else:
                    return cmp( a["Package"], b["Package"] )

            names = self._binary_db.get_pkg_names_in_state(state)
            packages = [self._binary_db.get_package(name) for name in names]
            packages.sort( cmp_func )

            for package in packages:
                vlist += "<li id=\"%s\">%s" % (
                                         package["Package"],
                                         self.link_to_source_summary(package["Package"]))
                if with_counts:
                    vlist += " (%d, %d)" % (package.rrdep_count(), package.block_count())
                vlist += " (%s)" % html_protect(package["Maintainer"])
                all_deps = package.all_dependencies()
                if all_deps:
                    vlist += "\n<ul>\n"
                    for alternatives in all_deps:
                        dep = alternatives[0]
                        vlist += "<li>dependency %s is %s</li>\n" % \
                                  (self.link_to_state_page(self._config.section,dep,dep),
                                  emphasize_reason(html_protect(self._binary_db.get_package_state(dep, resolve_virtual=False))))
                        vlist += self._show_providers(dep)
                        if len(alternatives) > 1:
                            vlist += "\n<ul>\n"
                            for dep in alternatives[1:]:
                                vlist += "<li>alternative dependency %s is %s</li>\n" % \
                                          (self.link_to_state_page(self._config.section,dep,dep),
                                          emphasize_reason(html_protect(self._binary_db.get_package_state(dep, resolve_virtual=False))))
                                vlist += self._show_providers(dep)
                            vlist += "</ul>\n"
                    vlist += "</ul>\n"
                vlist += "</li>\n"
            htmlpage = string.Template(HTML_HEADER + STATE_BODY_TEMPLATE + HTML_FOOTER)
            write_file(os.path.join(self._output_directory, "state-%s.html" % state), htmlpage.safe_substitute( {
                                        "page_title": html_protect("Packages in state "+state+" in "+self._config.section),
                                        "section_navigation": create_section_navigation(self._section_names,self._config.section,self._doc_root),
                                        "time": time.strftime("%Y-%m-%d %H:%M %Z"),
                                        "state": html_protect(state),
                                        "section": html_protect(self._config.section),
                                        "list": vlist,
                                        "aside": aside,
                                        "doc_root": self._doc_root,
                                       }))


    def archive_logfile(self, vdir, log):
        archivedir = os.path.join("archive", vdir)
        if not os.path.exists(archivedir):
            os.makedirs(archivedir)
        os.rename(os.path.join(vdir, log), os.path.join("archive", vdir, log))


    def cleanup_removed_packages(self, logs_by_dir):
        for vdir in logs_by_dir.keys():
            for log in sorted(logs_by_dir[vdir]):
                if log.endswith(".log"):
                    package, version = log[:-len(".log")].split("_")
                    if not self._binary_db.has_package(package):
                        logging.debug("Package %s was removed, archiving %s/%s" % (package, vdir, log))
                        self.archive_logfile(vdir, log)
                        logs_by_dir[vdir].remove(log)


    def generate_html(self):
        logging.debug("Finding log files")
        dirs = ["pass", "fail", "bugged", "affected", "reserved", "untestable"]
        logs_by_dir = {}
        for vdir in dirs:
            logs_by_dir[vdir] = find_files_with_suffix(vdir, ".log")

        logging.debug("Archiving logs of obsolete packages")
        self.cleanup_removed_packages(logs_by_dir)

        logging.debug("Copying log files")
        copy_logs(logs_by_dir, self._output_directory)

        logging.debug("Removing old log files")
        remove_old_logs(logs_by_dir, self._output_directory)

        logging.debug("Writing per-dir HTML pages")
        self.print_by_dir(self._output_directory, logs_by_dir)

        total_packages = self.write_counts_summary()

        self.create_package_summaries(logs_by_dir)

        logging.debug("Writing section index page")
        self.write_section_index_page(dirs, total_packages)

        logging.debug("Writing stats pages for %s" % self._config.section)
        self.write_state_pages()


    def generate_output(self, output_directory, section_names):
        # skip output generation for disabled sections
        if int(self._config["max-reserved"]) == 0:
            return

        self._section_names = section_names
        self._output_directory = os.path.abspath(os.path.join(output_directory, self._config.section))
        if not os.path.exists(self._output_directory):
            os.makedirs(self._output_directory)

        oldcwd = os.getcwd()
        os.chdir(self._master_directory)
        self.generate_html()
        os.chdir(oldcwd)


def main():
    setup_logging(logging.DEBUG, None)

    if len(sys.argv) > 1:
        print 'piuparts-report takes no command line parameters.'
        sys.exit(1)

    global_config = Config(section="global")
    global_config.read(CONFIG_FILE)
    section_names = global_config["sections"].split()
    master_directory = global_config["master-directory"]
    output_directory = global_config["output-directory"]
    doc_root = global_config["doc-root"].strip()
    if not doc_root.startswith("/"):
        doc_root = "/" + doc_root
    if doc_root.endswith("/"):
        doc_root = doc_root[:-1]

    if os.path.exists(master_directory):
        for section_name in section_names:
            section = Section(section_name, master_directory, doc_root)
            section.generate_output(output_directory=output_directory, section_names=section_names)

        # static pages
        logging.debug("Writing static pages")
        for page in ("index", "bug_howto"):
          tpl = os.path.join(output_directory,page+".tpl")
          INDEX_BODY = "".join(read_file(tpl))
          htmlpage = string.Template(HTML_HEADER + INDEX_BODY + HTML_FOOTER)
          write_file(os.path.join(output_directory,page+".html"), htmlpage.safe_substitute( {
                                 "page_title": "About piuparts.d.o and News",
                                 "section_navigation": create_section_navigation(section_names,"sid",doc_root),
                                 "time": time.strftime("%Y-%m-%d %H:%M %Z"),
                                 "doc_root": doc_root,
                              }))

    else:
        logging.debug("Warning: %s does not exist!?! Creating it for you now." % master_directory)
        os.makedirs(master_directory)


if __name__ == "__main__":
    main()

# vi:set et ts=4 sw=4 :
