/***************************************************************************
 *            pilot-qof.c
 *
 *   Sat Feb  5 10:40:03 GMT 2005
 *  Copyright  2005, 2006  Neil Williams <linux@codehelp.co.uk>
 *
 *  plu_connect and pq_findcategory based on pilot-link/src/userland.c
 *  Copyright  2004 Adriaan de Groot <groot@kde.org>
 ****************************************************************************/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/** @addtogroup QOF
    @{ */
/** @addtogroup pilot
@{ */

/** @file pilot-qof.c
  @brief Executable interface to the QOF external framework
  @author Copyright (c) 2005, 2006 Neil Williams <linux@codehelp.co.uk>
  @author Copyright (c) 2004 Adriaan de Groot <groot@kde.org>
*/

#define _GNU_SOURCE
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <regex.h>
#include <ctype.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <glib/gprintf.h>
#include <qof.h>
#include <popt.h>
#include "pi-source.h"
#include "pi-debug.h"
#include "pi-socket.h"
#include "pi-file.h"
#include "pi-header.h"
#include "pi-util.h"
#include "qof-main.h"
#include "palm.h"
#include "pilot-qof.h"
#include "pilot-todo.h"
#include "qof-address.h"
#include "qof-datebook.h"
#ifdef HAVE_QOFEXPENSES
#include <qof-expenses.h>
#endif
#include "pilot-expenses.h"

/** used to print debug logs. */
static QofLogModule log_module = PQ_MOD_CLI;
/** pack routines for communication with pilot */
static GList *pilot_modules = NULL;

static const gchar *env_pilotport = "PILOTPORT";

#define ARGUMENT_BAD_OPTION	17227

#define EXCLUDE_REPEATER_SQL "SELECT * from pilot_datebook " \
"where DATEBOOK_REPEATER == TRUE;"

#define ENUM_LIST_Q(_)    \
	_(qof_op_noop, = 0)   \
	_(qof_op_offline, )   \
	_(qof_op_list,)       \
	_(qof_op_hotsync,)    \
	_(qof_op_empty,)      \
	_(qof_op_category,)   \
	_(qof_op_database,)   \
	_(qof_op_time,)       \
	_(qof_op_exclude,)    \
	_(qof_op_sql,)        \
	_(qof_op_sql_file,)   \
	_(qof_op_write, )     \
	_(qof_op_upload, )    \
	_(qof_op_explain,)    \
	_(qof_op_vers,)       \
	_(qof_op_compress,)   \
	_(qof_op_debug,)      \
	_(qof_op_inv_city,)   \
	_(qof_op_inv_vendor,) \
	_(qof_op_use_locale,)

DEFINE_ENUM (qof_op_type, ENUM_LIST_Q)

static PQContext *
pilot_qof_create (void)
{
	PQContext *context;
	gint name_count;

	context = g_new0 (PQContext, 1);
	for (name_count = 0; name_count < 16; name_count++)
	{
		g_sprintf (context->names[name_count], "%s", "");
	}
	return context;
}

PQContext *
pilot_qof_init (void)
{
	qof_init ();
	g_return_val_if_fail (AddressRegister (), NULL);
	/** @bug need to get Expenses from gpe-expenses and then
	 add the pisock layer. */
#ifdef HAVE_QOFEXPENSES
	g_return_val_if_fail (ExpensesRegister (), NULL);
#endif
	g_return_val_if_fail (PQExpensesRegister (), NULL);
	g_return_val_if_fail (DateBookRegister (), NULL);
	g_return_val_if_fail (ToDoRegister (), NULL);
	g_return_val_if_fail (packing_registration(), NULL);
	return pilot_qof_create ();
}

typedef enum {
	PLU_CAT_NOFLAGS = 0,
	PLU_CAT_CASE_INSENSITIVE = 0x0001,
	PLU_CAT_DEFAULT_UNFILED = 0x0002,
	PLU_CAT_MATCH_NUMBERS = 0x0004,
	PLU_CAT_WARN_UNKNOWN = 0x0008
} plu_findcategory_flags_t;

/* copied from pilot-link 0.12 source */
static gint 
plu_connect(gchar* plu_port, gint plu_quiet)
{
	gint sd = -1;
	gint result;

	struct  SysInfo sys_info;

	if (plu_port == NULL)
		plu_port = getenv(env_pilotport);
	if (plu_port == NULL) {
		fprintf (stderr, "\n   ");
		fprintf (stderr, _("Unable to determine port to bind"));
		fprintf (stderr, "\n   ");
		fprintf (stderr, _("Please use --help for more information"));
		fprintf (stderr, "\n\n");
		return -1;
	}

	if ((sd = pi_socket(PI_AF_PILOT,
			PI_SOCK_STREAM, PI_PF_DLP)) < 0) {
		fprintf(stderr, "\n   ");
		fprintf(stderr, _("Unable to create socket '%s'"), plu_port);
		fprintf(stderr, "\n");
		return -1;
	}

	result = pi_bind(sd, plu_port);

	if (result < 0) {
		fprintf (stderr, "\n   ");
		fprintf (stderr, _("Unable to bind to port: %s"), plu_port);;
		fprintf (stderr, "\n   ");
		fprintf (stderr, _("Please use --help for more information"));
		fprintf (stderr, "\n\n");
		return result;
	}

	if (!plu_quiet && isatty(fileno(stdout))) {
		fprintf (stdout, "\n   ");
		fprintf (stdout, _("Listening for incoming connection on %s... "),
			plu_port);
		fflush(stdout);
	}

	if (pi_listen(sd, 1) < 0) {
		fprintf (stderr, "\n   ");
		fprintf(stderr, _("Error listening on %s"), plu_port);
		fprintf (stderr, "\n   ");
		pi_close(sd);
		return -1;
	}

	sd = pi_accept(sd, 0, 0);
	if (sd < 0) {
		fprintf (stderr, "\n   ");
		fprintf(stderr, _("Error accepting data on %s"), plu_port);
		fprintf (stderr, "\n   ");
		pi_close(sd);
		return -1;
	}

	if (!plu_quiet && isatty(fileno(stdout))) {
		fprintf(stdout, _("connected!"));
		fprintf(stdout, "\n\n");
	}

	if (dlp_ReadSysInfo(sd, &sys_info) < 0) {
		fprintf (stderr, "\n   ");
		fprintf (stderr, _("Error read system info on %s"), plu_port);
		fprintf (stderr, "\n   ");
		pi_close(sd);
		return -1;
	}

	dlp_OpenConduit(sd);
	return sd;
}

static gint 
pq_findcategory(PQContext *context, const gchar *name, gint flags)
{
	gint cat_index, match_category;

	match_category = -1;
	for (cat_index = 0; cat_index < 16; cat_index += 1) {
		if (context->names[cat_index][0]) {
			if (flags & PLU_CAT_CASE_INSENSITIVE) {
				if (strncasecmp(context->names[cat_index], name, 15) == 0) {
					match_category = cat_index;
					break;
				}
			} else {
				if (strncmp(context->names[cat_index],name,15) == 0) {
					match_category = cat_index;
					break;
				}
			}
		}
	}
	
	if ((match_category == -1)  && (flags & PLU_CAT_MATCH_NUMBERS)) {
		while (isspace(*name)) {
			name++;
		}
		if (isdigit(*name)) {
			match_category = atoi(name);
		}
	
		if ((match_category < 0) || (match_category > 15)) {
			match_category = -1;
		}
	}
	
	if (flags & PLU_CAT_WARN_UNKNOWN) {
		if (match_category == -1) {
			fprintf(stderr, _("WARNING: Unknown category '%s'%s.\n"),
				name,
				(flags & PLU_CAT_DEFAULT_UNFILED) ? _(", using 'Unfiled'") : "");
		}
	}
	
	if (flags & PLU_CAT_DEFAULT_UNFILED) {
		if (match_category == -1) {
			match_category = 0;
		}
	}
	
	return match_category;
}

void
pilot_qof_pack (QofEntity * ent, gpointer user_data)
{
	const gchar *category_name;
	const QofParam *param;
	gint size, result;
	QofPack pack_func;
	PQContext *context;
	const PQPack *p;

	context = (PQContext *) user_data;
	size = result = 0;
	g_return_if_fail (context != NULL);
	p = pilot_qof_pack_lookup (ent->e_type);
	if (!p)
		return;
	context->pi_buf = pi_buffer_new (PQ_DEF_BUFSZ);
	pack_func = p->pack_func;
	if (pack_func == NULL)
	{
		context->qof.error = TRUE;
		return;
	}
	size = pack_func (ent, context);
	if (size == -1)
		return;
	param = qof_class_get_parameter (ent->e_type, CATEGORY_NAME);
	category_name = (const gchar *) param->param_getfcn (ent, param);
	context->ent_category = pq_findcategory(context, 
		category_name, PLU_CAT_CASE_INSENSITIVE | PLU_CAT_DEFAULT_UNFILED);
/*	context->ent_category = plu_findcategory (context->pi_cat,
		category_name, PLU_CAT_CASE_INSENSITIVE | PLU_CAT_DEFAULT_UNFILED);*/
	if (context->ent_category == 0)
		PWARN (" Category: '%s' not found or not set, using 'Unfiled'",
			category_name);
	result = dlp_WriteRecord (context->sd, context->db,
		PQ_DLP_REC_ATTR, PQ_DLP_NEW_REC, context->ent_category,
		context->pi_buf->data, size, PQ_DLP_SET_ID);
	if (result < 0)
	{
		PERR (" record could not be written: error %d", result);
		return;
	}
}

void
pilot_qof_unpack (QofEntity * ent, gpointer user_data)
{
	QofPack unpack_func;
	const PQPack *p;
	PQContext *context;
	gint result;

	context = (PQContext *) user_data;
	g_return_if_fail (context && ent);
	p = pilot_qof_pack_lookup (ent->e_type);
	g_return_if_fail (p);
	unpack_func = p->unpack_func;
	if (unpack_func == NULL)
	{
		context->qof.error = TRUE;
		PERR ("No unpack routine was defined for the %s object!",
			ent->e_type);
		return;
	}
	result = unpack_func (ent, context);
	if (result < 0)
	{
		qof_entity_release (ent);
		g_free (ent);
	}
}

void
pilot_app_unpack (QofIdTypeConst e_type, gpointer user_data)
{
	PQContext *context;
	QofPack app_unpack;
	const PQPack *p;

	context = (PQContext *) user_data;
	g_return_if_fail (context != NULL);
	p = pilot_qof_pack_lookup (e_type);
	if (!p)
		return;
	app_unpack = p->app_info_unpack;
	if (app_unpack == NULL)
	{
		context->qof.error = TRUE;
		PERR (" no app_info_unpack routine for %s", e_type);
		return;
	}
	/* no entity available for the appInfo, pass NULL and work only in the context. */
	app_unpack (NULL, context);
}

static void
pilot_entity_free (QofEntity * ent, gpointer user_data)
{
	QofPack free_pack_func;
	const PQPack *p;

	p = pilot_qof_pack_lookup (ent->e_type);
	if (!p)
		return;
	free_pack_func = p->free_pack_func;
	free_pack_func (ent, NULL);
}

static void
pilot_object_free (QofObject * obj, gpointer user_data)
{
	QofBook *book;

	book = (QofBook *) user_data;
	qof_object_foreach (obj->e_type, book, pilot_entity_free, NULL);
}

void
pilot_entity_finaliser (QofBook * book, gpointer key, gpointer data)
{
	qof_object_foreach_type (pilot_object_free, book);
}

gboolean
pilot_qof_pack_register (const PQPack * p)
{
	if (g_list_index (pilot_modules, (gpointer) p) == -1)
		pilot_modules = g_list_prepend (pilot_modules, (gpointer) p);
	else
		return FALSE;
	return TRUE;
}

const PQPack *
pilot_qof_pack_lookup (QofIdTypeConst object_type)
{
	GList *piter;
	PQPack *p;

	if (!object_type)
		return NULL;
	for (piter = pilot_modules; piter; piter = piter->next)
	{
		p = piter->data;
		if (0 == safe_strcmp (p->e_type, object_type))
			return p;
	}
	return NULL;
}

static void
pilot_qof_free (PQContext * data)
{
	gint name_count;

	for (name_count = 0; name_count < 16; name_count++)
		g_sprintf (data->names[name_count], "%s", "");
	qof_main_free (&data->qof);
}

static void
write_ent_cb (QofEntity * ent, gpointer user_data)
{
	PQContext *context;

	context = (PQContext *) user_data;
	g_return_if_fail (context && ent);
	if (context->qof.error)
		return;
	context->pi_buf = pi_buffer_new (PQ_DEF_BUFSZ);
	pilot_qof_pack (ent, context);
}

static void
pilot_database_open (QofObject * obj, gpointer user_data)
{
	gchar *db_name, *log;
	PQContext *context;
	const PQPack *p;
	QofPack pref_unpack;
	gint db, attr, category, i, len;
	QofBook *book;
	QofEntity *ent;
	QofInstance *inst;

	context = (PQContext *) user_data;
	if (context->qof.error)
	{
		return;
	}
	pref_unpack = NULL;
	len = 0;
	i = 0;
	p = pilot_qof_pack_lookup (obj->e_type);
	if (!p)
		return;
	db_name = g_strdup (p->palm_db_name);
	if (db_name == NULL)
	{
		context->qof.error = TRUE;
		PERR (" object %s has no database name", obj->e_type);
		return;
	}
	if (0 == safe_strcmp (obj->e_type, context->qof.exclude))
		return;
	/* If no database specified, query all except any excluded database. */
	if ((context->qof.database != NULL) &&
		(0 != safe_strcmp (obj->e_type, context->qof.database)))
		return;
	if (p->db_pref_unpack)
	{
		gchar * creator;
		gint pref_result = 0;
		/* Optionally, unpack preferences here */
		pref_unpack = p->db_pref_unpack;
		/* no entity available for the preferences, */
		/* pass NULL and work only in the context. */
		creator = g_strdup (p->pref_creator);
		pref_result = dlp_ReadAppPreference (context->sd,
			makelong (creator), p->pref_flag,
			PQ_PREF_USE_BACKUP, PQ_DEF_BUFSZ, context->pref_buf,
			PQ_PREF_USE_SIZE, PQ_PREF_VERSION);
		pref_unpack (NULL, context);
		g_free (creator);
	}
	if (dlp_OpenDB (context->sd, PQ_DLP_CARD,
			PI_DLP_ARG_FLAG_SHORT | PI_DLP_ARG_FLAG_LONG,
			db_name, &db) < 0)
	{
		PWARN (" Unable to open %s database on Palm.\n", db_name);
		log =
			g_strdup_printf (_
			("%s: Unable to open %s database on Palm.\n"),
			PACKAGE, db_name);
		dlp_AddSyncLogEntry (context->sd, log);
		pi_close (context->sd);
		g_free (log);
		context->qof.error = TRUE;
		return;
	}
	context->db = db;
	context->app_buf = pi_buffer_new (PQ_DEF_BUFSZ);
	/* each database has it's own category list. */
	dlp_ReadAppBlock (context->sd, context->db, PQ_DLP_OFFSET,
		PQ_DLP_APPREAD, context->app_buf);
	pilot_app_unpack (obj->e_type, context);
	/* Write each entity if an upload file was specified. */
	if (context->qof.input_file)
	{
		book = qof_session_get_book (context->qof.input_session);
		qof_object_foreach (obj->e_type, book, write_ent_cb, context);
	}
	context->pi_buf = pi_buffer_new (PQ_DEF_BUFSZ);
	book = qof_session_get_book (context->qof.input_session);
	for (i = 0; len >= 0; i++)
	{
		len = dlp_ReadRecordByIndex (context->sd, context->db, i,
			context->pi_buf, PQ_DLP_RECORD, &attr, &category);
		/* attr holds archive - when deleted from Palm with option to archive on PC */
		if ((attr & dlpRecAttrDeleted) || (attr & dlpRecAttrArchived))
			continue;
		/* category holds the index of the category in this database category list. */
		context->ent_category = category;
		/* Create new entity to hold the unpacked data. */
		inst = (QofInstance *) qof_object_new_instance (obj->e_type, book);
		g_return_if_fail (inst != NULL);
		ent = &inst->entity;
		pilot_qof_unpack (ent, context);
		if (context->qof.error == TRUE)
		{
			len = -1;
			break;
		}
	}
	pi_buffer_free (context->app_buf);
	pi_buffer_free (context->pi_buf);
	g_free (db_name);
	dlp_CloseDB (context->sd, context->db);
}

static void
pilot_error (PQContext * context, const gchar * message)
{
	PERR (" %s - Error code: %d", message, pi_error(context->sd));
	if(context->sd > 0)
	{
		pi_close(context->sd);
	}
	qof_session_end(context->qof.input_session);
	qof_session_end(context->qof.export_session);
	context->qof.error = TRUE;
}

static void
find_invoice_contact (QofEntity * ent, gpointer data)
{
	GSList *field_list, *category_param_list;
	QofQueryPredData *appt_pred, *exp_pred, *category_pred;
	PQContext *context;
	const QofParam *param;
	const gchar *string;
	GList *results;
	QofBook *book;

	context = (PQContext *) data;
	g_return_if_fail (context && ent);
	param = NULL;
	string = NULL;
	results = NULL;
	category_param_list = NULL;
	if (0 != safe_strcmp (ent->e_type, PILOT_LINK_QOF_EXPENSES)
		&& (0 != safe_strcmp (ent->e_type, PILOT_LINK_QOF_DATEBOOK)))
		return;
	ENTER (" ent=%s", ent->e_type);
	book = qof_session_get_book (context->qof.input_session);
	qof_query_set_book (context->qof.query, book);
	field_list = NULL;
	/* remember: query runs in address using strings from 
	expenses or datebook. */
	if (context->qof.category)
	{
		category_param_list =
			qof_query_build_param_list (CATEGORY_NAME, NULL);
		category_pred =
			qof_query_string_predicate (QOF_COMPARE_EQUAL,
			context->qof.category,
			QOF_STRING_MATCH_CASEINSENSITIVE, FALSE);
		qof_query_add_term (context->qof.query, category_param_list,
			category_pred, QOF_QUERY_AND);
	}
	/* lookup appointment or expenses strings */
	if (0 == safe_strcmp (ent->e_type, PILOT_LINK_QOF_EXPENSES))
	{
		if (context->invoice_vendor)
		{
			param = qof_class_get_parameter (ent->e_type, EXP_VENDOR);
			string = param->param_getfcn (ent, param);
			field_list = qof_query_build_param_list (ADDR_CITY, NULL);
		}
		if (context->invoice_city)
		{
			param = qof_class_get_parameter (ent->e_type, EXP_CITY);
			string = param->param_getfcn (ent, param);
			field_list =
				qof_query_build_param_list (ADDR_COMPANY, ADDR_TITLE,
				NULL);
		}
		if (string)
		{
			exp_pred = qof_query_string_predicate (QOF_COMPARE_EQUAL,
				string, QOF_STRING_MATCH_CASEINSENSITIVE, FALSE);
			qof_query_add_term (context->qof.query, field_list,
				exp_pred, QOF_QUERY_AND);
			results = qof_query_run (context->qof.query);
			if (results != NULL)
				qof_entity_copy_list (context->qof.export_session,
					results);
		}
	}
	if (0 == safe_strcmp (ent->e_type, PILOT_LINK_QOF_DATEBOOK))
	{
		param =
			qof_class_get_parameter (ent->e_type, DATEBOOK_DESCRIPTION);
		string = param->param_getfcn (ent, param);
		field_list = qof_query_build_param_list (ADDR_COMPANY,
			ADDR_CITY, ADDR_TITLE, NULL);
		if (string)
		{
			appt_pred = qof_query_string_predicate (QOF_COMPARE_EQUAL,
				string, QOF_STRING_MATCH_CASEINSENSITIVE, FALSE);
			qof_query_add_term (context->qof.query, field_list,
				appt_pred, QOF_QUERY_AND);
			results = qof_query_run (context->qof.query);
			if (results != NULL)
				qof_entity_copy_list (context->qof.export_session,
					results);
		}
	}
	if (context->qof.query)
		qof_query_clear (context->qof.query);
	LEAVE (" ");
}

static void
check_invoice_handler (PQContext * context)
{
	if (!context->qof.min_qt)
	{
		fprintf (stderr, _("%s: Error: Please specify a time period "
				"using -t."), PACKAGE);
		fprintf (stderr, "\n\n");
		qof_session_end (context->qof.input_session);
		context->qof.error = TRUE;
		return;
	}
	if (context->qof.exclude)
	{
		g_free (context->qof.exclude);
		context->qof.exclude = NULL;
	}
	context->qof.exclude = g_strdup (PILOT_LINK_QOF_TODO);
	if (context->qof.database)
	{
		g_free (context->qof.database);
		context->qof.database = NULL;
	}
	if (context->qof.input_file)
	{
		g_free (context->qof.input_file);
		context->qof.input_file = NULL;
	}
	if (context->qof.sql_str)
	{
		g_free (context->qof.sql_str);
		context->qof.sql_str = NULL;
	}
	if (context->qof.sql_list)
	{
		g_list_free (context->qof.sql_list);
		context->qof.sql_list = NULL;
	}
}

void
qof_cmd_hotsync (PQContext * context)
{
	struct PilotUser QUser;
	QofBook *book;
	gchar *log_msg;

	if (0 == safe_strcmp (context->qof.exclude, context->qof.database)
		&& (context->qof.exclude != NULL))
	{
		/* Translators: The first string is the package name.
		   The second and third are database names. */
		qof_main_wrap_line (stderr, ERR_INDENT,
			_("%s: Error: Cannot exclude "
				"database \"%s\" with option -e because option -d is set to the "
				"same database: \"%s\""), PACKAGE, context->qof.exclude,
			context->qof.database);
		qof_session_end (context->qof.input_session);
		return;
	}
	if ((context->invoice_city) || (context->invoice_vendor))
		check_invoice_handler (context);
	if (context->qof.error)
		return;
	if (context->qof.input_file)
	{
		PINFO (" Trying to upload %s", context->qof.input_file);
		qof_session_begin (context->qof.input_session,
			context->qof.input_file, TRUE, FALSE);
		qof_session_load (context->qof.input_session, NULL);
	}
	else
		qof_session_begin (context->qof.input_session, QOF_STDOUT, TRUE,
			FALSE);
	context->qof.export_session = qof_session_new ();

	context->sd = plu_connect (context->port, context->quiet);

	if (context->sd < 0)
	{
		pilot_error (context, _("Unable to connect to the Palm"));
		return;
	}
	if (dlp_ReadUserInfo (context->sd, &QUser) < 0)
	{
		pilot_error (context, _("Unable to read Palm user info"));
		return;
	}
	context->qof.error = FALSE;
	qof_object_foreach_type (pilot_database_open, context);
	QUser.lastSyncPC = 0x00010000;
	QUser.lastSyncDate = QUser.successfulSyncDate = time (0);
	if (dlp_WriteUserInfo (context->sd, &QUser) < 0)
	{
		pilot_error (context, _("Unable to write user info"));
		return;
	}
	/* Translators: each string is the package name. */
	log_msg = g_strdup_printf (_("%s hotsync\n\n"
			"Thank you for using %s.\n"), PACKAGE, PACKAGE);
	dlp_AddSyncLogEntry (context->sd, log_msg);
	g_free (log_msg);
	dlp_EndOfSync (context->sd, 0);
	pi_close (context->sd);
	/* Now run the query, copy the objects, destroy the input session */
	/*and write out export_session */
	if (context->qof.write_file)
	{
		qof_session_begin (context->qof.export_session,
			context->qof.write_file, TRUE, TRUE);
		qof_mod_compression (context->qof.gz_level, &context->qof);
	}
	else
		qof_session_begin (context->qof.export_session, QOF_STDOUT, TRUE,
			FALSE);
	/* Note if no query is given, ignore repeater clones?
	   Nice idea, but in practice, difficult. It also makes recursive
	   queries of the XML harder as the XML needs to understand datebook
	   repeats. Instead, we simply ignore all repeater clones when it comes
	   to write data to the Palm by a simple check in datebook_pack.
	 */
	qof_main_moderate_query (&context->qof);
	/* if invoice_hook, create a second query and lookup in contacts */
	if ((context->invoice_city) || (context->invoice_vendor))
	{
		book = qof_session_get_book (context->qof.export_session);
		context->qof.query = qof_query_create_for (PILOT_LINK_QOF_ADDRESS);
		qof_object_foreach (PILOT_LINK_QOF_DATEBOOK, book,
			find_invoice_contact, context);
		qof_object_foreach (PILOT_LINK_QOF_EXPENSES, book,
			find_invoice_contact, context);
	}
	qof_session_save (context->qof.export_session, NULL);
	qof_main_show_error (context->qof.export_session);
	qof_session_end (context->qof.input_session);
	qof_session_end (context->qof.export_session);
}

static void
pq_invoice_xmlfile (PQContext * context)
{
	QofBook *book;

	ENTER (" ");
	qof_session_begin (context->qof.input_session, context->qof.filename,
		TRUE, FALSE);
	qof_session_load (context->qof.input_session, NULL);
	context->qof.export_session = qof_session_new ();
	if (context->qof.write_file)
	{
		qof_session_begin (context->qof.export_session,
			context->qof.write_file, TRUE, TRUE);
		qof_mod_compression (context->qof.gz_level, &context->qof);
	}
	else
		qof_session_begin (context->qof.export_session, QOF_STDOUT,
			TRUE, TRUE);
	qof_main_moderate_query (&context->qof);
	book = qof_session_get_book (context->qof.export_session);
	context->qof.query = qof_query_create_for (PILOT_LINK_QOF_ADDRESS);
	qof_object_foreach (PILOT_LINK_QOF_DATEBOOK, book,
		find_invoice_contact, context);
	qof_object_foreach (PILOT_LINK_QOF_EXPENSES, book,
		find_invoice_contact, context);
	qof_session_save (context->qof.export_session, NULL);
	qof_main_show_error (context->qof.export_session);
	qof_main_show_error (context->qof.input_session);
	qof_session_end (context->qof.input_session);
	qof_session_end (context->qof.export_session);
	LEAVE (" ");
}

int
main (int argc, const char *argv[])
{
	const gchar *help_header_text, *input_file, *plu_port;
	gint plu_quiet, optc;
	PQContext *pilot_qof_context;
	gboolean debug_on;
	poptContext pc;
	gint64 gz_level;
	qof_op_type palm_command;

	QOF_OP_VARS struct poptOption options[] = {
		{"port", 'p', POPT_ARG_STRING, &plu_port, 0,
			_("Use the device <port> to communicate with Palm"),
				"<port>"},
		{ "quiet",   'q', POPT_ARG_NONE,  &plu_quiet,  0 ,
		 _("Suppress HotSync connection messages"), NULL},
		{"input-file", 'i', POPT_ARG_STRING, &filename, qof_op_offline,
		 _("Query the data in <filename>"), _("filename")},
		QOF_CLI_OPTIONS 
		{"hot-query", 'a', POPT_ARG_NONE,
				NULL, qof_op_hotsync,
			_("Activate/HotSync and query the Palm."), NULL},
		{"upload", 'u', POPT_ARG_STRING, &input_file, qof_op_upload,
			_("Upload data from <filename> to the Palm. Requires -a"),
				"filename"},
		{"invoice-vendor", 0, POPT_ARG_NONE, NULL, qof_op_inv_vendor,
			_
			("Shorthand to relate an event or expense to a contact, by vendor. "
					"Requires -t."), NULL},
		{"invoice-city", 0, POPT_ARG_NONE, NULL, qof_op_inv_city,
			_
			("Shorthand to relate an event or expense to a contact, by city. "
					"Requires -t."), NULL},
		{"use-locale", 0, POPT_ARG_NONE, NULL, qof_op_use_locale,
				_
			("Write XML using the current locale encoding instead of UTF-8."),
			NULL},
		POPT_TABLEEND
	};

	palm_command = qof_op_noop;
	debug_on = FALSE;
	QOF_OP_INIT;
	input_file = NULL;
	plu_quiet = 0;
	plu_port = NULL;

#ifdef ENABLE_NLS
	setlocale (LC_ALL, "");
	bindtextdomain (GETTEXT_PACKAGE, LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	help_header_text = _("\n"
		"   Query Palm databases as objects and save to XML.\n"

		"   pilot-qof provides a query interface to data on a Palm device,\n"
		"   using pilot-link and QOF - the Query Object Framework.\n\n"
		"   pilot-qof supports reading addressbook, datebook, expenses and\n"
		"   ToDo data from a Palm device to XML files. pilot-qof runs SQL-type\n"
		"   queries on the live data or a QSF XML file and results can be imported\n"
		"   into other QOF applications or converted to other text based formats,\n"
		"   including non-XML formats like vcard.\n"
		"   See http://pilot-qof.sourceforge.net/\n\n"
		"   Commands are -x -a -l --explain -s or -f.\n"
		"   Options are -c -t -w, -d or -e.\n"
		"   option -u requires -a\n\n");

	pc = poptGetContext (PACKAGE, argc, argv, options, 0);

	poptSetOtherOptionHelp (pc, help_header_text);

	if (argc < 2)
	{
		poptPrintUsage (pc, stderr, 0);
		return EXIT_FAILURE;
	}
	pilot_qof_context = pilot_qof_init ();
	if (!pilot_qof_context)
	{
		qof_main_wrap_line (stderr, ERR_INDENT,
			_("%s: Failed to initialise "
				"the query object framework."), PACKAGE);
		return EXIT_FAILURE;
	}
	/* convert deprecated date fields into the newly supported
	time values. */
	qof_mod_convert_deprecated (1, &pilot_qof_context->qof);
	while ((optc = poptGetNextOpt (pc)) >= 0)
	{
		switch (optc)
		{
			/* commands - mutually exclusive */
		case qof_op_offline:
		case qof_op_list:
		case qof_op_explain:
		case qof_op_hotsync:
			{
				palm_command = optc;
				break;
			}
		case qof_op_sql:
			{
				qof_mod_sql (sql_query, &pilot_qof_context->qof);
				if (!filename)
				{
					filename = g_strdup (QOF_STDOUT);
				}
				palm_command = qof_op_empty;
				break;
			}
		case qof_op_sql_file:
			{
				qof_mod_sql_file (sql_file, &pilot_qof_context->qof);
				palm_command = qof_op_empty;
				break;
			}
		case qof_op_vers:
			{
				fprintf (stdout, _("\n This is %s v%s\n"), PACKAGE,
					VERSION);
				fprintf (stdout,
					_
					(" Query interface for Palm databases as objects.\n"));
				fprintf (stdout,
					"\n Copyright (c) 2005-2006 "
					"Neil Williams <linux@codehelp.co.uk>\n");
				fprintf (stdout,
					_(" For %s support, join the QOF-devel "
						"mailing list at\n"), PACKAGE);
				fprintf (stdout,
					" http://lists.sourceforge.net/mailman/listinfo/qof-devel\n\n");
/* Translators: Add or subtract dots to keep the translated lines aligned vertically */
				fprintf (stdout, _(" Build target............: %s\n"),
					HOST_OS);
/* Translators: Add or subtract dots to keep the translated lines aligned vertically */
				fprintf (stdout, _(" Build date..............: %s %s\n"),
					__DATE__, __TIME__);
/* Translators: Add or subtract dots to keep the translated lines aligned vertically */
				fprintf (stdout, _(" Built for pilot-link ...: %s\n"),
					PILOT_LINK_SUPPORT);
/* Translators: Add or subtract dots to keep the translated lines aligned vertically */
				fprintf (stdout, _(" --debug logs to.........: %s/%s\n\n"),
					g_get_tmp_dir (), PILOT_QOF_LOG);
				fprintf (stdout,
					_
					(" Please use --help for more detailed options.\n\n"));
				return EXIT_SUCCESS;
			}
			/* optional modifiers - store to act on later. */
		case qof_op_category:
			{
				qof_mod_category (category, &pilot_qof_context->qof);
				break;
			}
		case qof_op_database:
			{
				qof_mod_database (database, &pilot_qof_context->qof);
				break;
			}
		case qof_op_time:
			{
				qof_mod_time (date_time, &pilot_qof_context->qof);
				break;
			}
		case qof_op_exclude:
			{
				qof_mod_exclude (exclude, &pilot_qof_context->qof);
				break;
			}
		case qof_op_write:
			{
				qof_mod_write (write_file, &pilot_qof_context->qof);
				break;
			}
		case qof_op_upload:
			{
				if (palm_command != qof_op_hotsync)
				{
					fprintf (stderr,
						_("%s: Error: Please specify -a if you use -u\n"),
						PACKAGE);
					poptPrintUsage (pc, stderr, 0);
					return EXIT_FAILURE;
				}
				pilot_qof_context->qof.input_file = g_strdup (input_file);
				break;
			}
		case qof_op_debug:
			{
				gchar *log;

				log =
					g_strconcat (g_get_tmp_dir (), "/", PILOT_QOF_LOG,
					NULL);
				qof_log_init_filename (log);
				g_free (log);
				qof_log_set_default (QOF_LOG_DETAIL);
				qof_log_set_level (PQ_MOD_CLI, QOF_LOG_DETAIL);
				qof_log_set_level (PQ_MOD_PILOT, QOF_LOG_DETAIL);
				qof_log_set_level (QOF_MAIN_CLI, QOF_LOG_DETAIL);
				qof_log_set_level (QOF_MOD_QSF, QOF_LOG_DETAIL);
				qof_log_set_level ("qof-sqlite-module", QOF_LOG_DETAIL);
				debug_on = TRUE;
				/** \todo Output debug messages earlier. */
				break;
			}
		case qof_op_compress:
			{
				pilot_qof_context->qof.gz_level = gz_level;
				break;
			}
		case qof_op_inv_vendor:
			{
				if (!pilot_qof_context->invoice_city)
					pilot_qof_context->invoice_vendor = TRUE;
				break;
			}
		case qof_op_inv_city:
			{
				pilot_qof_context->invoice_city = TRUE;
				pilot_qof_context->invoice_vendor = FALSE;
				break;
			}
		case qof_op_use_locale:
			{
				const gchar *locale_encoding;
				gboolean test;
				locale_encoding = NULL;
				test = g_get_charset (&locale_encoding);
				if (!test)
					qof_mod_encoding (locale_encoding,
						&pilot_qof_context->qof);
				break;
			}
		default:
			{
				fprintf (stderr, _("%s: ERROR: got option %d, arg %s\n"),
					PACKAGE, optc, poptGetOptArg (pc));
				return EXIT_FAILURE;
			}
		}
	}
	if (qof_op_noop == palm_command)
	{
		qof_main_wrap_line (stderr, ERR_INDENT,
			_("%s: ERROR: specify a command "
				"-x, -a, -l, -s or -f, or --explain.\n"), PACKAGE);
		poptPrintUsage (pc, stderr, 0);
		return EXIT_FAILURE;
	}
	if (qof_op_noop == palm_command)
	{
		fprintf (stderr, _("%s: ERROR: specify a command "
		"-x, -a, -l, -s or -f, or --explain.\n"), PACKAGE);
		poptPrintUsage(pc, stderr, 0);
		return EXIT_FAILURE;
	}
	if (optc < -1)
	{
		fprintf (stderr, "%s: %s %s\n\n", PACKAGE,
			poptBadOption (pc, POPT_BADOPTION_NOALIAS),
			poptStrerror (optc));
		poptPrintUsage (pc, stderr, 0);
		return EXIT_FAILURE;
	}
	/* If we get this far, we should have sensible options: start the work. */
	pilot_qof_context->qof.input_session = qof_session_new ();
	switch (palm_command)
	{
	case qof_op_empty:
	{
		pilot_qof_context->qof.filename = g_strdup (filename);
		if ((pilot_qof_context->invoice_city)
			|| (pilot_qof_context->invoice_vendor))
		{
			check_invoice_handler (pilot_qof_context);
			if (pilot_qof_context->qof.error)
				return EXIT_FAILURE;
			pq_invoice_xmlfile (pilot_qof_context);
		}
		else
			qof_cmd_xmlfile (&pilot_qof_context->qof);
		break;
	}
	case qof_op_offline:
		{
			pilot_qof_context->qof.filename = g_strdup (filename);
			if ((pilot_qof_context->invoice_city)
				|| (pilot_qof_context->invoice_vendor))
			{
				check_invoice_handler (pilot_qof_context);
				if (pilot_qof_context->qof.error)
					return EXIT_FAILURE;
				pq_invoice_xmlfile (pilot_qof_context);
			}
			else
				qof_cmd_xmlfile (&pilot_qof_context->qof);
			break;
		}
	case qof_op_list:
		{
			DEBUG (" list mode");
			qof_cmd_list ();
			break;
		}
	case qof_op_explain:
		{
			if (!pilot_qof_context->qof.database)
			{
				qof_main_wrap_line (stderr, ERR_INDENT,
					_("%s: Error: please specify which database "
						"you would like explained.\n\n"), PACKAGE);
				break;
			}
			DEBUG (" explain mode");
			qof_cmd_explain (&pilot_qof_context->qof);
			break;
		}
	case qof_op_hotsync:
		{
			DEBUG (" hotsync mode");
			pilot_qof_context->port = g_strdup (plu_port);
			pilot_qof_context->quiet = plu_quiet;
			qof_cmd_hotsync (pilot_qof_context);
			break;
		}
	case qof_op_noop:
	case qof_op_category:
	case qof_op_database:
	case qof_op_time:
	case qof_op_exclude:
	case qof_op_sql:
	case qof_op_sql_file:
	case qof_op_write:
	case qof_op_upload:
	case qof_op_vers:
	case qof_op_compress:
	case qof_op_debug:
	case qof_op_inv_city:
	case qof_op_inv_vendor:
	case qof_op_use_locale:
	default:
			break;
	}
	poptFreeContext (pc);
	pilot_qof_free (pilot_qof_context);
	if (debug_on)
		qof_log_shutdown ();
	qof_close ();
	return EXIT_SUCCESS;
}

/** @} */
/** @} */
