<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:xlink="http://www.w3.org/1999/xlink"
                xmlns:func="http://exslt.org/functions"
                xmlns:str="http://exslt.org/strings"
                xmlns:date="http://exslt.org/dates-and-times"
                xmlns:pml="http://wohlberg.net/xml/photoml"
                extension-element-prefixes="func str date"
                version="1.0">

<!-- 
     This file contains templates for extracting relevant PhotoML data
     for use by the pmlvalid tool.

     Copyright © 2005-2007 Brendt Wohlberg <photoml@wohlberg.net>

     This is free software; you can redistribute it and/or modify it 
     under the terms of version 2 of the GNU General Public License 
     at http://www.gnu.org/licenses/gpl-2.0.txt.

     This software is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
     GNU General Public License for more details.
-->

  <xsl:output method="text"/>

  <!-- Key matching all frame elements with specific id -->
  <xsl:key name="fid" match="frame"
           use="concat(ancestor::*[name()='roll' or name()='digital']/@id,
                       '+',@id)"/>
  <!-- Key matching all frame elements with specific fn -->
  <xsl:key name="fn" match="frame"
           use="concat(ancestor::*[name()='roll' or name()='digital']/@id,
                       '+',@fn)"/>

  <!-- Handle roll elements. -->
  <xsl:template match="roll">
    <xsl:text>roll: </xsl:text>
    <xsl:value-of select="concat('&quot;',@id,'&quot;')"/>
    <xsl:text>&#10;</xsl:text>
    <xsl:apply-templates/>
  </xsl:template>


  <!-- Handle sheet elements. -->
  <xsl:template match="sheet">
    <xsl:text>sheet: </xsl:text>
    <xsl:value-of select="concat('&quot;',@id,'&quot;')"/>
    <xsl:text>&#10;</xsl:text>
    <xsl:apply-templates/>
    <xsl:for-each select="collection">
      <xsl:text> collection: </xsl:text>
      <xsl:value-of select="concat('&quot;',@id,'&quot; ',
                                   '&quot;',@cgid,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:for-each>
  </xsl:template>
  

  <!-- Handle digital elements. -->
  <xsl:template match="digital">
    <xsl:text>digital: </xsl:text>
    <xsl:value-of select="concat('&quot;',@id,'&quot;')"/>
    <xsl:text>&#10;</xsl:text>
    <xsl:apply-templates/>
  </xsl:template>


  <!-- Handle digimage elements. -->
  <xsl:template match="digimage">
    <xsl:variable name="group-id">
      <xsl:choose>
        <xsl:when test="parent::frame/parent::digital">
          <xsl:value-of select="parent::frame/parent::digital/@id"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="@group-id"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="frame-id">
      <xsl:choose>
        <xsl:when test="parent::frame">
          <xsl:value-of select="parent::frame/@id"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="@frame-id"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:text>digimage: </xsl:text>
    <xsl:value-of select="concat('&quot;',$group-id,'&quot; ',
                                 '&quot;',$frame-id,'&quot; ',
                                 '&quot;',@image-id,'&quot;')"/>
    <xsl:text>&#10;</xsl:text>
    <xsl:if test="@xlink:href">
      <xsl:text> href: </xsl:text>
      <xsl:value-of select="concat('&quot;',@xlink:href,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:if test="properties/file-hash">
      <xsl:text> file-hash: </xsl:text>
      <xsl:value-of select="concat('&quot;',properties/file-hash,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:if test="properties/image-hash">
      <xsl:text> image-hash: </xsl:text>
      <xsl:value-of select="concat('&quot;',properties/image-hash,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:apply-templates/>
    <xsl:if test="parent::frame and not(parent::frame/ancestor::digital)">
      <xsl:text> error: "parent-frame-not-digital-descendant"&#10;</xsl:text>
    </xsl:if>
    <xsl:if test="parent::frame and (@group-id or @frame-id)">
      <xsl:text> error: "parent-frame-and-explicit-id"&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Allow empty content in descendants of defaults elements -->
  <xsl:template match="*[ancestor::defaults and .='']">
    <xsl:apply-templates/>
  </xsl:template>


  <!-- Report and check collection element attributes -->
  <xsl:template match="collection">
    <xsl:text> collection: </xsl:text>
    <xsl:value-of select="concat('&quot;',@id,'&quot; ',
                          '&quot;',@cgid,'&quot; ',
                          '&quot;',@fstfid,'&quot; ',
                          '&quot;',@lstfid,'&quot;')"/>
    <xsl:text>&#10;</xsl:text>
    <xsl:if test="@fstfid != '' and not(../frame[@id=current()/@fstfid])">
      <xsl:text> error: "collection-fstfid-out-of-range" </xsl:text>
      <xsl:value-of select="concat('&quot;',@id,'&quot; ',
                            '&quot;',@cgid,'&quot; ',
                            '&quot;',@fstfid,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:if test="@lstfid != '' and not(../frame[@id=current()/@lstfid])">
      <xsl:text> error: "collection-lstfid-out-of-range" </xsl:text>
      <xsl:value-of select="concat('&quot;',@id,'&quot; ',
                            '&quot;',@cgid,'&quot; ',
                            '&quot;',@lstfid,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Determine whether frame id attributes are all defined and unique. -->
  <xsl:template match="frame">
    <xsl:if test="not(@id) and not(parent::defaults)">
      <xsl:text> error: "frame-without-id" </xsl:text>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:if test="@id and parent::defaults">
      <xsl:text> error: "defaults-frame-with-id" </xsl:text>
      <xsl:value-of select="concat('&quot;',@id,'&quot; ')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:if test="@fn and parent::defaults">
      <xsl:text> error: "defaults-frame-with-fn" </xsl:text>
      <xsl:value-of select="concat('&quot;',@fn,'&quot; ')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:if test="@id and count(key('fid',
                  concat(ancestor::*[name()='roll' or name()='digital']/@id,
                  '+',@id))) &gt; 1 and generate-id(.) = generate-id(key('fid',
                  concat(ancestor::*[name()='roll' or name()='digital']/@id,
                  '+',@id))[1])">
      <xsl:text> error: "frame-nonunique-id" </xsl:text>
      <xsl:value-of select="concat('&quot;',@id,'&quot; ')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:if test="@fn and count(key('fn',
                  concat(ancestor::*[name()='roll' or name()='digital']/@id,
                  '+',@fn))) &gt; 1 and generate-id(.) = generate-id(key('fn',
                  concat(ancestor::*[name()='roll' or name()='digital']/@id,
                  '+',@fn))[1])">
      <xsl:text> error: "frame-nonunique-fn" </xsl:text>
      <xsl:value-of select="concat('&quot;',@fn,'&quot; ')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:apply-templates/>
  </xsl:template>


  <!-- Check format of elements with positive integer content. -->
  <xsl:template match="speed|rated-speed|width|height|bit-depth">
    <xsl:if test="not(pml:is-positive-integer(.))">
      <xsl:text> error: "invalid-pos-int" </xsl:text>
      <xsl:value-of select="concat('&quot;',name(.),'&quot; ')"/>
      <xsl:value-of select="concat('&quot;',.,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Check format of elements with real content. -->
  <xsl:template match="shift|temperature|aperture|focal-length|focal-distance">
    <xsl:if test="not(pml:is-real(.))">
      <xsl:text> error: "invalid-real" </xsl:text>
      <xsl:value-of select="concat('&quot;',name(.),'&quot; ')"/>
      <xsl:value-of select="concat('&quot;',.,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Check format of elements with fraction content. -->
  <xsl:template match="shutter|exp-comp|flash-comp">
    <xsl:if test="not(pml:is-fraction(.))">
      <xsl:text> error: "invalid-fraction" </xsl:text>
      <xsl:value-of select="concat('&quot;',name(.),'&quot; ')"/>
      <xsl:value-of select="concat('&quot;',.,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Check format of elements with hex content. -->
  <xsl:template match="file-hash|image-hash">
    <xsl:if test="not(pml:is-hex(.))">
      <xsl:text> error: "invalid-hex-string" </xsl:text>
      <xsl:value-of select="concat('&quot;',name(.),'&quot; ')"/>
      <xsl:value-of select="concat('&quot;',.,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Check format of elements with date content. -->
  <xsl:template match="date|expiry">
    <xsl:if test="not(pml:is-date(.))">
      <xsl:text> error: "invalid-date" </xsl:text>
      <xsl:value-of select="concat('&quot;',name(.),'&quot; ')"/>
      <xsl:value-of select="concat('&quot;',.,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Check format of elements with time content. -->
  <xsl:template match="time">
    <xsl:if test="not(pml:is-time(.))">
      <xsl:text> error: "invalid-time" </xsl:text>
      <xsl:value-of select="concat('&quot;',name(.),'&quot; ')"/>
      <xsl:value-of select="concat('&quot;',.,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
    <xsl:if test="@zone and not(pml:is-time-zone(@zone))">
      <xsl:text> error: "invalid-time-zone" </xsl:text>
      <xsl:value-of select="concat('&quot;',name(.),'&quot; ')"/>
      <xsl:value-of select="concat('&quot;',@zone,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Check format of latitude element. -->
  <xsl:template match="latitude">
    <xsl:if test="not(pml:is-real(.) and . &gt;= -90.0 and . &lt;= 90.0)">
      <xsl:text> error: "invalid-latitude" </xsl:text>
      <xsl:value-of select="concat('&quot;',name(.),'&quot; ')"/>
      <xsl:value-of select="concat('&quot;',.,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Check format of longitude element. -->
  <xsl:template match="longitude">
    <xsl:if test="not(pml:is-real(.) and . &gt;= -180.0 and . &lt;= 180.0)">
      <xsl:text> error: "invalid-longitude" </xsl:text>
      <xsl:value-of select="concat('&quot;',name(.),'&quot; ')"/>
      <xsl:value-of select="concat('&quot;',.,'&quot;')"/>
      <xsl:text>&#10;</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- High priority null template to avoid application of content tests 
       to elements playing the role of defaults merge pruning. -->
  <xsl:template match="*[child::processing-instruction('merge') = 'prune']"
                priority="10"/>


  <!-- Null template to avoid copying of all text nodes into output. -->
  <xsl:template match="text()"/>


  <!-- Extension function validating positive integer content. -->
  <func:function name="pml:is-positive-integer">
    <xsl:param name="x" select="''"/>

    <func:result select="translate($x,'+0123456789','')='' and
                           pml:occurrences($x,'+') &lt; 2 and
                           substring-before($x,'+') = '' and
                           not($x = '')"/>
  </func:function>


  <!-- Extension function validating integer content. -->
  <func:function name="pml:is-integer">
    <xsl:param name="x" select="''"/>

    <func:result select="translate($x,'+-0123456789','')='' and
                           (pml:occurrences($x,'+') +
                           pml:occurrences($x,'-')) &lt; 2 and
                           substring-before($x,'+') = '' and
                           substring-before($x,'-') = '' and
                           not($x = '')"/>
  </func:function>


  <!-- Extension function validating real value content. -->
  <func:function name="pml:is-real">
    <xsl:param name="x" select="''"/>

    <func:result select="translate($x,'+-.0123456789','')='' and
                           (pml:occurrences($x,'+') +
                           pml:occurrences($x,'-')) &lt; 2 and
                           pml:occurrences($x,'.') &lt; 2 and
                           not($x = '')"/>
  </func:function>


  <!-- Extension function validating rational value content. -->
  <func:function name="pml:is-rational">
    <xsl:param name="x" select="''"/>

    <func:result select="(contains($x,'/') and 
                         pml:is-integer(substring-before($x,'/')) and
                         pml:is-integer(substring-after($x,'/'))) or
                         pml:is-real($x) and
                         not($x = '')"/>
  </func:function>


  <!-- Extension function validating fraction valued content. -->
  <func:function name="pml:is-fraction">
    <xsl:param name="x" select="''"/>

    <func:result select="(contains($x,'/') and 
                         pml:is-real(substring-before($x,'/')) and
                         pml:is-real(substring-after($x,'/'))) or
                         pml:is-real($x) and
                         not($x = '')"/>
  </func:function>
  

 <!-- Extension function validating hexadecimal string content. -->
  <func:function name="pml:is-hex">
    <xsl:param name="x" select="''"/>

    <func:result select="translate($x,'0123456789abcdef','')='' and
                         not($x = '')"/>
  </func:function>


  <!-- Extension function validating date string content. -->
  <func:function name="pml:is-date">
    <xsl:param name="x" select="''"/>

    <xsl:variable name="dtokens" select="str:tokenize($x, '-')"/>
    <xsl:variable name="Y" select="string($dtokens[1])"/>
    <xsl:variable name="M" select="string($dtokens[2])"/>
    <xsl:variable name="D" select="string($dtokens[3])"/>

    <xsl:variable name="leap" select="number($M=2 and date:leap-year($Y))"/>
    <xsl:variable name="monthdays" 
                  select="str:tokenize(
                           '31 28 31 30 31 30 31 31 30 31 30 31',
                           ' ')"/>
    <xsl:variable name="maxday" 
                  select="($monthdays[number($M)])*(1-$leap) + 29*$leap"/>

    <func:result select="count($dtokens) &lt;= 3 and not(contains($x,'--')) and
                         pml:is-integer($Y) and string-length($Y)=4 and
                         (not($dtokens[2]) or
                           (pml:is-integer($M) and string-length($M)=2 and
                            $M &gt; 0 and $M &lt;= 12)) and
                         (not($dtokens[3]) or
                           (pml:is-integer($D) and string-length($D)=2 and
                            $D &gt; 0 and $D &lt;= $maxday))"/>
  </func:function>


  <!-- Extension function validating time string content. -->
  <func:function name="pml:is-time">
    <xsl:param name="x" select="''"/>

    <xsl:variable name="ttokens" select="str:tokenize(string($x), ':')"/>
    <xsl:variable name="h" select="string($ttokens[1])"/>
    <xsl:variable name="m" select="string($ttokens[2])"/>
    <xsl:variable name="s" select="string($ttokens[3])"/>

    <func:result select="count($ttokens) &lt;= 3 and not(contains($x,'::')) and
                         pml:is-integer($h) and string-length($h)=2 and
                         $h &gt;= 0 and $h &lt; 24 and
                         (not($ttokens[2]) or
                           (pml:is-integer($m) and string-length($m)=2 and
                            $m &gt;= 0 and $m &lt; 60)) and
                         (not($ttokens[3]) or
                            (pml:is-real($s) and not(contains($x,'+')) and
                            not(contains($x,'-')) and
                            string-length($s) &gt;= 2 and
                            $s &gt;= 0 and $s &lt; 60))"/>
  </func:function>


  <!-- Extension function validating time zone string content. -->
  <func:function name="pml:is-time-zone">
    <xsl:param name="x" select="''"/>

    <xsl:variable name="ztokens" select="str:tokenize(string($x), ':')"/>
    <xsl:variable name="zh" select="string($ztokens[1])"/>
    <xsl:variable name="zm" select="string($ztokens[2])"/>
    <xsl:variable name="zhsp" select="translate($zh,'+','')"/>

    <func:result select="count($ztokens) &lt;= 2 and not(contains($x,'::')) and
                         pml:is-integer($zh) and string-length($zh) &lt;= 3 and
                         $zhsp &gt;= -12 and $zhsp &lt;= 14 and
                         (not($ztokens[2]) or
                            (pml:is-integer($zm) and string-length($zm)=2 and
                            $zm &gt;= 0 and $zm &lt; 60))"/>
  </func:function>


  <!-- Extension function counting the number of occurences of the
       second argument string in the first argument string. -->
  <func:function name="pml:occurrences">
    <xsl:param name="x" select="''"/>
    <xsl:param name="y" select="''"/>

    <xsl:variable name="n">
      <xsl:choose>
        <xsl:when test="contains($x, $y)">
          <xsl:value-of select="1 + pml:occurrences(substring-after($x, $y),
                                                    $y)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="number('0')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <func:result select="$n"/>
  </func:function>


</xsl:stylesheet>
