#!/bin/sh
# the next line calls 
#     tclsh "$0" "$@"
# where - "$0" is the filename of the script being executed, i.e.  pfm.tcl
#       - "$@" stands for all arguments with which the script was called
# The backslash at the end of this comment makes the next line a continuation
# of the comment for tclsh, but not for sh. So, the next line is only
# executed in 'sh', not in 'tclsh' \
exec tclsh "$0" "$@"
#############################################################################
#
# This is Postgres Forms (pfm), a client application for PostgreSQL.
#
# Copyright (C) 2004-2007 Willem Herremans
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
# 
# Please send bug reports and feature requests by E-mail to the address
# that you can find in procedure cmdAbout.
#
# The home page of pfm can be found at
#
# http://pgfoundry.org/projects/pfm/
#############################################################################

############################################################################
# pfm.tcl script arguments                                                 #
############################################################################

# The pfm.tcl script can be called with 0 or 1 argument(s).
#
# The optional argument should be the fully qualified name of the directory
# in which pfm is installed.
#
# If pfm.tcl is called without arguments, pfm.tcl assumes that it is
# installed in the current working directory. If this assumption is
# wrong, pfm will not be able to find the on-line help files and the
# SQL files that are needed to install the pfm_* tables.
#

package require Tk

########################################################################
# Begin namespace gen                                                  #
########################################################################

# This namespace contains generic components that can be used throughout
# pfm. There are a few global variables, but they have no use outside
# this namespace.

namespace eval gen {

    proc Dialog {parent title message msgWidth defaultButton buttonList} {
	variable dialogChoice

	# This procedure displays a message and a number of buttons.
	# It returns, after the user has pressed a button, the
	# index (starting from 0) of the button pressed.

	# parent: is the toplevel window to which the dialog "belongs".

	# title: is the title for the dialog window

	# message: is the text that is displayed on the dialog

	# msgWidth: width in number of characters of the message.

	# defaultButton: the button that initially gets the input focus.

	# buttonList: a list containing the strings that are displayed
	#      on the buttons

	set dialogChoice 0
	destroy .tpDialog
	toplevel .tpDialog -class Toplevel
	wm transient .tpDialog $parent
	wm title .tpDialog $title
	message .tpDialog.message -width $msgWidth -justify left \
	    -text $message
	set lastButton [expr [llength $buttonList] - 1]
	grid .tpDialog.message -row 0 -column 0 \
	    -columnspan [expr $lastButton + 1]
	for {set button 0} {$button <= $lastButton} {incr button} {
	    set cmdButton [list set ::gen::dialogChoice ${button}]
	    append cmdButton "\ndestroy .tpDialog"
	    button .tpDialog.button${button} \
		-text [lindex $buttonList $button] \
		-command $cmdButton
	    grid .tpDialog.button${button} -row 1 -column $button
	}
	focus .tpDialog.button${defaultButton}
	tkwait window .tpDialog
	return $dialogChoice
    }

    proc PressButton {button} {

	# This procedure simulates a "press button" action and it
	# invokes the command related to the button. It is used
	# as a binding to the keyboard shortcuts.

	if {([info command $button] ne {}) && \
		([$button cget -state] ne {disabled}) } then {
 	    if {[$button cget -relief] eq {raised} } then {
 		$button configure -relief sunken
 		set command "$button configure -relief raised"
 		append command "\n$button invoke"
 		after 100 $command
 	    } else {
 		after idle "$button invoke"
 	    }
	}
	return
    }

    proc ListBox {boundWidget width height title listVar dispListVar currVal} {
	# This procedure displays a listbox and returns the item
	# selected by the user.

	# boundWidget: the widget to which the listbox is related,
	#        usually a button or an entry. It is used to calculate the
	#        listbox's position.

	# width: the width of the listbox window

	# height: the height of the listbox window

	# title: the title of the listbox window

	# listVar : the name of the global variable containing the list
	#        of values related to the listbox

	# dispListVar: the name of the global variable containing
	#        the list of values displayed in the listbox. This
	#        can be the same as list, but the display list can also show 
	#        a longer version of the values as the values in list, e.g. 
	#        a description of the value in list. At any rate both lists
	#        must have the same length and the items with the same index
	#        must be 'related'. The CurrVal and the value returned by 
	#        ListBox are items from list, not from dispList.
	
	# currVal: the current value of the variable that ListBox tries to
	#        get from the user. It is used to determine the initially
	#        selected item of the listbox and also to return the initial
	#        value if the user destroys the ListBox without selecting
	#        anything

	variable SelVal
	variable SearchString
	upvar #0 $listVar list
	upvar #0 $dispListVar dispList


	set SelVal $currVal
	set currIndex [lsearch $list $currVal]
	if {$currIndex < 0} then {
	    set currIndex 0
	}
	set parent [winfo toplevel $boundWidget]
	set x [winfo rootx $boundWidget]
	set y [winfo rooty $boundWidget]
	set geometry "${width}x${height}+${x}+${y}"
	destroy .tpListBox
	toplevel .tpListBox -class Toplevel	
	wm transient .tpListBox $parent
	wm geometry .tpListBox $geometry
	wm title .tpListBox $title
	set SearchString {}
	frame .tpListBox.fm1
	label .tpListBox.fm1.lbSearch -text "Search for:"
	entry .tpListBox.fm1.entSearch -textvariable ::gen::SearchString \
	    -background white -width 0
	set command [list ::gen::SearchListBox $dispListVar]
	append command { $::gen::SearchString}
	# puts $command
	button .tpListBox.fm1.bnSearch -text "Next" -command $command \
	    -takefocus 0 -underline 0 -pady 0
	pack .tpListBox.fm1.lbSearch -side left
	pack .tpListBox.fm1.entSearch -side left -expand 1 -fill x
	pack .tpListBox.fm1.bnSearch -side left
	pack .tpListBox.fm1 -side top -fill x
	frame .tpListBox.fm2
	listbox .tpListBox.fm2.lsb -background white \
	    -yscrollcommand {.tpListBox.fm2.vsb set}
	focus .tpListBox.fm2.lsb
	scrollbar .tpListBox.fm2.vsb -orient vertical \
	    -command {.tpListBox.fm2.lsb yview} -takefocus 0
	pack .tpListBox.fm2.lsb -side left -expand 1 -fill both
	pack .tpListBox.fm2.vsb -side left -fill y
	pack .tpListBox.fm2 -side top -expand 1 -fill both
	foreach item $dispList {
	    .tpListBox.fm2.lsb insert end $item
	}
	.tpListBox.fm2.lsb selection clear 0 end
	.tpListBox.fm2.lsb selection set $currIndex $currIndex
	.tpListBox.fm2.lsb activate $currIndex
	.tpListBox.fm2.lsb see $currIndex
	set command [list ::gen::SetSelected $listVar]
	bind .tpListBox.fm2.lsb <ButtonPress-1> [list after idle $command]
	bind .tpListBox.fm2.lsb <KeyPress-Return> $command
	bind .tpListBox <KeyPress-Escape> {
	    after idle {
		destroy .tpListBox
	    }
	}
	bind .tpListBox <Alt-KeyPress-n> \
	    [list ::gen::PressButton .tpListBox.fm1.bnSearch]
	bind .tpListBox.fm1.entSearch <KeyPress-Return> \
	    [list ::gen::PressButton .tpListBox.fm1.bnSearch]
	tkwait window .tpListBox
	return $SelVal
    }

    proc SetSelected {listVar} {
	variable SelVal
	upvar #0 $listVar list

	# Only meant to be called from ListBox.

	set SelVal [lindex $list [.tpListBox.fm2.lsb curselection]]
	destroy .tpListBox
	return
    }

    proc SearchListBox {listVar searchString} {
	upvar #0 $listVar list

	# Only meant to be called from ListBox.

	set lastIndex [expr [llength $list] - 1]
	set startPosition [expr [lindex [.tpListBox.fm2.lsb curselection] 0] + 1]
	if {$startPosition > $lastIndex} then {
	    .tpListBox.fm2.lsb selection clear 0 end
	    .tpListBox.fm2.lsb selection set 0 0
	    .tpListBox.fm2.lsb see 0
	    .tpListBox.fm2.lsb activate 0
	} else {
	    set newPosition -1
	    for {set index $startPosition} {$index <= $lastIndex} {incr index} {
		if {[string match -nocase "*$searchString*" \
			 [lindex $list $index]]} then {
		    set newPosition $index
		    break
		}
	    }
	    if { $newPosition >= 0 } then {
		.tpListBox.fm2.lsb selection clear 0 end
		.tpListBox.fm2.lsb selection set $newPosition $newPosition
		.tpListBox.fm2.lsb see $newPosition
		.tpListBox.fm2.lsb activate $newPosition
	    } else {
		.tpListBox.fm2.lsb selection clear 0 end
		.tpListBox.fm2.lsb selection set 0 0
		.tpListBox.fm2.lsb see 0
		.tpListBox.fm2.lsb activate 0
	    }
	}
	focus .tpListBox.fm2.lsb
	return
    }

    proc TextEdit {title textvar readOnly buttonList} {
	variable wrapOn
	upvar #0 $textvar text
	variable TextEditId
	variable TextEditGeometry
	variable TextEditVar

	# This procedure displays a text window and, if readOnly=0, it
	# allows the user to edit the text. It returns an identifier
	#  (integer) that can be used later to do some operations on
	# the TextEdit object. See DestroyTextEdit, TextEditExists
	# and UpdateTextEdit.

	# title: the title for the toplevel window that is created.

	# textvar: the NAME of a global variable that contains the text to
	#     be displayed or edited. The modified text is also stored
	#     in that variable

	# readOnly: 0 or 1. Indicates whether the user is allowed to
	#      modify the text.

	# buttonList: a Tcl list in which each item is itself a list with
	# the following items:
	#   1. label: text to display on the button
	#   2. shortcut: 1 character that will be used as an ALT-<char>
	#            shortcut to invoke the button. It must be a character
	#            that is in the 'label'.
	#   3. command: the command to be executed when the button is
	#            invoked. The text widget may be referred to as
	#            TextEdit and may be used in this command.

	if {[info exists TextEditId] } then {
	    incr TextEditId
	} else {
	    set TextEditId 0
	}
	set wrapOn($TextEditId) 1
	set tpName .textEdit$TextEditId
	set TextEditVar($TextEditId) $textvar
	destroy $tpName
	toplevel $tpName -class Toplevel
	if {[info exists TextEditGeometry(width)] && \
		[info exists TextEditGeometry(height)]} then {
	    set geometry "$TextEditGeometry(width)"
	    append geometry "x$TextEditGeometry(height)"
	    wm geometry $tpName $geometry
	} else {
	    wm geometry $tpName "600x300"
	}
	wm title $tpName $title
 	set tags [bindtags $tpName]
 	lappend tags TextEdit$TextEditId
 	bindtags $tpName $tags
 	set command "set ::gen::TextEditGeometry(width) "
 	append command "\[winfo width .textEdit$TextEditId\]"
 	append command "\nset ::gen::TextEditGeometry(height) "
 	append command "\[winfo height .textEdit$TextEditId\]"
 	bind TextEdit$TextEditId <Configure> \
 	    [list after idle [list catch $command]]
	

	# text and vertical scrollbar widgets
	frame ${tpName}.fm1
	text ${tpName}.text -background white -wrap word \
	    -yscrollcommand [list ${tpName}.vsb set] \
	    -xscrollcommand [list ${tpName}.hsb set] \
	    -width 1 -height 1
	scrollbar ${tpName}.vsb -orient vertical \
	    -command [list ${tpName}.text yview] \
	    -takefocus 0
	pack ${tpName}.text -in ${tpName}.fm1 \
	    -side left -expand 1 -fill both
	pack ${tpName}.vsb  -in ${tpName}.fm1 \
	    -side left -fill y

	# horizontal scrollbar + filler widgets

	frame ${tpName}.fm2
	scrollbar ${tpName}.hsb -orient horizontal \
	    -command [list ${tpName}.text xview] \
	    -takefocus 0
	frame ${tpName}.filler1 -width 20 -height 20
	pack ${tpName}.hsb -in ${tpName}.fm2 \
	    -side left -expand 1 -fill x
	pack ${tpName}.filler1 -in ${tpName}.fm2 \
	    -side left

	# buttons bar

	frame ${tpName}.fm3

	# Cancel button
	button ${tpName}.bnCancel -text Cancel -underline 0 \
	    -command [list destroy ${tpName}]
	bind ${tpName} <Alt-KeyPress-c> \
	    [list ::gen::PressButton ${tpName}.bnCancel]
	bind ${tpName} <KeyPress-Escape> \
	    [list ::gen::PressButton ${tpName}.bnCancel]
	pack ${tpName}.bnCancel -in ${tpName}.fm3 -side right

	# OK button
	if {!$readOnly} then {
	    button ${tpName}.bnOK -text OK -underline 0 -command \
		[list ::gen::ReturnText $tpName $textvar]
	    bind ${tpName} <Alt-KeyPress-o> \
		[list ::gen::PressButton ${tpName}.bnOK]
	    pack ${tpName}.bnOK -in ${tpName}.fm3 -side right
	}
	# Custom buttons
	set bnr 0
	foreach buttonDef $buttonList {
	    set label [lindex $buttonDef 0]
	    set shortcut [lindex $buttonDef 1]
	    set command [lindex $buttonDef 2]
	    set command [string map "TextEdit ${tpName}.text" $command]
	    set underline [string first $shortcut [string tolower $label]]
	    button ${tpName}.button${bnr} -text $label \
		-underline $underline -command $command
	    bind ${tpName} <Alt-KeyPress-${shortcut}> \
		[list ::gen::PressButton ${tpName}.button${bnr}]
	    pack ${tpName}.button${bnr} -in ${tpName}.fm3 -side right
	    incr bnr
	}

	# Readonly label
	if {$readOnly} then {
	    label ${tpName}.lbState -text {READ ONLY} -takefocus 0 \
		-foreground {medium blue}
	    pack ${tpName}.lbState -in ${tpName}.fm3 -side right
	}

	# Search

	frame ${tpName}.frmFind -borderwidth 2 -relief ridge
 	button ${tpName}.btnFind -text {Find} -pady 0 \
 	    -underline 0 -takefocus 0 -command \
 	    [list ::gen::cmdSearchText ${tpName}]
 	pack ${tpName}.btnFind -in ${tpName}.frmFind -side right \
	    -padx 2 -pady 2
	entry ${tpName}.entSearch -text {} -width 10 -background white
	pack ${tpName}.entSearch -in ${tpName}.frmFind -side right \
	    -fill both -expand 1 -padx 2 -pady 2
	pack ${tpName}.frmFind -in ${tpName}.fm3 -side right -expand 1 -fill both
 	bind ${tpName} <Alt-KeyPress-f> \
 	    [list ::gen::PressButton ${tpName}.btnFind]
 	bind ${tpName}.entSearch <KeyPress-Return> \
 	    [list ::gen::PressButton ${tpName}.btnFind]
	    
	# radio buttons for Wrap/Truncate
	radiobutton ${tpName}.rnTruncate -text {Truncate} -value 0 \
	    -variable ::gen::wrapOn($TextEditId) -takefocus 0 \
	    -command [list ${tpName}.text configure -wrap none] -underline 0
	pack ${tpName}.rnTruncate -in ${tpName}.fm3 -side right
	radiobutton ${tpName}.rbWrap -text {Wrap} -value 1 \
	    -underline 0 \
	    -variable ::gen::wrapOn($TextEditId) -takefocus 0 \
	    -command [list ${tpName}.text configure -wrap word]
	pack ${tpName}.rbWrap -in ${tpName}.fm3 -side right
	bind ${tpName} <Alt-KeyPress-w> \
	    [list ::gen::PressButton ${tpName}.rbWrap]
	bind ${tpName} <Alt-KeyPress-t> \
	    [list ::gen::PressButton ${tpName}.rnTruncate]
	
	# Pack all the frames
	pack ${tpName}.fm1 -in $tpName -side top \
	    -expand 1 -fill both
	pack ${tpName}.fm2 -in $tpName -side top -fill x
	pack ${tpName}.fm3 -in $tpName -side top -fill x

	# Fill text widget

	${tpName}.text insert end $text
	${tpName}.text mark set insert 1.0
	if {$readOnly} then {
	    ${tpName}.text configure -background $::pfm::readOnlyBackground \
		-state disabled
	}
	focus ${tpName}.text
	return $TextEditId
    }

    proc ReturnText {tpName textvar} {
	upvar #0 $textvar text

	# Only meant to be called from TextEdit

	set text [${tpName}.text get 1.0 "end -1 chars"]
	destroy $tpName
	return
    }

    proc DestroyTextEdit {Id} {
	variable wrapOn
	variable TextEditVar

	set tpName .textEdit$Id
	destroy $tpName
	unset wrapOn($Id)
	unset TextEditVar($Id)
	return
    }

    proc TextEditExists {Id} {

	return [winfo exists .textEdit$Id]
    }

    proc TextEditName {Id} {
	
	if {[winfo exists .textEdit$Id]} then {
	    set name [wm title .textEdit$Id]
	} else {
	    set name {}
	}
	return $name
    }

    proc UpdateTextEdit {Id} {
	variable TextEditVar
	upvar #0 $TextEditVar($Id) text

	set tpName .textEdit$Id
	${tpName}.text configure -state normal
	${tpName}.text delete 1.0 end
	${tpName}.text insert end $text
	${tpName}.text configure -state disabled
	return
    }

    proc cmdSearchText {tpName} {

	focus ${tpName}.entSearch
	set pattern [${tpName}.entSearch get]
	if {[string length $pattern]} then {
	    set searchPosition [${tpName}.text index insert]
	    ${tpName}.text tag delete match
	    set searchPosition [${tpName}.text search -nocase $pattern $searchPosition end]
	    if {$searchPosition ne {}} then {
		set endmatch [${tpName}.text index "$searchPosition +[string length $pattern] chars"]
		${tpName}.text tag add match $searchPosition $endmatch
		${tpName}.text tag configure match -background yellow
		${tpName}.text mark set insert $endmatch
		${tpName}.text see insert
	    } else {
		set message "End of file reached.\nFind will return to begin of file now."
		tk_messageBox -type ok -icon info -parent ${tpName} \
		    -message $message
		${tpName}.text mark set insert 1.0
		${tpName}.text see insert
	    }
	}
	return
    }


}

########################################################################
# End of namespace gen                                                 #
########################################################################


########################################################################
# Begin namespace options                                              #
########################################################################

# pfmOptions is an array with all the options for Postgres Forms.
# Up to now, the options are:
#
#     - browser : a web browser used to display the help file.
#
#     - dblist : a list of data base names from which the user
#       can choose one at Open data base.
#
#     - host, port, user and dbname: the default values for the
#       connection parameters proposed by Open data base.
#
#     - fontmonospace: a monospace font to use for the GUI
#
#     - fontproportional : a proportional font to use for the GUI
#
#     - printcmd : UNIX command which accepts data on stdin and sends
#       them to the printer.
#
#     - printencoding: character encoding used by pfm to send text to print command.
#
#     - psql: the command used to call psql.
#
#     - tmpdir : the temporary directory used by pfm
#
#     - usePGPASSWORD: whether or not to use the environment variable PGPASSWORD
#                      to store the password.
#
# pfmOptions are stored in file ~/.pfmrc. This file is read by proc initOptions.
#
# On Windows platforms the options are stored in %APPDATA%\pfm\pfm.conf
#
# pfmOptions can be modified by menu Tools -> Options.
#


namespace eval options {
    variable pfmOptions
    variable newOptions
    variable optionList
    variable fontFam
    variable fontSize
	
    proc getDefault {option} {
	global env
	global tcl_platform
	variable ::pfm::pfmConfig

	switch -- $option {
	    "browser" {
		switch -- $tcl_platform(platform) {
		    "unix" {
			set value $pfmConfig(defaultBrowser)
		    }
		    "windows" {
			if {[info exists env(ProgramFiles)]} then {
			    set iexplorer \
				[file normalize [file join $env(ProgramFiles) \
						     {Internet Explorer} \
						     {iexplore.exe}]]
			    set value [list $iexplorer %s]
			} else {
			    set value {iexplore.exe %s}
			}
		    }
		    default {
			set value {}
		    }
		}
	    }
	    "dblist" {
		set value [list $tcl_platform(user)]
	    }
	    "dbname" {
		set value $tcl_platform(user)
	    }
	    "fontmonospace" {
		set value [list courier -12]
	    }
	    "fontproportional" {
		set value [list helvetica -12]
	    }
	    "fontstyle" {
		set value normal
	    }
	    "host" {
		set value {}
	    }
	    "port" {
		set value {5432}
	    }
	    "printcmd" {
		switch -- $tcl_platform(platform) {
		    "unix" {
			set value $pfmConfig(defaultPrintcmd)
		    }
		    "windows" {
			if {[info exists env(ProgramFiles)]} then {
			    set wordpad \
				[file normalize [file join $env(ProgramFiles) \
						     {Windows NT} \
						     {Bureau-accessoires} \
						     {wordpad.exe}]]
			    set value [list $wordpad %s]
			} else {
			    set value {wordpad.exe %s}
			}
		    }
		}
	    }
	    "printencoding" {
		set value [encoding system]
	    }
	    "psql" {
		switch -- $tcl_platform(platform) {
		    "unix" {
			set value {psql}
		    }
		    "windows" {
			set value {psql.exe}
		    }
		    default {
			set value {}
		    }
		}
	    }
	    "tmpdir" {
		switch -- $tcl_platform(platform) {
		    "unix" {
			set value {/tmp}
		    }
		    "windows" {
			if {[info exists env(TEMP)]} then {
			    set value [file normalize $env(TEMP)]
			} else {
			    set value [file normalize "~/tmp"]
			}
		    }
		    default {
			set value {}
		    }
		}
	    }
	    "usePGPASSWORD" {
		set value {yes}
	    }
	    "user" {
		set value $tcl_platform(user)
	    }
	    default {
		set value {}
	    }
	}
	return $value
    }

    proc setOneDefault {option} {
	variable newOptions
	variable fontFam
	variable fontSize

	set newOptions($option) [getDefault $option]
	if {($option eq {fontproportional}) || \
		($option eq {fontmonospace})} then {
	    set fontFam($option) [lindex $newOptions($option) 0]
	    set fontSize($option) [expr - [lindex $newOptions($option) 1]]
	}
	return
    }

    proc setDefaultOptions {OptionsName erase} {
	# If $erase, all options are set to their default.
	# Else, only the options that are not defined yet are set to
	# their defaults.
	# This makes it possible to add new options in new versions of pfm
	# without having to convert the .pfmrc file.
	# From version 1.0.4. on, this procedure does not call
	# pg_conndefaults anymore.

	upvar $OptionsName Options
	variable optionList
	variable fontFam
	variable fontSize

	foreach option $optionList {
	    if {![info exists Options($option)] || $erase} then {
		set Options($option) [getDefault $option]
		if {[lsearch {fontmonospace fontproportional} $option] \
			>= 0} then {
		    set fontFam($option) [lindex $Options($option) 0]
		    set fontSize($option) \
			[expr - [lindex $Options($option) 1]]
		}
	    }
	}
	return
    }

    proc saveOptions {} {

	variable pfmOptions
	variable optionList
	global tcl_platform
	global env

	if {$tcl_platform(platform) eq {windows}} then {
	    set filename [file join $env(APPDATA) pfm pfm.conf]
	    set dirname [file join $env(APPDATA) pfm]
	    if {![file exists $dirname]} then {
		file mkdir $dirname
	    }
	} else {
	    set filename "~/.pfmrc"
	}
	set rcFile [open $filename w]
	foreach option $optionList {
	    puts $rcFile [list $option $pfmOptions($option)]
	}
	close $rcFile
	return
    }

    proc initOptions { } {

	variable pfmOptions
	variable optionList
	global tcl_platform
	global env
	variable TextEditList

	set optionList {}
	lappend optionList browser
	lappend optionList dblist
	lappend optionList dbname
	lappend optionList fontmonospace
	lappend optionList fontproportional
	lappend optionList fontstyle
	lappend optionList host
	lappend optionList port
	lappend optionList printcmd
	lappend optionList printencoding
	lappend optionList psql
	lappend optionList tmpdir
	lappend optionList usePGPASSWORD
	lappend optionList user
	set TextEditList {}
	if {$tcl_platform(platform) eq {windows}} then {
	    set filename [file join $env(APPDATA) pfm pfm.conf]
	} else {
	    set filename "~/.pfmrc"
	}
	if { [file exists $filename]} then {
	    set rcFile [open $filename r]
	    while {![eof $rcFile]} {
		set line [gets $rcFile]
		if {[string length $line] != 0} then {
		    if {[catch {
			set pfmOptions([lindex $line 0]) [lindex $line 1]
		    } errMsg]} then {
			append errMsg "\nError parsing '${line}' in options file '$filename'."
			append errMsg "\nCheck options using Tools -> Options."
			tk_messageBox -type ok -icon error -message $errMsg
		    }
		}
	    }
	    close $rcFile
	    # Next lines make it possible to add new options in new versions of
	    # pfm without having to convert the .pfmrc file: all options not
	    # present in .pfmrc, are set to their default values.
	    setDefaultOptions pfmOptions 0
	    setFonts
	    saveOptions
	} else {
	    setDefaultOptions pfmOptions 1
	    setFonts
	    saveOptions
	}
	return
    }

    proc setFonts {} {
	# This procedure defines the named fonts prop, propbold, mono and
	# monobold.
	# It also defines the rules for the fonts in the options database.
	# From version 1.4.3 on, all fonts are set here, instead of at the
	# time of the widget creation.

	variable pfmOptions

	if {[lsearch -exact [font names] prop] >= 0} then {
	    font delete prop
	}
	if {[lsearch -exact [font names] propbold] >= 0} then {
	    font delete propbold
	}
	if {[lsearch -exact [font names] mono] >= 0} then {
	    font delete mono
	}
	if {[lsearch -exact [font names] monobold] >= 0} then {
	    font delete monobold
	}
	if {$pfmOptions(fontstyle) eq {bold}} then {
	    set weight bold
	} else {
	    set weight normal
	}
	set fontsdefined 0
	while {!$fontsdefined} {
	    if {[catch {
		font create prop \
		    -family [lindex $pfmOptions(fontproportional) 0] \
		    -size [lindex $pfmOptions(fontproportional) 1] \
		    -weight normal
		font create propbold \
		    -family [lindex $pfmOptions(fontproportional) 0] \
		    -size [lindex $pfmOptions(fontproportional) 1] \
		    -weight $weight
	    } errMsg]} then {
		set pfmOptions(fontproportional) [getDefault fontproportional]
		append errMsg "\nOption 'fontproportional' has been reset to $pfmOptions(fontproportional)"
		tk_messageBox -type ok -icon warning -message $errMsg
	    } else {
		set fontsdefined 1
	    }
	}
	set fontsdefined 0
	while {!$fontsdefined} {
	    if {[catch {
		font create mono \
		    -family [lindex $pfmOptions(fontmonospace) 0] \
		    -size [lindex $pfmOptions(fontmonospace) 1] \
		    -weight normal
		font create monobold \
		    -family [lindex $pfmOptions(fontmonospace) 0] \
		    -size [lindex $pfmOptions(fontmonospace) 1] \
		    -weight $weight
	    } errMsg]} then {
		set pfmOptions(fontmonospace) [getDefault fontmonospace]
		append errMsg "\nOption 'fontmonospace' has been reset to $pfmOptions(fontmonospace)"
		tk_messageBox -type ok -icon warning -message $errMsg
	    } else {
		set fontsdefined 1
	    }
	}

	# At http://wiki.tcl.tk/1062, I have found how
	# to change the font of the tk_messageBox:
	#     option add *Dialog.msg.font prop.
	# At http://wiki.tcl.tk/10424, I have found how to put all font
	# settings at 1 point. I have used that here:
	option clear
	option add *font propbold
	option add *report.fmReport.lsb.font monobold
	option add *Entry.font prop
	option add *Text.font mono
	option add *Entry.readonlyBackground $::pfm::readOnlyBackground
	option add *Entry.highlightThickness 1
	option add *Button.highlightThickness 1
	return
    }

    proc cmdOptions {} {

	variable pfmOptions
	variable newOptions
	variable optionList
	variable fontFam
	variable fontSize

	destroy .options
	toplevel .options -class Toplevel
	wm title .options "pfm - Options"
	frame .options.table -borderwidth 2 -relief sunken
	set rowidx 0
	foreach option $optionList {
	    set newOptions($option) $pfmOptions($option)
	    label .options.lb$option -text $option
	    frame .options.fm$option
	    button .options.def$option -text {Default} -underline 0 \
		-command [list ::options::setOneDefault $option] \
		-takefocus 0 -pady 0
	    button .options.help$option -text {Help} -underline 0 \
		-command [list ::options::cmdOptionHelp $option] \
		-takefocus 0 -pady 0
	    grid .options.lb$option -in .options.table \
		-column 0 -row $rowidx
	    grid .options.fm$option -in .options.table \
		-column 1 -row $rowidx -sticky we
	    grid .options.def$option -in .options.table \
		-column 3 -row $rowidx
	    grid .options.help$option -in .options.table \
		-column 4 -row $rowidx
	    switch -- $option {
		"fontmonospace" -
		"fontproportional" {
		    set fontFam($option) [lindex $newOptions($option) 0]
		    set fontSize($option) [lindex $newOptions($option) 1]
		    set fontSize($option) [expr - $fontSize($option)]
		    button .options.fm$option.entrybutton -pady 0 -padx 0 \
			-textvariable ::options::fontFam($option) \
			-command [list ::options::getFontFam $option] \
			-width 0 -anchor w
		    spinbox .options.fm$option.spinbox \
			-textvariable ::options::fontSize($option) \
			-from -30.0 -to 30.0 -increment 1.0 -format %1.0f \
			-width 4 -background white -takefocus 0
		    pack .options.fm$option.entrybutton \
			-in .options.fm$option \
			-side left -expand 1 -fill x
		    pack .options.fm$option.spinbox \
			-in .options.fm$option \
			-side left
		}
		"fontstyle" {
		    radiobutton .options.fm$option.entrybutton -text "Normal" \
			-variable ::options::newOptions($option) \
			-value {normal}
		    radiobutton .options.fm$option.entrybutton2 -text "Bold" \
			-variable ::options::newOptions($option) \
			-value {bold}
		    pack .options.fm$option.entrybutton -in .options.fm$option \
			-side left -expand 1 -fill x
		    pack .options.fm$option.entrybutton2 -in .options.fm$option \
			-side left -expand 1 -fill x
		}
		"printencoding" {
		    button .options.fm$option.entrybutton -pady 0 -padx 0 \
			-textvariable ::options::newOptions($option) \
			-anchor w -command \
			[list ::options::getPrintEncoding \
			     ::options::newOptions(printencoding) \
			     .options.fm$option.entrybutton]
		    pack .options.fm$option.entrybutton -in .options.fm$option \
			-side left -expand 1 -fill x
		}
		"psql" {
		    button .options.fm$option.entrybutton -pady 0 -padx 0 \
			-textvariable ::options::newOptions($option) \
			-anchor w -command ::options::getPsqlLocation
		    pack .options.fm$option.entrybutton -in .options.fm$option \
			-side left -expand 1 -fill x
		}
		"tmpdir" {
		    button .options.fm$option.entrybutton -pady 0 -padx 0 \
			-textvariable ::options::newOptions($option) \
			-anchor w -command ::options::getTmpdir
		    pack .options.fm$option.entrybutton -in .options.fm$option \
			-side left -expand 1 -fill x
		}
		"usePGPASSWORD" {
		    radiobutton .options.fm$option.entrybutton -text "Yes" \
			-variable ::options::newOptions($option) \
			-value {yes}
		    radiobutton .options.fm$option.entrybutton2 -text "No" \
			-variable ::options::newOptions($option) \
			-value {no}
		    pack .options.fm$option.entrybutton -in .options.fm$option \
			-side left -expand 1 -fill x
		    pack .options.fm$option.entrybutton2 -in .options.fm$option \
			-side left -expand 1 -fill x
		}
		default {
		    entry .options.fm$option.entrybutton \
			-textvar ::options::newOptions($option) \
			-width 30 -background white
		    button .options.exp$option -text "Expand" \
			-underline 1 -pady 0 -takefocus 0 \
			-command [list ::options::cmdExpand $option \
				     ::options::newOptions($option) ]
		    bind .options.fm$option.entrybutton <Alt-KeyPress-x> \
			[list ::gen::PressButton .options.exp$option]
		    pack .options.fm$option.entrybutton -in .options.fm$option \
			-side left -expand 1 -fill x
		    grid .options.exp$option -in .options.table \
			-column 2 -row $rowidx

		}
	    }
	    bind .options.fm$option.entrybutton <Alt-KeyPress-d> \
		[list ::gen::PressButton .options.def$option]
	    bind .options.fm$option.entrybutton <Alt-KeyPress-h> \
		[list ::gen::PressButton .options.help$option]
	    if {[winfo exists .options.fm$option.entrybutton2]} then {
		bind .options.fm$option.entrybutton2 <Alt-KeyPress-d> \
		    [list ::gen::PressButton .options.def$option]
		bind .options.fm$option.entrybutton2 <Alt-KeyPress-h> \
		    [list ::gen::PressButton .options.help$option]
	    }
	    incr rowidx
	}
	pack .options.table -in .options -side top
	bind .options <KeyPress-Down> \
	    {focus [tk_focusNext [focus -displayof .options]]}
	bind .options <KeyPress-Up> \
	    {focus [tk_focusPrev [focus -displayof .options]]}
	frame .options.buttons -borderwidth 2 -relief raised
	button .options.btnOK -text {OK} \
	    -command [namespace code cmdOptionOK] -underline 0
	button .options.btnCancel -text {Cancel} \
	    -command [namespace code cmdOptionCancel] -underline 0
	pack .options.btnCancel -in .options.buttons -side right
	pack .options.btnOK -in .options.buttons -side right
	pack .options.buttons -in .options -side top -fill x
	focus .options.fm[lindex $optionList 0].entrybutton
	bind .options <Alt-KeyPress-o> "::gen::PressButton .options.btnOK"
	bind .options <Alt-KeyPress-c> "::gen::PressButton .options.btnCancel"
	bind .options <KeyPress-Escape> "::gen::PressButton .options.btnCancel"
	return
    }

    proc getFontFam {option} {
	variable fontFam
	variable fontfamilies

	set fontfamilies {}
	foreach family [lsort [font families]] {
	    switch -- $option {
		"fontproportional" {
		    if {![font metrics [list $family] -fixed]} then {
			lappend fontfamilies $family
		    }
		}
		"fontmonospace" {
		    if {[font metrics [list $family] -fixed]} then {
			lappend fontfamilies $family
		    }
		}
	    }
	}
	set fontFam($option) \
	    [::gen::ListBox .options.fm${option}.entrybutton \
		 300 200 $option ::options::fontfamilies \
		 ::options::fontfamilies $fontFam($option)]
	return
    }

    proc getDefaultFont {option} {
	variable fontFam
	variable fontSize

	set defFont [getDefault $option]
	set fontFam($option) [lindex $defFont 0]
	set fontSize($option) [expr - [lindex $defFont 1]]
	return
    }

    proc getPrintEncoding {printencodingVar boundWidget} {
	upvar #0 $printencodingVar printencoding
	variable encodingList

	set title "Select printencoding"
	set encodingList [lsort [encoding names]]
	set printencoding \
	    [::gen::ListBox $boundWidget 300 300 $title \
		 ::options::encodingList ::options::encodingList \
		 $printencoding]
	return
    }

    proc getPsqlLocation {} {
	variable newOptions
	global tcl_platform
	global env

	switch -- $tcl_platform(platform) {
	    "windows" {
		if {[info exists env(ProgramFiles)]} then {
			set initialdir [file join $env(ProgramFiles) PostgreSQL]
		} else {
			set initialdir {C:/Program Files/PostgreSQL}
		}
		set filetypes {
		    {{Executable} {.exe}}
		    {{All Files} *}
		}
	    }
	    "unix" -
	    default {
		set initialdir {/usr/bin}
		set filetypes {
		    {{All Files} *}
		}
	    }
	}
	set newvalue \
	    [tk_getOpenFile \
		 -initialdir $initialdir \
		 -parent .options \
		 -title "Choose psql executable" \
		 -filetypes $filetypes]
	if {[string length $newvalue]} then {
	    set newOptions(psql) $newvalue
	}
	return
    }

    proc getTmpdir {} {
	variable newOptions

	set newvalue \
	    [tk_chooseDirectory \
		 -initialdir [list $newOptions(tmpdir)] \
		 -parent .options \
		 -title "Choose tmpdir"]
	if {[string length $newvalue]} then {
	    set newOptions(tmpdir) $newvalue
	}
	return
    }

    proc cmdExpand {option textvar} {
	variable TextEditList

	set title "pfm - Options: $option"
	set buttonList {}
	if {($option eq {browser}) || ($option eq {printcmd}) \
		|| ($option eq {psql})} then {
	    # button Paste filename
	    set label {Paste filename}
	    set shortcut p
	    set command {
		TextEdit insert insert \
		    [file normalize [tk_getOpenFile -title {Paste filename} \
					 -parent TextEdit]]
	    }
	    lappend buttonList [list $label $shortcut $command]
	}
	# Button Default
	set label {Default}
	set shortcut d
	set command "TextEdit delete 1.0 end\n"
	set defVal [getDefault $option]
	append command \
	    "TextEdit insert end [list $defVal]"
	lappend buttonList [list $label $shortcut $command]
	# Button Help
	set label Help
	set shortcut h
	set command "::options::cmdOptionHelp $option"
	lappend buttonList [list $label $shortcut $command]
	lappend TextEditList \
	    [::gen::TextEdit $title $textvar 0 $buttonList]
	return
    }

    proc cmdOptionHelp {option} {
	variable optionHelp
	variable TextEditList
	variable ::pfm::installDir
	variable ::pfm::pfmConfig
	
	set fileName [file join $installDir $pfmConfig(helpDir) \
			  help_${option}.txt]
	if {[catch {open $fileName r} helpChannel]} then {
	    set optionHelp {No help available}
	} else {
	    set optionHelp [read $helpChannel]
	    close $helpChannel
	}
	set title "pfm - Help for option '$option'"
	lappend TextEditList \
	    [::gen::TextEdit $title ::options::optionHelp 1 {}]
	return
    }

    proc cmdOptionOK {} {

	variable pfmOptions
	variable newOptions
	variable fontFam
	variable fontSize
	variable ::pfm::currentDB
	variable optionList
	variable TextEditList

	foreach Id $TextEditList {
	    ::gen::DestroyTextEdit $Id
	}
	set TextEditList {}
	destroy .options
	foreach option $optionList {
	    switch -- $option {
		"fontmonospace" -
		"fontproportional" {
		    if {[catch {
			set pfmOptions($option) \
			    [list $fontFam($option) [expr -$fontSize($option)]]
			# puts $pfmOptions($option)
		    } errMsg]} then {
			append errMsg "\nOption '$option' has not been modified."
			tk_messageBox -type ok -icon error -message $errMsg
		    }
		}
		default {
		    set pfmOptions($option) $newOptions($option)
		}
	    }
	}
	setFonts
	saveOptions
	return
    }

    proc cmdOptionCancel {} {
	variable TextEditList

	foreach Id $TextEditList {
	    ::gen::DestroyTextEdit $Id
	}
	set TextEditList {}
	destroy .options
	return
    }

    proc addToDBlist {dbName} {
	variable pfmOptions

	if { [lsearch $pfmOptions(dblist) $dbName] == -1 } then {
	    lappend pfmOptions(dblist) $dbName
	    set pfmOptions(dblist) [lsort $pfmOptions(dblist)]
	}
	set pfmOptions(dbname) $dbName
	saveOptions
	return
    }


    ###############################################################
    # Main of namespace options                                   #
    ###############################################################

    # initOptions has to be postponed until configuration has
    # been read. It is moved to the main of namespace pfm


}

###############################################################
# End of namespace options                                    #
###############################################################


################################################################
#                                                              #
# Begin of namespace pfm                                       #
#                                                              #
################################################################

# widget is an associative array containing aliases for widget path names
#
# dbName is the name of the currently open data base. It is filled out
# by proc cmdOpenOK.
#
# passMatrix an associative array which is filled by readPgPass with the 
# contents of the ~/.pgpass file:
#      passMatrix($i,hostname) = hostname of $i-th entry in pgpass
#      passMatrix($i,port) = port of $i-th entry in pgpass
#      passMatrix($i,database) = database name of $i-th entry in pgpass
#      passMatrix($i,username) = username of $i-th entry in pgpass
#      passMatrix($i,password) = password of $i-th entry in pgpass
#
# psqlChannel the channel linked to the psql command pipeline. It is set
# in cmdOpenOK and unset in cmdCloseDataBase.
#
# currentDB contains the postgres data base handle for the open data base
# It is filled in by proc cmdOpenDataBase, and it is used throughout this
# application.
#
# formsArray contains all the tuples of pfm_form, where
# formsArray($name,$attribute) contains the value of attribute '$attribute'
# in the tuple of pfm_form for which name=$name. It is also filled by 
# refreshFormsList.
#
# formsIndex is an array for which formsIndex($n) contains the
# name of the form in the n-th form in the forms listbox. It is
# filled by refreshFormsList.
#
# connInfoList: the list of parameters for the conninfo in pg_connect.
#
# pfmMode can be either:
#
#    -  "normal": only the forms with showform = true are displayed;  or
#
#    -  "design": only the forms with showform = false are displayed.
#
# pfmConfig is an array containing configuration parameters. On UNIX platforms,
# pfmConfig is read from the config-file pfm.conf in pfm's installation directory 
# or in /etc. On the Windows platform pfmConfig is hardcoded.
# Currently the array elements are:
#
#    - docDir : the directory in which the on-line documentation is installed.
#    - helpDir : the directory in which the options on-line help texts and the
#                license are installed.
#    - exampleDir : the directory in which the example databases are installed.
#    - defaultBrowser : the default value for the "browser" option
#    - defaultPrintcmd : the default value for the "printcmd" option.

namespace eval pfm {

    variable pfmVersion {1.5.4}
    variable API
    variable pfmConfig
    variable installDir
    variable currentDB
    variable psqlChannel
    variable errChannel
    variable formsArray
    variable formsIndex
    variable widget
    variable dbName
    variable pfmMode {normal}
    variable connInfoList
    variable passMatrix
    variable readOnlyBackground

    proc initRootWindow { } {

	variable widget

	wm title . "pfm - No data base opened"
	wm geometry . 350x300

	# Menubar of root window
	menu .menubar -tearoff 0
	.menubar add cascade -menu .menubar.database -label {Database} \
	    -underline 0
	.menubar add command -label {SQL} -underline 0 \
	    -command {::report::cmdReportSQL sql} -state disabled
	.menubar add command -label {Reports} -underline 0 \
	    -command {::report::cmdReportSQL report} -state disabled
	.menubar add cascade -menu .menubar.tools -label {Tools} \
	    -underline 0
	.menubar add cascade -menu .menubar.help -label {Help} \
	    -underline 0

	# Make it the root window's bar menu
	. configure -menu .menubar

	# Database pull down menu
	menu .menubar.database -tearoff 0
	.menubar.database add command \
	    -command [namespace code {cmdOpenDataBase}] \
	    -label "Open ..." -underline 0 \
	    -accelerator {Cntrl-o}
	.menubar.database add command \
	    -command [namespace code cmdCloseDataBase] -label Close \
	    -state disabled -underline 0 \
	    -accelerator {Cntrl-w}
	.menubar.database add command \
	    -command {destroy .} -label Quit \
	    -underline 0 \
	    -accelerator {Cntrl-q}

	# Tools pull down menu
	menu .menubar.tools -tearoff 0
	.menubar.tools add command \
	    -command ::pfm::installPfm \
	    -label "Install pfm_* tables" -state disabled \
	    -underline 8
	.menubar.tools add command \
	    -command ::pfm::cmdInstallExample \
	    -label "Install example database ..." -state disabled \
	    -underline 8
	.menubar.tools add command \
	    -command ::options::cmdOptions -label "Options" \
	    -underline 0

	# Help menu
	menu .menubar.help -tearoff 0
	.menubar.help add command \
	    -command ::help::cmdDisplayManual -label {Help file} \
	    -underline 0 -accelerator {F1}
	.menubar.help add command \
	    -command ::help::cmdLicense -label {License} \
	    -underline 0
	.menubar.help add command \
	    -command [namespace code cmdAbout] -label About \
	    -underline 0
	bind . <KeyPress-F1> ::help::cmdDisplayManual

	# The radio buttons
	frame .fmRadioButtons
	radiobutton .rbNormal -text {Normal mode} -value {normal} \
	    -variable ::pfm::pfmMode -command ::pfm::refreshFormsList \
	    -state disabled -takefocus 0 \
	    -underline 0
	radiobutton .rbDesign -text {Design mode} -value {design} \
	    -variable ::pfm::pfmMode -command ::pfm::refreshFormsList \
	    -state disabled -takefocus 0 \
	    -underline 1
	bind . <Alt-KeyPress-n> "::gen::PressButton .rbNormal"
	bind . <Alt-KeyPress-e> "::gen::PressButton .rbDesign"
	# The open button

	button .btnOpen \
	    -command {::pfm::cmdOpenQuery} \
	    -padx 0 -text {Open form} \
	    -underline 0 \
	    -state disabled
	bind . <Alt-KeyPress-o> {::gen::PressButton .btnOpen}

	# Listbox + Scrollbar  + Title
	frame .fmListBox
	label .lbList -text "List of forms"
	listbox .lsbForms \
	    -yscrollcommand {.vsb set} -background white -width 1 -height 1
	scrollbar .vsb -orient vertical \
	    -command {.lsbForms yview} -takefocus 0
	set widget(lsbForms) .lsbForms
	bind .lsbForms <KeyPress-Return> ::pfm::cmdOpenQuery

	# Now pack all the widgets
	pack .rbNormal -in .fmRadioButtons -side left -expand 1 -fill x
	pack .rbDesign -in .fmRadioButtons -side left -expand 1 -fill x
	pack .lsbForms -in .fmListBox -side left -expand 1 -fill both
	pack .vsb -in .fmListBox -side left -fill y

	pack .fmRadioButtons -in . -side top -fill x
	pack .lbList -in . -side top -fill x
	pack .fmListBox -in . -side top -expand 1 -fill both
	pack .btnOpen -in . -side top -fill x
	focus .lsbForms


	# Bindings

	# Accelerators

	bind . <Control-KeyPress-o> {::pfm::cmdOpenDataBase}
	bind . <Control-KeyPress-w> {::pfm::cmdCloseDataBase}
	bind . <Control-KeyPress-q> {destroy .}
	bind . <Control-KeyPress-s> {::report::cmdReportSQL sql}
	bind . <Control-KeyPress-r> {::report::cmdReportSQL report}

	# Cleanup no matter how user exits pfm.
	# Bug 1052: The initial solution:
	#     "bind . <Destroy> ::pfm::cmdExit"
	# caused cmdExit to be called for every <Destroy> event to any of
	# the root window's child widgets. So, when pfm's main window is
	# destroyed, cmdExit, which in turn calls "exit", wass called multiple
	# times. Sometimes, this caused the Tcl/Tk interpreter to get locked.
	# Here is the more sofisticated solution.

	# Add a bindtag to the toplevel .
	set tags [bindtags .]
	lappend tags MainWindow
	bindtags . $tags
	bind MainWindow <Destroy> ::pfm::cmdExit
	# This binding causes cmdExit to be called only when
	# pfm's main window is destroyed.

	# Use Return rather than Space to invoke buttons.
	bind Button <KeyPress-Return> {event generate %W <KeyPress-space>}

	return
    }

    proc cmdAbout {} {
	variable ::pfm::installDir
	variable API
	variable pfmVersion

	set aboutMsg \
	    "Postgres Forms (pfm) Version $pfmVersion. Copyright (C)"
	append aboutMsg \
	    " Willem Herremans 2004-2007\n\nPostgres Forms comes with ABSOLUTELY NO WARRANTY;"
	append aboutMsg \
	    " see 'Help -> License' for details."
	append aboutMsg \
	    "\n\nThis is free software, and you are welcome to redistribute it under"
	append aboutMsg " certain conditions; see 'Help -> License' for details."
	append aboutMsg \
	    "\n\nSend bug reports or other comments to comments.pfm@scarlet.be."
        append aboutMsg "\n\n$API"
	append aboutMsg "\n\npfm is installed in $installDir"
	::gen::Dialog {} "pfm - About" $aboutMsg 400 0 "OK"
	return
    }


    proc cmdOpenDataBase { } {
	# Let the user specify a data base to open

	variable ::options::pfmOptions
	variable connInfoList
	variable currentDB
	variable dbName
	if { [string equal -nocase -length 1 $pfmOptions(usePGPASSWORD) y] } then {
	    set connInfoList {host port user password}
	    # The password will be prompted for and stored in PGPASSWORD.
	} else {
	    set connInfoList {host port user}
	    # The password will not be prompted for. A properly configured
	    # ~/.pgpass file is required.
	}
	if { ![info exists currentDB]} then {
	    set dbName $pfmOptions(dbname)
	    destroy .opendb
	    toplevel .opendb -class Toplevel
	    wm transient .opendb .
	    set x [expr [winfo rootx .] + 20]
	    set y [expr [winfo rooty .] + 20]
	    wm geometry .opendb +$x+$y
	    wm title .opendb "pfm - Open data base"
	    set rowidx 0
	    foreach connItem $connInfoList {
		if { [info exists pfmOptions($connItem)] } then {
		    set connInfo $pfmOptions($connItem)
		} else {
		    set connInfo {}
		}
		label .opendb.lbl$connItem -text $connItem
		entry .opendb.val$connItem -width 30 -background white
		if { [string equal $connItem {password}] } then {
		    .opendb.val$connItem configure -show *
		}
		grid .opendb.lbl$connItem -in .opendb -row $rowidx -column 0 \
			-rowspan 1 -columnspan 1
		grid .opendb.val$connItem -in .opendb -row $rowidx -column 1 \
			-rowspan 1 -columnspan 3 -sticky {we}
		.opendb.val$connItem insert end $connInfo
		incr rowidx
	    }
	    label .opendb.lbldbname -text {dbname}
	    grid .opendb.lbldbname -in .opendb -row $rowidx -column 0 \
		-rowspan 1 -columnspan 1
	    entry .opendb.valdbname -width 30 -background white \
		-textvariable ::pfm::dbName
	    if {[string equal -nocase -length 1 $pfmOptions(usePGPASSWORD) y]} then {
		focus .opendb.valpassword
	    } else {
		focus .opendb.valdbname
	    }
	    button .opendb.btndbname -image ::img::down \
		-command ::pfm::cmdSelectDB
	    grid .opendb.valdbname -in .opendb -row $rowidx -column 1 \
		-rowspan 1 -columnspan 2
	    grid .opendb.btndbname -in .opendb -row $rowidx -column 3 \
		-rowspan 1 -columnspan 1
	    button .opendb.btnOK -text OK \
		-command {::pfm::cmdOpenOK} -underline 0
	    button .opendb.btnCancel -text Cancel \
		-command {::pfm::cmdOpenCancel} -underline 0
	    bind .opendb.valdbname <KeyPress-Return> \
		"::gen::PressButton .opendb.btnOK"
	    bind .opendb <Alt-KeyPress-o> \
		"::gen::PressButton .opendb.btnOK"
	    bind .opendb <Alt-KeyPress-c> \
		"::gen::PressButton .opendb.btnCancel"
	    bind .opendb <KeyPress-Escape> \
		"::gen::PressButton .opendb.btnCancel"
	    incr rowidx
	    grid .opendb.btnOK -column 1 -row $rowidx -sticky we
	    grid .opendb.btnCancel -column 2 -row $rowidx -sticky we
	} else {
	    tk_messageBox -message "First close data base $dbName" -type ok \
		-icon info -parent .
	}
	return
    }

    proc cmdSelectDB {} {
	variable ::options::pfmOptions
	variable dbName

	# Prepare argumnts for ::gen::ListBox
	set title "pfm - Select database"
	set dbName [::gen::ListBox .opendb.valdbname 300 200 $title \
			::options::pfmOptions(dblist) \
			::options::pfmOptions(dblist) $dbName]
	focus .opendb.btnOK
	return
    }

    proc cmdOpenOK {} {
	global env
	variable connInfoList
	variable currentDB
	variable dbName
	variable ::options::pfmOptions

	set env(PGCLIENTENCODING) "UNICODE"
	set connInfo {}
	lappend connInfo "dbname='$dbName'"
	set dbHost {}
	set dbPort {}
	set dbUser {}
	set dbPassword {}
	foreach connItem $connInfoList {
	    set connItemValue [.opendb.val$connItem get]
	    if { ![string equal $connItemValue {}] } then {
		lappend connInfo "$connItem='$connItemValue'"
	    }
	    switch $connItem {
		host {
		    set dbHost $connItemValue
		}
		port {
		    set dbPort $connItemValue
		}
		user {
		    set dbUser $connItemValue
		}
		password {
		    set dbPassword $connItemValue
		}
	    }
	}
	if { [string equal -nocase -length 1 $pfmOptions(usePGPASSWORD) n] } then {
	    # Get password from ~/.pgpass
	    # Pgtcl is able to read the .pgpass file, but pgin.tcl is not.
	    # That is why pfm reads it.
	    set dbPassword [findPassword $dbHost $dbPort $dbName $dbUser]
	    if { $dbPassword ne {} } then {
		lappend connInfo "password='$dbPassword'"
	    }
	}
	set connInfo [join $connInfo]
	if { [catch {set currentDB [pg_connect -conninfo $connInfo]} errorMsg]} then {
	    # Database could not be opened for some reason.
	    tk_messageBox -message $errorMsg -type ok -icon error -parent .opendb
	} else {
	    openPsql $dbName $dbHost $dbPort $dbUser $dbPassword
	    destroy .opendb
	    wm title . "pfm - Database : $dbName"
	    check_pfm_tables
	    refreshFormsList
	    .rbNormal configure -state normal
	    .rbDesign configure -state normal
	    .btnOpen configure -state normal
	    # disable Database -> Open
	    .menubar.database entryconfigure 0 -state disabled
	    # enable Database -> Close
	    .menubar.database entryconfigure 1 -state normal
	    # enable SQL and Reports
	    .menubar entryconfigure 1 -state normal
	    .menubar entryconfigure 2 -state normal
	    ::options::addToDBlist $dbName
	}
	return
    }

    proc openPsql {dbName dbHost dbPort dbUser dbPassword} {
	variable psqlChannel
	variable errChannel
	global env
	variable ::options::pfmOptions

	if { [info exists psqlChannel] } then {
	    close $psqlChannel
	    unset psqlChannel
	}
	set openCommand "|"
	lappend openCommand $pfmOptions(psql)
	lappend openCommand {--echo-queries}
	if { ![string equal $dbName {}] } then {
	    lappend openCommand "--dbname"
	    lappend openCommand $dbName
	}
	if { ![string equal $dbHost {}] } then {
	    lappend openCommand "--host"
	    lappend openCommand $dbHost
	}
	if { ![string equal $dbPort {}] } then {
	    lappend openCommand "--port"
	    lappend openCommand $dbPort
	}
	if { ![string equal $dbUser {}] } then {
	    lappend openCommand "--username"
	    lappend openCommand $dbUser
	}
	if { [string equal -nocase -length 1 $pfmOptions(usePGPASSWORD) y] } then {
	    set env(PGPASSWORD) $dbPassword
	}
	lappend openCommand "2>@$errChannel"
	# Connecting to psql.
	if { [catch { open $openCommand RDWR } psqlChannel] } then {
	    # For security reasons, the PGPASSWORD environment variable is cleared.
	    if { [string equal -nocase -length 1 $pfmOptions(usePGPASSWORD) y] } then {
		unset env(PGPASSWORD)
	    }
	    tk_messageBox -message $psqlChannel -type ok -icon error -parent .opendb
	    unset psqlChannel
	} else {
	    # For security reasons, the PGPASSWORD environment variable is cleared.
	    if { [string equal -nocase -length 1 $pfmOptions(usePGPASSWORD) y] } then {
		unset env(PGPASSWORD)
	    }
	    # When openeing a database, pfm puts the environment variable 
	    # PGCLIENTENCODING UNICODE, which means in fact utf-8 (see
	    # chapter '20.2 Character set support' of PostgreSQL documentation.
	    # The next statement puts the command pipeline to utf-8 as well.
	    fconfigure $psqlChannel -encoding utf-8
	    fileevent $psqlChannel readable ::report::showResult
	}
	return
    }

    proc readPgPass { } {
	variable passMatrix
	global tcl_platform
	global env

	# This procedure reads the ~/.pgpass file if it exists and if it
	# has the right permissions (00600, i.e. rw for owner only).
	# It parses this file and stores the result in passMatrix.
	# This procedure supports the backslash escape for : and backslash.
	# backslash backslash is read as backslash
	# backslash ':' is read as ':' and not interpreted as entry separator
	# backslash 'anything else' is read as 'anything else'
	#                                      (i.e. backslash is dropped)
	# ':' is interpreted as entry separator

	# On Windows platforms, the pgpass file is
	# %APPDATA%\postgresql\pgpass.conf

	set seqnr 0
	if {$tcl_platform(platform) eq {windows}} then {
	    set filename [file join $env(APPDATA) postgresql pgpass.conf]
	} else {
	    set filename "~/.pgpass"
	}
	if { [file exists $filename] } then {
	    if {$tcl_platform(platform) eq {unix}} then {
	        set filePermission [file attributes $filename -permissions]
	        set first [expr [string length $filePermission] - 3]
	        set filePermission [string range $filePermission $first end]
	    } else {
	        set filePermission "600"
	    }
	    if { $filePermission ne "600" } then {
		set map {0 --- 1 --x 2 -w- 3 -wx 4 r-- 5 r-x 6 rw- 7 rwx}
		set filePermission [string map $map $filePermission]
		set errMsg "The permissions on ~/.pgpass are '$filePermission'."
		set errMsg "$errMsg\nThey should be 'rw-------'"
		tk_messageBox -type ok -icon error -message $errMsg -parent .opendb
	    } else {
		if { [catch {open $filename r} pgPass ] } then {
		    tk_messageBox -type ok -icon error -message $pgPass \
			-parent .opendb
		} else {
		    set argList {hostname port database username password}
		    while { ![eof $pgPass] } {
			if {[gets $pgPass current_line] > 0} then {
			    incr seqnr
			    foreach name $argList {
				set passMatrix($seqnr,$name) {}
			    }
			    set arg {}
			    set argNr 0
			    set last [expr [string length $current_line] - 1]
			    for {set i 0} {$i <= $last} {incr i} {
				set curChar [string index $current_line $i]
				switch -- $curChar {
				    "\\" {
					# This is the way to write 1 backslash:
					# NOT with curly braces.
					# Skip the backslash and copy the next character
					incr i
					append arg [string index $current_line $i]
				    }
				    ":" {
					# end of an arg
					set name [lindex $argList $argNr]
					if {$name ne {}} then {
					    set passMatrix($seqnr,$name) $arg
					}
					# puts "$seqnr, $name : $arg"
					set arg {}
					incr argNr
				    }
				    default {
					# just copy the character
					append arg $curChar
				    }
				}
			    }
			    # We are at end of line. Just copy the last arg.
			    set name [lindex $argList $argNr]
			    if {$name ne {}} then {
				set passMatrix($seqnr,$name) $arg
			    }
			    # puts "$seqnr, $name : $arg"
			    set arg {}
			    incr argNr
			}
		    }
		    close $pgPass
		}
	    }
	}
	return $seqnr
    }

    proc findPassword {hostname port database username} {
	variable passMatrix

	# This procedure tries to get the password from ~/.pgpass
	# It returns the found password. If it does not find
	# a password, it returns the empty string.

	set nr_of_lines [readPgPass]
	set found 0
	set password {}
	for {set seqnr 1} {($seqnr <= $nr_of_lines) && (!$found)} {incr seqnr} {
	    if { ([string equal $hostname $passMatrix($seqnr,hostname)] || \
		      [string equal {*} $passMatrix($seqnr,hostname)]) && \
		     ([string equal $port $passMatrix($seqnr,port)] || \
			  [string equal {*} $passMatrix($seqnr,port)]) && \
		     ([string equal $database $passMatrix($seqnr,database)] || \
			  [string equal {*} $passMatrix($seqnr,database)]) && \
		     ([string equal $username $passMatrix($seqnr,username)] || \
			  [string equal {*} $passMatrix($seqnr,username)]) } then {
		set found 1
		set password $passMatrix($seqnr,password)
	    }
	}
	array unset passMatrix
	return $password
    }



    proc cmdOpenCancel {} {

	destroy .opendb
	return
    }

    proc cmdCloseDataBase {} {
	#Close data base that is currently open

	variable currentDB
	variable dbName
	variable formsArray
	variable widget
	variable psqlChannel
	variable ::form::openQueryAfterForm

	set openQueryAfterForm 0
	destroy .form
	set openQueryAfterForm 1
	destroy .query
	destroy .report
	if { [info exists currentDB] } then {
	    pg_disconnect $currentDB
	    unset currentDB
	}
	if { [info exists psqlChannel] } then {
	    if { [catch {close $psqlChannel} errMsg] } then {
		tk_messageBox -message $errMsg -type ok -icon error -parent .
	    }
	    unset psqlChannel
	}
	array unset formsArray
	$widget(lsbForms) delete 0 end
	$widget(lsbForms) see 0
	wm title . "pfm - No data base opened"
	.btnOpen configure -state disabled
	.rbNormal configure -state disabled
	.rbDesign configure -state disabled
	# enable Database -> Open ...
	.menubar.database entryconfigure 0 -state normal
	# disable Database -> Close
	.menubar.database entryconfigure 1 -state disabled
	# disable reports/querries
	.menubar entryconfigure 1 -state disabled
	.menubar entryconfigure 2 -state disabled
	# disable install pfm_* tables and install example database
	.menubar.tools entryconfigure 0 -state disabled
	.menubar.tools entryconfigure 1 -state disabled
	return
    }

    proc cmdExit {} {

	variable currentDB
	variable ::options::pfmOptions

	if {[info exists currentDB]} {
	    cmdCloseDataBase
	}
	# remove temporary files of this session
	set tmpFile [file join $pfmOptions(tmpdir) "pfm_$pfmOptions(user)_[pid].sql"]
	if {[file exists $tmpFile]} then {
	    catch {file delete $tmpFile}
	}
	set tmpFile [file join $pfmOptions(tmpdir) "pfm_$pfmOptions(user)_[pid].txt"]
	if {[file exists $tmpFile]} then {
	    catch {file delete $tmpFile}
	}
	exit
    }

    proc refreshFormsList {} {

	variable currentDB
	variable ::options::pfmOptions
	variable formsArray
	variable formsIndex
	variable widget
	variable pfmMode

	# reworked because of bug 1058: formsArray is now filled
	# with all the forms, regardless of pfmMode

	if {[info exists currentDB]} then {
	    array unset formsArray
	    array unset formsIndex
	    set formQuery "SELECT name,tablename,pkey,showform,view,sqlselect,sqlfrom,groupby,sqlorderby,sqllimit"
	    append formQuery " FROM pfm_form ORDER BY name"
	    set resQuery [pg_exec $currentDB $formQuery]
	    pg_result $resQuery -assignbyidx formsArray
	    set lastTuple [expr [pg_result $resQuery -numTuples] -1]
	    $widget(lsbForms) delete 0 end
	    set listIndex 0
	    for {set tupleNr 0} {$tupleNr <= $lastTuple} {incr tupleNr} {
		set tuple [pg_result $resQuery -getTuple $tupleNr]
		set form [lindex $tuple 0]
		if {$formsArray($form,pkey) eq {}} then {
		    set formsArray($form,view) {t}
		}		
		set showform [lindex $tuple 3]
		if {(($pfmMode eq "design") && ($showform eq "f")) || \
			(($pfmMode eq "normal") && ($showform eq "t"))} then {
		    $widget(lsbForms) insert end $form
		    set formsIndex($listIndex) $form
		    incr listIndex
		}
	    }
	    pg_result $resQuery -clear
	    $widget(lsbForms) selection clear 0 end
	    $widget(lsbForms) selection set 0 0
	    $widget(lsbForms) see 0
	    $widget(lsbForms) activate 0
	}
	return
    }


    proc installPfm {} {

	variable installDir
	variable currentDB

	if { ![info exists currentDB] } then {
	    tk_messageBox -message "There is no database open!" -type ok \
		-icon error
	} else {
	    ::report::cmdReportSQL {sql}
	    set fileName [file join $installDir install_pfm.sql]
	    if { ![file exists $fileName] } then {
		set errMsg "$fileName does not exist!"
		tk_messageBox -message $errMsg -icon error -type ok -parent .report
	    } else {
		.report.fmSQL.text delete 1.0 end
		set convertedFile [::report::ConvertToUTF-8 $fileName {iso8859-1}]
		if {$convertedFile ne {}} then {
		    .report.fmSQL.text insert end "\\i '$convertedFile'"
		    set msgInfo "The pfm_* tables will be installed."
		    append msgInfo "\nPlease, check the output that will be generated."
		    append msgInfo "\nThen close window \"Run SQL\"."
		    tk_messageBox -message $msgInfo -type ok -icon info -parent .report
		    ::report::cmdRunSQL
		    tkwait window .report
		    refreshFormsList
		    # disable install pfm_* tables and install example database
		    .menubar.tools entryconfigure 0 -state disabled
		    .menubar.tools entryconfigure 1 -state disabled
		} 
	    }
	}
	return
    }

    proc cmdInstallExample {} {

	variable installDir
	variable currentDB
	variable dbName
	variable pfmConfig

	if { ![info exists currentDB] } then {
	    tk_messageBox -message "There is no database open!" -type ok \
		-icon error
	} else {
	    set message \
		"You are about to install an example database in the "
	    append message \
		"currently opened database, which is '${dbName}'. "
	    append message \
		"This database should be completely empty. "
	    append message \
		"It should not even contain the pfm_*  tables. "
	    append message \
		"Is this OK?"
	    set proceed [tk_messageBox -message $message -type yesno \
			     -icon warning -parent .]
	    if { [string equal $proceed "yes"] } then {
		set initialDir $pfmConfig(exampleDir)
		set title "Select an example database"
		set fromEncoding {iso8859-1}
		set fileTypes {
		    {{SQL statements} {.sql} }
		    {{All files} *}
		}
		set defaultExt ".sql"
		::report::cmdReportSQL {sql}
		set fileName [tk_getOpenFile -title $title -filetypes $fileTypes \
				  -defaultextension $defaultExt -parent .report \
				  -initialdir $initialDir]
		if {![string equal $fileName {}]} then {
		    set example [file tail $fileName]
		    .report.fmSQL.text delete 1.0 end
		    set convertedFile [::report::ConvertToUTF-8 $fileName {iso8859-1}]
		    if {$convertedFile ne {}} then {
			.report.fmSQL.text insert end "\\i '$convertedFile'"
			set msgInfo "The example database '${example}' will be installed."
			append msgInfo "\nPlease, check the output that will be generated."
			append msgInfo "\nThen close window \"Run SQL\"."
			tk_messageBox -message $msgInfo -type ok -icon info -parent .report
			::report::cmdRunSQL
			tkwait window .report
			refreshFormsList
			# disable install pfm_* tables and install example database
			.menubar.tools entryconfigure 0 -state disabled
			.menubar.tools entryconfigure 1 -state disabled
		    }
		}
	    }
	}
	return
    }

    proc check_pfm_tables {} {

	variable currentDB
	variable pfmVersion

	set queryDef {
	    SELECT COUNT(*) AS pfm_exists FROM pg_tables WHERE
	    tablename IN ('pfm_form','pfm_attribute','pfm_value','pfm_value_list',
			  'pfm_link','pfm_report','pfm_section')
	}
	set queryRes [pg_exec $currentDB $queryDef]
	set nrInstalled [lindex [pg_result $queryRes -getTuple 0] 0]
	pg_result $queryRes -clear
	if { $nrInstalled == 0} then {
	    # enable Install pfm_* tables and Install example database
	    .menubar.tools entryconfigure 0 -state normal
	    .menubar.tools entryconfigure 1 -state normal
	    set msgWarn \
		"The pfm_* tables have not been installed in this database. "
	    append msgWarn \
		"You are currently not able to use or define forms or reports."
	    append msgWarn \
		"\nHint: Use 'Tools -> Install pfm_* tables' or 'Tools -> "
	    append msgWarn \
		"Install example database ...'."
	    tk_messageBox -type ok -icon warning -message $msgWarn -parent .
	} else {
	    # disable Install pfm_* tables and Install example database
	    .menubar.tools entryconfigure 0 -state disabled
	    .menubar.tools entryconfigure 1 -state disabled	    
	    set queryDef {SELECT version FROM pfm_version ORDER BY seqnr DESC}
	    set queryRes [pg_exec $currentDB $queryDef]
	    set status [pg_result $queryRes -status]
	    if { [string equal $status "PGRES_TUPLES_OK"] } then {
		set numTuples [pg_result $queryRes -numTuples]
		if { $numTuples > 0 } then {
		    pg_result $queryRes -assign versionArray
		    set DBversion $versionArray(0,version)
		} else {
		    set msgWarn "pfm_version exists, but contains no data. "
		    append msgWarn "pfm allows you to open this database, "
		    append msgWarn "but it is an indication that the pfm_* tables "
		    append msgWarn "are not properly installed."
		    tk_messageBox -message $msgWarn -type ok -icon warning -parent .
		    # We assume that no conversion is needed.
		    set DBversion {1.5.0}
		}
	    } else {
		# Older versions of pfm did not record version in the database.
		set DBversion "1.0.4"
	    }
	    pg_result $queryRes -clear
	    switch -- [versionCompare $pfmVersion $DBversion] {
		1 {
		    # pfm version is newer than database version
		    if {![convertDB $DBversion]} then {
			set msgWarn "This version of the pfm_* tables (${DBversion}) may be "
			append msgWarn "incompatible with this version of pfm (${pfmVersion}). "
			append msgWarn "Continue at your own risk."
			tk_messageBox -type ok -icon warning -message $msgWarn -parent .
		    }
		}
		0 {
		    # pfm version = database version, no action required
		}
		-1 {
		    # pfm version is older than database version
		    set msgWarn "This version of the pfm_* tables (${DBversion}) may be "
		    append msgWarn "incompatible with this version of pfm (${pfmVersion}). "
		    append msgWarn "Continue at your own risk."
		    tk_messageBox -type ok -icon warning -message $msgWarn -parent .
		}
	    }
	}
	return
    }

    proc versionCompare {v1 v2} {

	set v1List [split $v1 "."]
	set v2List [split $v2 "."]
	set result 0
	for {set i 0} {($i <= 2) && ($result == 0)} {incr i} {
	    if {[lindex $v1List $i] < [lindex $v2List $i]} then {
		set result -1
	    } else {
		if {[lindex $v1List $i] > [lindex $v2List $i]} then {
		    set result 1
		}
	    }
	}
	return $result
    }

    proc convertDB {DBversion} {
       	variable installDir
	variable currentDB

	# From pfm version 1.3.0 on, we do not convert the database to
	# the latest version of pfm. We only convert it to the
	# earliest compatible version, which is 1.5.0 now.

	# Similarly, the SQL-scripts install_pfm.sql and
	# install_<exampledb>.sql, of this and future versions of pfm
	# will also create pfm_tables of version 1.5.0, until there is
	# a need to change the structure of the pfm_tables.

	set toVersion {1.5.0}
	switch $DBversion {
	    {1.0.4} {
		set proceed [tk_messageBox -message \
		       "Convert pfm-* tables from version $DBversion to version ${toVersion}" \
				 -type yesno -icon question -parent .]
		if { [string equal $proceed "yes"] } then {
		    set converted [specialConversion {1.0.4} {1.1.0}]
		    set converted [specialConversion {1.1.0} {1.2.0}]
		    set converted [specialConversion {1.2.0} $toVersion]
		} else {
		    set converted 0
		}
	    }
	    {1.1.0} -
	    {1.1.1} {
		set proceed [tk_messageBox -message \
		       "Convert pfm-* tables from version $DBversion to version ${toVersion}?" \
				 -type yesno -icon question -parent .]
		if { [string equal $proceed "yes"] } then {
		    set converted [specialConversion {1.1.0} {1.2.0}]
		    set converted [specialConversion {1.2.0} $toVersion]
		} else {
		    set converted 0
		}
	    }
	    {1.2.0} -
	    {1.2.1} -
	    {1.2.3} -
	    {1.2.4} -
	    {1.2.5} {
		set proceed [tk_messageBox -message \
		       "Convert pfm-* tables from version $DBversion to version ${toVersion}?" \
				 -type yesno -icon question -parent .]
		if { [string equal $proceed "yes"] } then {
		    set converted [specialConversion {1.2.0} $toVersion]
		} else {
		    set converted 0
		}		
	    }
	    default {
		# The database version is between 1.5.0 and the
		# current pfm version.  Since the structure of the
		# pfm_tables has not changed between version 1.5.0 and
		# the current pfm version, there is no need to touch
		# the pfm tables.
		set converted 1
	    }
	}
	return $converted
    }

    proc specialConversion {DBversion toVersion} {
       	variable installDir
	variable currentDB

	# When this procedure is called, the file 
	# '$installDir/convert_from_$DBversion.sql' should exist.

	::report::cmdReportSQL {sql}
	set fileName [file join $installDir convert_from_$DBversion.sql]
	if {![file exists $fileName]} then {
	    set errMsg "$fileName does not exist!"
	    tk_messageBox -message $errMsg -icon error -type ok -parent .report
	    set converted 0
	} else {
	    .report.fmSQL.text delete 1.0 end
	    set convertedFile [::report::ConvertToUTF-8 $fileName {iso8859-1}]
	    if {$convertedFile ne {}} then {
		.report.fmSQL.text insert end "\\i '$convertedFile'"
		set msgInfo "The database will be converted to $toVersion."
		append msgInfo "\nPlease, check the output that will be generated."
		append msgInfo "\nThen close window \"Run SQL\"."
		tk_messageBox -message $msgInfo -type ok -icon info -parent .report
		::report::cmdRunSQL
		tkwait window .report
		refreshFormsList
		set converted 1
	    } else {
		set converted 0
	    }
	}
	return $converted
    }
    

    proc cmdOpenQuery {} {
	variable formsIndex
	variable widget

	set formNr [$widget(lsbForms) curselection]
	if { $formNr == {} } then {
	    set formNr 0
	}
	if {[info exists formsIndex($formNr)]} then {
	    set formName $formsIndex($formNr)
	    ::form::OpenQuery $formName 1
	}
	return
    }

    proc defineEditMenu {} {
	
	menu .mnEdit -tearoff 0
	.mnEdit add command \
	    -command {::pfm::cmdEditCopy} -label Copy
	.mnEdit add command \
	    -command {::pfm::cmdEditCut} -label Cut
	.mnEdit add command \
	    -command {::pfm::cmdEditPaste} -label Paste
	# Entry and Text are class names. They refer to all entry and text widgets.
	bind Entry <ButtonPress-3> ::pfm::popUpEditMenu
	bind Text <ButtonPress-3> ::pfm::popUpEditMenu
    }

    proc popUpEditMenu {} {

	# Set focus to the widget on which button 3 (right button) is pressed.
	set x [winfo pointerx .]
	set y [winfo pointery .]
	set widget [winfo containing -displayof . $x $y]
	focus $widget
	if {[selection own -displayof . -selection PRIMARY] ne $widget ||
	    [catch {selection get -displayof . -selection PRIMARY \
			-type STRING}]} then {
	    # The widget does not own the selection or there is no selection
	    # in the widget
	    .mnEdit entryconfigure 0 -state disabled
	    .mnEdit entryconfigure 1 -state disabled
	} else {
	    .mnEdit entryconfigure 0 -state normal
	    if {[$widget cget -state] eq {normal}} then {
		.mnEdit entryconfigure 1 -state normal
	    } else {
		.mnEdit entryconfigure 1 -state disabled
	    }
	}
	if {[catch {clipboard get -displayof . -type STRING}]} then {
	    .mnEdit entryconfigure 2 -state disabled
	} else {
	    if {[$widget cget -state] eq {normal}} then {
		.mnEdit entryconfigure 2 -state normal
	    } else {
		.mnEdit entryconfigure 2 -state disabled
	    }
	}
	tk_popup .mnEdit $x $y
	return
    }

    proc cmdEditCopy {} {

	clipboard clear -displayof .
	clipboard append -displayof . -format STRING -type STRING -- \
	    [selection get -displayof . -selection PRIMARY -type STRING]
	selection clear -displayof . -selection PRIMARY
	return
    }
 
    proc cmdEditCut {} {

	# selection own -selection PRIMARY $widget
	clipboard clear -displayof .
	clipboard append -displayof . -format STRING -type STRING -- \
	    [selection get -displayof . -selection PRIMARY -type STRING]
	set widget [focus -displayof .]
	$widget delete sel.first sel.last
	selection clear -displayof . -selection PRIMARY
	return
    }
    
    proc cmdEditPaste {} {
	set widget [focus -displayof .]
	$widget insert insert [clipboard get -displayof . -type STRING]
	return
    }


    #########################################################
    # Main of namespace pfm                                 #
    #########################################################

    if { $argc == 0 } then {
	set installDir [file normalize [pwd]]
    } else {
	set installDir [lindex $argv 0]
    }
    if {[string equal $tcl_platform(platform) "unix"]} then {
	set configFile [file join $installDir {pfm.conf}]
	if {![file exists $configFile]} then {
	    set configFile {/etc/pfm.conf}
	}
	if {[catch {open $configFile r} confChan]} then {
	    tk_messageBox -type ok -icon error -message $confChan
	    exit
	} else {
	    while {![eof $confChan]} {
		set line [gets $confChan]
		if {([string length $line] != 0) && \
			([string compare -length 1 $line "#"] != 0)} then {
		    set pfmConfig([lindex $line 0]) [lindex $line 1]
		}
	    }
	    close $confChan
	    if {[string compare -length 1 $pfmConfig(docDir) "/"] != 0} then {
		# The directory indicated is relative to the installDir
		set pfmConfig(docDir) [file join $installDir $pfmConfig(docDir)]
	    }
	    if {[string compare -length 1 $pfmConfig(helpDir) "/"] != 0} then {
		# The directory indicated is relative to the installDir
		set pfmConfig(helpDir) [file join $installDir $pfmConfig(helpDir)]
	    }
	    if {[string compare -length 1 $pfmConfig(exampleDir) "/"] != 0} then {
		# The directory indicated is relative to the installDir
		set pfmConfig(exampleDir) [file join $installDir $pfmConfig(exampleDir)]
	    }
	}
    } else {
	# On non-UNIX platforms there is no configuration file.
	set pfmConfig(docDir) [file join $installDir {doc}]
	set pfmConfig(helpDir) [file join $installDir {help}]
	set pfmConfig(exampleDir) [file join $installDir {examples}]
	# The default values for "browser" and "printcmd" are not read
	# from pfmConfig. They are calculated in the procedure getDefault.
    }
    set readOnlyBackground {gray92}
    ::options::initOptions
    initRootWindow
    defineEditMenu


    # Init errChannel. This the channel to which stderr of psql is redirected.

    switch -- $tcl_platform(platform) {
	"unix" {
	    # We use UNIX's cat instead of cat.tcl
	    set openCommand "| cat"
	}
	"windows" {
	    # Cat does not require Tk. Hence it is better to use tclkitsh
	    # for faster start up.
	    set tclkit [file join $installDir tclkit tclkitsh.exe]
	    set cat [file join $installDir "cat.kit"]
	    set openCommand "|"
	    lappend openCommand $tclkit
	    lappend openCommand $cat
	}
    }
    if {[catch {open $openCommand RDWR} errChannel]} then {
	tk_messageBox -type ok -icon error -message $errChannel
    } else {
	fconfigure $errChannel -encoding utf-8
	fileevent $errChannel readable ::report::showError
    }

}

#############################################################################
# end of namespace pfm                                                      #
#############################################################################

#############################################################################
# begin of namespace help                                                   #
#############################################################################
#

namespace eval help {

    proc cmdDisplayManual {} {

	variable ::pfm::pfmConfig
	variable ::options::pfmOptions
	
	set url "file://${pfmConfig(docDir)}/index.html"
	set command {exec}
	set map {%s}
	lappend map $url
	foreach arg $pfmOptions(browser) {
		lappend command [string map $map $arg]
	}
	    lappend command {&}
	# puts $command
	if { [catch $command errMsg]} then {
	    tk_messageBox -type ok -icon error -parent . \
		-message "$command failed\n$errMsg"
	}
	return
    }

    proc cmdLicense {} {

	variable ::pfm::pfmConfig
	variable licenseText

	set fileName [file join $pfmConfig(helpDir) {gpl.txt}]
	if { [catch {open $fileName r} license_ch] } then {
	    fileNotFound {license} $license_ch
	} else {
	    set title {pfm - License}
	    set licenseText [read $license_ch]
	    close $license_ch
	    ::gen::TextEdit $title ::help::licenseText 1 {}
	}
	unset licenseText
	return
    }

    proc fileNotFound {what reason} {

	switch $what {
	    help {
		set msgText "Normally you should see the help file now, but $reason"
	    }
	    license {
		set msgText "Normally you should see the GNU General Public License now, but $reason"
	    }
	    default {
		set msgText $reason
	    }
	}
	tk_messageBox -message $msgText -type ok -icon error
	return
    }

}
######################################################################
# End of namespace help                                              #
######################################################################

######################################################################
# Begin of namespace form                                            #
######################################################################

# activeForm contains the name of the currently opened form.
# It is filled by several procedures:
#     1. OpenQuery
#     2. cmdFollowLink
#     3. cmdBack
#
# formAttribList contains the list of attributes of the active form,
# as defined in data base table pfm_attributes.
#
# attributeArray contains the complete attributes defintion of the
# active form as defined by pfm_attributes.
#
# tableAttribList contains only the attributes that are not "tgReadOnly".
# The purpose is to include only the attributes of the table referred
# to by pfm_form.tablename.
#
# formAttribList, tableAttribList and attributeArray are filled by 
# proc getAttributes.
#
# getAttributes is called by:
#     1. OpenQuery
#     2. cmdFollowLink
#     3. cmdBack
#
# WhereSelected is a boolean which indicates whether the user is
# pasting into the "where" or into the "order by" text entry
# of the query. Its value is controlled by the radio buttons
# on the query window.
#
# recordArray contains all the records selected by the query defined
# by the user in the query window. recordArray($tupleNr,$attribute)
# indicates the value of $attribute of $tupleNr. It is filled
# by proc OpenForm. It is the so called "internal buffer".
# On top of the attribute values, recordArray also contains
# a status for each record: recordArray($tupleNr,23status47)
# contains the status the record indicated by $tupleNr. The
# status can be : "Not modified", "Updated", "Deleted", "Added",
# "After last", "Not added", "Updating", "Adding". 
# The attribute name "23status47" has been chosen
# to avoid name conflicts with real table attributes.
#
# lastRecord is the tupleNr of the last tuple in recordArray. This
# is in fact a dummy, empty record, functioning as a sentinel.
#
# curRecord is the tupleNr of the record that is currently displayed
# on the screen.
#
# formState is the state of the user interface of of the form. It can be
# browse, update or add.
#
# formOffset is the offset value used in the LIMIT clause, to load records
# in the internal buffer.
#
# formLastChunk is a boolean indicating whether this is the last chunk
# of form data. It is determined by loadDataChunk
#
# txtRecord is the textvar linked to the attribute labels,
# entries or buttons on the form. When the current record
# is displayed, the values of recordArray($curRecord,$attribute)
# are copied to txtRecord($attribute).
#
# formStack contains the subsequent queries issued by the user
# as a result of following links. lastFormOnStack is a stack
# pointer on this stack. The first query is pushed on the stack
# by proc cmdExecuteQuery. If the user clicks on a link button
# another query is pushed on the stack by proc cmdFollowLink.
# Any time the user presses back, a query defintion is popped
# of the stack. The elements that are kept on the stack are:
#     formId
#     queryDef
#     intro: the information displayed on top of the form
#     displayKey: the list of pkey attribute values of the record 
#                 that was displayed at
#                 the time a link button was pressed and which
#                 is displayed again when the user presses "Back"
#     offset: the offset that was used for displaying the record
#                 that was displayed at the time the user presses
#                 a link button
#
# linksArray is loaded with all the links originating from the
# active form. It is filled by proc displayLinks, which is called
# from OpenForm. Its structure is linksArray($link,$attribute)
# where $link is an index for the link (starting from 0) and
# where $attribute is any attribute of pfm_link.
#
# widget is an array containing aliases for some widget
# path names
#
# windowSize is and associative array in which the window size
# and window position of the form windows are stored, such that
# when the user resizes and replaces a window, pfm remembers size
# and position until pfm is closed.
#
# focusRow is de attribute row that has the focus on the form window.
#
# TextEditList is the list of opened TextEdit windows


namespace eval form {
    variable activeForm
    variable formState
    variable formOffset
    variable formLastChunk
    variable formAttribList
    variable tableAttribList
    variable attributeArray
    variable WhereSelected
    variable queryWhere
    variable queryOrderBy
    variable lastRecord
    variable recordArray
    variable curRecord
    variable txtRecord
    variable formStack
    variable lastFormOnStack
    variable linksArray
    variable widget
    variable windowSize
    variable focusRow
    variable TextEditList

    #############################################################
    # Procedures that are called from the corresponding cmdXXX  #
    # procedures in namespace pfm                               #    
    #############################################################


    proc OpenQuery {formName clear} {

	variable ::pfm::formsArray
	variable activeForm
	variable WhereSelected
	variable queryWhere
	variable queryOrderBy
	variable formAttribList
	variable attributeArray
	variable widget

	set activeForm $formName
	set WhereSelected true
	if {$clear} then {
	    set queryWhere {}
	    set queryOrderBy $formsArray($activeForm,sqlorderby)
	}
	set groupBy [expr {$formsArray($activeForm,groupby) ne {}}]
	showQueryWindow $groupBy
	wm title .query "pfm - Open form : $formName"
	$widget(txtSelect) insert end \
	    "SELECT $formsArray($activeForm,sqlselect)\nFROM $formsArray($activeForm,sqlfrom)"
	if {$groupBy} then {
	    $widget(txtSelect) insert end \
		"\nGROUP BY $formsArray($activeForm,groupby)"
	}
	$widget(txtSelect) configure -state disabled
	getAttributes $activeForm
	fillPasteMenus
	return
    }

    proc fillPasteMenus {} {
	variable ::pfm::formsArray
	variable formAttribList
	variable attributeArray
	variable widget

	set names 0
	set values 0
	foreach attribute $formAttribList {
	    .query.menubar.mnNames add command \
		-label $attribute \
		-command [list ::form::cmdPasteAttribute $attribute]
	    set names 1
	    switch $attributeArray($attribute,typeofget) {
		"tgDirect" -
		"tgReadOnly" -
		"tgExpression" { }
		"tgList" -
		"tgLink" {
		    .query.menubar.mnValues add command \
			-label $attribute \
			-command [list ::form::cmdSelectFromList .query.txtWhere \
				      300 300 $attribute paste]
		    set values 1
		}
	    }
	}
	if {$names} then {
	    .query.menubar entryconfigure 1 -state normal
	}
	if {$values} then {
	    .query.menubar entryconfigure 2 -state normal
	}
	return
    }


    ################################################################
    #                                                              #
    # Procedures for query window (now renamed to "Open form")     #
    #                                                              #
    ################################################################

    proc showQueryWindow {groupBy} {

	variable widget
	variable windowSize
	variable openQueryAfterForm
	variable queryWhere
	variable queryOrderBy

	set openQueryAfterForm 0
	destroy .form
	set openQueryAfterForm 1
	destroy .query
	toplevel .query -class Toplevel
	set tags [bindtags .query]
	lappend tags QueryWindow
	bindtags .query $tags
	bind QueryWindow <Destroy> {::form::cmdQuitQuery}
	if {![info exists windowSize(.query)]} then {
	    set windowSize(.query) {500x300}
	}
	wm geometry .query $windowSize(.query)
	set command {set ::form::windowSize(.query) }
	append command {[string map {{+0+0} {}} [wm geometry .query]]}
 	bind QueryWindow <Configure> \
	    [list after idle [list catch $command]]
	# The frames
	frame .query.fmGrid -relief sunken -borderwidth 2
	frame .query.buttons
	pack .query.fmGrid -side top -expand 1 -fill both
	pack .query.buttons -side top -fill x

	# Text for sql statement
	text .query.txtSelect -wrap word -background $::pfm::readOnlyBackground \
	    -yscrollcommand {.query.vsbSelect set} -takefocus 0 \
	    -width 0 -height 0
	scrollbar .query.vsbSelect -orient vertical \
	    -command {.query.txtSelect yview} -takefocus 0
	set widget(txtSelect) .query.txtSelect
	grid .query.txtSelect -in .query.fmGrid \
	    -column 0 -row 0 -columnspan 2 -sticky nswe
	grid .query.vsbSelect -in .query.fmGrid \
	    -column 2 -row 0 -sticky ns

	# radio buttons and text widgets
	if {$groupBy} then {
	    set WhereHaving "HAVING"
	} else {
	    set WhereHaving "WHERE"
	}
	radiobutton .query.rabWhere \
	    -text $WhereHaving -value true \
	    -variable ::form::WhereSelected \
	    -underline 0 -takefocus 0 -command {focus .query.txtWhere}
	text .query.txtWhere -background white \
	    -yscrollcommand {.query.vsbWhere set} \
	    -width 0 -height 0
	.query.txtWhere insert end $queryWhere
	scrollbar .query.vsbWhere -orient vertical \
	    -command {.query.txtWhere yview}
	radiobutton .query.rabOrderBy \
	    -text {ORDER BY} -value false  -variable ::form::WhereSelected \
	    -underline 0 -takefocus 0 -command {focus .query.txtOrderBy}
	text .query.txtOrderBy -background white \
	    -yscrollcommand {.query.vsbOrderBy set} \
	    -width 0 -height 0
	.query.txtOrderBy insert end $queryOrderBy
	scrollbar .query.vsbOrderBy -orient vertical \
	    -command {.query.txtOrderBy yview}
	grid .query.rabWhere -in .query.fmGrid \
	    -column 0 -row 1 -sticky nw
	grid .query.txtWhere -in .query.fmGrid \
	    -column 1 -row 1 -sticky nswe
	grid .query.vsbWhere -in .query.fmGrid \
	    -column 2 -row 1 -sticky ns
	grid .query.rabOrderBy -in .query.fmGrid \
	    -column 0 -row 2 -sticky nw
	grid .query.txtOrderBy -in .query.fmGrid \
	    -column 1 -row 2 -sticky nswe
	grid .query.vsbOrderBy -in .query.fmGrid \
	    -column 2 -row 2 -sticky ns
	grid rowconfigure .query.fmGrid 0 -weight 1
	grid rowconfigure .query.fmGrid 1 -weight 1
	grid rowconfigure .query.fmGrid 2 -weight 1
	grid columnconfigure .query.fmGrid 1 -weight 1

	# Menubar

	menu .query.menubar -tearoff 0
	.query configure -menu .query.menubar
	.query.menubar add cascade \
	    -menu .query.menubar.mnForm \
	    -label {Form} -underline 0 \
	    -state normal
	.query.menubar add cascade \
	    -menu .query.menubar.mnNames \
	    -label {paste attribute Name} -underline 16 \
	    -state disabled
	.query.menubar add cascade \
	    -menu .query.menubar.mnValues \
	    -label {paste attribute Value} -underline 16 \
	    -state disabled

	# The From menu
	menu .query.menubar.mnForm -tearoff 0
	.query.menubar.mnForm add command \
	    -label {Close} -underline 0 -accelerator {Esc} \
	    -command {destroy .query}

	# The attribute Names menu
	menu .query.menubar.mnNames -tearoff 0
	# This menu is filled by fillPasteMenus

	# The attribute Values menu
	menu .query.menubar.mnValues -tearoff 0
	# This menu is filled by fillPasteMenus

	# The query buttons

	button .query.buttons.execute -command {
	    set ::form::queryWhere [.query.txtWhere get 1.0 "end -1 chars"]
	    set ::form::queryOrderBy [.query.txtOrderBy get 1.0 "end -1 chars"]
	    ::form::cmdExecuteQuery $::form::queryWhere $::form::queryOrderBy \
		"Open form \'$::form::activeForm\'"
	} -text Run -underline 0
	button .query.buttons.quit -command {destroy .query} -text Quit \
	    -underline 0
	pack .query.buttons.quit -side right
	pack .query.buttons.execute -side right
	pack .query.buttons -side top -fill x

	focus .query.txtWhere
	bind .query <Alt-KeyPress-w> {::gen::PressButton .query.rabWhere}
	bind .query <Alt-KeyPress-h> {::gen::PressButton .query.rabWhere}
	bind .query <Alt-KeyPress-o> {::gen::PressButton .query.rabOrderBy}
	bind .query <Alt-KeyPress-r> \
	    {::gen::PressButton .query.buttons.execute}
	bind .query <KeyPress-Escape> \
	    {::gen::PressButton .query.buttons.quit}
	bind .query <Alt-KeyPress-q> \
	    {::gen::PressButton .query.buttons.quit}
	bind .query.txtWhere <FocusIn> {set ::form::WhereSelected true}
	bind .query.txtOrderBy <FocusIn> {set ::form::WhereSelected false}
	return
    }


    proc cmdExecuteQuery {sqlWhere sqlOrderBy intro} {
	# This procedure prepares the SQL SELECT statement of the
	# query to be made, it initialises the formStack and
	# it calls OpenForm. It is called when the user presses
	# the Execute button on the query window.

	variable ::pfm::formsArray
	variable activeForm
	variable formStack
	variable lastFormOnStack
	variable windowSize

	set sqlAttrib $formsArray($activeForm,sqlselect)
	if { !$formsArray($activeForm,view) && \
		 ([lsearch $formsArray($activeForm,pkey) {oid}] >= 0) && \
		 ([regexp {\moid\M} $sqlAttrib] == 0)} then {
	    set sqlAttrib \
		"\"$formsArray($activeForm,tablename)\".oid, $sqlAttrib"
	}
	set sqlFrom $formsArray($activeForm,sqlfrom)
	set groupby $formsArray($activeForm,groupby)
	set queryDef "SELECT $sqlAttrib\nFROM $sqlFrom"
	if { ![string equal $groupby {}] } then {
	    append queryDef "\nGROUP BY $groupby"
	    if { ![string equal $sqlWhere {}] } then {
		# If there is a GROUP BY clause, the sqlWhere must become
		# a HAVING clause.
		append queryDef "\nHAVING $sqlWhere"
	    }
	} else {
	    if { ![string equal $sqlWhere {}] } then {
		append queryDef "\nWHERE $sqlWhere"
	    }
	}
	if { ![string equal $sqlOrderBy {}] } then {
	    append queryDef "\nORDER BY $sqlOrderBy"
	}
	array unset formStack
	set lastFormOnStack 0
	set formStack($lastFormOnStack,formId) $activeForm
	set formStack($lastFormOnStack,queryDef) $queryDef
	set formStack($lastFormOnStack,intro) $intro
	set formStack($lastFormOnStack,displayKey) {}
	destroy .query
	OpenForm $queryDef $intro {}
	return
    }

    proc cmdPasteAttribute {attribute} {
	# The user pastes an attribute name into either the "where" or
	# "order by" text entries, depending on WhereSelect, the value
	# of which is determined by the radio buttons .query.rabWhere
	# and .query.rabOrderBy

	variable WhereSelected
	if {$WhereSelected} then {
	    .query.txtWhere insert insert \"$attribute\"
	} else {
	    .query.txtOrderBy insert insert \"$attribute\"
	}
	return
    }


    proc cmdQuitQuery {} {
	
	# set ::form::windowSize(.query) \
	#    [string map {{+0+0} {}} [wm geometry .query]]
	# ::pfm::refreshFormsList Not necessary, already called in
	# cmdQuitForm
	return
    }



    ################################################################
    #                                                              #
    # Procedures for form window                                   #
    #                                                              #
    ################################################################

    proc showFormWindow {title} {

	variable widget
	variable windowSize

	# destroy .query
	if { [winfo exists .form] } then {
	    wm title .form $title
	    displayRecordFrame
 	    destroy .form.frmLinkBtn
 	    frame .form.frmLinkBtn -borderwidth 2 -relief sunken \
		-width 0 -height 0
	    pack .form.frmLinkBtn -in .form.pLink -side top -expand 1 -fill both
	    clearTextOnForm .form.txtQuery
	    clearTextOnForm .form.txtResult
	} else {
	    toplevel .form -class Toplevel
	    wm title .form $title
	    set tags [bindtags .form]
	    lappend tags FormWindow
	    bindtags .form $tags
	    if {![info exists windowSize(.form)]} then {
		set windowSize(.form) {700x500}
	    }
	    wm geometry .form $windowSize(.form)
	    set command {set ::form::windowSize(.form) }
	    append command {[string map {{+0+0} {}} [wm geometry .form]]}
	    bind FormWindow <Configure> \
		[list after idle [list catch $command]]
	    bind FormWindow <Destroy> ::form::cmdQuitForm
	    makeFormPanes

	    # Main menu
	    menu .form.menu -tearoff 0
	    .form.menu add cascade -menu .form.menu.mnForm -label {Form} \
		-underline 0
	    .form.menu add cascade -menu .form.menu.mnRecord -label {Record} \
		-underline 0
	    .form.menu add cascade -menu .form.menu.mnGoto \
		-label {Go to} -underline 0
	    .form.menu add cascade -menu .form.menu.mnSearch \
		-label {Search} -underline 0
	    .form.menu add command -label {Help} -underline 0 \
		-command ::form::cmdHelp
	    bind .form <KeyPress-F1> ::form::cmdHelp

	    # Make it the main menu
	    .form configure -menu .form.menu

	    # Menu Form
	    menu .form.menu.mnForm -tearoff 0
	    .form.menu.mnForm add command \
		-command {destroy .form} \
		-label {Close} -underline 0 -accelerator {Esc}

	    # Menu Record
	    menu .form.menu.mnRecord -tearoff 0
	    .form.menu.mnRecord add command \
		-command {::form::cmdAdd} \
		-label {Add} -underline 0 -accelerator {Alt-a}
	    .form.menu.mnRecord add command \
		-command {::form::cmdDelete} \
		-label {Delete} -underline 0 -accelerator {Alt-d}
	    .form.menu.mnRecord add command \
		-command {::form::cmdUpdate} \
		-label {Update} -underline 0 -accelerator {Alt-u}

	    # Menu Go to
	    menu .form.menu.mnGoto -tearoff 0
	    .form.menu.mnGoto add command \
		-command {::form::cmdFirst} \
		-label {First} -underline 0 -accelerator {Home, Cntrl-Home}
	    .form.menu.mnGoto add command \
		-command {::form::cmdPrev} \
		-label {Previous} -underline 0 -accelerator {Page Up, Left}
	    .form.menu.mnGoto add command \
		-command {::form::cmdNext} \
		-label {Next} -underline 0 -accelerator {Page Down, Right}
	    .form.menu.mnGoto add command \
		-command {::form::cmdLast} \
		-label {Last} -underline 0 -accelerator {End, Cntrl-End}

	    # Menu Search is defined in displayAttribNames

	    # Bind up and down arrows to move input focus to
	    # higher or lower attribute on the form.

	    bind .form <KeyPress-Down> \
		{focus [tk_focusNext [focus -displayof .form]]}
	    bind .form <KeyPress-Up> \
		{focus [tk_focusPrev [focus -displayof .form]]}


	    # the contents of pQuery pane

	    text .form.txtQuery -background white \
		-wrap word -yscrollcommand {.form.scrollQuery set} \
		-takefocus 0 -width 1 -height 1
	    .form.txtQuery tag configure blueTag -foreground {medium blue}
	    .form.txtQuery tag configure greenTag -foreground {green4}
	    .form.txtQuery tag configure redTag -foreground {red3}
	    scrollbar .form.scrollQuery -orient vertical \
		-command {.form.txtQuery yview} -takefocus 0
	    pack .form.txtQuery -in .form.pQuery -side left -expand 1 -fill both
	    pack .form.scrollQuery -in .form.pQuery -side left -fill y

	    # text area for Result

	    text .form.txtResult -background white \
		-wrap word -yscrollcommand {.form.scrollResult set} \
		-takefocus 0 -width 1 -height 1
	    .form.txtResult tag configure blueTag -foreground {medium blue}
	    .form.txtResult tag configure greenTag -foreground {green4}
	    .form.txtResult tag configure redTag -foreground {red3}
	    scrollbar .form.scrollResult -orient vertical \
		-command {.form.txtResult yview} -takefocus 0
	    pack .form.txtResult -in .form.pResult -side left -expand 1 -fill both
	    pack .form.scrollResult -in .form.pResult -side left -fill y

	    # The form body

	    # The status bar

	    frame .form.frmStatus -borderwidth 2 -relief raised
	    pack .form.frmStatus -in .form.pForm -side top -fill x
 	    label .form.frmStatus.lblRecord \
		-textvar ::form::txtRecord(23nr47)
	    label .form.frmStatus.lblBlankLeft -text {}
	    label .form.frmStatus.lblFormName \
		-textvar ::form::activeForm \
		-foreground {medium blue}
	    label .form.frmStatus.lblBlankRight -text {}
	    label .form.frmStatus.lblStatus \
		-textvar ::form::txtRecord(23status47)
	    pack .form.frmStatus.lblRecord -in .form.frmStatus -side left
	    pack .form.frmStatus.lblBlankLeft -in .form.frmStatus \
		-side left -expand 1 -fill x
	    pack .form.frmStatus.lblFormName -in .form.frmStatus -side left
	    pack .form.frmStatus.lblBlankRight -in .form.frmStatus \
		-side left -expand 1 -fill x
	    pack .form.frmStatus.lblStatus -in .form.frmStatus -side left

	    # The record frame

	    displayRecordFrame

	    # The buttons bar

	    frame .form.frmButtons -borderwidth 2 -relief raised
	    pack .form.frmButtons -in .form.pForm -side top -fill x

	    # The buttons are defined but not displayed yet. That is
	    # up to the procedures displayBrowseButtons and
	    # displayEditButtons

	    label .form.frmButtons.left
	    button .form.frmButtons.btnUpdate \
		-text Update -command ::form::cmdUpdate -takefocus 0 \
		-underline 0 -pady 0
	    button .form.frmButtons.btnAdd \
		-text Add -command ::form::cmdAdd -takefocus 0 \
		-underline 0 -pady 0
	    button .form.frmButtons.btnDelete \
		-text Delete -command ::form::cmdDelete -takefocus 0 \
		-underline 0 -pady 0
	    button .form.frmButtons.btnOK \
		-text OK -command ::form::cmdOK \
		-underline 0 -pady 0
	    button .form.frmButtons.btnCancel \
		-text Cancel -command ::form::cmdCancel \
		-underline 0 -pady 0
	    label .form.frmButtons.right

	    # The link frame

	    # The title bar

	    frame .form.frmLink1 -borderwidth 2 -relief raised
	    pack .form.frmLink1 -in .form.pLink -side top -fill x
	    label .form.frmLink1.lblLinks -text Links -width 0
	    pack .form.frmLink1.lblLinks -in .form.frmLink1 -side top
	    # The body of the links frame

	    frame .form.frmLinkBtn -borderwidth 2 -relief sunken \
		-width 0 -height 0
	    pack .form.frmLinkBtn -in .form.pLink -side top -expand 1 -fill both

	    # The links are displayed by displayLinks procedure
	}
	return
    }

    proc makeFormPanes {} {
	variable windowSize

	panedwindow .form.pw1 -orient vertical
	pack .form.pw1 -side top -expand 1 -fill both
	if {![info exists windowSize(.form.pQuery)]} then {
	    set windowSize(.form.pQuery) 80
	}
	if {![info exists windowSize(.form.pMid)]} then {
	    set windowSize(.form.pMid) 300
	}
	frame .form.pQuery -height $windowSize(.form.pQuery)
	frame .form.pMid -height $windowSize(.form.pMid)
	frame .form.pResult
	set command {set ::form::windowSize(.form.pQuery) }
	append command {[winfo height .form.pQuery]}
	bind .form.pQuery <Configure> \
	    [list after idle [list catch $command]]
	set command {set ::form::windowSize(.form.pMid) }
	append command {[winfo height .form.pMid]}
	bind .form.pMid <Configure> \
	    [list after idle [list catch $command]]
 	pack propagate .form.pQuery 0
 	pack propagate .form.pMid 0
 	pack propagate .form.pResult 0
	.form.pw1 add .form.pQuery .form.pMid .form.pResult
	panedwindow .form.pw2 -orient horizontal
	pack .form.pw2 -in .form.pMid -side top -expand 1 -fill both
	if {![info exists windowSize(.form.pForm)]} then {
	        set windowSize(.form.pForm) 550
	}
	frame .form.pForm -width $windowSize(.form.pForm)
	set command {set ::form::windowSize(.form.pForm) }
	append command {[winfo width .form.pForm]}
	bind .form.pForm <Configure> \
	    [list after idle [list catch $command]]
        frame .form.pLink
 	pack propagate .form.pForm 0
	pack propagate .form.pLink 0
	.form.pw2 add .form.pForm .form.pLink
	return
    }

    proc displayRecordFrame {} {
	variable widget
	global tcl_platform

	destroy .form.fmCanvas
	destroy .form.canvas
	destroy .form.vscroll
	frame .form.fmCanvas
	canvas .form.canvas -yscrollcommand {.form.vscroll set} \
	    -borderwidth 2 -relief sunken -width 0 -height 0
	scrollbar .form.vscroll -orient vertical \
	    -command {.form.canvas yview} -takefocus 0
	frame .form.canvas.frmRecord
	.form.canvas create window 20 20 \
	    -window .form.canvas.frmRecord -anchor nw
	set widget(record) .form.canvas.frmRecord
	pack .form.fmCanvas -in .form.pForm -after .form.frmStatus \
	    -side top -expand 1 -fill both
	pack .form.canvas -in .form.fmCanvas -side left -expand 1 -fill both
	pack .form.vscroll -in .form.fmCanvas -side left -fill y
	switch -- $tcl_platform(platform) {
	    "windows" {
		bind .form <MouseWheel> {
		    if {%D < 0} then {
			::form::mouseWheelCanvas 1
		    } else {
			::form::mouseWheelCanvas -1
		    }
		}
	    }
	    "unix" -
	    default {
		# On X Window system, mouse wheel sends <4> and <5> events.
		bind .form <4> {::form::mouseWheelCanvas -1}
		bind .form <5> {::form::mouseWheelCanvas 1}
	    }
	}
	return
    }

    proc mouseWheelCanvas {direction} {

	set mouse_x [winfo pointerx .form]
	set mouse_y [winfo pointery .form]
	set x1 [winfo rootx .form.pForm]
	set y1 [winfo rooty .form.pForm]
	set x2 [expr $x1 + [winfo width .form.pForm]]
	set y2 [expr $y1 + [winfo height .form.pForm]]
	if {($x1 <= $mouse_x) && ($mouse_x <= $x2) && \
		($y1 <= $mouse_y) && ($mouse_y <= $y2)} then {
	    .form.canvas yview scroll $direction unit
	}
	return
    }

    ################################################################
    # Commands for browse, update, add and quit buttons            #
    ################################################################

    proc identCurRecord {withTable} {
	variable ::pfm::formsArray
	variable activeForm
	variable recordArray
	variable curRecord
	variable attributeArray

	set whereClause {}
	foreach pkey $formsArray($activeForm,pkey) {
	    if {[info exists recordArray($curRecord,$pkey)]} then {
		if {$whereClause ne {}} then {
		    set whereClause "$whereClause AND"
		}
		if {$withTable} then {
		    set attribute "\"$formsArray($activeForm,tablename)\".\"$pkey\""
		} else {
		    set attribute "\"$pkey\""
		}
		if {[info exists attributeArray($pkey,typeofattrib)] } then {
		    if {$attributeArray($pkey,typeofattrib) eq {taQuoted}} then {
			set value [string map {' ''} $recordArray($curRecord,$pkey)]
			set whereClause \
			    "$whereClause ($attribute = '$value')"
		    } else {
			set whereClause \
			    "$whereClause ($attribute = $recordArray($curRecord,$pkey))"
		    }
		} else {
		    set whereClause \
			"$whereClause ($attribute = $recordArray($curRecord,$pkey))"
		}
	    } else {
		set whereClause FALSE
		set errMsg "The pkey attribute '$pkey' of form '$activeForm' is not returned by the form's SQL statement. Check the form's definition!"
		tk_messageBox -type ok -icon error -message $errMsg -parent .form
		break
	    }
	}
	return $whereClause
    }

    proc cmdHelp {} {

	variable ::pfm::currentDB
	variable activeForm
	variable formHelp

	set queryDef "SELECT help FROM pfm_form WHERE name='$activeForm'"
	set queryRes [pg_exec $currentDB $queryDef]
	set formHelp [lindex [pg_result $queryRes -getTuple 0] 0]
	if { [string length $formHelp] == 0 } then {
	    set formHelp "No help available for $activeForm."
	}
	pg_result $queryRes -clear
	set title "Help for form '$activeForm'"
	::gen::TextEdit $title ::form::formHelp 1 {}
	unset formHelp
	return
    }

    proc cmdAdd {} {
	variable attributeArray
	variable tableAttribList
	variable txtRecord
	variable ::pfm::currentDB
	variable activeForm
	variable windowSize

	if {[winfo exists .form.search]} then {
	    destroy .form.search
	}
	clearTextOnForm .form.txtResult
	foreach attribute $tableAttribList {
	    set defVal $attributeArray($attribute,default)
	    if {$defVal ne {}} then {
		if { [string index $defVal 0] eq {=}} then {
		    set queryDef [string range $defVal 1 end]
		    set queryRes [pg_exec $currentDB $queryDef]
		    if { [pg_result $queryRes -status] eq {PGRES_TUPLES_OK}} then {
			if {[pg_result $queryRes -numTuples] == 1} then {
			    set txtRecord($attribute) \
				[lindex [pg_result $queryRes -getTuple 0] 0]
			} else {
			    set errMsg "$queryDef\nhas returned [pg_result $queryRes -numTuples] records."
			    set errMsg "$errMsg\nCheck the definition of the default value for $attribute of $activeForm in pfm_attribute.\n"
			    bell
			    displayOnForm .form.txtResult $errMsg {red}
			}
		    } else {
			set errMsg "$queryDef\n[pg_result $queryRes -error]"
			set errMsg "$errMsg\nCheck the definition of the default value for $attribute of $activeForm in pfm_attribute.\n"
			bell
			displayOnForm .form.txtResult $errMsg {red}
		    }
		    pg_result $queryRes -clear
		} else {
		    set txtRecord($attribute) $defVal
		}
	    }
	}
	UpdateAllTextEdit 1
	newFormState add
	return
    }

    proc cmdUpdate {} {
	variable txtRecord
	variable windowSize
	# Bug 679 : The start of the transaction is postponed until
	# the user presses [OK]. Instead a 'reload record' is executed
	# to minimize the time window during which another user can
	# modify or delete the current record.

	if { ![string equal $txtRecord(23status47) {After last}] && \
		![string equal $txtRecord(23status47) {Deleted}] && \
		![string equal $txtRecord(23status47) {Not added}] } then {
	    clearTextOnForm .form.txtResult
	    if {[winfo exists .form.search]} then {
		destroy .form.search
	    }
	    if { [reloadRecord] } then {
		newFormState update
		UpdateAllTextEdit 1
	    }	
	}
	return
    }


    proc cmdDelete {} {
	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable curRecord
	variable activeForm
	variable txtRecord
	variable recordArray
	variable formAttribList

	if { ($txtRecord(23status47) ne {After last}) && \
		 ($txtRecord(23status47) ne {Deleted}) && \
		 ($txtRecord(23status47) ne {Not added}) } then {
	    set question "Do you really want to delete the current record?"
	    set answer [tk_messageBox -type yesno -default no -icon question \
			    -message $question -parent .form]
	    if {$answer eq {yes}} then {
		set queryDef "DELETE FROM \"$formsArray($activeForm,tablename)\""
		set queryDef "$queryDef WHERE [identCurRecord 0]"
		clearTextOnForm .form.txtResult
		displayOnForm .form.txtResult "$queryDef\n" {black}
		set queryRes [pg_exec $currentDB $queryDef]
		set status [pg_result $queryRes -status]
		if { [string equal $status {PGRES_COMMAND_OK}] } then {
		    set recordArray($curRecord,23status47) "Deleted"
		    setFormStatus "Deleted"
		    foreach attribute $formAttribList {
			set recordArray($curRecord,$attribute) ""
			set txtRecord($attribute) ""
		    }
		    displayOnForm .form.txtResult $status {green}
		} else {
		    set status $status\n[pg_result $queryRes -error]
		    bell
		    displayOnForm .form.txtResult $status {red}
		}
		pg_result $queryRes -clear
	    }
	    UpdateAllTextEdit 1
	}
	return
    }

    proc cmdFirst {} {
	variable curRecord

	if {[getFormOffset] == 0} then {
	    if {$curRecord != 0} {
		set curRecord 0
		filltxtRecord $curRecord
		clearTextOnForm .form.txtResult
		UpdateAllTextEdit 1
	    }
	} else {
	    loadDataChunk 0
	    set curRecord 0
	    filltxtRecord $curRecord
	    UpdateAllTextEdit 1
	}
	return
    }

    proc cmdNext {} {
	variable curRecord
	variable lastRecord
	variable ::pfm::formsArray

	if {$curRecord <= [expr $lastRecord - 2]} then {
	    incr curRecord
	    filltxtRecord $curRecord
	    clearTextOnForm .form.txtResult
	    UpdateAllTextEdit 1
	} else {
	    if {![lastChunkLoaded]} then {
		loadDataChunk 1
		set curRecord 0
		filltxtRecord $curRecord
		clearTextOnForm .form.txtResult
		UpdateAllTextEdit 1
	    } else {
		set curRecord $lastRecord
		filltxtRecord $curRecord
		clearTextOnForm .form.txtResult
		UpdateAllTextEdit 1
	    }
	}
	return
    }

    proc cmdPrev {} {
	variable curRecord
	variable lastRecord

	if {$curRecord >= 1} then {
	    incr curRecord -1
	    filltxtRecord $curRecord
	    clearTextOnForm .form.txtResult
	    UpdateAllTextEdit 1
	} else {
	    if {[getFormOffset] > 0} then {
		loadDataChunk -1
		set curRecord [expr $lastRecord - 1]
		filltxtRecord $curRecord
		clearTextOnForm .form.txtResult
		UpdateAllTextEdit 1
	    }
	}
	return
    }

    proc cmdLast {} {
	variable curRecord
	variable lastRecord
	variable ::pfm::formsArray
	variable activeForm

	while {![lastChunkLoaded]} {
	    loadDataChunk 1
	}
	set curRecord [expr $lastRecord - 1]
	if {$curRecord < 0} then {
	    set curRecord 0
	}
	filltxtRecord $curRecord
	clearTextOnForm .form.txtResult
	UpdateAllTextEdit 1
 	return
    }

    proc cmdQuitForm {} {
	variable windowSize
	variable formStack
	variable openQueryAfterForm
	# This procedure is called by the <Destroy> event on FormWindow

	DestroyAllTextEdit 0
	DestroyAllTextEdit 1
 	::pfm::refreshFormsList
	if {$openQueryAfterForm} then {
	    # OpenQuery must not be called on "forced" quit: when
	    # closing the database or when opening another form
	    OpenQuery $formStack(0,formId) 0
	}
	return
    }

    proc DestroyAllTextEdit {readonly} {
	variable TextEditList

	foreach Id $TextEditList($readonly) {
	    ::gen::DestroyTextEdit $Id
	}
	set TextEditList($readonly) {}
	return
    }

    proc TextEditPending {readonly} {
	variable TextEditList

	set pending {}
	foreach Id $TextEditList($readonly) {
	    set name  [::gen::TextEditName $Id]
	    if {$name ne {}} then {
		lappend pending $name
	    }
	}
	return $pending
    }

    proc UpdateAllTextEdit {readonly} {
	variable TextEditList
	
	foreach Id $TextEditList($readonly) {
	    if {[::gen::TextEditExists $Id]} then {
		::gen::UpdateTextEdit $Id
	    }
	}
	return
    }


    ################################################################
    # Commands for OK and Cancel buttons                           #
    ################################################################

    proc cmdOK {} {

	variable ::pfm::currentDB
	variable formState

	set pending [TextEditPending 0]
	if {$pending ne {} } then {
	    set pending [join $pending ",\n    "]
	    set message "You still have expanded edit windows for "
	    append message \
		"the attribute(s):\n\n    $pending.\n\n"
	    append message \
		"First decide, for each of these windows, what you want to do:"
	    append message \
		" accept (OK) or discard (Cancel) your edit."
	    tk_messageBox -type ok -icon warning -message $message \
		-parent .form
	} else {
	    # reworked because of bug 679
	    clearTextOnForm .form.txtResult
	    switch $formState {
		"update" {
		    set success [updateRecord]
		    reloadRecord
		}
		"add" {
		    if {[addRecord]} then {
			reloadRecord
		    }
		}
	    }
	    newFormState browse
	    UpdateAllTextEdit 1
	}
	return
    }

    proc cmdCancel {} {

	variable ::pfm::currentDB
	variable curRecord
	variable formState

	DestroyAllTextEdit 0
	# reworked because of bug 679
	filltxtRecord $curRecord
	if { [string equal $formState {update}] } then {
	    set rollbackStatus "Update cancelled."
	} else {
	    set rollbackStatus "No record inserted."
	}
	clearTextOnForm .form.txtResult
	displayOnForm .form.txtResult $rollbackStatus {blue}
	newFormState browse
	UpdateAllTextEdit 1
	return
    }

    ################################################################
    # Commands for selecting a value from a list                   #
    ################################################################

    
    proc cmdSelectFromList {boundWidget width height attribute action} {

	variable ::pfm::currentDB
	variable activeForm
	variable attributeArray
	variable txtRecord
	variable WhereSelected
	variable valueList
	variable displayList

	set title "pfm - Select value for $attribute"

	# Get data for list
	switch $attributeArray($attribute,typeofget) {
	    "tgList" {
		set queryDef "SELECT value,description FROM pfm_value "
		append queryDef \
		    "WHERE valuelist='$attributeArray($attribute,valuelist)'"
	    }
	    "tgLink" {
		set queryDef $attributeArray($attribute,sqlselect)
	    }
	}
	set queryRes [pg_exec $currentDB $queryDef]
	set lastItem [expr [pg_result $queryRes -numTuples] - 1]
	set valueList {}
	set displayList {}
	for {set item 0} {$item <= $lastItem} {incr item } {
	    set listItem [pg_result $queryRes -getTuple $item]
	    lappend valueList [lindex $listItem 0]
	    set displayItem {}
	    set itemNr 0
	    foreach subItem $listItem {
		set displayItem "$displayItem $subItem"
		if {$itemNr == 0} then {
		    set displayItem "$displayItem :"
		}
		incr itemNr
	    }
	    lappend displayList $displayItem
	}
	pg_result $queryRes -clear

	# Get current value
	if {$action eq {fillout}} then {
	    set currVal $txtRecord($attribute)
	} else {
	    set currVal {}
	}
	set selectedValue \
	    [::gen::ListBox $boundWidget $width $height $title \
		 ::form::valueList ::form::displayList $currVal]
	unset valueList
	unset displayList
	switch $action {
	    "fillout" {
		set txtRecord($attribute) $selectedValue
	    }
	    "paste" {
		if {$attributeArray($attribute,typeofattrib) eq {taQuoted}} then {
		    set selectedValue "'${selectedValue}'"
		}
		if {$WhereSelected} then {

		    .query.txtWhere insert insert $selectedValue
		} else {
		    .query.txtOrderBy insert insert $selectedValue
		}
	    }
	}
	return
    }


    ########################################################################
    # Procedures for opening the active form with a defined query          #
    ########################################################################

    proc OpenForm {queryDef intro displayKey} {
	variable activeForm
	variable formQuery
	variable formIntro
	variable curRecord
	variable focusRow
	variable txtRecord
	variable formState

	array unset txtRecord
	set formQuery $queryDef
	set formIntro $intro
	showFormWindow "pfm - Form : $activeForm"
	loadDataChunk 0
	set curRecord [locateRecord $displayKey]
	filltxtRecord $curRecord
	set focusRow 0
	set formState "browse"
	hideBrowseButtons
	hideEditButtons
	displayBrowseButtons
	displayAttribNames
	displayAttribLabels
	displayLinks
	return
    }

    proc loadDataChunk {firstPrevNext} {
	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable activeForm
	variable formQuery
	variable formIntro
	variable lastRecord
	variable recordArray
	variable formAttribList
	variable formOffset
	variable formLastChunk

	set sqlLimit $formsArray($activeForm,sqllimit)
	switch -- $firstPrevNext {
	    0 {# first
		set formOffset 0
	    }
	    -1 {# previous
		if {($formOffset > 0) && ($sqlLimit ne {})} then {
		    incr formOffset -$sqlLimit
		    if {$formOffset < 0} then {
			set formOffset 0
		    }
		}
	    }
	    1 {# next
		if {$sqlLimit ne {}} then {
		    incr formOffset $sqlLimit
		}
	    }
	}
	clearTextOnForm .form.txtQuery
	array unset recordArray
	displayOnForm .form.txtQuery "${formIntro}\n" {blue}
	displayOnForm .form.txtQuery "${formQuery}\n" {black}
	if {$sqlLimit ne {}} then {
	    set limitDef "LIMIT $sqlLimit OFFSET $formOffset"
	    displayOnForm .form.txtQuery "$limitDef\n" {black}
	    set loadDef "${formQuery}\n${limitDef}"
	} else {
	    set limitDef {}
	    set loadDef $formQuery
	}
	set queryRes [pg_exec $currentDB $loadDef]
	set status [pg_result $queryRes -status]
	if { [string equal $status {PGRES_TUPLES_OK}] } then {
	    displayOnForm .form.txtQuery "$status\n" {green}
	    # lastRecord is a dummy empty record.
	    set lastRecord [pg_result $queryRes -numTuples]
	    pg_result $queryRes -assign recordArray
	} else {
	    set lastRecord 0
	    set status $status\n[pg_result $queryRes -error]
	    bell
	    displayOnForm .form.txtQuery "$status\n" {red}
	}
	pg_result $queryRes -clear
	if {$sqlLimit ne {}} then {
	    set formLastChunk [expr $lastRecord < $sqlLimit]
	} else {
	    set formLastChunk 1
	}
	for {set recordNr 0} {$recordNr < $lastRecord} {incr recordNr} {
	    set recordArray($recordNr,23status47) "Not modified"
	}
	foreach attribute $formAttribList {
	    set recordArray($lastRecord,$attribute) {}
	}
	set recordArray($lastRecord,23status47) "After last"
	return
    }

    proc getFormOffset {} {
	variable formOffset

	return $formOffset
    }

    proc lastChunkLoaded {} {
	variable formLastChunk

	return $formLastChunk
    }

    proc locateRecord {displayKey} {
	variable ::pfm::formsArray
	variable activeForm
	variable recordArray
	variable lastRecord

	set locatedRecord 0
	if {$displayKey ne {}} then {
 	    set searching 1
	    while {$searching} {
		# bug 1054: next statement was missing, which caused
		# 'match' to be undefined if lastRecord = 0, leading to
		# a run-time error at the 1st statement after for loop.
		set match 0
		for {set recordNr 0} {$recordNr < $lastRecord} \
		    {incr recordNr} {
			set match 1
			set i 0
			foreach pkey $formsArray($activeForm,pkey) {
			    if {[info exist recordArray($recordNr,$pkey)]} then {
				if {[lindex $displayKey $i] ne \
					$recordArray($recordNr,$pkey)} then {
				    set match 0
				    break
				}
				incr i
			    } else {
				set errMsg "The pkey attribute '$pkey' of form '$activeForm' is not returned by the form's SQL statement. Check the form's definition!"
				tk_messageBox -type ok -icon error -message $errMsg \
				    -parent .form
				break
			    }
			}
			if {$match} then {
			    set locatedRecord $recordNr
			    break
			}
		    }
		if {$match} then {
		    set searching 0
		} else {
		    if {![lastChunkLoaded]} then {
			loadDataChunk 1
		    } else {
			loadDataChunk 0
			set searching 0
		    }
		}
	    }
	}
	return $locatedRecord
    }

    proc displayAttribNames {} {
	variable formAttribList
	variable widget

	set rowidx 0
	foreach attribute $formAttribList {
	    label $widget(record).lb1$attribute \
		-text $attribute -anchor w \
		-relief sunken -borderwidth 2
	    grid $widget(record).lb1$attribute \
		-column 0 -row $rowidx \
		-columnspan 1 -rowspan 1 -sticky we
	    incr rowidx
	}
	# Menu Search
	destroy .form.menu.mnSearch
	menu .form.menu.mnSearch -tearoff 0
	foreach attribute $formAttribList {
	    .form.menu.mnSearch add command \
		-command [list ::form::cmdOpenSearch $attribute] \
		-label $attribute
	}
	return
    }

    proc cmdOpenSearch {attribute} {
	variable formState
	variable windowSize

	if {($formState ne {update}) && ($formState ne {add})} then {
	    if {[winfo exists .form.search]} then {
		destroy .form.search
	    }
	    toplevel .form.search -class Toplevel
	    wm transient .form.search .form
	    set x [expr [winfo rootx .form] + 20]
	    set y [expr [winfo rooty .form] + 5]
	    wm geometry .form.search "+$x+$y"
	    wm title .form.search "pfm - Form: Search $attribute"
	    label .form.search.lb1 -text "Search for next record in buffer with"
	    label .form.search.lb2 -text "$attribute ="
	    entry .form.search.ent -background white -width 50
	    set command {::form::searchInBuffer}
	    lappend command $attribute
	    append command { [.form.search.ent get]}
	    append command { $bnCase}
	    button .form.search.bnSearch -text Next -command $command \
		-takefocus 0 -underline 0 -pady 0
	    button .form.search.bnCancel -text Cancel \
		-command {destroy .form.search} -underline 0 \
		-pady 0 -takefocus 0
	    checkbutton .form.search.bnCase -text {Match case} \
		-underline 0 -takefocus 0
	    set help "*     matches any sequence of characters;"
	    set help "$help\n?     matches any single character."
	    message .form.search.msgHelp -width 500 -text $help -borderwidth 2 \
		-relief groove
	    focus .form.search.ent
	    bind .form.search.ent <KeyPress-Return> $command
	    bind .form.search <KeyPress-Escape> \
		"::gen::PressButton .form.search.bnCancel"
	    bind .form.search <Alt-KeyPress-c> \
		"::gen::PressButton .form.search.bnCancel"
	    bind .form.search <Alt-KeyPress-m> \
		"::gen::PressButton .form.search.bnCase"
	    bind .form.search <Alt-KeyPress-n> \
		"::gen::PressButton .form.search.bnSearch"
	    grid .form.search.lb1 -column 0 -row 0 -columnspan 3
	    grid .form.search.lb2 -column 0 -row 1 -columnspan 1
	    grid .form.search.ent -column 1 -row 1 -columnspan 1
	    grid .form.search.bnSearch -column 2 -row 1 -columnspan 1 -sticky we
	    grid .form.search.bnCase -column 0 -row 2 -columnspan 1
	    grid .form.search.msgHelp -column 1 -row 2 -columnspan 1
	    grid .form.search.bnCancel -column 2 -row 2 -columnspan 1 -sticky we
	}
	return
    }

    proc searchInBuffer {attribute pattern matchcase} {
	variable curRecord
	variable lastRecord
	variable recordArray

	set searching 1
	set startSearch [expr $curRecord + 1]
	while {$searching} {
	    set found 0
	    for {set tuple $startSearch} {$tuple < $lastRecord} {incr tuple} {
		if {$matchcase} then {
		    set found [string match $pattern $recordArray($tuple,$attribute)]
		} else {
		    set found \
			[string match -nocase $pattern $recordArray($tuple,$attribute)]
		}
		if {$found} then {
		    set curRecord $tuple
		    filltxtRecord $curRecord
		    clearTextOnForm .form.txtResult
		    UpdateAllTextEdit 1
		    break
		}
	    }
	    if {$found} then {
		set searching 0
	    } else {
		if {![lastChunkLoaded]} then {
		    loadDataChunk 1
		    set startSearch 0
		} else {
		    set answer \
			[tk_messageBox -type yesno -icon question \
			     -parent .form.search \
			     -message "Last record reached. Wrap around?"]
		    if {$answer eq {yes}} then {
			loadDataChunk 0
			set startSearch 0
			set searching 1
		    } else {
			set searching 0
		    }
		}
	    }
	}
	return
    }

    proc displayAttribLabels {} {

	variable formAttribList
	variable txtRecord
	variable attributeArray
	variable widget
	variable focusRow

	set rowidx 0
	foreach attribute $formAttribList {
	    entry $widget(record).ent$attribute -state readonly -width 40 \
		-textvar ::form::txtRecord($attribute)
		# -highlightthickness 1
	    bind $widget(record).ent$attribute <Alt-KeyPress-x> \
		[list ::gen::PressButton $widget(record).bn$attribute]
	    grid $widget(record).ent$attribute -column 1 -row $rowidx \
		-columnspan 1 -rowspan 1
	    button $widget(record).bn$attribute -text {Expand} -pady 0 \
		-command [list ::form::cmdExpand $attribute 1] \
	        -takefocus 0 -underline 1
	    grid $widget(record).bn$attribute -column 2 -row $rowidx \
		-columnspan 1 -rowspan 1
	    bind $widget(record).ent$attribute <FocusIn> \
		[list ::form::scrollForm $rowidx]
	    incr rowidx
	}
	# update is necessary to get a correct result from the canvas
	# bbox command
	update
	set bbox [.form.canvas bbox all]
	set rightEdge [expr [lindex $bbox 2] + 20]
	set bottomEdge [expr [lindex $bbox 3] + 20]
	.form.canvas configure -scrollregion \
	    [list 0 0 $rightEdge $bottomEdge]
	set attribute [lindex $formAttribList $focusRow]
	focus $widget(record).ent${attribute}
	return
    }

    proc hideAttribLabels {} {

	variable formAttribList
	variable widget
	foreach attribute $formAttribList {
	    destroy $widget(record).ent$attribute
	    destroy $widget(record).bn$attribute
	}
	return
    }

    proc displayAttribEntries {} {

	variable formAttribList
	variable txtRecord
	variable attributeArray
	variable widget
	variable focusRow

	set rowidx 0
	foreach attribute $formAttribList {
	    switch $attributeArray($attribute,typeofget) {
		"tgDirect" -
		"tgExpression" {
		    entry $widget(record).ent$attribute \
			-textvar ::form::txtRecord($attribute) \
			-relief sunken -width 40 \
			-background white 
			# -highlightthickness 1
		    button $widget(record).bn$attribute -text {Expand} \
			-pady 0 \
			-command [list ::form::cmdExpand $attribute 0] \
		        -takefocus 0 -underline 0
		    grid $widget(record).bn$attribute -column 2 \
			-row $rowidx -columnspan 1 -rowspan 1
		}
		"tgReadOnly" {
		    entry $widget(record).ent$attribute \
			-textvar ::form::txtRecord($attribute) \
			-relief sunken \
			-width 40 \
			-state readonly
			# -highlightthickness 1
		    button $widget(record).bn$attribute -text {Expand} \
			-pady 0 \
			-command [list ::form::cmdExpand $attribute 1] \
		        -takefocus 0 -underline 0
		    grid $widget(record).bn$attribute -column 2 \
			-row $rowidx -columnspan 1 -rowspan 1
		}
		"tgList" -
		"tgLink" {
		    button $widget(record).ent$attribute -pady 0 -padx 0 \
			-anchor w -textvar ::form::txtRecord($attribute) \
			-relief raised -width 40
			# -highlightthickness 1
		    set command [list ::form::cmdSelectFromList \
				     $widget(record).ent$attribute \
				     300 300 $attribute fillout]
		    $widget(record).ent$attribute configure -command $command
		    bind $widget(record).ent$attribute \
			<KeyPress-Return> $command
		    button $widget(record).bn$attribute -text {Expand} \
			-pady 0  \
			-command [list ::form::cmdExpand $attribute 0] \
			-takefocus 0 -underline 1
		    grid $widget(record).bn$attribute -column 2 \
			-row $rowidx -columnspan 1 -rowspan 1
		}
	    }
	    grid $widget(record).ent$attribute -column 1 -row $rowidx \
		-columnspan 1 -rowspan 1
	    bind $widget(record).ent$attribute <Alt-KeyPress-x> \
		[list ::gen::PressButton $widget(record).bn$attribute]
	    bind $widget(record).ent$attribute <FocusIn> \
		[list ::form::scrollForm $rowidx]
	    incr rowidx
	}
	# update is necessary to get a correct result from the canvas
	# bbox command
	update
	set bbox [.form.canvas bbox all]
	set rightEdge [expr [lindex $bbox 2] + 20]
	set bottomEdge [expr [lindex $bbox 3] + 20]
	.form.canvas configure -scrollregion \
	    [list 0 0 $rightEdge $bottomEdge]
	set attribute [lindex $formAttribList $focusRow]
	focus $widget(record).ent${attribute}
	return
    }

    proc cmdExpand {attribute readonly} {
	variable TextEditList

	lappend TextEditList($readonly) \
	    [::gen::TextEdit $attribute \
		 ::form::txtRecord($attribute) \
		 $readonly {}]
	return
    }

    proc scrollForm {i} {
	variable formAttribList
	variable focusRow

	# Scroll form such that attribute i is visible in the middle
	# of the form.
	#
	# Let: n1 be index of first visible attribute before scrolling
	#      n2 be index of last visible attribute before scrolling
	#      nn1 be index of first visible attribute after scrolling
	#      nn2 be index of last visible attribute after scrolling
	#      f1 = 1st fraction returned by yview before scrolling (known)
	#      f2 = 2nd fraction returned by yview before scrolling (known)
	#      nf1 = 1st fraction to be given to yview for scrolling
	#      nf2 = 2nd fraction returned by yview after scrolling
	#      n = nr of attributes on form (known)
	#      i = index of attribute with input focus (arg of function)
	#      s = number of visible attributes on screen
	# Known data:
	#     n, f1, f2, i
	# To be calculated:
	#     nf1
	# Calculation:
	#     f1 = (y-coord of first visible horizontal line)
	#                                     /(height of canvas)
	#        = (n1 + 1)/(n + 2)  
	#     f2 = (y-coord of last visible horizontal line)
	#                                     /(height of canvas)
	#        = (n2 + 1)/(n + 2)
	#     -- to take into account the empty space at top and bottom
	#     -- we assume 2 dummy attributes, one at top (index "-1"),
	#     -- one at bottom (index "n")
	#     n1 = (n + 2) * f1 - 1
	#     n2 = (n + 2) * f2 - 1
	#     s = n2 - n1 + 1
	#       = (n + 2) * (f2 - f1) + 1   (1)
	#     Afters scrolling, same relationship:
	#     nn1 = (n + 2) * nf1 - 1
	#     nn2 = (n + 2) * nf2 - 1
	#     s = (n + 2) * (nf2 - nf1) + 1  (2)
	#     After scrolling we want i to be in the middle between nn1 and nn2
	#     i = (nn1 + nn2)/2
	#       = (nf1 + nf2) * (n + 2)/2 - 1 (3)
	#     s we can calculate with (1)
	#     From (2) and (3) it follows:
	#     nf1 = (2*i - s + 3)/(2 * (n + 2))

	# Remeber which row gets focus
	set focusRow $i
	set n [llength $formAttribList]
	set yviewList [.form.canvas yview]
	set f1 [lindex $yviewList 0]
	set f2 [lindex $yviewList 1]
	set s [expr ($n + 2) * ($f2 - $f1) + 1.0]
	set nf1 [expr (2.0*$i - $s + 3.0)/(2.0 * ($n + 2.0))]
	if {$nf1 < 0} then {
	    set nf1 0
	}
	if {$nf1 > 1} then {
	    set nf1 1
	}
	.form.canvas yview moveto $nf1
	return
    }

    proc hideAttribEntries {} {

	variable formAttribList
	variable widget
	foreach attribute $formAttribList {
	    destroy $widget(record).ent$attribute
	    destroy $widget(record).bn$attribute
	}
	return
    }

    proc displayBrowseButtons {} {

	variable activeForm
	variable ::pfm::formsArray

	# enable Form menu
	.form.menu entryconfigure 0 -state normal
	# enable Go to menu
	.form.menu entryconfigure 2 -state normal
	# enable Help menu
	.form.menu entryconfigure 3 -state normal
	set view $formsArray($activeForm,view)
	pack .form.frmButtons.left -side left -expand 1
	if { [string equal $view {f}] } then {
	    # enable Record menu
	    .form.menu entryconfigure 1 -state normal
	    pack .form.frmButtons.btnUpdate -side left
	    pack .form.frmButtons.btnAdd -side left
	    pack .form.frmButtons.btnDelete -side left
	    bind .form <Alt-KeyPress-u> \
		{::gen::PressButton .form.frmButtons.btnUpdate}
	    bind .form <Alt-KeyPress-a> \
		{::gen::PressButton .form.frmButtons.btnAdd}
	    bind .form <Alt-KeyPress-d> \
		{::gen::PressButton .form.frmButtons.btnDelete}
	}
	pack .form.frmButtons.right -side left -expand 1
	bind .form <KeyPress-Next> {::form::cmdNext}
	bind .form <KeyPress-Right> {::form::cmdNext}
	bind .form <KeyPress-Prior> {::form::cmdPrev}
	bind .form <KeyPress-Left> {::form::cmdPrev}
	bind .form <KeyPress-Escape> {
	    if {[winfo exists .form.frmLinkBtn.btnBack]} then {
		::gen::PressButton .form.frmLinkBtn.btnBack
	    } else {
		destroy .form
	    }
	}
	bind .form <KeyPress-Home> {::form::cmdFirst}
	bind .form <KeyPress-End> {::form::cmdLast}
	return
    }

    proc hideBrowseButtons {} {

	foreach widget [pack slaves .form.frmButtons] {
	    pack forget $widget
	}
	# unbind keys
	bind .form <KeyPress-Next> {}
	bind .form <KeyPress-Right> {}
	bind .form <KeyPress-Prior> {}
	bind .form <KeyPress-Left> {}
	bind .form <Alt-KeyPress-u> {}
	bind .form <Alt-KeyPress-a> {}
	bind .form <Alt-KeyPress-d> {}
	bind .form <KeyPress-Escape> {}
	bind .form <KeyPress-Home> {}
	bind .form <KeyPress-End> {}
	# disable all menus
	.form.menu entryconfigure 0 -state disabled
	.form.menu entryconfigure 1 -state disabled
	.form.menu entryconfigure 2 -state disabled
	.form.menu entryconfigure 3 -state disabled
	return
    }

    proc displayEditButtons {} {

	pack .form.frmButtons.left -side left -expand 1
	pack .form.frmButtons.btnOK -side left
	pack .form.frmButtons.btnCancel -side left
	pack .form.frmButtons.right -side left -expand 1
	bind .form <Alt-KeyPress-o> \
	    [list ::gen::PressButton .form.frmButtons.btnOK]
	bind .form <Alt-KeyPress-c> \
	    [list ::gen::PressButton .form.frmButtons.btnCancel]
	bind .form <KeyPress-Escape> \
	    [list ::gen::PressButton .form.frmButtons.btnCancel]
	return
    }

    proc hideEditButtons {} {

	foreach widget [pack slaves .form.frmButtons] {
	    pack forget $widget
	}
	bind .form <Alt-KeyPress-o> {}
	bind .form <Alt-KeyPress-c> {}
	bind .form <KeyPress-Escape> {}

	return
    }

    proc newFormState {newState} {

	variable formState
	variable txtRecord

	set formState $newState
	displayRecordFrame
	displayAttribNames
	switch $newState {
	    "update" {
		hideAttribLabels
		displayAttribEntries
		hideBrowseButtons
		displayEditButtons
		newLinkState disabled
		setFormStatus "Updating"
	    }
	    "add" {
		hideAttribLabels
		displayAttribEntries
		hideBrowseButtons
		displayEditButtons
		newLinkState disabled
		setFormStatus "Adding"
	    }
	    "browse" {
		hideAttribEntries
		displayAttribLabels
		hideEditButtons
		displayBrowseButtons
		newLinkState normal
	    }
	}
	return
    }

    proc setFormStatus {status} {
	variable txtRecord
	
	set txtRecord(23status47) $status
	switch -- $status {
	    "Updated" -
	    "Added" -
	    "Deleted" {
		set colour {green4}
	    }
	    "Updating" -
	    "Adding" {
		set colour {red3}
	    }
	    "Not added" -
	    "After last" {
		set colour {medium blue}
	    }
	    "Not modified" -
	    default {
		set colour {black}
	    }
	}
	.form.frmStatus.lblStatus configure -foreground $colour
	return
    }

    proc filltxtRecord {recordNr} {
	variable recordArray
	variable txtRecord
	variable formAttribList
	variable lastRecord
	variable activeForm
	variable ::pfm::formsArray

	foreach attribute $formAttribList {
	    if {[info exists recordArray($recordNr,$attribute)]} then {
		set txtRecord($attribute) $recordArray($recordNr,$attribute)
	    } else {
		set txtRecord($attribute) "Undefined"
		set errMsg "The attribute '$attribute' of form '$activeForm' is not returned by the form's SQL statement. Check the form's definition!"
		tk_messageBox -type ok -icon error -message $errMsg -parent .form
	    }
	}
	set absRecordNr [expr [getFormOffset] + $recordNr + 1]
	if {[lastChunkLoaded]} then {
	    set absLastRecord [expr [getFormOffset] + $lastRecord]
	} else {
	    set absLastRecord {?}
	}
	set txtRecord(23nr47) "Record ${absRecordNr}/${absLastRecord}"
	setFormStatus $recordArray($recordNr,23status47)
	return
    }

    proc getAttributes {formName} {

	variable ::pfm::currentDB
	variable attributeArray
	variable formAttribList
	variable tableAttribList
	array unset attributeArray
	set fields "attribute,typeofattrib,typeofget,valuelist,sqlselect,\"default\""
	set queryDef \
	    "SELECT $fields FROM pfm_attribute WHERE form = '$formName' ORDER BY nr"
	set queryRes [pg_exec $currentDB $queryDef]
	set formAttribList [list]
	set tableAttribList [list]
	set lastAttribute [expr [pg_result $queryRes -numTuples] - 1]
	pg_result $queryRes -assign attribRecords
	for {set attribNr 0} {$attribNr <= $lastAttribute} {incr attribNr} {
	    set attribute [string trim $attribRecords($attribNr,attribute)]
	    lappend formAttribList $attribute
	    set typeofattrib [string trim $attribRecords($attribNr,typeofattrib)]
	    set attributeArray($attribute,typeofattrib) $typeofattrib
	    set typeofget [string trim $attribRecords($attribNr,typeofget)]
	    set attributeArray($attribute,typeofget) $typeofget
	    set valuelist [string trim $attribRecords($attribNr,valuelist)]
	    set attributeArray($attribute,valuelist) $valuelist
	    set sqlselect [string trim $attribRecords($attribNr,sqlselect)]
	    set attributeArray($attribute,sqlselect) $sqlselect
	    set defVal [string trim $attribRecords($attribNr,default)]
	    set attributeArray($attribute,default) $defVal
	    if { ($typeofget ne {tgReadOnly}) || ($defVal ne {}) } then {
		lappend tableAttribList $attribute
	    }
	}
	pg_result $queryRes -clear
	array unset attribRecords
	return
    }

    ########################################################################
    # Procedures that modify the data base                                 #
    ########################################################################

    proc addRecord {} {
	# insert txtRecord to data base and to recordArray

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable recordArray
	variable txtRecord
	variable curRecord
	variable activeForm
	variable lastRecord
	variable attributeArray
	variable tableAttribList
	variable formAttribList

	set success 1
	set colDef "("
	set valueDef "("
	foreach attribute $tableAttribList {
	    set colDef "$colDef \"$attribute\","
	    if { [string equal $attributeArray($attribute,typeofget) {tgExpression}] } then {
		set txtRecord($attribute) [expr $txtRecord($attribute)]
	    }
	    switch $attributeArray($attribute,typeofattrib) {
		"taQuoted" {
		    set convertedValue [string map {' ''} $txtRecord($attribute)]
		    append valueDef " '$convertedValue',"
		}
		"taNotQuoted" {
		    append valueDef " $txtRecord($attribute),"
		}
	    }
	}
	set colDef "[string trimright $colDef ","])"
	set valueDef "[string trimright $valueDef ","])"
	set queryDef "INSERT INTO \"$formsArray($activeForm,tablename)\" $colDef\nVALUES $valueDef"
	displayOnForm .form.txtResult "$queryDef\n" {black}
	set queryRes [pg_exec $currentDB $queryDef]
	set status [pg_result $queryRes -status]
	if { [string equal $status {PGRES_COMMAND_OK}] } then {
	    displayOnForm .form.txtResult "$status\n" {green}
	    set curRecord $lastRecord
	    incr lastRecord
	    set recordArray($curRecord,23status47) "Added"
	    set recordArray($lastRecord,23status47) "After last"
	    if {[lsearch $formsArray($activeForm,pkey) {oid}] >= 0} then {
		set oid [pg_result $queryRes -oid]
		set recordArray($curRecord,oid) $oid
	    }
	    foreach attribute $tableAttribList {
		set recordArray($curRecord,$attribute) $txtRecord($attribute)
	    }
	    foreach attribute $formAttribList {
		set recordArray($lastRecord,$attribute) {}
	    }
	    filltxtRecord $curRecord
	} else {
	    set status $status\n[pg_result $queryRes -error]
	    set success 0
	    displayOnForm .form.txtResult "$status\n" {red}
	    # set curRecord $lastRecord
	    # We don't filltxtRecord $curRecord. This leaves the user's
	    # input on the form which makes it easy to correct the input.
	    # However, as soon as the user moves away from this record
	    # both data and status are lost forever.
	    set txtRecord(23nr47) "Record ?/?"
	    setFormStatus "Not added"
	}
	pg_result $queryRes -clear
	if {!$success} then {
	    bell
	    tk_messageBox -type ok -icon warning -message "Add record has failed" \
		-parent .form
	}
	return $success
    }


    proc updateRecord {} {
	# copy txtRecord to data base and to recordArray

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable recordArray
	variable txtRecord
	variable curRecord
	variable activeForm
	variable attributeArray
	variable tableAttribList

	# Reworked because of bugs 679 and 680.
	# selectForUpdate starts a transaction and checks that the record
	# still exists and has not been modified by another user. If that check
	# fails, selectForUpdate returns 'false' and the update operation is
	# canceled.
	set success 1
	if { [selectForUpdate] } then {
	    set queryDef {}
	    foreach attribute $tableAttribList {
		if {[info exists recordArray($curRecord,$attribute)]} then {
		    if { $txtRecord($attribute) ne \
			     $recordArray($curRecord,$attribute)} then {
			if { $attributeArray($attribute,typeofget) eq \
				 {tgExpression}} then {
			    set txtRecord($attribute) [expr $txtRecord($attribute)]
			}
			switch $attributeArray($attribute,typeofattrib) {
			    "taQuoted" {
				set quotesDoubled [string map {' ''} $txtRecord($attribute)]
				append queryDef " \"${attribute}\"='$quotesDoubled',"
			    }
			    "taNotQuoted" {
				append queryDef " \"${attribute}\"=$txtRecord($attribute),"
			    }
			}
		    }
		} else {
		    set success 0
		    set errMsg "The attribute '$attribute' of form '$activeForm' is not returned by the form's SQL statement. Check the form's definition!\n"
		    displayOnForm .form.txtResult $errMsg {red}
		}
	    }
	    if { ![string equal $queryDef {}] && $success } then {
		set queryDef [string trimright $queryDef ","]
		set queryDef "UPDATE \"$formsArray($activeForm,tablename)\"\nSET $queryDef"
		append queryDef "\nWHERE [identCurRecord 0]"
		displayOnForm .form.txtResult "$queryDef\n" {black}
		set queryRes [pg_exec $currentDB $queryDef]
		set status [pg_result $queryRes -status]
		if { [string equal $status {PGRES_COMMAND_OK}] } then {
		    set recordArray($curRecord,23status47) "Updated"
		    setFormStatus "Updated"
		    foreach attribute $tableAttribList {
			set recordArray($curRecord,$attribute) $txtRecord($attribute)
		    }
		    displayOnForm .form.txtResult "$status\n" {green}
		    set endTransaction {COMMIT WORK}
		    displayOnForm .form.txtResult "$endTransaction\n" {black}
		} else {
		    set status $status\n[pg_result $queryRes -error]
		    set success 0
		    displayOnForm .form.txtResult "$status\n" {red}
		    set endTransaction {ROLLBACK WORK}
		    displayOnForm .form.txtResult "$endTransaction\n" {black}
		}
		pg_result $queryRes -clear
		filltxtRecord $curRecord
	    } else {
		set status {No updates.}
		set success 0
		displayOnForm .form.txtResult "$status\n" {blue}
		set endTransaction {ROLLBACK WORK}
		displayOnForm .form.txtResult "$endTransaction\n" {black}
		setFormStatus $recordArray($curRecord,23status47)
	    }
	} else {
	    set status {Your update is cancelled.}
	    set success 0
	    displayOnForm .form.txtResult "$status\n" {blue}
	    set endTransaction {ROLLBACK WORK}
	    displayOnForm .form.txtResult "$endTransaction\n" {black}
	    setFormStatus $recordArray($curRecord,23status47)
	}
	set commitResult [pg_exec $currentDB $endTransaction]
	set commitStatus [pg_result $commitResult -status]
	if { ![string equal $commitStatus {PGRES_COMMAND_OK}] } then {
	    set commitStatus "$commitStatus\n[pg_result $commitResult -error]"
	    displayOnForm .form.txtResult "$commitStatus\n" {red}
	    set success 0
	    setFormStatus $recordArray($curRecord,23status47)
	} else {
	    displayOnForm .form.txtResult "$commitStatus\n" {green}
	}
	pg_result $commitResult -clear
	if {!$success} then {
	    bell
	    tk_messageBox -type ok -icon warning -message "No updates done" \
		-parent .form
	}
	return $success
    }

#########################################################################
# selectForUpdate and reloadRecord                                      #
#########################################################################

    proc selectForUpdate { } {

	# Introduced because of bugs 679 and 680.
	# selectForUpdate starts a transaction and checks that the record
	# still exists and has not been modified by another user. If that check
	# fails, selectForUpdate returns 'false'.

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable txtRecord
	variable recordArray
	variable curRecord
	variable tableAttribList
	variable activeForm
	set success 1
	set status {}
	set queryDef {}
	set sqlAttrib \"[join $tableAttribList "\", \""]\"
	set beginWork [pg_exec $currentDB "BEGIN WORK"]
	displayOnForm .form.txtResult "BEGIN WORK\n" {black}
	set beginStatus [pg_result $beginWork -status]
	if { [string equal $beginStatus {PGRES_COMMAND_OK}] } then {
	    displayOnForm .form.txtResult "$beginStatus\n" {green}
	    set queryDef "SELECT $sqlAttrib FROM \"$formsArray($activeForm,tablename)\""
	    set queryDef "$queryDef\nWHERE [identCurRecord 0] FOR UPDATE"
	    displayOnForm .form.txtResult "$queryDef\n" {black}
	    set queryRes [pg_exec $currentDB $queryDef]
	    set status [pg_result $queryRes -status]
	    if { [string equal $status {PGRES_TUPLES_OK}] } then {
		displayOnForm .form.txtResult "$status\n" {green}
		switch -- [pg_result $queryRes -numTuples] {
		    1 {
			pg_result $queryRes -assign recordForUpdate
			foreach attribute $tableAttribList {
			    if {[info exists recordArray($curRecord,$attribute)]} then {
				if { ![string equal $recordArray($curRecord,$attribute) \
					   $recordForUpdate(0,$attribute)] } then {
				    set success 0
				    set status \
					"Record was modified by another user after you pressed the \[Update\] button.\nReconsider your input."
				    displayOnForm .form.txtResult "$status\n" {red}
				    break
				}
			    } else {
				set success 0
				set status \
				    "Attribute '$attribute' of form '$activeForm' was not returned by the form's SQL statement.\nCheck the form's definition!"
				displayOnForm .form.txtResult "$status\n" {red}
				break
			    }
			}
		    }
		    0 {
			set success 0
			# Bug 680 : selectForUpdate has to take into account that the the
			# current record may have been deleted by another user.
			set status \
			    "Record was deleted by another user after you pressed \[Update\]."
			displayOnForm .form.txtResult "$status\n" {red}
		    }
		    default {
			set success 0
			set status "Error in definition of form '$activeForm'."
			set status "$status\npkey '$formsArray($activeForm,pkey)' does not uniquely define a record."
			set status "$status\n[pg_result $queryRes -numTuples] records where returned by the query."
			displayOnForm .form.txtResult "$status\n" {red}
		    }
		}
	    } else {
		set success 0
		set status $status\n[pg_result $queryRes -error]
		displayOnForm .form.txtResult "$status\n" {red}
	    }
	    pg_result $queryRes -clear
	} else {
	    set success 0
	    set beginStatus "$beginStatus\n[pg_result $beginWork -error]"
	    displayOnForm .form.txtResult "$beginStatus\n" {red}
	}
	pg_result $beginWork -clear
	if {!$success} then {
	    bell
	    tk_messageBox -type ok -icon warning -parent .form \
		-message "Select for update failed"
	}
	return $success
    }

    proc reloadRecord { } {

	variable ::pfm::currentDB
	variable ::pfm::formsArray
	variable txtRecord
	variable recordArray
	variable curRecord
	variable formAttribList
	variable activeForm
	set success 1
	if { ![string equal $txtRecord(23status47) {After last}] && \
		![string equal $txtRecord(23status47) {Deleted}] && \
		![string equal $txtRecord(23status47) {Not added}] } then {
	    set sqlAttrib $formsArray($activeForm,sqlselect)
	    set sqlFrom $formsArray($activeForm,sqlfrom)
	    set groupby $formsArray($activeForm,groupby)
	    set tableName $formsArray($activeForm,tablename)
	    set queryDef "SELECT $sqlAttrib\nFROM $sqlFrom"
	    if { [string length $groupby] != 0 } then {
		set queryDef "$queryDef\nGROUP BY $groupby"
		# If there is a GROUP BY clause, the WHERE clause must become a
		# HAVING clause
		set queryDef "$queryDef\nHAVING [identCurRecord 1]"
	    } else {
		set queryDef "$queryDef\nWHERE [identCurRecord 1]"
	    }
	    displayOnForm .form.txtResult "Reload record:\n" {blue}
	    displayOnForm .form.txtResult "$queryDef\n" {black}
	    set queryRes [pg_exec $currentDB $queryDef]
	    set status [pg_result $queryRes -status]
	    if { [string equal $status {PGRES_TUPLES_OK}]} then {
		displayOnForm .form.txtResult "$status\n" {green}
		switch -- [pg_result $queryRes -numTuples] {
		    1 {
			pg_result $queryRes -assign reloadedRecord
			foreach attribute $formAttribList {
			    if {[info exists reloadedRecord(0,$attribute)]} then {
				set recordArray($curRecord,$attribute) $reloadedRecord(0,$attribute)
				set txtRecord($attribute) $recordArray($curRecord,$attribute)
			    } else {
				set errMsg "The attribute '$attribute' of form '$activeForm' is not returned by the form's SQL statement. Check the form's definition!"
				displayOnForm .form.txtResult "$errMsg\n" {red}
				set succes 0
			    }
			}
		    }
		    0 {
			set success 0
			set status "The query did not return any tuples."
			displayOnForm .form.txtResult "$status\n" {red}
			set status "The record is deleted from the internal buffer,"
			append status " but the database is not modified."
			displayOnForm .form.txtResult "$status\n" {blue}
			set recordArray($curRecord,23status47) "Deleted"
			setFormStatus "Deleted"
			foreach attribute $formAttribList {
			    set recordArray($curRecord,$attribute) ""
			    set txtRecord($attribute) ""
			}
		    }
		    default {
			set success 0
			set status "Error in definition of form $activeForm."
			append status "\npkey '$formsArray($activeForm,pkey)' does not uniquely define a record."
			append status "\n[pg_result $queryRes -numTuples] records where returned by the query.\n"
			displayOnForm .form.txtResult $status {red}
		    }
		}
	    } else {
		set success 0
		set status "$status\n[pg_result $queryRes -error]"
		displayOnForm .form.txtResult "$status\n" {red}
	    }
	    pg_result $queryRes -clear
	} else {
	    set success 0
	    set status {Record not reloaded}
	    displayOnForm .form.txtResult "$status\n" {blue}
	}
	if {!$success} then {
	    bell
	    tk_messageBox -type ok -icon warning -parent .form -message \
		"Reload record has failed"
	}
	return $success
    }

    ###########################################################################
    #                                                                         #
    # Procedure for writing text on the text windows txtQuery and txtResult   #
    #                                                                         #
    ###########################################################################

    proc displayOnForm {txtWidget text colour} {

	$txtWidget configure -state normal
	if {$colour ne {black}} then {
	    set begin [$txtWidget index "end - 1 chars"]
	    $txtWidget insert end $text
	    set end [$txtWidget index "end -1 chars"]
	    $txtWidget tag add ${colour}Tag $begin $end
	} else {
	    $txtWidget insert end $text
	}
	$txtWidget configure -state disabled
	return
    }

    proc clearTextOnForm {txtWidget} {

	$txtWidget configure -state normal
	$txtWidget delete 1.0 end
	$txtWidget configure -state disabled
	return
    }

    #######################################################################
    #                                                                     #
    #  Procedures treating the links                                      #
    #                                                                     #
    #######################################################################

    proc displayLinks {} {

	variable ::pfm::currentDB
	variable activeForm
	variable linksArray
	variable lastFormOnStack

	if {$lastFormOnStack != 0} then {
	    button .form.frmLinkBtn.btnBack -text "0: Back (Esc)" -pady 0 -anchor w \
		-command ::form::cmdBack -takefocus 0 \
		-underline 0
	    bind .form <Alt-KeyPress-0> \
		[list ::gen::PressButton .form.frmLinkBtn.btnBack]
	    grid .form.frmLinkBtn.btnBack -row 0 \
		    -column 0 -rowspan 1 -columnspan 1 -sticky we
	}
	array unset linksArray
	set queryDef "SELECT * from pfm_link WHERE fromform = '$activeForm'"
	set queryRes [pg_exec $currentDB $queryDef]
	pg_result $queryRes -assign linksArray
	set lastLink [expr [pg_result $queryRes -numTuples] -1]
	pg_result $queryRes -clear
	set rowidx 1
	for {set link 0} {$link <= $lastLink} {incr link} {
	    set linkName $linksArray($link,linkname)
	    button .form.frmLinkBtn.btn$link -text "   $linkName" \
		-pady 0 -anchor w \
		-command [list ::form::cmdFollowLink $link] \
		-takefocus 0
	    if {$rowidx <= 9} then {
		.form.frmLinkBtn.btn$link configure -text "${rowidx}: $linkName" \
		    -underline 0
		set key "<Alt-KeyPress-${rowidx}>"
		bind .form $key [list ::gen::PressButton .form.frmLinkBtn.btn${link}]
	    }
	    grid .form.frmLinkBtn.btn$link -row $rowidx \
		    -column 0 -rowspan 1 -columnspan 1 -sticky we
	    incr rowidx
	}
	return
    }

    proc newLinkState {newState} {

	if { [winfo exists .form.frmLinkBtn.btnBack] } then {
	    .form.frmLinkBtn.btnBack configure -state $newState
	}
	set link 0
	while { [winfo exists .form.frmLinkBtn.btn$link] } {
	    .form.frmLinkBtn.btn$link configure -state $newState
	    incr link
	}
	return
    }

    proc cmdFollowLink {link} {

	variable ::pfm::formsArray
	variable linksArray
	variable txtRecord
	variable attributeArray
	variable activeForm
	variable recordArray
	variable curRecord
	variable lastFormOnStack
	variable formStack
	variable windowSize

	DestroyAllTextEdit 0
	DestroyAllTextEdit 1
	if { ![string equal $txtRecord(23status47) {After last}] && \
		![string equal $txtRecord(23status47) {Deleted}] && \
		![string equal $txtRecord(23status47) {Not added}] } then {
	    if {[winfo exists .form.search]} then {
		destroy .form.search
	    }
	    set whereDef [expandSqlWhere $linksArray($link,sqlwhere) $link]
	    set orderDef $linksArray($link,orderby)
	    set dispAttribList $linksArray($link,displayattrib)
	    set displayDef {}
	    foreach attribute $dispAttribList {
		if {[info exists txtRecord($attribute)]} then {
		    lappend displayDef $txtRecord($attribute)
		} else {
		    set errMsg "Wrong definition of 'displayattrib' for link '$linksArray($link,linkname)'."
		    set errMsg "$errMsg\n'$attribute' is not an attribute of form '$activeForm'"
		    tk_messageBox -type ok -icon error -parent .form -message $errMsg
		}
	    }
	    set displayDef [join $displayDef {, }]
	    set From "$activeForm \($displayDef\)"
	    set To "$linksArray($link,toform)"
	    # remember view attribute of 'fromform'
	    set view $formsArray($activeForm,view)
	    # remember active form
	    set fromForm $activeForm
	    # prepare form pointed to by the link
	    set activeForm $linksArray($link,toform)
	    getAttributes $activeForm
	    set sqlAttrib $formsArray($activeForm,sqlselect)
	    if {!$formsArray($activeForm,view) && \
		     ([lsearch $formsArray($activeForm,pkey) {oid}] >= 0) && \
		     ([regexp {\moid\M} $sqlAttrib] == 0)} then {
		set sqlAttrib \
		    "\"$formsArray($activeForm,tablename)\".oid, $sqlAttrib"
	    }
	    set sqlFrom $formsArray($activeForm,sqlfrom)
	    set groupby $formsArray($activeForm,groupby)
	    set queryDef "SELECT $sqlAttrib\nFROM $sqlFrom"
	    set intro $formStack($lastFormOnStack,intro)
	    append intro "\nLink \'$linksArray($link,linkname)\': $From -> $To"
	    if { [string length $groupby] != 0 } then {
		append queryDef "\nGROUP BY $groupby"
		# If there is a GROUP BY clause, the whereDef must become a
		# HAVING clause.
		append queryDef "\nHAVING $whereDef"
	    } else {
		append queryDef "\nWHERE $whereDef"
	    }
	    if { [string length $orderDef] != 0 } then {
		append queryDef "\nORDER BY $orderDef"
	    }
	    set displayKey {}
	    foreach pkey $formsArray($fromForm,pkey) {
		if {[info exists recordArray($curRecord,$pkey)]} then {
		    lappend displayKey $recordArray($curRecord,$pkey)
		} else {
		    set errMsg "The pkey attribute '$pkey' of form '$activeForm' is not returned by the form's SQL statement. Check the form's definition!"
		    tk_messageBox -type ok -icon error -parent .form \
			-message $errMsg
		}
	    }
	    set formStack($lastFormOnStack,displayKey) $displayKey
	    incr lastFormOnStack
	    set formStack($lastFormOnStack,formId) $activeForm
	    set formStack($lastFormOnStack,queryDef) $queryDef
	    set formStack($lastFormOnStack,intro) $intro
	    OpenForm $queryDef $intro {}
	}
	return
    }

    proc cmdBack {} {
	variable activeForm
	variable lastFormOnStack
	variable formStack

	DestroyAllTextEdit 0
	DestroyAllTextEdit 1
	if {$lastFormOnStack >= 1} then {
	    incr lastFormOnStack -1
	    set activeForm $formStack($lastFormOnStack,formId)
	    getAttributes $activeForm
	    set queryDef $formStack($lastFormOnStack,queryDef)
	    set intro $formStack($lastFormOnStack,intro)
	    set displayKey $formStack($lastFormOnStack,displayKey)
	    OpenForm $queryDef $intro $displayKey
	}
	return
    }

    proc expandSqlWhere {sqlWhere link} {
	# This procedure finds the $(name) variables,
	# and replaces them with $txtRecord(name)
	# and also doubles all single quotes

	variable txtRecord
	variable linksArray

	set expandWhere $sqlWhere
	set first [string first "\$(" $expandWhere]
	while {$first >= 0} {
	    set last [string first ")" $expandWhere $first]
	    set parName [string range $expandWhere [expr $first + 2] [expr $last -1]]
	    if {[info exists txtRecord($parName)]} then {
		set parameter [string map {' ''} $txtRecord($parName)]
		set expandWhere [string replace $expandWhere $first $last $parameter]
	    } else {
		set errMsg \
		    "Error in 'sqlwhere' of link '$linksArray($link,linkname)' "
		append errMsg \
		    "from '$linksArray($link,fromform)' to '$linksArray($link,toform)': "
		append errMsg \
		    "No attribute '$parName' in form '$linksArray($link,fromform)'"
		tk_messageBox -type ok -icon error -parent .form -message $errMsg
	    }
	    set first [string first "\$(" $expandWhere $last]
	}
	return $expandWhere
    }

    ############################################################
    # init namespace form                                      #
    ############################################################

    set TextEditList(0) {}
    set TextEditList(1) {}

}


##################################################################
# End of namespace form                                          #
##################################################################

##################################################################
# Begin namespace report                                         #
##################################################################

# commandHistory is an associative array where
#
#     - commandHistory(top) is the index in the history where
#           the next command will be stored, i.e. it points
#           to the first free element in the command history array.
#
#     - commandHistory(cursor) is the index in the history of
#               the command that is currently displayed
#
#     - commandHistory($n) is the $n-th command in the history
#
# reportDef is an array which contains the data stored in table
#           pfm_report. It is filled out by cmdDisplayReportList.
#
# reportData($tuple,$field) contains the data returned by the report's
#      SQL SELECT statement
#
# reportLastTuple is the last tuple in reportData.
#
# sectionFields($level) contains, for each level, the list
# of field names to be displayed in that level.
#
# fieldLabel($field) contains the label to be used for $field.
# It is filled by readSectionDef
#
# fieldAlignment($field) contains the alignment info:
#    l: left
#    r: right
# It is filled by readSectionDef
#
# fieldMaxLength($field) specifies the maximum line length for the data
#    of $field. Lines exceeding that length are wrapped by inserting
#    line breaks before printing the report
#
# sectionLayout($level) = row, column or table
#         - "row" means that the labels and values are printed on 1 row
#         - "column" means that the labels are printed in the
#            first column and the values in the second column
#         - "table" means that the values are printed in a
#            table with the labels as table header
#               
# It is filled out by readSectionDef
#
# lastLevel contains the last section number. Is is calculated by
# readSectionDef.
#
# summaryList($level) the list of summary field defintions for $level.
# Every list item is itself a list of 'field', 'aggregate' and 'format'
#
# summaryList is filled by readSectionDef.
#
# reportTree($node) is a tree in which each $node refers to a tuple of
# reportData. It is filled by fillReportTree.
#
# Each node of the tree is uniquely identified by $node, a string of the form
#
#    0.n1.n2.n3 ...
#
# where 0 is the root node,
#       0.1 through 0.N1 are the 'level 1' nodes. They are all children of the
#           root node,
#       0.n1.1 through 0.n1.N2 are the children of node 0.n1. They are nodes of
#           level 2
#       etc.
#
# At level 1, nodes are only created for tuples of reportData that have
# different values for at least one level 1 field.

# At level n (1 < n < $lastLevel):
#    - each node is a child of a level (n-1) node;
#    - all children of the same level (n-1) node, have the same data
#      for all fields of levels lower than n; in other words, the parent
#      node determines the lower level data;
#    - all children of the same level (n-1) node, have different data
#      for at least one of the level n fields

# At level $lastLevel:
#    - each node is a child of a level ($lastLevel-1) node;
#    - all children of the same level ($lastLevel-1) node, have the same data
#      for all fields of levels lower than $lastLevel;
#    - a node is created for every tuple of reportData, regardless of
#      whether the data for the $lastLevel fields are equal or different.

# lastSeqnr($parent) contains the last seqnr of the children of $parent where
#       $parent.$seqnr
# is the node identifier of the chidren of $parent.

# summaryData($parent,$field,$aggregate) contains the data for the
# summary data that are calculated from the children and higher level
# descendants of $parent.  It is filled by calculateSummaries.

# MaxLabelWidth($level) contains the maximum label width for all the
# fields of $level.

# MaxColWidth($field) contains the maximum column width for a table
# layout where the data of $field are printed in the column and the
# fieldLabel($field) is printed in the column header. Multi-line data
# are taken into account.

# MaxSummaryWidth($level) contains the maximum length of
# "$aggregate($field)" for all the summary fields of this level.


namespace eval report {

    variable ::pfm::currentDB
    variable ::pfm::dbName
    variable ::pfm::psqlChannel
    variable ::pfm::errChannel
    variable commandHistory
    variable reportMode {sql}
    variable printCommand
    variable parmlist
    variable windowSize
    variable reportDef
    variable reportData
    variable reportLastTuple
    variable sectionFields
    variable fieldLabel
    variable fieldAlignment
    variable fieldMaxLength
    variable sectionLayout
    variable lastLevel
    variable summaryList
    variable reportTree
    variable lastSeqnr
    variable summaryData
    variable MaxLabelWidth
    variable MaxColWidth
    variable MaxSummaryWidth


########################################################################
#  This is the entry point for this namespace                          #
########################################################################

    proc cmdReportSQL {requestedMode} {
	variable ::pfm::currentDB
	variable ::pfm::dbName
	variable reportMode
	variable wrapOn
	variable windowSize

	set wrapOn 0
	set reportMode $requestedMode
	if { [info exists currentDB] } then {
	    if {[winfo exists .report]} {
		destroy .report
	    }
	    toplevel .report -class Toplevel
	    set tags [bindtags .report]
	    lappend tags ReportWindow
	    bindtags .report $tags
	    set command {set ::report::windowSize(.report) }
	    append command {[string map {{+0+0} {}} [wm geometry .report]]}
	    bind ReportWindow <Configure> \
		[list after idle [list catch $command]]
	    initCommandHistory
	    if {![info exists windowSize(.report)]} then {
		set windowSize(.report) {600x550}
	    }
	    wm geometry .report $windowSize(.report)
	    # wm title .report "pfm - Reports and Queries : $dbName"

	    # Define menubar

	    menu .report.menubar -tearoff 0
	    .report.menubar add cascade \
		-menu .report.menubar.mnWindow \
		-label {Window} -underline 0
	    .report.menubar add cascade \
		-menu .report.menubar.mnSQL \
		-label {SQL} -underline 0
	    .report.menubar add cascade \
		-menu .report.menubar.mnOutput \
		-label {Output} -underline 0
	    .report.menubar add cascade \
		-menu .report.menubar.mnHelp \
		-label {Help} -underline 0

	    # Make it the menubar
	    .report configure -menu .report.menubar

	    # Window menu
	    menu .report.menubar.mnWindow -tearoff 0
	    .report.menubar.mnWindow add radiobutton \
		-label {SQL} -underline 0 -accelerator {Cntrl-s} \
		-variable ::report::reportMode \
		-value {sql} \
		-command ::report::cmdEnterSQLmode
	    .report.menubar.mnWindow add radiobutton \
		-label {Report} -underline 0 -accelerator {Cntrl-r} \
		-variable ::report::reportMode \
		-value {report} \
		-command ::report::cmdEnterReportMode
	    .report.menubar.mnWindow add command \
		-label {Close} -underline 0 -accelerator {Esc} \
		-command ::report::cmdQuitReport
	    bind .report <KeyPress-Escape> ::report::cmdQuitReport
	    bind .report <Control-KeyPress-s> {
		set ::report::reportMode {sql}
		::report::cmdEnterSQLmode
	    }
	    bind .report <Control-KeyPress-r> {
		set ::report::reportMode {report}
		::report::cmdEnterReportMode
	    }
	    
	    # SQL menu
	    menu .report.menubar.mnSQL -tearoff 0
	    .report.menubar.mnSQL add command \
		-label {Import SQL from file} -underline 0 \
		-command "::report::cmdImportFile {Open SQL-file}"
	    .report.menubar.mnSQL add command \
		-label {Save SQL to file} -underline 0 \
		-command {::report::cmdSaveFile .report.fmSQL.text \
			      "Save SQL as" "sql"}
	    .report.menubar.mnSQL add command \
		-label {List of Databases} -underline 8 \
		-command {::report::cmdShortCut {\l}}
	    .report.menubar.mnSQL add command \
		-label {List of Relations in database} -underline 8 \
		-command {::report::cmdShortCut {\d}}

	    # Output menu
	    menu .report.menubar.mnOutput -tearoff 0
	    .report.menubar.mnOutput add command \
		-label {Save} -underline 0 \
		-command {::report::cmdSaveFile .report.txtOutput \
			      "Save Output as" "text"}
	    .report.menubar.mnOutput add command \
		-label {Print} -underline 0 \
		-command {::report::cmdPrint .report.txtOutput}
	    .report.menubar.mnOutput add command \
		-label {Clear} -underline 0 \
		-command {.report.txtOutput delete 1.0 end}
	    .report.menubar.mnOutput add separator
	    .report.menubar.mnOutput add radiobutton \
		-label {Wrap} -underline 0 \
		-variable ::report::wrapOn -value 1 \
		-command {.report.txtOutput configure -wrap char}
	    .report.menubar.mnOutput add radiobutton \
		-label {Truncate} -underline 0 \
		-variable ::report::wrapOn -value 0 \
		-command {.report.txtOutput configure -wrap none}
	    .report.menubar.mnOutput add separator
	    .report.menubar.mnOutput add command \
		-label {Scroll Up} -underline 7 -accelerator {Alt-Up} \
		-command {.report.txtOutput yview scroll -1 pages}
	    .report.menubar.mnOutput add command \
		-label {Scroll Down} -underline 7 -accelerator {Alt-Down} \
		-command {.report.txtOutput yview scroll 1 pages}
	    .report.menubar.mnOutput add command \
		-label {Scroll Left} -underline 7 -accelerator {Alt-Left} \
		-command {.report.txtOutput xview scroll -1 pages}
	    .report.menubar.mnOutput add command \
		-label {Scroll Right} -underline 7 -accelerator {Alt-Right} \
		-command {.report.txtOutput xview scroll 1 pages}
	    bind .report <Alt-KeyPress-Up> \
		{.report.txtOutput yview scroll -1 pages}
	    bind .report <Alt-KeyPress-Down> \
		{.report.txtOutput yview scroll 1 pages}
	    bind .report <Alt-KeyPress-Left> \
		{.report.txtOutput xview scroll -1 pages}
	    bind .report <Alt-KeyPress-Right> \
		{.report.txtOutput xview scroll 1 pages}
	    
	    # Help menu
	    menu .report.menubar.mnHelp -tearoff 0
	    .report.menubar.mnHelp add command \
		-label {List of SQL-commands} -underline 0 \
		-command {::report::cmdShortCut {\h}}
	    .report.menubar.mnHelp add command \
		-label {psql \-commands} -underline 0 \
		-command {::report::cmdShortCut {\?}}
	    set helpText \
		"-- Type name of SQL-command on the next line after the \\h."
	    append helpText "\n\\h "
	    set command "::report::cmdClear\n"
	    append command ".report.fmSQL.text insert end \{${helpText}\}"
	    .report.menubar.mnHelp add command \
		-label {Help for a particular SQL command} -underline 0 \
		-command $command

	    # define panes

	    if {![info exists windowSize(.report.fm1)]} then {
		set windowSize(.report.fm1) 200
	    }
	    panedwindow .report.pw -orient vertical
	    pack .report.pw -side top -expand 1 -fill both
	    frame .report.fm1 -height $windowSize(.report.fm1)
	    frame .report.fm2
	    pack propagate .report.fm1 0
	    pack propagate .report.fm2 0
	    .report.pw add .report.fm1 .report.fm2
	    set command {set ::report::windowSize(.report.fm1) }
	    append command {[winfo height .report.fm1]}
	    bind .report.fm1 <Configure> \
		[list after idle [list catch $command]]

	    # define middle frame and buttons for SQL command

	    frame .report.fmMiddle -borderwidth 2 -relief groove
	    pack .report.fmMiddle -in .report.fm1 -side bottom \
		-fill x
	    button .report.fmMiddle.bnRun -text {Run} \
		-command {::report::cmdRun} \
		-underline 0 -takefocus 0 -pady 0
	    bind .report <Alt-KeyPress-r> \
		"::gen::PressButton .report.fmMiddle.bnRun"
	    button .report.fmMiddle.bnClear -text {Clear} -underline 0 \
 		-command ::report::cmdClear -takefocus 0 -pady 0
	    bind .report <Alt-KeyPress-c> \
		"::gen::PressButton .report.fmMiddle.bnClear"
	    button .report.fmMiddle.bnNext -text {Next} -underline 0 \
		-command ::report::cmdNext -takefocus 0 -pady 0
	    bind .report <Alt-KeyPress-n> \
		"::gen::PressButton .report.fmMiddle.bnNext"
	    bind .report <Control-KeyPress-Down> \
		"::gen::PressButton .report.fmMiddle.bnNext"
	    button .report.fmMiddle.bnPrev -text {Previous} -underline 0 \
		-command ::report::cmdPrev -takefocus 0 -pady 0
	    bind .report <Alt-KeyPress-p> \
		"::gen::PressButton .report.fmMiddle.bnPrev"
	    bind .report <Control-KeyPress-Up> \
		"::gen::PressButton .report.fmMiddle.bnPrev"
	    
	    pack .report.fmMiddle.bnRun -side right
	    pack .report.fmMiddle.bnNext -side right
	    pack .report.fmMiddle.bnPrev -side right
	    pack .report.fmMiddle.bnClear -side right

	    # define scrolled text widget for displaying Outputs

	    label .report.lblOutput -text {Output}
	    frame .report.fm2.fmtext
	    frame .report.fm2.fmhsb
	    text .report.txtOutput -wrap none -background white \
		-xscrollcommand {.report.hscrollOutput set} \
		-yscrollcommand {.report.vscrollOutput set} \
		-width 1 -height 1 -takefocus 0

	    # Tag errTag which will be used to highlight error
	    # messages from psql.

	    .report.txtOutput tag configure errTag -foreground red3

	    # Scrollbars
	    scrollbar .report.hscrollOutput -orient horizontal \
		-command {.report.txtOutput xview} -takefocus 0
	    scrollbar .report.vscrollOutput -orient vertical \
		-command {.report.txtOutput yview} -takefocus 0

	    # Pack text and vertical scrollbar

	    pack .report.txtOutput -in .report.fm2.fmtext \
		-side left -expand 1 -fill both
	    pack .report.vscrollOutput -in .report.fm2.fmtext \
		-side left -fill y

	    # Pack horizontal scrollbar

	    frame .report.fm2.fmhsb.filler -width 20 -height 20
	    pack .report.hscrollOutput -in .report.fm2.fmhsb \
		-side left -expand 1 -fill x
	    pack .report.fm2.fmhsb.filler -in .report.fm2.fmhsb \
		-side left

	    # Pack the output widgets

	    pack .report.lblOutput -in .report.fm2 -side top -fill x
	    pack .report.fm2.fmtext -in .report.fm2 -side top \
		-expand 1 -fill both
	    pack .report.fm2.fmhsb -in .report.fm2 -side top -fill x

	    # Initially enter mode according to $reportMode

	    switch $reportMode {
		sql {
		    cmdEnterSQLmode
		}
		report {
		    cmdEnterReportMode
		}
	    }
	} else {
	    tk_messageBox -message "There is no data base open!" -type ok \
		-icon error -parent .
	}
	return
    }

###########################################################################
# Common procedures, i.e. for both SQL and report mode                    #
###########################################################################

    proc cmdSaveFile {txtWidget title type} {
	
	set textToSave [$txtWidget get 1.0 end]
	switch $type {
	    "sql" {
		set fileTypes {
		    {{SQL statements} {.sql} }
		    {{All files} *}
		}
		set defaultExt ".sql"
	    }
	    "text" {
		set fileTypes {
		    {{Text files} {.txt} }
		    {{All files} *}
		}
		set defaultExt ".txt"
	    }
	}
	set fileName [tk_getSaveFile -title $title -filetypes $fileTypes \
			  -defaultextension $defaultExt -parent .report]
	if { $fileName !=  "" } then {
	    set file_ch [open $fileName w]
	    puts $file_ch $textToSave
	    close $file_ch
	}
	return
    }


    proc cmdImportFile {title} {
	variable ::pfm::pfmConfig
	
	set txtWidget ".report.fmSQL.text"
	set initialDir [file normalize ~]
	set fromEncoding [encoding system]
	set fileTypes {
	    {{SQL statements} {.sql} }
	    {{All files} *}
	}
	set defaultExt ".sql"
	set fileName [tk_getOpenFile -title $title -filetypes $fileTypes \
			  -defaultextension $defaultExt -parent .report \
			  -initialdir $initialDir]
	    
	if {![string equal $fileName {}]} then {
	    toplevel .report.import -class Toplevel
	    wm transient .report.import .report
	    wm title .report.import "Import SQL - Select encoding"
	    set shortFileName [file tail $fileName]
	    set message \
		"Select the character encoding with which ${shortFileName} was made.\n"
	    append message \
		"Your system's default encoding is '${fromEncoding}'\n"
	    append message \
		"That is usually OK, but if you know that ${shortFileName}\n"
	    append message \
		"was made with another encoding, you can select it now."
	    label .report.import.lbMessage -text $message
	    label .report.import.lbEncoding -text {Selected encoding:}
	    button .report.import.bnEncoding -text $fromEncoding \
		-command ::report::GetEncoding
	    set cmdContinue "::report::ProcessSQLscript \{${fileName}\} "
	    append cmdContinue "\[.report.import.bnEncoding cget -text\] $txtWidget"
	    button .report.import.bnContinue -text {Continue} -command $cmdContinue
	    grid .report.import.lbMessage -row 0 -column 0 -columnspan 3
	    grid .report.import.lbEncoding -row 1 -column 0 -columnspan 1 -sticky we
	    grid .report.import.bnEncoding -row 1 -column 1 -columnspan 1 -sticky we
	    grid .report.import.bnContinue -row 1 -column 2 -columnspan 1 -sticky we
	    focus .report.import.bnEncoding
	}
	return
    }

    proc GetEncoding {} {
	variable encodingList

	set boundWidget .report.import.bnEncoding
	set title "Select encoding"
	set encodingList [lsort [encoding names]]
	set currEncoding [.report.import.bnEncoding cget -text]
	set encoding [::gen::ListBox $boundWidget 300 300 $title \
			  ::report::encodingList ::report::encodingList \
			  $currEncoding]
	.report.import.bnEncoding configure -text $encoding
	return
    }

    proc ProcessSQLscript {fileName fromEncoding txtWidget} {

	destroy .report.import
	set title "Import SQL"
	set message "Offer file to psql using \\i (recommended for large files),"
	append message "\nor import directly in SQL window?"
	set buttonList {}
	lappend buttonList {Offer file to psql}
	lappend buttonList {Import file in SQL window}
	lappend buttonList {Cancel}
	switch -- \
	    [::gen::Dialog .report $title $message 400 0 $buttonList] {
		0 {
		    $txtWidget insert end "\\i '[ConvertToUTF-8 $fileName $fromEncoding]'"
		}
		1 {
		    set file_ch [open $fileName r]
		    fconfigure $file_ch -encoding $fromEncoding
		    $txtWidget insert end [read $file_ch]
		    close $file_ch
		}
		default {
		}
	    }
	return
    }

    proc ConvertToUTF-8 {fileName fromEncoding} {
	# This procedures converts $fileName from $fromEncoding to UTF-8.
	# It writes the converted file in pfmOptions(tmpdir) and returns
	# the name of the converted file.
	#
	# Even if the $fromEncoding = utf-8, we execute this conversion.
	# Tcl seems to be rather clever to recognise encodings such that
	# even if the user has specifed utf-8 when that is not correct,
	# Tcl converts it to utf-8.

	variable ::options::pfmOptions

	set outFileName {}
	if {![file exists $pfmOptions(tmpdir)]} then {
	    if {[catch {file mkdir $pfmOptions(tmpdir)} errMsg]} then {
		tk_messageBox -type ok -icon error -message $errMsg
	    }
	}
	if {[catch {open $fileName r} inFile]} then {
	    tk_messageBox -type ok -icon error -message $inFile
	} else {
	    fconfigure $inFile -encoding $fromEncoding
	    set tmpName "pfm_$pfmOptions(user)_[pid].sql"
	    set outFileName [file join $pfmOptions(tmpdir) $tmpName]
	    if {[catch {open $outFileName w} outFile]} then {
		tk_messageBox -type ok -icon error -message $outFile
		set outFileName {}
	    } else {
		# bug 1057 "-translation lf" added in version 1.5.2
		# Without this modification, tcl would use CR LF
		# as line ending on the Windows platform. psql would
		# interpret LF as line ending and it would consider CR
		# as an extra character.
		fconfigure $outFile -encoding utf-8 -translation lf
		while {![eof $inFile]} {
		    puts $outFile [gets $inFile]
		}
		close $inFile
		close $outFile
	    }
	}
	return $outFileName
    }

    proc cmdPrint {txtWidget} {
	variable ::options::pfmOptions
	variable printCommand
	variable parmlist

	set x [expr [winfo rootx .report] +20]
	set y [expr [winfo rooty .report] +20]
	toplevel .report.tpPrint -class Toplevel
	wm transient .report.tpPrint .report
	wm geometry .report.tpPrint +$x+$y
	wm title .report.tpPrint "pfm - Print Output"
	set printCommand $pfmOptions(printcmd)
	message .report.tpPrint.msgPrintcmd -width 500 -justify center -text \
	    "$printCommand\n\nLongest line is: [longestLine .report.txtOutput]"
	grid .report.tpPrint.msgPrintcmd -column 0 -row 0 -columnspan 3
	set n 1
	# Get the parameters for the printcommand
	set parmlist {}
	set startOfParm [string first "\$(" $printCommand 0]
	if { $startOfParm >= 0 } then {
	    set n 1
	    while { $startOfParm >= 0 } {
		set endOfParm [string first ")" $printCommand $startOfParm]
		if { $endOfParm >= 0 } then {
		    set parm [string range $printCommand $startOfParm $endOfParm]
		    set equalSign [string first "=" $parm 0]
		    if { $equalSign >= 0 } then {
			set defVal [string range $parm [expr $equalSign + 1] "end-1"]
			set parmName [string range $parm 2 [expr $equalSign - 1]]
		    } else {
			set defVal {}
			set parmName [string range $parm 2 "end-1"]
		    }
		    label .report.tpPrint.lb$n -text $parmName
		    grid .report.tpPrint.lb$n -column 0 -row $n
		    entry .report.tpPrint.en$n -width 40 -background white
		    .report.tpPrint.en$n insert end $defVal
		    grid .report.tpPrint.en$n -column 1 -columnspan 2 -row $n
		    lappend parmlist $parm
		    set startOfParm \
			[string first "\$(" $printCommand [expr $endOfParm + 1]]
		    incr n
		} else {
		    set startOfParm -1
		}
	    }
	    focus .report.tpPrint.en1
	}
	button .report.tpPrint.bnOK -text OK \
	    -command [list ::report::cmdPrintOK $txtWidget] \
	    -underline 0
	button .report.tpPrint.bnCancel -text Cancel \
	    -command {destroy .report.tpPrint} \
	    -underline 0
	grid .report.tpPrint.bnOK -column 1 -row $n -sticky we
	grid .report.tpPrint.bnCancel -column 2 -row $n -sticky we
	bind .report.tpPrint <Alt-KeyPress-o> \
	    "::gen::PressButton .report.tpPrint.bnOK"
	bind .report.tpPrint <Alt-KeyPress-c> \
	    "::gen::PressButton .report.tpPrint.bnCancel"
	bind .report.tpPrint <KeyPress-Escape> \
	    "::gen::PressButton .report.tpPrint.bnCancel"
	return
    }

    proc cmdPrintOK {txtWidget} {
	variable printCommand
	variable parmlist
	variable ::options::pfmOptions

	# This part was completely rewritten.
	# The lappend function ensures that the arguments of the openCommand
	# are properly delimited with { and } where necessary.
	# Also possibility to use temp file was added: %s in printCommand
	# represents the temporary file.

	if {[string first {%s} $printCommand] <= 0} then {
	    # input via stdin
	    set tempFile 0
	} else {
	    # input via temporary file
	    if {![file exists $pfmOptions(tmpdir)]} then {
		if {[catch {file mkdir $pfmOptions(tmpdir)} errMsg]} then {
		    tk_messageBox -type ok -icon error -message $errMsg
		}
	    }
	    set tempFile 1
	    set tmpName "pfm_$pfmOptions(user)_[pid].txt"
	    set fileName [file normalize [file join $pfmOptions(tmpdir) $tmpName]]
	    set mapTemp {%s}
	    lappend mapTemp $fileName
	}
	set execCommand {}
	foreach arg $printCommand {
	    set n 1
	    foreach parm $parmlist {
		set value [.report.tpPrint.en$n get]
		set map $parm
		lappend map $value
		set arg [string map $map $arg]
		incr n
	    }
	    if {$tempFile} then {
		set arg [string map $mapTemp $arg]
	    }
	    lappend execCommand $arg
	}
	destroy .report.tpPrint
	if {$tempFile} then {
	    set openCommand $fileName
	} else {
	    set openCommand [linsert $execCommand 0 {|}]
	}
	# puts $openCommand
	if { [catch {open $openCommand w} printch] } then {
	    tk_messageBox -type ok -icon error -message $printch
	} else {
	    # Next line added because of feature request 693.
	    fconfigure $printch -encoding $pfmOptions(printencoding)
	    puts $printch [$txtWidget get 1.0 end]
	    if { [catch {close $printch} errMsg] } then {
		tk_messageBox -message $errMsg -type ok -icon info
	    }
	    if {$tempFile} then {
		set execCommand [linsert $execCommand 0 {exec}]
		lappend execCommand {&}
		# puts $execCommand
		if {[catch $execCommand errMsg]} then {
		    tk_messageBox -type ok -icon error -message $errMsg
		}
	    }
	}
	return
    }

    proc longestLine {txtWidget} {

	set longest 0
	set lastIndex [$txtWidget index end]
	set index [$txtWidget index 1.0]
	while { $index < $lastIndex } {
	    set thisLineLength [string length [$txtWidget get $index "$index lineend"]]
	    if { $longest < $thisLineLength } then {
		set longest $thisLineLength
	    }
	    set index [$txtWidget index "$index +1 lines"]
	}
	return $longest
    }

    proc cmdRun {} {
	variable reportMode
	variable reportDef

	switch $reportMode {
	    sql {
		cmdRunSQL
	    }
	    report {
		set selectedReport [.report.fmReport.lsb curselection]
		if {[info exists reportDef($selectedReport,sqlselect)]} then {
		    cmdRunReport $selectedReport
		} else {
		    set message "First select a report!"
		    tk_messageBox -parent .report -icon warning -type ok \
			-message $message
		}
	    }
	}
	return
    }


    proc {cmdQuitReport} {} {
	variable commandHistory
	variable windowSize

	destroy .report
	array unset commandHistory
    }

###########################################################################
#  Procedures for queries (SQL mode)                                      #
###########################################################################

    proc cmdEnterSQLmode { } {
	variable ::pfm::dbName
	variable reportMode

	wm title .report "pfm - Run SQL : $dbName"
	destroy .report.fmReport
	destroy .report.fmSQL
	frame .report.fmSQL
	label .report.fmSQL.title -text {SQL statement}
	frame .report.fmSQL.fmtext
	frame .report.fmSQL.fmhsb
	text .report.fmSQL.text -wrap none \
	    -background white -width 1 -height 1 \
	    -xscrollcommand {.report.fmSQL.hscroll set} \
	    -yscrollcommand {.report.fmSQL.vscroll set}
	scrollbar .report.fmSQL.hscroll -orient horizontal \
	    -command {.report.fmSQL.text xview} -takefocus 0
	frame .report.fmSQL.fmhsb.filler -width 20 -height 20
	scrollbar .report.fmSQL.vscroll -orient vertical \
	    -command {.report.fmSQL.text yview} -takefocus 0

	# pack text and vsb in .report.fmSQL.fmtext
	pack .report.fmSQL.text -in .report.fmSQL.fmtext \
	    -side left -expand 1 -fill both
	pack .report.fmSQL.vscroll -in .report.fmSQL.fmtext \
	    -side left -fill y

	# pack hsb and filler in .report.fmSQL.fmhsb

	pack .report.fmSQL.hscroll -in .report.fmSQL.fmhsb \
	    -side left -expand 1 -fill x
	pack .report.fmSQL.fmhsb.filler -in .report.fmSQL.fmhsb \
	    -side left

	# pack the frames in .report.fmSQL

	pack .report.fmSQL.title -in .report.fmSQL -side top -fill x
	pack .report.fmSQL.fmtext -in .report.fmSQL -side top \
	    -expand 1 -fill both
	pack .report.fmSQL.fmhsb -in .report.fmSQL -side top -fill x

	# pack .report.fmSQL in .report.fm1 in which .report.fmMiddle
	# is already present

	pack .report.fmSQL -in .report.fm1 -side bottom \
	    -expand 1 -fill both
		    
	focus .report.fmSQL.text
	# enable clear button
	.report.fmMiddle.bnClear configure -state normal

	# the status of the Previous and Next buttons is controlled by
	# procedures handling the command history and by
	# cmdEnterReportMode.

	# enable SQL menu
	.report.menubar entryconfigure 1 -state normal
	# enable Help menu
	.report.menubar entryconfigure 3 -state normal
	cmdClear
	return
    }

    proc showResult { } {
	variable ::pfm::psqlChannel

	if { ![winfo exists .report.txtOutput] } then {
	    # If psql sends output when the .report window is not existing,
	    # we create the .report window.
	    ::report::cmdReportSQL {sql}
	}
	if { ![eof $psqlChannel] } then {
	    .report.txtOutput insert end "[gets $psqlChannel]\n"
	    .report.txtOutput see end
	} else {
	    # Bug 690: pfm hangs when psql exits.
	    # This branch is necessary to avoid endless loops when the user types '\q',
	    # or in the unlikely event that psql dies.
	    fileevent $psqlChannel readable {}
	    unset psqlChannel
	    puts [info exists psqlChannel]
	    .report.txtOutput insert end "Connection with psql is closed.\n"
	    .report.txtOutput see end
	}
	return
    }

    proc showError { } {
	variable ::pfm::errChannel

	if { ![winfo exists .report.txtOutput] } then {
	    # If psql sends output when the .report window is not existing,
	    # we create the .report window.
	    ::report::cmdReportSQL {sql}
	}
	if { ![eof $errChannel] } then {
	    set begin [.report.txtOutput index "end - 1 chars"]
	    .report.txtOutput insert end "[gets $errChannel]\n"
	    .report.txtOutput see end
	    set end [.report.txtOutput index "end - 1 chars"]
	    .report.txtOutput tag add errTag $begin $end
	} else {
	    fileevent $errChannel readable {}
	    unset errChannel
	    .report.txtOutput insert end "Error channel has died.\n"
	    .report.txtOutput see end
	}
	return
    }

    proc cmdClear {} {
	variable commandHistory

	.report.fmSQL.text delete 1.0 end
	set commandHistory($commandHistory(top)) {}
	# Added because of bug 691. If user presses [Clear], cursor must
	# be reset to top of command history.
	set commandHistory(cursor) $commandHistory(top)
	if { $commandHistory(top) > 0 } then {
	    .report.fmMiddle.bnPrev configure -state normal
	} else {
	    .report.fmMiddle.bnPrev configure -state disabled
	}
	.report.fmMiddle.bnNext configure -state disabled
	return
    }

    proc initCommandHistory {} {
	variable commandHistory

	set commandHistory(top) 0
	set commandHistory(cursor) 0
	set commandHistory(0) {}
	return
    }

    proc storeCommand {} {
	variable commandHistory

	# Store command in first free element of commandHistory array
	# Do not store \i commands
	set commandStart [.report.fmSQL.text get 1.0 1.2]
	if {$commandStart ne {\i}} then {
	    set commandHistory($commandHistory(top)) [.report.fmSQL.text get 1.0 "end -1 chars"]
	    # Increment top to first free element.
	    incr commandHistory(top)
	    set commandHistory($commandHistory(top)) {}
	    set commandHistory(cursor) $commandHistory(top)
	    .report.fmMiddle.bnPrev configure -state normal
	    .report.fmMiddle.bnNext configure -state disabled
	}
	return
    }

    proc cmdPrev {} {
	variable commandHistory

	if { $commandHistory(cursor) == $commandHistory(top) } then {
	    set commandHistory($commandHistory(top)) [.report.fmSQL.text get 1.0 "end -1 chars"]
	}
	# Reworked because of bug 691.
	
	if { $commandHistory(cursor) > 0 } then {
	    incr commandHistory(cursor) -1
	    .report.fmSQL.text delete 1.0 end
	    .report.fmSQL.text insert end $commandHistory($commandHistory(cursor))
	    .report.fmMiddle.bnNext configure -state normal
	    if { $commandHistory(cursor) == 0} then {
		.report.fmMiddle.bnPrev configure -state disabled
	    }
	} else {
	    bell
	}
	return
    }

    proc cmdNext {} {
	variable commandHistory

	# Reworked because of bug 691.
	if { $commandHistory(cursor) < $commandHistory(top) } then {
	    incr commandHistory(cursor)
	    .report.fmSQL.text delete 1.0 end
	    .report.fmSQL.text insert end $commandHistory($commandHistory(cursor))	    
	    if { $commandHistory(cursor) == $commandHistory(top) } then {
		.report.fmMiddle.bnNext configure -state disabled
	    }
	    .report.fmMiddle.bnPrev configure -state normal
	} else {
	    bell
	}
	return
    }

    proc cmdRunSQL {} {
	variable ::pfm::psqlChannel
	variable ::pfm::errChannel

	storeCommand
        set sqlCmd [.report.fmSQL.text get 1.0 end]
	if { [info exists psqlChannel] } then {
	    if { [catch {
		puts $psqlChannel $sqlCmd
		flush $psqlChannel
	    } errMsg] } then {
		tk_messageBox -message $errMsg -type ok -icon error
	    }
	} else {
	    puts $errChannel \
		"No connection with psql.\nTry to close and reopen the database.\n"
	    flush $errChannel
	}
	.report.fmSQL.text delete 1.0 end
	return
    }

    proc cmdShortCut {command} {

	.report.fmSQL.text insert end $command
	cmdRunSQL
	return
    }

###################################################################
#  Procedures for report mode                                     #
###################################################################

    proc cmdEnterReportMode { } {
	variable reportMode
	variable ::pfm::dbName

	wm title .report "pfm - Run report : $dbName"
	destroy .report.fmSQL
	destroy .report.fmReport
	.report.fmMiddle.bnClear configure -state disabled
	.report.fmMiddle.bnNext configure -state disabled
	.report.fmMiddle.bnPrev configure -state disabled

	# disable SQL menu
	.report.menubar entryconfigure 1 -state disabled

	# disable Help Menu
	.report.menubar entryconfigure 3 -state disabled
	cmdDisplayReportList
	return
    }


    proc cmdDisplayReportList {} {
	variable ::pfm::currentDB
	variable reportDef
	variable ::options::pfmOptions

	if { ![info exists currentDB] } then {
	    tk_messageBox -message "There is no data base open!" -type ok \
		-icon error
	} else {
	    set queryDef "SELECT * FROM pfm_report ORDER BY name"
	    set queryRes [pg_exec $currentDB $queryDef]
	    set lastTuple [expr [pg_result $queryRes -numTuples] - 1]
	    pg_result $queryRes -assign reportDef
	    pg_result $queryRes -clear
	    if { $lastTuple < 0} then {
		tk_messageBox -message "There are no reports defined!" \
		    -type ok -icon info -parent .report
	    } else {
		set maxNameWidth 0
		for {set tuple 0} {$tuple <= $lastTuple} {incr tuple } {
		    set nameLength [string length $reportDef($tuple,name)]
		    if { $nameLength > $maxNameWidth } then {
			set maxNameWidth $nameLength
		    }
		}
		set reportList {}
		for {set tuple 0} {$tuple <= $lastTuple} {incr tuple } {
		    set name [format "%-$maxNameWidth\s" $reportDef($tuple,name)]
		    lappend reportList \
			    "$name : $reportDef($tuple,description)"
		}
		destroy .report.fmReport
		frame .report.fmReport
		label .report.fmReport.title -text "List of reports"
		frame .report.fmReport.fmlist
		listbox .report.fmReport.lsb \
		    -background white \
		    -yscrollcommand {.report.fmReport.vscroll set} \
		    -width 1 -height 1
		scrollbar .report.fmReport.vscroll -orient vertical \
		    -command {.report.fmReport.lsb yview} -takefocus 0
		foreach report $reportList {
		    .report.fmReport.lsb insert end $report
		}
		.report.fmReport.lsb selection clear 0 end
		.report.fmReport.lsb selection set 0 0
		.report.fmReport.lsb activate 0
		bind .report.fmReport.lsb <KeyPress-Return> {
		    ::gen::PressButton .report.fmMiddle.bnRun
		}
		bind .report.fmReport.lsb <ButtonPress-1> {
		    focus .report.fmReport.lsb
		}

		# pack listbox and vsb

		pack .report.fmReport.lsb -in .report.fmReport.fmlist \
		    -side left -expand 1 -fill both
		pack .report.fmReport.vscroll \
		    -in .report.fmReport.fmlist \
		    -side left -fill y

		# pack label + listbox + vsb

		pack .report.fmReport.title -in .report.fmReport \
		    -side top -fill x
		pack .report.fmReport.fmlist -in .report.fmReport \
		    -side top -expand 1 -fill both

		# pack .report.fmReport in .rpeort.fm1 in which
		# .report.fmMiddle is already present
		pack .report.fmReport -in .report.fm1 -side bottom \
		    -expand 1 -fill both

		focus .report.fmReport.lsb
	    }
	}
	return
    }

    proc cmdRunReport {selectedReport} {
	variable reportDef
	variable ::pfm::currentDB
	variable parmlist

	# Get the parameters necessary to execute the query
	set sqlselect $reportDef($selectedReport,sqlselect)
	set startOfParm [string first "\$(" $sqlselect 0]
	if { $startOfParm >= 0 } then {
	    set parmlist {}
	    set x [expr [winfo rootx .report] + 20]
	    set y [expr [winfo rooty .report] + 20]
	    toplevel .report.getparm -class Toplevel
	    wm transient .report.getparm .report
	    wm geometry .report.getparm 600x400+$x+$y
	    wm title .report.getparm "pfm - Get report parameters"
	    panedwindow .report.getparm.pw -orient vertical
	    pack .report.getparm.pw -side top -expand 1 -fill both
	    frame .report.getparm.fmtext -height 150
	    frame .report.getparm.lower
	    pack propagate .report.getparm.fmtext 0
	    pack propagate .report.getparm.lower 0
	    .report.getparm.pw add .report.getparm.fmtext \
		.report.getparm.lower
	    text .report.getparm.txtselect -wrap word \
		-yscrollcommand {.report.getparm.vscroll set} \
		-background white -takefocus 0 -width 1 -height 1
	    scrollbar .report.getparm.vscroll -orient vertical \
		-command {.report.getparm.txtselect yview} \
		-takefocus 0
	    pack .report.getparm.txtselect -in .report.getparm.fmtext \
		-side left -expand 1 -fill both
	    pack .report.getparm.vscroll -in .report.getparm.fmtext \
		-side left -fill y

	    .report.getparm.txtselect insert end \
		$reportDef($selectedReport,sqlselect)
	    frame .report.getparm.frmparm -relief sunken -borderwidth 2
	    pack .report.getparm.frmparm -in .report.getparm.lower \
		-side top -expand 1 -fill both
	    set frmparm .report.getparm.frmparm
	    set n 0
	    while { $startOfParm >= 0 } {
		set endOfParm [string first ")" $sqlselect $startOfParm]
		if { $endOfParm >= 0 } then {
		    set parm [string range $sqlselect $startOfParm $endOfParm]
		    if { [lsearch -exact $parmlist $parm] == -1} then {
			# It is a new parameter. If the parameter is already
			# in parmlist, there is no need to prompt the user again.
			set labelText [string range $sqlselect \
			    [expr $startOfParm +2] [expr $endOfParm - 1]]
			label .report.getparm.frmparm.lb$n -text $labelText
			grid .report.getparm.frmparm.lb$n -column 0 -row $n
			entry .report.getparm.frmparm.en$n -width 40 \
			    -background white
			grid .report.getparm.frmparm.en$n -column 1 -row $n
			lappend parmlist $parm
			incr n
		    }
		    set startOfParm \
			[string first "\$(" $sqlselect [expr $endOfParm + 1]]
		} else {
		    set startOfParm -1
		}
	    }
	    focus .report.getparm.frmparm.en0
	    frame .report.getparm.fmbuttons
	    button .report.getparm.bnOK -text OK \
		-command [list ::report::completeSqlselect $selectedReport] \
		-underline 0
	    button .report.getparm.bnCancel -text Cancel \
		-command {destroy .report.getparm} \
		-underline 0
	    pack .report.getparm.bnCancel -in .report.getparm.fmbuttons \
		-side right
	    pack .report.getparm.bnOK -in .report.getparm.fmbuttons \
		-side right
	    pack .report.getparm.fmbuttons -in .report.getparm.lower \
		-side right -fill x
	    bind .report.getparm <Alt-KeyPress-o> \
		"::gen::PressButton .report.getparm.bnOK"
	    bind .report.getparm <Alt-KeyPress-c> \
		"::gen::PressButton .report.getparm.bnCancel"
	    bind .report.getparm <KeyPress-Escape> \
		"::gen::PressButton .report.getparm.bnCancel"
	} else {
	    executeQuery $selectedReport $sqlselect
	}
	return
    }

    proc completeSqlselect {selectedReport} {
	variable parmlist
	variable reportDef

	set sqlselect $reportDef($selectedReport,sqlselect)
	set n 0
	foreach parm $parmlist {
	    set value [.report.getparm.frmparm.en$n get]
	    set sqlselect [string map "$parm \"$value\"" $sqlselect]
	    incr n
	}
	destroy .report.getparm
	executeQuery $selectedReport $sqlselect
	return
    }

    proc executeQuery {selectedReport sqlselect} {
	variable reportDef
	variable ::pfm::currentDB
	variable ::pfm::errChannel
	variable reportData
	variable reportLastTuple
	variable sectionFields
	variable fieldLabel
	variable fieldAlignment
	variable sectionLayout
	variable lastLevel
	variable summaryList
	variable reportTree
	variable lastSeqnr
	variable summaryData
	variable MaxLabelWidth
	variable MaxColWidth
	variable MaxSummaryWidth

	# Execute the query for the report and store the result in
	# reportData Then fill reportTree and fill summaryData

	.report.txtOutput delete 1.0 end
	set report $reportDef($selectedReport,name)
	set queryRes [pg_exec $currentDB $sqlselect]
	set queryStatus [pg_result $queryRes -status]
	if { [string equal $queryStatus "PGRES_TUPLES_OK"] } then {
	    pg_result $queryRes -assign reportData
	    set reportLastTuple [expr [pg_result $queryRes -numTuples] - 1]
	    if {$reportLastTuple >= 0} then {
		if {[readSectionDef $report]} then {
		    wrapTooLongLines
		    fillReportTree 0 $reportLastTuple 1 0
		    calculateSummaries 1 0
		    printReport $selectedReport $sqlselect
		}
	    } else {
		printReportHeader $selectedReport $sqlselect
		printString \
		    "\nThe report's SQL statement did not return any data.\n"
	    }
	} else {
	    set errmsg "\nERROR: $sqlselect failed\n"
	    append errmsg "[pg_result $queryRes -error]\n"
	    puts $errChannel $errmsg
	    flush $errChannel
	}
	pg_result $queryRes -clear
	unset -nocomplain reportData
	unset -nocomplain reportLastTuple
	unset -nocomplain sectionFields
	unset -nocomplain fieldLabel
	unset -nocomplain fieldAlignment
	unset -nocomplain fieldMaxLength
	unset -nocomplain sectionLayout
	unset -nocomplain lastLevel
	unset -nocomplain summaryList
	unset -nocomplain reportTree
	unset -nocomplain lastSeqnr
	unset -nocomplain summaryData
	unset -nocomplain MaxLabelWidth
	unset -nocomplain MaxColWidth
	unset -nocomplain MaxSummaryWidth
	return
    }

    proc readSectionDef {report} {
	variable ::pfm::currentDB
	variable ::pfm::errChannel
	variable sectionFields
	variable sectionLayout
	variable fieldLabel
	variable fieldAlignment
	variable fieldMaxLength
	variable lastLevel
	variable summaryList
	variable reportData

	set lastLevel 1
	set queryDef "SELECT level, fieldlist, layout, summary\n"
	append queryDef "FROM pfm_section WHERE report = '$report'\n"
	append queryDef "ORDER BY level"
	set queryRes [pg_exec $currentDB $queryDef]
	set queryStatus [pg_result $queryRes -status]
	if {$queryStatus eq "PGRES_TUPLES_OK"} then {
	    pg_result $queryRes -assign sectionDef
	    set lastTuple [expr [pg_result $queryRes -numTuples] - 1]
	    if {$lastTuple >= 0} then {
		set level 0
		for {set tuple 0} {$tuple <= $lastTuple} {incr tuple} {
		    # We don't use the level numbers from pfm_section. We only
		    # rely on the numbers for pfm_section for properly
		    # ordering the sections. First level should be 1, next
		    # levels should be consecutive integers.
		    incr level
		    set sectionLayout($level) $sectionDef($tuple,layout)
		    # First check syntax of fieldlist
		    if {[catch {
		    foreach item $sectionDef($tuple,fieldlist) {
			set field [lindex $item 0]
			set label [lindex $item 1]
			set alignment [lindex $item 2]
			set maxLength [lindex $item 3]
		    }} tclerror]} then {
			set errMsg \
			    "\nERROR: fieldlist $sectionDef($tuple,fieldlist) in section $level could not be parsed.\n"
			append errMsg "${tclerror}\n"
			puts $errChannel $errMsg
			flush $errChannel
			set sectionDef($tuple,fieldlist) {}
		    }
		    set sectionFields($level) {}
		    foreach item $sectionDef($tuple,fieldlist) {
			set field [lindex $item 0]
			set label [lindex $item 1]
			set alignment [lindex $item 2]
			set maxLength [lindex $item 3]
			if {[info exists reportData(0,$field)]} then {
			    lappend sectionFields($level) $field
			    set fieldLabel($field) $label
			    set fieldAlignment($field) $alignment
			    set fieldMaxLength($field) $maxLength
			} else {
			    set errMsg \
				"\nERROR: The report's SQL SELECT statement does not return data"
			    append errMsg \
				"\nfor field '$field' defined in section '$level' of the report."
			    append errMsg \
				"\nCheck the report definition!\n"
			    puts $errChannel $errMsg
			    flush $errChannel
			}
		    }
		    set summaryList($level) $sectionDef($tuple,summary)
		}
		set lastLevel $level
		set success 1
	    } else {
		set errmsg "\nERROR: $queryDef did not return any data\n"
		puts $errChannel $errmsg
		flush $errChannel
		set success 0
	    }
	} else {
	    set errmsg "\nERROR: $queryDef failed\n"
	    append errmsg " [pg_result $queryRes -error]\n"
	    puts $errChannel $errmsg
	    flush $errChannel
	    set success 0
	}
	pg_result $queryRes -clear
	array unset sectionDef
	return $success
    }

    proc fillReportTree {firstTuple lastTuple level parent} {
	variable reportData
	variable lastLevel
	variable reportTree
	variable lastSeqnr

	# the range $firstTuple .. $lastTuple is the range of tuples
	# in reportData that have the same vaules for all the fields
	# of all the levels lower than $level.

	# $parent is the node in the reportTree that points to a tuple
	# in reportData that has the values for the fields of all the
	# levels lower than $level.

	# We now create a node at $level for all records in the range
	# that have different values for at least one of the fields of
	# this level and we call fillReportTree recursively for each
	# node created at this level to also fill the higher levels.

	# At $lastLevel, we create a node for all tuples in the range,
	# and we don't call fillReportTree anymore.

	set seqnr 1
	set reportTree($parent.$seqnr) $firstTuple
	set nextLevelFirstTuple $firstTuple
	if {$level < $lastLevel} then {
	    for {set tuple [expr $firstTuple + 1]} {$tuple <= $lastTuple} \
		{incr tuple} {
		    if {[differentData $level $tuple]} then {
			fillReportTree $nextLevelFirstTuple [expr $tuple - 1] \
			    [expr $level + 1] "$parent.$seqnr"
			incr seqnr
			set reportTree($parent.$seqnr) $tuple
			set nextLevelFirstTuple $tuple
		    }
		}
	    fillReportTree $nextLevelFirstTuple $lastTuple \
		[expr $level + 1] "$parent.$seqnr"

	} else {
	    for {set tuple [expr $firstTuple + 1]} {$tuple <= $lastTuple} \
		{incr tuple} {
		    incr seqnr
		    set reportTree($parent.$seqnr) $tuple
		}
	}
	set lastSeqnr($parent) $seqnr
	return
    }

    proc differentData {level tuple} {
	variable reportData
	variable sectionFields

	set different 0
	foreach field $sectionFields($level) {
	    if {$reportData([expr $tuple - 1],$field) ne \
		    $reportData($tuple,$field)} then {
		set different 1
		break
	    }
	}
	return $different
    }

    proc calculateSummaries {level parent} {
	variable reportData
	variable reportTree
	variable summaryList
	variable lastSeqnr
	variable lastLevel
	variable ::pfm::errChannel
	variable summaryData
	
	# Check summary definition
	if {[catch {
	    foreach item $summaryList($level) {
		set field [lindex $item 0]
		set aggregate [lindex $item 1]
		set format [lindex $item 2]
	    }
	} tclerror]} then {
	    set errMsg \
		"\nERROR: summary '$summaryList($level)' of section $level could not be parsed.\n"
	    append errMsg "${tclerror}\n"
	    puts $errChannel $errMsg
	    flush $errChannel
	    set summaryList($level) {}
	}
	# Now we really begin
	foreach item $summaryList($level) {
	    set field [lindex $item 0]
	    set aggregate [lindex $item 1]
	    set format [lindex $item 2]
	    set values {}
	    if {[accumulateValues $level $parent $field values]} then {
		if {[info commands ::aggregate::$aggregate] ne {}} then {
		    set summaryData($parent,$field,$aggregate) \
			[aggregate::$aggregate $values]
		} else {
		    set errMsg \
			"\nERROR: Summary field $aggregate\($field\) defined in section $level."
		    append errMsg \
			"\nAggregate function $aggregate is unknown.\n"
		    puts $errChannel $errMsg
		    flush $errChannel
		    set index [lsearch $summaryList($level) $item]
		    set summaryList($level) \
			[lreplace $summaryList($level) $index $index]

		}
	    } else {
		set errMsg \
		    "\nERROR: Summary field $aggregate\($field\) defined in section $level."
		append errMsg \
		    "\n$field does not appear in this or any higher numbered section.\n"
		puts $errChannel $errMsg
		flush $errChannel
		set index [lsearch $summaryList($level) $item]
		set summaryList($level) \
		    [lreplace $summaryList($level) $index $index]
	    }
	}
	if {$level < $lastLevel} then {
	    for {set seqnr 1} {$seqnr <= $lastSeqnr($parent)} \
		{incr seqnr} {
		    calculateSummaries [expr $level + 1] "$parent.$seqnr"
		}
	}
	return
    }

    proc accumulateValues {level parent field valueListName} {
	upvar 1 $valueListName values
	variable sectionFields
	variable reportTree
	variable reportData
	variable lastSeqnr
	variable lastLevel

	if {[lsearch $sectionFields($level) $field] >= 0} then {
	    for {set seqnr 1} {$seqnr <= $lastSeqnr($parent)} \
		{incr seqnr} {
		    set tuple $reportTree($parent.$seqnr)
		    lappend values $reportData($tuple,$field)
		}
	    set success 1
	} else {
	    if {$level < $lastLevel} then {
	    for {set seqnr 1} {$seqnr <= $lastSeqnr($parent)} \
		{incr seqnr} {
		    set success [accumulateValues [expr $level + 1] \
				     "$parent.$seqnr" $field values]
		}
	    } else {
		set success 0
	    }
	}
	return $success
    }

    proc printString {string} {

	.report.txtOutput insert end $string
	# .report.txtOutput see end
	return
    }

    proc printReport {report sqlselect} {

	labelAndColumnWidths
	printReportHeader $report $sqlselect
	printReportTree 1 0
    }

    proc labelAndColumnWidths {} {
	variable MaxLabelWidth
	variable MaxColWidth
	variable MaxSummaryWidth
	variable lastLevel
	variable sectionFields
	variable summaryList
	variable fieldLabel
	variable reportData
	variable reportLastTuple

	# MaxLabelWidth($level) is needed for a column layout.
	# It determines the larges label for each level
	for {set level 1} {$level <= $lastLevel} {incr level} {
	    set MaxLabelWidth($level) 0
	    foreach field $sectionFields($level) {
		set length [string length $fieldLabel($field)]
		if {$length > $MaxLabelWidth($level)} then {
		    set MaxLabelWidth($level) $length
		}
	    }
	}
	# MaxDataWith($field) is needed for a table layout
	# It takes into account that there may be line breaks in the data
	for {set level 1} {$level <= $lastLevel} {incr level} {
	    foreach field $sectionFields($level) {
		set MaxColWidth($field) [string length $fieldLabel($field)]
		for {set tuple 0} {$tuple <= $reportLastTuple} {incr tuple} {
		    set data $reportData($tuple,$field)
		    set moreLines 1
		    while {$moreLines} {
			set moreLines 0
			set newLineIndex [string first "\n" $data]
			if {$newLineIndex >= 0} then {
			    set moreLines 1
			    set oneLine [string range $data 0 [expr $newLineIndex - 1]]
			    set data [string range $data [expr $newLineIndex + 1] end]
			} else {
			    set oneLine $data
			    set data {}
			}
			set length [string length $oneLine]
			if {$length > $MaxColWidth($field)} then {
			    set MaxColWidth($field) $length
			}
		    }
		}
	    }
	}
	# MaxSummaryWidth($level) is needed for properly formatting
	# the summaries
	for {set level 1} {$level <= $lastLevel} {incr level} {
	    set MaxSummaryWidth($level) 0
	    foreach item $summaryList($level) {
		set field [lindex $item 0]
		set aggregate [lindex $item 1]
		set length [string length "$aggregate\($field\)"]
		if {$length > $MaxSummaryWidth($level)} then {
		    set MaxSummaryWidth($level) $length
		}
	    }
	}
	return
    }

    proc wrapTooLongLines {} {
	variable reportData
	variable reportLastTuple
	variable fieldMaxLength
	variable lastLevel
	variable sectionFields
	
	for {set level 1} {$level <= $lastLevel} {incr level} {
	    foreach field $sectionFields($level) {
		if {$fieldMaxLength($field) ne {}} then {
		    for {set tuple 0} {$tuple <= $reportLastTuple} {incr tuple} {
			set data $reportData($tuple,$field)
			set newData {}
			set moreLines 1
			while {$moreLines} {
			    set moreLines 0
			    set newLineIndex [string first "\n" $data]
			    if {$newLineIndex >= 0} then {
				set moreLines 1
				set oneLine [string range $data 0 \
						 [expr $newLineIndex - 1]]
				set data [string range $data \
					      [expr $newLineIndex + 1] end]
				wrapLine oneLine $fieldMaxLength($field)
				append newData "${oneLine}\n"
			    } else {
				set oneLine $data
				wrapLine oneLine $fieldMaxLength($field)
				append newData $oneLine
			    }
			}
			set reportData($tuple,$field) $newData
		    }
		}
	    }
	}
	return
    }

    proc wrapLine {lineVar maxLength} {
	upvar 1 $lineVar line

	if {[string length $line] > $maxLength} then {
	    # We try to find a space between 30% of maxLength and maxLength
	    set breakTo [expr $maxLength - 1]
	    set breakFrom [expr int(0.3 * $maxLength)]
	    set wrappedLine {}
	    while {[string length $line] > $maxLength} {
		set oneLine [string range $line 0 $breakTo]
		set breakAt [string last { } $oneLine]
		if {$breakAt >= $breakFrom} then {
		    # If possible we replace a space with a line break
		    # So, we don't copy the space
		    set oneLine [string range $oneLine 0 [expr $breakAt - 1]]
		    set line [string range $line [expr $breakAt + 1] end]
		} else {
		    # If not possible we just insert a line break
		    set line [string range $line [expr $breakTo + 1] end]
		}
		append wrappedLine "${oneLine}\n"
	    }
	    append wrappedLine $line
	    set line $wrappedLine
	}
	return
    }

    proc printReportHeader {report sqlselect} {
	variable reportDef
	
	printString "$reportDef($report,name)\n"
	printString \
	    "[string repeat - [string length $reportDef($report,name)]]\n\n"
	printString \
	    "Description: $reportDef($report,description)\n"
	set formattedSQL [string map {\n "\n             "} $sqlselect]
	printString "SQL        : $formattedSQL\n"
	printString \
	    "Date       : [clock format [clock seconds] -format {%d-%b-%Y}]\n"
	return
    }

    proc printReportTree {level parent} {
	variable lastLevel
	variable lastSeqnr

	set firstChild 1
	for {set seqnr 1} {$seqnr <= $lastSeqnr($parent)} {incr seqnr} {
	    printNode $level $parent.$seqnr $firstChild
	    set firstChild 0
	    if {$level < $lastLevel} then {
		printReportTree [expr $level + 1] $parent.$seqnr
	    }
	}
	printSummary $level $parent
	return
    }

    proc printNode {level node firstRecord} {
	variable reportData
	variable reportTree
	variable sectionLayout
	

	set tuple $reportTree($node)
	set margin [string repeat " " [expr 6 * $level]]
	set sectionNr "[string range $node 2 end].  "
	set nrLength [string length $sectionNr]
	set beforeSectionNr [string repeat " " [expr (6 * $level) - $nrLength]]
	set sectionNr "${beforeSectionNr}${sectionNr}"
	switch -- $sectionLayout($level) {
	    "row" {
		printRow $level $tuple $firstRecord $margin $sectionNr
	    }
	    "table" {
		printTable $level $tuple $firstRecord $margin $sectionNr
	    }
	    default {
		printColumn $level $tuple $firstRecord $margin $sectionNr
	    }
	}
	return
    }

    proc printSummary {level parent} {
	variable summaryData
	variable summaryList
	variable MaxSummaryWidth
	variable ::pfm::errChannel


	set margin [string repeat " " [expr 6 * ($level - 1)]]
	set summaryText "Summary [string range $parent 2 end] : "
	set summaryTextLength [string length $summaryText]
	set firstLine 1
	foreach item $summaryList($level) {
	    if {$firstLine} then {
		set output "\n${margin}${summaryText}"
	    } else {
		set output \
		    "${margin}[string repeat { } $summaryTextLength]"
	    }
	    set firstLine 0
	    set field [lindex $item 0]
	    set aggregate [lindex $item 1]
	    set format [lindex $item 2]
	    set label "${aggregate}\($field\)"
	    set formatLabel "%-$MaxSummaryWidth($level)\s"
	    append output "[format $formatLabel $label] = "
	    set value $summaryData($parent,$field,$aggregate)
	    if {$format ne {}} then {
		if {[catch {format $format $value} formattedValue]} then {
		    set errMsg "\nERROR: Format string '$format' in summary field $aggregate\($field\) in section $level\n"
		    append errMsg "$formattedValue\n"
		    puts $errChannel $errMsg
		    flush $errChannel
		    set formattedValue $value
		}
		append output "$formattedValue\n"
	    } else {
		append output "${value}\n"
	    }
	    printString $output
	}
	return
    }

    proc printRow {level tuple firstRecord margin sectionNr} {
	variable reportData
	variable sectionFields
	variable fieldLabel
	variable lastLevel
	
	if {$firstRecord || ($level != $lastLevel)} then {
	    set output "\n"
	} else {
	    set output {}
	}
	if {$level != $lastLevel} then {
	    append output $sectionNr
	} else {
	    append output $margin
	}
	foreach field $sectionFields($level) {
	    append output "$fieldLabel($field): $reportData($tuple,$field); "
	}
	append output "\n"
	printString $output
	return
    }

    proc printColumn {level tuple firstRecord margin sectionNr} {
	variable reportData
	variable sectionFields
	variable fieldLabel
	variable MaxLabelWidth
	variable lastLevel

	set output "\n"
	set firstLine 1
	foreach field $sectionFields($level) {
	    set formatString "%-$MaxLabelWidth($level)\s"
	    if {$firstLine} then {
		set label $sectionNr
	    } else {
		set label $margin
	    }
	    set firstLine 0
	    append label "[format $formatString $fieldLabel($field)] : "
	    append output $label
	    set nextLineOffset [string repeat { } [string length $label]]
	    set startIdx 0
	    set nlIdx [string first "\n" $reportData($tuple,$field) $startIdx]
	    while { $nlIdx >= 0 } {
		append output \
		    [string range $reportData($tuple,$field) $startIdx $nlIdx]
		append output $nextLineOffset
		set startIdx [expr $nlIdx + 1]
		set nlIdx [string first "\n" $reportData($tuple,$field) $startIdx]
	    }
	    append output \
		"[string range $reportData($tuple,$field) $startIdx end]\n"
	}
	printString $output
	return
    }

    proc printTable {level tuple firstRecord margin sectionNr} {
	variable reportData
	variable sectionFields
	variable fieldLabel
	variable fieldAlignment
	variable MaxColWidth
	variable lastLevel

	if {$firstRecord || ($level != $lastLevel)} then {
	    printTableHeader $level $margin $sectionNr
	}
	foreach field $sectionFields($level) {
	    switch -- $fieldAlignment($field) {
		"r" {
		    set formatString($field) "%$MaxColWidth($field)\s"
		}
		"l" -
		default {
		    set formatString($field) "%-$MaxColWidth($field)\s"
		}
	    }
	    set data($field) $reportData($tuple,$field)
	}
	set moreLines 1
	while {$moreLines} {
	    set output {}
	    set moreLines 0
	    foreach field $sectionFields($level) {
		set newLineIdx [string first "\n" $data($field)]
		if {$newLineIdx >= 0} then {
		    set oneLine \
			[string range $data($field) 0 [expr $newLineIdx - 1]]
		    set data($field) \
			[string range $data($field) [expr $newLineIdx + 1] end]
		    set moreLines 1
		} else {
		    set oneLine $data($field)
		    set data($field) {}
		}
		append output \
		    "| [format $formatString($field) $oneLine] "
	    }
	    set output "${margin}[string range $output 1 end]\n"
	    printString $output
	}
	return
    }

    proc printTableHeader {level margin sectionNr} {
	variable sectionFields
	variable fieldLabel
	variable fieldAlignment
	variable MaxColWidth
	variable MaxLabelWidth
	variable lastLevel

	set header {}
	set underline {}
	foreach field $sectionFields($level) {
	    switch $fieldAlignment($field) {
		r {
		    set formatString "%$MaxColWidth($field)\s"
		}
		l -
		default {
		    set formatString "%-$MaxColWidth($field)\s"
		}
	    }
	    append header \
		"| [format $formatString $fieldLabel($field)] "
	    append underline \
		"+-[string repeat - $MaxColWidth($field)]-"
	}
	set header [string range $header 1 end]
	set underline [string range $underline 1 end]
	set output "\n${margin}${header}\n"
	set nrLength [string length $sectionNr]
	if {$level != $lastLevel} then {
	    append output "${sectionNr}${underline}\n"
	} else {
	    append output "${margin}${underline}\n"
	}
	printString $output
	return
    }


}

########################################################################
# End of namespace report                                              #
########################################################################

########################################################################
# Begin of namespace aggregate                                         #
########################################################################

namespace eval aggregate {

    proc SUM {List} {

	set sum 0
	foreach item $List {
	    if {[string is double -strict $item]} then {
		set sum [expr $sum + $item]
	    }
	}
	return $sum
    }

    proc COUNT {List} {

	return [llength $List]
    }

    proc AVG {List} {

	set sum 0.0
	set count 0
	foreach item $List {
	    if {[string is double -strict $item]} then {
		set sum [expr $sum + $item]
		incr count
	    }
	}
	if {$count != 0} then {
	    set avg [expr double($sum) / double($count)]
	} else {
	    set avg 0.0
	}
	return $avg
    }

    proc STDDEV {List} {

	set sum 0.0
	set count 0
	foreach item $List {
	    if {[string is double -strict $item]} then {
		set sum [expr $sum + $item]
		incr count
	    }
	}
	if {$count != 0} then {
	    set avg [expr double($sum) / double($count)]
	} else {
	    set avg 0.0
	}
	set squareDev 0.0
	foreach item $List {
	    if {[string is double -strict $item]} then {
		set dev [expr double($item) - double($avg)]
		set squareDev [expr $squareDev + $dev * $dev]
	    }
	}
	if {$count != 0} then {
	    set stddev [expr sqrt(double($squareDev) / double($count))]
	} else {
	    set stddev 0.0
	}
	return $stddev
    }

    proc MIN {List} {

	set numeric 1
	set min {}
	foreach item $List {
	    if {$item ne {}} then {
		set min $item
		if {![string is double -strict $item]} then {
		    set numeric 0
		}
	    }
	}
	if {$numeric} then {
	    foreach item $List {
		if {$item ne {}} then {
		    if {$item < $min} then {
			set min $item
		    }
		}
	    }
	} else {
	    foreach item $List {
		if {[string compare -nocase $min $item] == 1} then {
		    set min $item
		}
	    }
	}
	return $min
    }

    proc MAX {List} {

	set numeric 1
	set max {}
	foreach item $List {
	    if {$item ne {}} then {
		set max $item
		if {![string is double -strict $item]} then {
		    set numeric 0
		}
	    }
	}
	if {$numeric} then {
	    foreach item $List {
		if {$item ne {}} then {
		    if {$item > $max} then {
			set max $item
		    }
		}
	    }
	} else {
	    foreach item $List {
		if {[string compare -nocase $max $item] == -1} then {
		    set max $item
		}
	    }
	}
	return $max
    }

# test aggregate functions

#     set lists {
# 	{5 6 3 {} 8 11}
# 	{}
# 	{{}}
# 	{23.5 een twee 13.0 7}
# 	{-2 3.45 6.25 9}
# 	{aarde Aarde aardappel {} appel grondpeer peer}
#     }
#     foreach list $lists {
# 	puts "list = $list"
# 	puts "COUNT = [COUNT $list]"
# 	puts "SUM = [SUM $list]"
# 	puts "AVG = [AVG $list]"
# 	puts "STDDEV = [STDDEV $list]"
# 	puts "MIN = [MIN $list]"
# 	puts "MAX = [MAX $list]"
# 	puts "------------------------"
#     }


}

########################################################################
# End of namespace aggregate                                           #
########################################################################


###################################################################
# Intialisation of the application programmers  interface (API)   #
###################################################################

if { [catch {source [file join $::pfm::installDir pgin.tcl]} errMsg1 ] } then {
    if { [catch {package require Pgtcl} PgtclVersion] } then {
	set errMsg "$errMsg1\n$PgtclVersion."
	set errMsg "$errMsg\nNeither Pgtcl, nor pgintcl were found."
	set errMsg "$errMsg\npfm cannot connect to postgreSQL."
	tk_messageBox -message $errMsg -type ok -icon error
	set ::pfm::API "pfm cannot communicate with postgreSQL. "
	append ::pfm::API "Neither Pgtcl nor pgintcl are present."
    } else {
	set ::pfm::API "pfm is using Pgtcl $PgtclVersion "
	append ::pfm::API "to communicate with postgreSQL."
    }
} else {
    if { [catch { set pgintclVersion $::pgtcl::version } errMsg] } then {
	set pgintclVersion "???"
    }
    set ::pfm::API "pfm is using pgintcl $pgintclVersion "
    append ::pfm::API "to communicate with postgreSQL."
}



# Define arrow images

image create bitmap ::img::down \
    -file [file join $::pfm::installDir arrow_down.xbm] \
    -maskfile [file join $::pfm::installDir arrow_down_mask.xbm]

package provide pfm $::pfm::pfmVersion
