# Author:    Lisandro Dalcin
# Contact:   dalcinl@users.sourceforge.net
# Copyright: This module has been placed in the public domain.
# Id: $Id$

"""
PETSc Error
===========

Exception class for PETSc errors in the C side.
"""

# --------------------------------------------------------------------

__date__     = '$Date$'
__version__  = '$Version$'
__revision__ = '$Revision$'

__docformat__ = 'reStructuredText'

# --------------------------------------------------------------------

__all__ = ['Error']

# --------------------------------------------------------------------

import sys as _sys
from petsc4py.lib import _petsc

# --------------------------------------------------------------------

class Error(RuntimeError):

    """
    PETSc Error.
    """

    _traceback_ = []
    
    ## def __init__(self, ierr=0, msg=None):
    ##     txt, _ = PetscErrorMessage(ierr)
    ##     RuntimeError.__init__(self, ierr, txt, msg)

    @classmethod
    def getTraceBack(cls):
        """
        Gets a list of strings with traceback information
        about PETSc runtime errors originated in the C side.
        """
        return list(cls._traceback_)

    @classmethod
    def view(cls):
        """
        Displays (in sys.stderr) traceback information about PETSc
        runtime errors originated in the C side.
        """
        comm_self = _petsc.PetscGetCommSelf()
        comm_world = _petsc.PetscGetCommWorld()
        wsize = _petsc.PetscCommGetSize(comm_world)
        wrank = _petsc.PetscCommGetRank(comm_world)
        try:
            fd = _sys.stderr
        except AttributeError:
            fd = None
        if not isinstance(fd, file):
            fd = _sys.__stderr__
        for error in cls._traceback_:
            message = '[%*d] %s\n' % (len(str(wsize-1)), wrank, error)
            _petsc.PetscFPrintf(comm_self, fd, message)

    if _sys.version_info[:2] >= (2, 5):
        @property
        def ierr(self):
            "error code"
            return self.args[0]
        @property
        def message(self):
            "error message"
            return self.args[1:]
        
# --------------------------------------------------------------------

try:
    getattr(_petsc, 'ErrorTypeRegister')(Error)
    delattr(_petsc, 'ErrorTypeRegister')
except AttributeError:
    pass

# --------------------------------------------------------------------
