/*
 * Permafrost - Physical modelling framework
 *
 * Copyright (C) 2009, 2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>

#include <math.h>

#include "src/types.h"
#include "src/util.h"
#include "src/list.h"
#include "src/expr.h"
#include "src/schedule.h"

struct find_input_context
  {
	struct component	*comp;
	struct port		*port;
	struct conn_elem	*ret;
  };

static void
find_input(void *data, void *context)
{
	struct connection *conn;
	struct find_input_context *ctx;

	conn = (struct connection *)data;
	ctx = (struct find_input_context *)context;

	if (ctx->ret != NULL)
		return;

	if ((conn->output.c.p.component == ctx->comp)
	    && (conn->output.c.p.port == ctx->port))
	  {
		ctx->ret = &conn->input;
		return;
	  }

	if ((conn->output.type == conn_elem_type_port)
	    && ((conn->output.c.p.port->type == port_type_w)
	        || (conn->output.c.p.port->type == port_type_k)))
		if ((conn->input.c.p.component == ctx->comp)
		    && (conn->input.c.p.port == ctx->port))
		  {
			ctx->ret = &conn->output;
			return;
		  }
}

struct comp_port
  {
	struct component	*comp;
	struct port		*port;
  };

static int
scheduled_expr_cmp_comp_port(void *e, void *p)
{
	struct scheduled_expr *expr;
	struct comp_port *cp;

	expr = (struct scheduled_expr *)e;
	cp = (struct comp_port *)p;

	return !((expr->component == cp->comp) && (expr->port == cp->port));
}

static int
stmt_defines_port(void *s, void *p)
{
	struct stmt *stmt;
	struct port *port;

	stmt = (struct stmt *)s;
	port = (struct port *)p;

	return stmt->out.port != p;
}

struct port_add_context
  {
	struct scheduled_system *system;
	struct component	*comp;
	struct scheduled_expr	*sexpr;
  };

static struct scheduled_expr *
port_add_expr(struct scheduled_system *system, struct component *comp,
	      struct port *port);

static void
port_add(void *data, void *context)
{
	struct port *port;
	struct port_add_context *ctx;
	struct find_input_context f_ctx;
	struct scheduled_expr *s;

	port = (struct port *)data;
	ctx = (struct port_add_context *)context;

	f_ctx.comp = ctx->comp;
	f_ctx.port = port;
	f_ctx.ret = NULL;
	list_for_each(ctx->system->system->connections, find_input, &f_ctx);

	if (f_ctx.ret == NULL)
		expr_port_to_value(ctx->sexpr->expr, port, 0.0);
	else if (f_ctx.ret->type == conn_elem_type_value)
		expr_port_to_value(ctx->sexpr->expr, port, f_ctx.ret->c.value);
	else
	  {
		s = port_add_expr(ctx->system, f_ctx.ret->c.p.component,
				  f_ctx.ret->c.p.port);
		expr_port_to_ref(ctx->sexpr->expr, port, s);
		list_append(s->is_refd, ctx->sexpr);
		list_merge(s->delays, expr_get_delays(ctx->sexpr->expr, s));
	  }
}

static void
set_is_df_refd(void *data, void *context)
{
	struct scheduled_expr *sexpr;

	sexpr = (struct scheduled_expr *)data;

	sexpr->is_df_refd = 1;
}

static struct scheduled_expr *
port_add_expr(struct scheduled_system *system, struct component *comp,
	      struct port *port)
{
	struct scheduled_expr *sexpr, *p;
	struct find_input_context ctx;
	struct port_add_context p_ctx;
	struct comp_port c;
	struct stmt *s;
	list_t l;

	c.comp = comp;
	c.port = port;
	p = list_find(system->exprs, scheduled_expr_cmp_comp_port, &c);
	if (p != NULL)
		return p;

	sexpr = xmalloc(sizeof(struct scheduled_expr));
	sexpr->component = comp;
	sexpr->port = port;
	sexpr->is_output = 0;
	sexpr->is_refd = list_new();
	sexpr->is_df_refd = 0;
	sexpr->delays = list_new();

	if (comp == NULL)
	  {
		sexpr->refs = list_new();
		sexpr->df_refs = list_new();

		if (port->type == port_type_output)
		  {
			sexpr->is_output = 1;

			ctx.comp = comp;
			ctx.port = port;
			ctx.ret = NULL;
			list_for_each(system->system->connections, find_input,
				      &ctx);

			if (ctx.ret == NULL)
				sexpr->expr = expr_new_value(0.0);
			else if (ctx.ret->type == conn_elem_type_value)
				sexpr->expr = expr_new_value(ctx.ret->c.value);
			else
			  {
				p = port_add_expr(system,
						  ctx.ret->c.p.component,
						  ctx.ret->c.p.port);

				sexpr->expr =
					expr_new_signal_ref(p, NULL, NULL);
				list_append(sexpr->refs, p);
				list_append(sexpr->df_refs, p);
				list_append(p->is_refd, sexpr);
				p->is_df_refd = 1;
			  }
		  }

		list_append(system->exprs, sexpr);
	  }
	else
	  {
		s = list_find(((struct block *)comp->type.t.p)->stmts,
			      stmt_defines_port, port);
		sexpr->expr = expr_copy(s->expr);
		list_append(system->exprs, sexpr);
		l = expr_get_inputs(sexpr->expr);
		p_ctx.system = system;
		p_ctx.comp = comp;
		p_ctx.sexpr = sexpr;
		list_for_each(l, port_add, &p_ctx);
		list_free(l);
		sexpr->refs = expr_get_refs(sexpr->expr);
		sexpr->df_refs = expr_get_df_refs(sexpr->expr);
		list_for_each(sexpr->df_refs, set_is_df_refd, NULL);
	  }

	return sexpr;
}

static void
port_add_io(void *data, void *context)
{
	struct port *p;
	struct scheduled_system *s;

	p = (struct port *)data;
	s = (struct scheduled_system *)context;

	port_add_expr(s, NULL, p);
}

static int
ptr_cmp(void *d1, void *d2)
{
	return (char *)d1 - (char *)d2;
}

static void
print_port_ids(void *data, void *context)
{
	struct scheduled_expr *sexpr;

	sexpr = (struct scheduled_expr *)data;

	fprintf(stderr, " %s%s%s",
		(sexpr->component != NULL) ? sexpr->component->id : "",
		(sexpr->component != NULL) ? "." : "", sexpr->port->id);
}

struct check_delay_free_loop_context
  {
	struct scheduled_system	*ss;
	list_t			 refs;
  };

static void
check_delay_free_loop(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct check_delay_free_loop_context *ctx;

	sexpr = (struct scheduled_expr *)data;
	ctx = (struct check_delay_free_loop_context *)context;

	if (list_find(ctx->refs, ptr_cmp, sexpr))
	  {
		fprintf(stderr, "error: system `%s': delay free loop "
			"detected, involved ports:", ctx->ss->system->id);
		list_for_each(ctx->refs, print_port_ids, NULL);
		fprintf(stderr, "\n");
		exit(EXIT_FAILURE);
	  }

	list_push(ctx->refs, sexpr);
	list_for_each(sexpr->df_refs, check_delay_free_loop, ctx);
	list_pop(ctx->refs);
}

static void
check_delay_free_loops(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct scheduled_system *ss;
	struct check_delay_free_loop_context ctx;
	list_t refs;

	sexpr = (struct scheduled_expr *)data;
	ss = (struct scheduled_system *)context;

	refs = list_new();
	list_push(refs, sexpr);

	ctx.ss = ss;
	ctx.refs = refs;
	list_for_each(sexpr->df_refs, check_delay_free_loop, &ctx);

	list_free(refs);
}

struct schedule_add_context
  {
	struct scheduled_system	*ss;
	list_t			 visited;
	list_t			 visited_d;
  };

static void
schedule_add_df(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct schedule_add_context *ctx;

	sexpr = (struct scheduled_expr *)data;
	ctx = (struct schedule_add_context *)context;

	if (list_find(ctx->visited, ptr_cmp, sexpr) != NULL)
		return;

	list_append(ctx->visited, sexpr);
	list_for_each(sexpr->df_refs, schedule_add_df, ctx);
	list_append(ctx->ss->schedule, sexpr);
}

static void
schedule_add_output_df(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct scheduled_system *ss;
	struct schedule_add_context ctx;

	sexpr = (struct scheduled_expr *)data;
	ss = (struct scheduled_system *)context;

	if (sexpr->component != NULL)
		return;

	if (!sexpr->is_output)
		return;

	ctx.ss = ss;
	ctx.visited = list_copy(ss->schedule);
	schedule_add_df(sexpr, &ctx);
	list_free(ctx.visited);
}

static void
schedule_add(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct schedule_add_context *ctx;

	sexpr = (struct scheduled_expr *)data;
	ctx = (struct schedule_add_context *)context;

	if (list_find(ctx->visited_d, ptr_cmp, sexpr) != NULL)
		return;

	list_append(ctx->visited_d, sexpr);
	list_for_each(sexpr->refs, schedule_add_df, ctx);
	list_for_each(sexpr->refs, schedule_add, ctx);
	list_append(ctx->ss->schedule_d, sexpr);
}

static void
schedule_add_output(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct scheduled_system *ss;
	struct schedule_add_context ctx;

	sexpr = (struct scheduled_expr *)data;
	ss = (struct scheduled_system *)context;

	if (sexpr->component != NULL)
		return;

	if (!sexpr->is_output)
		return;

	ctx.ss = ss;
	ctx.visited = list_copy(ss->schedule);
	ctx.visited_d = list_copy(ss->schedule_d);
	schedule_add(sexpr, &ctx);
	list_free(ctx.visited);
	list_free(ctx.visited_d);
}

static int
sexpr_cmp_port(void *s, void *p)
{
	return ((struct scheduled_expr *)s)->port - (struct port *)p;
}

struct assign_io_indexes_context
{
	list_t	exprs;
	size_t	i;
};

static void
assign_io_indexes(void *data, void *context)
{
	struct port *port;
	struct assign_io_indexes_context *ctx;
	struct scheduled_expr *sexpr;

	port = (struct port *)data;
	ctx = (struct assign_io_indexes_context *)context;

	sexpr = list_find(ctx->exprs, sexpr_cmp_port, port);

	sexpr->index = ctx->i;
	ctx->i += 1;
}

static void
assign_indexes(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	size_t *cur;

	sexpr = (struct scheduled_expr *)data;
	cur = (size_t *)context;

	if (sexpr->component == NULL)
		return;

	sexpr->index = *cur;
	*cur = *cur + 1;
}

struct check_delay_context
  {
	struct scheduled_expr	*sexpr;
	struct scheduled_system	*ss;
  };

static void
check_delay(void *data, void *context)
{
	struct delay *d;
	struct check_delay_context *ctx;

	d = (struct delay *)data;
	ctx = (struct check_delay_context *)context;

	if (d->delay_max == NULL)
	  {
		if (!expr_is_rt_const(d->delay))
		  {
			fprintf(stderr, "error: system `%s': variable delay "
				"without maximum delay value "
				"(port: %s%s%s%s)\n", ctx->ss->system->id,
				(ctx->sexpr->component != NULL)
				? ctx->sexpr->component->id : "",
				(ctx->sexpr->component != NULL) ? "." : "",
				ctx->sexpr->port->id,
				((ctx->sexpr->port->type == port_type_w)
				 || (ctx->sexpr->port->type == port_type_k))
				 ? ".out" : "");
			exit(EXIT_FAILURE);
		  }
	  }
	else
	  {
		if (!expr_is_rt_const(d->delay_max))
		  {
			fprintf(stderr, "error: system `%s': maximum delay "
				"value is not runtime constant "
				"(port: %s%s%s%s)\n", ctx->ss->system->id,
				(ctx->sexpr->component != NULL)
				? ctx->sexpr->component->id : "",
				(ctx->sexpr->component != NULL) ? "." : "",
				ctx->sexpr->port->id,
				((ctx->sexpr->port->type == port_type_w)
				 || (ctx->sexpr->port->type == port_type_k))
				 ? ".out" : "");
			exit(EXIT_FAILURE);
		  }
	  }
}

static void
check_delays(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct scheduled_system *ss;
	struct check_delay_context ctx;

	sexpr = (struct scheduled_expr *)data;
	ss = (struct scheduled_system *)context;

	ctx.sexpr = sexpr;
	ctx.ss = ss;
	list_for_each(sexpr->delays, check_delay, &ctx);
}

static void
get_delay_type_min(void *data, void *context)
{
	struct delay *d;
	struct scheduled_expr *sexpr;
	double value;

	d = (struct delay *)data;
	sexpr = (struct scheduled_expr *)context;

	if (d->delay_max != NULL)
	  {
		if (expr_is_const(d->delay_max))
		  {
			value = ceil(expr_eval(d->delay_max));
			if ((value > 1.0) &&
			    (sexpr->db_type == delay_buf_type_single))
				sexpr->db_type = delay_buf_type_fixed;
			if (value > sexpr->delay_min)
				sexpr->delay_min = value;
		  }
		else
			sexpr->db_type = delay_buf_type_var;
	  }
	else if (expr_is_const(d->delay))
	  {
		value = ceil(expr_eval(d->delay));
		if ((value > 1.0) && (sexpr->db_type == delay_buf_type_single))
			sexpr->db_type = delay_buf_type_fixed;
		if (value > sexpr->delay_min)
			sexpr->delay_min = value;
	  }
	else
		sexpr->db_type = delay_buf_type_var;
}

static void
set_delay_data(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct scheduled_system *ss;

	sexpr = (struct scheduled_expr *)data;
	ss = (struct scheduled_system *)context;

	if (list_is_empty(sexpr->delays))
	  {
		sexpr->db_type = delay_buf_type_none;
		return;
	  }

	sexpr->db_type = delay_buf_type_fixed;
	sexpr->delay_min = 1.0;
	list_for_each(sexpr->delays, get_delay_type_min, sexpr);

	switch (sexpr->db_type)
	  {
		case delay_buf_type_single:
			ss->has_single_delay = 1;
			break;
		case delay_buf_type_fixed:
			ss->has_fixed_delay = 1;
			break;
		case delay_buf_type_var:
			ss->has_var_delay = 1;
			break;
		default:
			break;
	  }
}

static void
set_uses_sample_rate(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct scheduled_system *ss;

	sexpr = (struct scheduled_expr *)data;
	ss = (struct scheduled_system *)context;

	if ((sexpr->component == NULL) && !sexpr->is_output)
		return;

	if (!ss->uses_sample_rate)
		ss->uses_sample_rate = expr_contains_sample_rate(sexpr->expr);
}

struct scheduled_system *
schedule(struct system *system)
{
	struct scheduled_system *ss;
	struct assign_io_indexes_context ctx;

	ss = xmalloc(sizeof(struct scheduled_system));

	ss->system = system;
	ss->exprs = list_new();
	ss->schedule = list_new();
	ss->schedule_d = list_new();
	ss->has_single_delay = 0;
	ss->has_fixed_delay = 0;
	ss->has_var_delay = 0;
	ss->uses_sample_rate = 0;

	list_for_each(system->ports, port_add_io, ss);

	list_for_each(ss->exprs, check_delay_free_loops, ss);
	list_for_each(ss->exprs, check_delays, ss);

	ctx.i = 0;
	ctx.exprs = ss->exprs;
	list_for_each(ss->system->ports, assign_io_indexes, &ctx);
	list_for_each(ss->exprs, assign_indexes, &ctx.i);

	list_for_each(ss->exprs, schedule_add_output_df, ss);
	list_for_each(ss->exprs, schedule_add_output, ss);
	list_free(ss->schedule_d);

	list_for_each(ss->exprs, set_delay_data, ss);

	list_for_each(ss->exprs, set_uses_sample_rate, ss);

	return ss;
}

static void
free_delay(void *data, void *context)
{
	free(data);
}

static void
free_sexpr(void *data, void *context)
{
	struct scheduled_expr *sexpr;

	sexpr = (struct scheduled_expr *)data;

	if ((sexpr->component != NULL) || sexpr->is_output)
		expr_free(sexpr->expr);

	list_free(sexpr->refs);
	list_free(sexpr->df_refs);
	list_free(sexpr->is_refd);

	list_for_each(sexpr->delays, free_delay, NULL);
	list_free(sexpr->delays);

	free(sexpr);
}

void
scheduled_system_free(struct scheduled_system *system)
{
	list_for_each(system->exprs, free_sexpr, NULL);

	list_free(system->schedule);
	list_free(system->exprs);

	free(system);
}
