/*
 * STV0680 Vision Camera Chipset Driver
 * Copyright (C) 2000 Adam Harrison <adam@antispin.org> 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 *
 *  Modifications and additions by Kevin Sisson <kjsisson@bellsouth.net>
 *  November, 2001
 *
 */

#include <malloc.h>
#include <stdio.h>
#include <string.h>

#include <stdlib.h>
#include <term.h>
#include <ctype.h>
#include <math.h>
#include <usb.h>

#include "pencam2.h"

#define RED 0
#define GREEN 1
#define BLUE 2
#define AD(x, y, w) (((y)*(w)+(x))*3)


void bayer_unshuffle (unsigned char *raw, unsigned char *output, usb_stv *stv680)
{
        int x, y, i;
	int w = stv680->cwidth;
        int vw=stv680->vwidth, vh = stv680->vheight;
	unsigned int p=0;
        int colour = 0, bayer = 0;
//  	  float R=0, G=0, B=0;
        
        memset(output, 0, 3*vw*vh);  /* clear output matrix */

	/*  raw bayer data: 1st row, 1st half are the odd pixels (same color), 
	    2nd half (w/2) are the even pixels (same color) for that row 
	    top left corner of sub array is always green */

        for (y=0; y<vh; y++) {
		for (x=0; x<vw; x++) {

			if (x&1)
	    			p = *(raw + y*w + (x>>1));
			else
				p = *(raw + y*w + (x>>1) + (w>>1));
	    
			if (y&1)
				bayer = 2;
			else
				bayer = 0;
			if (x&1)
				bayer++;

			switch(bayer) {
				case 0:
				case 3: colour = 1; break;
				case 1: colour = 0; break;
				case 2: colour = 2; break;
			}
			i = (y*vw + x)*3;  /* output already zeroed out with memset */

	    /* sample a small region of pic of white object to get correction values */
/*	    if ((y&1) && (!(x&1)) && (x<=4) && (y<=5))
		R += p;		
	    if ((!(y&1)) && (x&1) && (x<=4) && (y<=5))
		B += p;		
	    if ((!(y&1)) && (!(x&1)) && (x<=4) && (y<=5))
		G += p;		
	    if ((y&1) && (x&1) && (x<=4) && (y<=5))
		G += p;		
	    if ((x==5) && (y==5)) {
		R = R/9;
		B = B/9;
		G = G/18;
		fprintf(stdout, "R=%i  G=%i  B=%i\n", (int)R, (int)G, (int)B);
	    }
*/		
			*(output + i + colour) = (unsigned char) p;
		} /* for x */
	}  /* for y */

}  /* bayer_unshuffle */

void bayer_demosaic (usb_stv *stv680, unsigned char *output)
{
        int x, y, bayer;
	int vw=stv680->vwidth, vh=stv680->vheight;
    
        for (y=1; y<(vh-1); y++) { 
	        for (x=1; x<(vw-1); x++) {   /* work out pixel type */
			if (y&1)
				bayer = 0;
			else
				bayer = 2;
			if (!(x&1))
				bayer++;

			switch(bayer) {

    				case 0:        /* green. blue lr, red tb */
					*(output + AD(x,y,vw) + BLUE) = 
					((int)*(output + AD(x-1,y,vw) + BLUE) + (int)*(output + AD(x+1,y,vw) + BLUE))>>1;
					*(output + AD(x,y,vw) + RED) = 
					((int)*(output + AD(x,y-1,vw) + RED) + (int)*(output + AD(x,y+1,vw) + RED))>>1;
					break;
		    
				case 1:        /* blue. green lrtb, red diagonals */
					*(output + AD(x,y,vw) + GREEN) = 
					((int)*(output + AD(x-1,y,vw) + GREEN)+(int)*(output + AD(x+1,y,vw) + GREEN)+(int)*(output + AD(x,y-1,vw) + GREEN)+(int)*(output + AD(x,y+1,vw) + GREEN))>>2;
					*(output + AD(x,y,vw) + RED) = 
					((int)*(output + AD(x-1,y-1,vw) + RED)+(int)*(output + AD(x-1,y+1,vw) + RED)+(int)*(output + AD(x+1,y-1,vw) + RED)+(int)*(output + AD(x+1,y+1,vw) + RED))>>2;
					break;
		    
				case 2:        /* red. green lrtb, blue diagonals */
					*(output + AD(x,y,vw) + GREEN) = 
					((int)*(output + AD(x-1,y,vw) + GREEN)+(int)*(output + AD(x+1,y,vw) + GREEN)+(int)*(output + AD(x,y-1,vw) + GREEN)+(int)*(output + AD(x,y+1,vw) + GREEN))>>2;
					*(output + AD(x,y,vw) + BLUE) = 
					((int)*(output + AD(x-1,y-1,vw) + BLUE)+(int)*(output + AD(x+1,y-1,vw) + BLUE)+(int)*(output + AD(x-1,y+1,vw) + BLUE)+(int)*(output + AD(x+1,y+1,vw) + BLUE))>>2;
					break;
		    
				case 3:        /* green. red lr, blue tb */
					*(output + AD(x,y,vw) + RED) = 
					((int)*(output + AD(x-1,y,vw) + RED)+(int)*(output + AD(x+1,y,vw) + RED))>>1;
					*(output + AD(x,y,vw) + BLUE) = 
					((int)*(output + AD(x,y-1,vw) + BLUE)+(int)*(output + AD(x,y+1,vw) + BLUE))>>1;
					break;
			}  /* switch */
		}   /* for x */
	}  /* for y */
}  /* bayer_demosaic */


/****** gamma correction from trans[], plus hardcoded white balance */
/* Thanks to Alexander Schwartx <alexander.schwartx@gmx.net> for this code.
   Gamma correction (trans[] values generated by (pow((i-17)/256, GAMMA)*255)
   where GAMMA=0.45, 1<i<255.  */

#define GAMMA 0.450
#define ZERO 17.0

void light_enhance (usb_stv *stv680, unsigned char *output, char *cam_snap)
{
        unsigned long int i;
	int vw = stv680->vwidth, vh = stv680->vheight;
        int lt=0;
	float wb[3][3];
        double x=0, y=0;
	unsigned int trans[258], v;    
    
        fprintf(stdout," Downloaded picture %i  ", stv680->pic_no);

        if (strcmp(cam_snap, "PENCAM") == 0) {
		lt = stv680->light_type;
    		x = stv680->brightness;
	} else if (strcmp(cam_snap, "PENSNAP") == 0) {
		lt = stv680->light_type_ps;
		x = stv680->brightness_ps;
	}

        /*  -1=skip,  0=natural,  1=fluorescent,  2=incandescent, 3=auto  */
	if (lt != 3) 
		fprintf (stdout, "\n");
	if (lt == -1) 
		return;
	if (lt == 3) {
	        light_enhance2 (stv680, output, cam_snap);
		return;
	}

/*   not used - see light_enhance2
	if (lt == 3) {
    		fprintf(stdout, "(F_Exp=%i  C_Exp=%i  APV=%i => filter = ",
			stv680->FineExp, stv680->CoarseExp, stv680->AvgPixVal); 

		if (stv680->FineExp >= (stv680->CoarseExp<<2)) {
			lt = 0;
			fprintf(stdout, "Natural)\n");
		} else if ((stv680->FineExp <= 5) && (stv680->CoarseExp <= 399)
			    && (stv680->CoarseExp >= 90)) {
			lt = 2;
			fprintf(stdout, "Incandescent)\n");
		} else {
		    lt = 1;
		    fprintf(stdout, "Fluorescent)\n");
		}
	}  
*/
        /*  white balance filter numbers */
	wb[0][0] = 1.08 * x;  wb[0][1] = 1.00 * x;  wb[0][2] = 0.95 * x;
	wb[1][0] = 1.05 * x;  wb[1][1] = 1.00 * x;  wb[1][2] = 1.00 * x;
	wb[2][0] = 0.90 * x;  wb[2][1] = 1.00 * x;  wb[2][2] = 1.10 * x;
	    
	for(i=1; i<256; ++i)
	{
		x = i; x -= ZERO;
		if (x < 1.0)
			x = 1.0;
		y = pow((x/256.0),GAMMA)*255.0; 	
		trans[i] = (unsigned int) y;
	}
	trans[0] = 0;

        for (i=0; i<(vw*vh*3); i+=3)
	{   
		y = trans[*(output+i)] * wb[lt][0];	v = (unsigned int)y;
		*(output+i) = (unsigned char) (v > 255) ? 255 : v;
		y = trans[*(output+i+1)] * wb[lt][1];	v = (unsigned int)y;
		*(output+i+1) = (unsigned char) (v > 255) ? 255 : v;
		y = trans[*(output+i+2)] * wb[lt][2];	v = (unsigned int)y;
		*(output+i+2) = (unsigned char) (v > 255) ? 255 : v;
	}  /* for */

}  /* light_enhance  */


/*  Thanks to Kurt Garloff <kurt@garloff.de> for this routine  */

/* KG: Looking at very dark parts of images, the sensor seems to produce 
 * only very few points below 0x11 and almost none below 14. Therefore we map 
 * everything below 14 to 0 and ev'thing below 17 to 1; then the power function 
 * reigns.
 *
 * KG: Some notes on these:
 * - Try to avoid strong deviations from 1.00 for the amplification,
 *   because this potentially results in not using the full range
 *   of colours (<1) or in clipping (>1) multiple colours to max,
 *   which would be a loss of information.
 * - The gamma mainly determines how fast values increase after ZERO1.
 *   Influence on the highlights is small; therefore the description
 *   with amplifiaction and gamma seems not very appropriate; a better
 *   correction function would allow to influence the slope for small
 *   and for large values indepentently without incurring loss of
 *   accuracy/information. It should not be hard to construct such a
 *   thing. (Splines or Bzier or Triginometric/Hyperbolic functions
 *   could be used, e.g.)
 * - The below parameters have been found by lots of experiments with
 *   pictures taken at different light levels. They're optimized for
 *   my PenCam (and my screens), of course. No theory behind this;
 *   I don't have insight into the physics of the imaging sensor.
 *   CCDs are linear, basically; but higher order effects may play
 *   a role as well as the electronics that controls the shutter
 *   and the one doing the readout.
*/

#define ZERO0 14 //  0--13 mapped to 0
#define ZERO1 17 // 14--16 mapped to 1

typedef struct _rgbgamma {
    	float ampl, gamma;
} rgbgamma;

rgbgamma gampar[6][3] = {
	{ { 1.02, 0.56 }, { 1.00, 0.61 }, { 0.99, 0.65 } }, // cold
	{ { 1.01, 0.56 }, { 1.00, 0.58 }, { 1.00, 0.61 } }, // coldish
	{ { 1.00, 0.55 }, { 1.00, 0.57 }, { 1.00, 0.59 } }, // mid
	{ { 1.00, 0.55 }, { 1.00, 0.56 }, { 1.01, 0.55 } }, // warmish
	{ { 1.01, 0.56 }, { 0.99, 0.57 }, { 1.03, 0.50 } }, // warm
//	{ { 1.03, 0.52 }, { 0.97, 0.57 }, { 1.04, 0.49 } }  // warm bright
	{ { 1.01, 0.52 }, { 1.01, 0.55 }, { 1.04, 0.49 } }  // warm bright
};


void light_enhance2 (usb_stv *stv680, unsigned char *output, char *cam_snap)
{
        unsigned long int i;
	unsigned char     trans[3][256], col;	
	int      vw = stv680->vwidth, vh = stv680->vheight;
	int      lt=0;
        double   y = 0, brightness = 0;
	int      fine, coarse, avg_pix;
	int      r,g,b;


        if (strcmp(cam_snap, "PENCAM") == 0) {
    		brightness  = stv680->brightness;
	} else if (strcmp(cam_snap, "PENSNAP") == 0) {
		brightness  = stv680->brightness_ps;
	}
    	
	fine = stv680->FineExp;		coarse = stv680->CoarseExp;	
	avg_pix = stv680->AvgPixVal;

	fprintf(stdout, "(F_Exp=%3i,  C_Exp=%3i,  APV=%3i => ",
		stv680->FineExp, stv680->CoarseExp, stv680->AvgPixVal); 

        if (fine > coarse) {
    		lt = 0; 	fprintf (stdout, "cold)\n");
	} else if (coarse < 100) {	    
		lt = 1; 	fprintf (stdout, "coldish)\n");
	} else if (coarse < 200) {
		lt = 2; 	fprintf (stdout, "mid)\n");
	} else if (coarse < 400) {
		lt = 3;		fprintf (stdout, "warmish)\n");
	} else if (avg_pix < 94) {
		lt = 4;		fprintf (stdout, "warm)\n");
	} else {
		lt = 5; 	fprintf (stdout, "warm, bright)\n");
	}

        for (col = 0; col < 3; col++) {

		rgbgamma *gp = gampar[lt] + col;

		for (i=0; i<256; ++i) {
			if (i < ZERO0)
				y = 0;
			else if (i < ZERO1)
				y = 1;
			else 
				y = brightness * gp->ampl * (2 + pow((i-ZERO1)/((double)254-ZERO1),gp->gamma) * 253.5);
			if (y > 255.0)
				y = 255.0;
			trans[col][i] = (unsigned char) y;
		}
	}
 
	for (i=0; i<(vw*vh*3); i+=3)
	{   
		r = *(output+i);
		g = *(output+i+1);
		b = *(output+i+2);

		*(output+i)   = trans[0][r];
		*(output+i+1) = trans[1][g];
		*(output+i+2) = trans[2][b];
         }  /* for */

}  /* light_enhance2  */

