#include <string.h>
#include <stdio.h>
#include <sys/timeb.h>

#include "mudclient.h"
#include "PapayaList.h"
#include "PluginHandler.h"
#include "TurfProtocol.h"
#include "Message.h"
#include "Stats.h"
#include "version.h"

struct stats {

  int min; // ms
  int max; // ms
  unsigned long total; // ms

  int num_cmds;
};

extern PluginHandler * phandler;
static Stats * st;
static List * sList = NULL;

#define MAJOR "1"
#define MINOR "1"

extern "C" char * plugin_query_name() {
  return "Stats";
}

extern "C" char * plugin_query_description() {
  return _("Attempts to perform tests on the lag to a MUD.  Requires TurfProtocol.  Recommended against - use LagIndicator instead.");
}

extern "C" char * plugin_query_major() {
  return MAJOR;
}

extern "C" char * plugin_query_minor() {
  return MINOR;
}

extern "C" void plugin_init(void) {
  st = new Stats();
}

extern "C" void plugin_cleanup(void) {
  delete st;
}

Stats::Stats() {
  sList = new List();
  version = 1.0;
  name = strdup("Connection statistics");
  phandler->registerPlugin(this, VERSION);
}

Stats::~Stats() {
  free(name);
  delete sList;
  sList = NULL;
  phandelr->unregisterPlugin(this);
}

void Stats::onEvent(Event * e, Connection * c) {

  if (e->getType() == EvConnect) {
    struct stats * data;
    ListElement * tmp = sList->findEntry(c);
    if (!tmp)
      tmp = sList->newEntry(c, NULL);

    data = (struct stats *)tmp->getData();
    if (!data) {
      data = (struct stats *)malloc(sizeof(struct stats));
      tmp->setData(data);
    }

    data->min = 0;
    data->max = 0;
    data->total = 0;
    data->num_cmds = 0;
    return;
  }

  if (e->getType() == EvDisconnect) {
    ListElement * tmp = sList->findEntry(c);
    // This will call a callback to get rid of the data.
    displayData(tmp);
    return;
  }
}

void Stats::displayData(ListElement * tmp) {

  struct stats * data = (struct stats *)tmp->getData();
  //  Connection * c = tmp->getConnection();

  if (data->num_cmds == 0) {
    return;
  }

  printf(_("Minimum command time: %d ms\n"
	 "Maximum command time: %d ms\n"
	 "Average command time: %ld ms\n"), data->min, data->max, data->total/data->num_cmds);

  printf(_("Total time waiting for %d commands: %ld ms\n"), data->num_cmds, data->total);

}

void Stats::input(Connection * c, char * buf) {
  TurfProtocol * tp = NULL;
  
  // Try and find the TurfProtocol object.
  void * handle = phandler->findPlugin("TurfProtocol");
  if (!handle) {
    return;
  }
  
  if (!findTurf(handle, (void **)&tp)) {
    return;
  }
  
  if (!tp) {
    return;
  }

  struct timeb * t = (struct timeb *)malloc(sizeof(struct timeb));
  ftime(t);

  tp->addCommand(c, buf, stats_callback, (void *)t);
  buf[0] = '\0';

}

void stats_callback(Connection * c, char * buf, void * d) {
  struct timeb start = *((struct timeb *)d);
  struct timeb end;
  struct stats * data;

  if (buf) {
    c->getVT()->append(c, buf);
    return;
  }

  c->getVT()->scroll();

  ListElement * tmp = sList->findEntry(c);
  if (!tmp)
    return;

  data = (struct stats *)tmp->getData();
  if (!data)
    return;

  ftime(&end);
  int e, s, diff;
  e = (end.time * 1000) + end.millitm;
  s = (start.time * 1000) + start.millitm;
  diff = e - s;

  data->total += diff;

  if (data->min == 0 || diff < data->min)
    data->min = diff;

  if (diff > data->max)
    data->max = diff;

  data->num_cmds++;
  // Otherwise we're leaky
  //  free(d);
}
