#include <stdio.h>
#include "papaya/system.h"
#include "papaya/ColourEditor.h"

ColourEditor::ColourEditor(gushort red, gushort green, gushort blue) {
  init(red, green, blue);

  change_callback = NULL;
  callback_data = NULL;
}

ColourEditor::ColourEditor(gushort r, gushort g, gushort b, CEChangeCallback cb, gpointer data) {
  init(r, g, b);

  change_callback = cb;
  callback_data = data;
}

void ColourEditor::init(gushort red, gushort green, gushort blue) {
  colour = (GdkColor*) g_malloc (sizeof(GdkColor));
  colour->red = red;
  colour->green = green;
  colour->blue = blue;

  wrapper = gtk_button_new();
  gtk_widget_show(wrapper);

  GtkWidget * frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
  gtk_container_add(GTK_CONTAINER(wrapper), frame);

  preview = gtk_drawing_area_new();
  gtk_widget_set_size_request(preview, 20, 10);
  gtk_widget_show(preview);
  g_signal_connect_data(G_OBJECT(preview), "expose_event",
					GTK_SIGNAL_FUNC(colour_editor_expose), this,
					NULL, (GConnectFlags)0);
  g_signal_connect_data(G_OBJECT(wrapper), "clicked",
					GTK_SIGNAL_FUNC(colour_editor_clicked), this,
					NULL, (GConnectFlags)0);

  gtk_container_add(GTK_CONTAINER(frame), preview);

  editor = NULL;

  sensitive = TRUE;

  /*
   * This cleans up memory automatically when the widgets are destroyed.
   * Nothing else should ever call delete on a ColourEditor object.
   */
  g_object_set_data_full(G_OBJECT(wrapper), "C++ destructor", this,
				colour_editor_destroy);
}

ColourEditor::~ColourEditor() {
  g_free(colour);

  if (editor)
    gtk_widget_destroy(editor);
}

GtkWidget * ColourEditor::getPreview() {
  return wrapper;
}

gushort ColourEditor::getRed() {
  return colour->red;
}

gushort ColourEditor::getGreen() {
  return colour->green;
}

gushort ColourEditor::getBlue() {
  return colour->blue;
}

void ColourEditor::setRGB(gushort red, gushort green, gushort blue) {
  colour->red = red;
  colour->green = green;
  colour->blue = blue;

  // Need to trigger the expose event.
  
  //  colour_editor_expose(preview, NULL, this);
}

void ColourEditor::setSensitive(bool b) {
  if (!b && editor) {
    gtk_widget_hide(editor);
    gtk_widget_destroy(editor);
    editor = (GtkWidget *)NULL;
  }

  gtk_widget_set_sensitive(wrapper, b);

  sensitive = b;
}

bool ColourEditor::isSensitive() {
  return sensitive;
}

void ColourEditor::createEditorDialog() {
  GtkWidget *ok_button1;
  GtkWidget *cancel_button1;

  if (editor)
    return;
 
  editor = gtk_color_selection_dialog_new (_("Select Color"));
  gtk_container_set_border_width (GTK_CONTAINER (editor), 10);
 
  ok_button1 = GTK_COLOR_SELECTION_DIALOG (editor)->ok_button;
  gtk_widget_show (ok_button1);
  GTK_WIDGET_SET_FLAGS (ok_button1, GTK_CAN_DEFAULT);

  cancel_button1 = GTK_COLOR_SELECTION_DIALOG (editor)->cancel_button;
  gtk_widget_show (cancel_button1);

  gtk_widget_hide(GTK_COLOR_SELECTION_DIALOG (editor)->help_button);

  g_signal_connect_data(G_OBJECT(ok_button1), "clicked",
						GTK_SIGNAL_FUNC(colour_editor_ok_clicked), this,
						NULL, (GConnectFlags)0);
  g_signal_connect_data(G_OBJECT(cancel_button1), "clicked",
						GTK_SIGNAL_FUNC(colour_editor_cancel_clicked), this,
						NULL, (GConnectFlags)0);
  g_signal_connect_data(G_OBJECT(editor), "delete-event",
						GTK_SIGNAL_FUNC(colour_editor_wm_close), this,
						NULL, (GConnectFlags)0);
}

gboolean colour_editor_expose(GtkWidget * widget, GdkEvent * event,
                              gpointer data) {
  ColourEditor * ce = (ColourEditor *) data;
  GdkGC * gc = gdk_gc_new(widget->window);
  gdk_gc_copy(gc, widget->style->white_gc);

  if (!gdk_color_alloc(gdk_colormap_get_system(), ce->colour)) {
    printf(_("colour_editor_expose: couldn't allocate colour.\n"));
  }
  gdk_gc_set_foreground(gc, ce->colour);
  gdk_gc_set_background(gc, ce->colour);

  gdk_draw_rectangle(widget->window, gc, TRUE, 0, 0,
                     widget->allocation.width, widget->allocation.height);

  if (!ce->sensitive) {
    GdkColor hashing;

    // We need to know the intensity of the high colour.
    int average = (ce->colour->red + ce->colour->green + ce->colour->blue) / 3;

    if (average > (65535 / 2)) {
      hashing.red   = 0;
      hashing.green = 0;
      hashing.blue  = 0;
    } else {
      hashing.red   = 65535;
      hashing.green = 65535;
      hashing.blue  = 65535;
    }

    if (!gdk_color_alloc(gdk_colormap_get_system(), &hashing)) {
      printf(_("colour_editor_expose: couldn't allocate colour.\n"));
    }

    gdk_gc_set_foreground(gc, &hashing);
    gdk_gc_set_background(gc, &hashing);

    for ( int i = -1 * widget->allocation.height;
          i < widget->allocation.width;
          i += 3)
    {
      gdk_draw_line(widget->window, gc,
                    i, 0,
                    i + widget->allocation.height, widget->allocation.height);
    }
  }

  gdk_gc_destroy(gc);
  return TRUE;
}


gboolean colour_editor_clicked(GtkWidget * widget, gpointer data) {
  ColourEditor * ce = (ColourEditor *) data;

  // Clicked on a colour.  Bring up a colour editing dialog for it and set
  // the colour in that dialog.

  if (ce->editor) {
    // Already editing this colour
    // Present the window
    gtk_window_present((GtkWindow *)ce->editor);
    // We processed this, so don't let the parents handle it
    return TRUE;
  }

  ce->createEditorDialog();

  // Figure out the colour for data.

  GtkColorSelection * csel = (GtkColorSelection *)((GtkColorSelectionDialog *)ce->editor)->colorsel;
  gtk_color_selection_set_current_color(csel, ce->colour);
  // Show the colour selector.
  gtk_widget_show(ce->editor);
  return TRUE;
}

gboolean colour_editor_ok_clicked(GtkButton * button, gpointer data) {
  ColourEditor * ce = (ColourEditor *) data;

  GtkColorSelection * csel = (GtkColorSelection *)((GtkColorSelectionDialog *)ce->editor)->colorsel;
  gtk_color_selection_get_current_color(csel, ce->colour);
  gtk_widget_hide(ce->editor);
  colour_editor_expose(ce->preview, NULL, ce);
  gtk_widget_destroy(ce->editor);
  ce->editor = (GtkWidget *)NULL;

  if (ce->change_callback)
    ce->change_callback(ce, ce->callback_data);

  return TRUE;
}

gboolean colour_editor_cancel_clicked(GtkButton * button, gpointer data) {
  ColourEditor * ce = (ColourEditor *) data;

  gtk_widget_hide(ce->editor);
  gtk_widget_destroy(ce->editor);
  ce->editor = (GtkWidget *)NULL;

  return TRUE;
}

gboolean colour_editor_wm_close(GtkWidget * widget, GdkEvent * event,
                                gpointer data) {
  ColourEditor * ce = (ColourEditor *) data;

  gtk_widget_hide(ce->editor);
  gtk_widget_destroy(ce->editor);
  ce->editor = (GtkWidget *)NULL;

  return TRUE;
}

void colour_editor_destroy(gpointer colour_editor) {
  delete (ColourEditor *) colour_editor;
}
