#include "mycrypt.h"

#ifdef MDH

const static struct {
    int size;
    char *name, *base, *prime;
} sets[] = {
#ifdef DH512
{ 
    64,
   "DH-512",
   "3",
   "7875166757558590412849031755863612581437009353583544009256809204514602324889"
   "4629104084970318723235592186356722189829916467310465696557594067708443966679"
   "47"
},
#endif
#ifdef DH768
{
   96,
   "DH-768",
   "2",
   "8506764189761809446256912233709688203260180211766226617875564517619113193197"
   "4416555038034781222372326151399672811174780386356064252016670553051085041482"
   "9797471384631926482095774990602978715508601368299682865616785599295763013500"
   "447"
},
#endif
#ifdef DH1024
{
   128,
   "DH-1024",
   "3",
   "1011240114611255733745700382106578814958412208352009214949934273277728754922"
   "5629955034388736052168383397704841664635967868548487982193838389649997214743"
   "0988786346417520119109038481897275742523465299987799057619813706950302531729"
   "7022728308669684540489780422329936041430269037062247486965194862959258047071"
   "76203"
},
#endif
#ifdef DH1280
{
   160,
   "DH-1280",
   "2",
   "2039079912681232603888428410353887554196625293350055295514068591085419165420"
   "8033333778459315070449626628457179853385599355143913757940791470793937515806"
   "0499997462264057712389132172008973805509957596554247493234650426732342042606"
   "6362502962957356596282090935360515445531510105668915419015175794607544875747"
   "5075019627391149198409192341234847338953543795290063350337040283149894084115"
   "473039"
},
#endif
#ifdef DH1536
{
   192,
   "DH-1536",
   "2",
   "2023562525307700117785992292096803154014501370424915215042233945137282514726"
   "6420444006963723185415667827743312547768217989188494898673450456121532505551"
   "7858951499556392482329850848162377411583564686775460471790255332240600866594"
   "8087403133969175805381856920070451562951904730579963150956216315662212483507"
   "7950049771592007542233718943104478134812113239871702362856063247485122566143"
   "2688380894868476270549736396582269945399917448374683619923333607320197398440"
   "9764423"
},
#endif
#ifdef DH1792
{
   224,
   "DH-1792",
   "3",
   "2629936743808560411708185464517638782895739026889869009804927317674103440226"
   "4236213860802159655662695262557672850213922267121294681395974906700736063420"
   "9932251019068619456196796247094872832738643606912818442826354500343664142884"
   "4073088591743499156899984409073573942599701229213366991106025520126138908313"
   "6554429929522060361152616080578866790731991546654938571833506230077856889589"
   "0703822902263965946576665944274654002362332099201314953627067387235628959551"
   "6681511773105594777872915732326713920424125330333098987282815358297778582810"
   "83492299"
},
#endif
#ifdef DH2048
{
   256,
   "DH-2048",
   "3",
   "2303967819157298457820983538298963225086964150462985438548012773533497902528"
   "3824926002995060399672823430521514565863093033047991222579426358248747008839"
   "1716361049010195641773881316970764537904156478821471169516823900128330287552"
   "8099471126889282135385945070584383886727991490581811447904820408526873159275"
   "9495269619825419252476832584352349544475781431351367749318384958213902506572"
   "0919965962961709140589454473470691067126569434681207849453889224086369408763"
   "8202020566323910079819460105406414307909521880581880258251607170349774742903"
   "9017191024775041828323249899475213230298994486518429566633208609029383647709"
   "218060723"
},
#endif
#ifdef DH2560
{
   320,
   "DH-2560",
   "2",
   "2450032170253157911090662843449749004690191008166001739198163130342854385314"
   "6407416803528330754701380534252669713799610626591663139873239704737180831241"
   "5061340224313692836154334521770434686985907336168081730767921141418525583881"
   "2184148828979254047721190126133734183200271487583358796617468332364511649228"
   "3948931787335406078652913526273453637527819970523843481884717651990978087083"
   "3381909426287160384826544822250458987463942802737906434661742923999597885957"
   "0072300336303225026457902667714829590579139307237443898009944729037073424071"
   "7081867826767100741254353885442402653847377987168215931964574726596403134169"
   "7350404328530408346479403244587934373727708053301364877413865015541847490369"
   "6498063942696297612262282049786932243359593827486311295944329463369195743069"
   "46482032759"
},
#endif
#ifdef DH3072
{
   384,
   "DH-3072",
   "2",
   "3548479356261408592012977158876942479693315121989799964323893909998059109942"
   "9698372525587390242655701963132667493511812212866751422510733790438376212914"
   "4014726045850967624957481265882888371097563469436248943656153998117201471047"
   "3874206945284627064358451005708934640305704968883483267302092815733581080333"
   "8416132287161200347126528994951008945131391740430373292742975856502832535933"
   "5821028509544367963550879696935220398508770926175291688301525142555995981076"
   "7886969726684583927927670659748594689045470473212718211971528959663685504446"
   "9660349871023505264937609866071875368494864404620079985581170965387740222553"
   "0497702953476870888007138222974729409347140711709419876332321326372128679117"
   "9602734685492690656836885039765096019043936438116952524646557092944601255466"
   "9649807950770581894461224054876998601346868416894765860003441823347935768368"
   "7226140805486329638753368469778013220102403020904760684130165615839534898091"
   "3562767896959"
},
#endif
#ifdef DH4096
{
   512,
   "DH-4096",
   "3",
   "6358539196446615807931022654140507280416734561024895426775153474314973125240"
   "4347402763938035264935368421122877986076049256341847343984058691414699239822"
   "6406308172677026734967135340473409901967897748721493157405324288903903717955"
   "2981983142207872615134636940383110675114566508117592995972746072870933859082"
   "1455364652219843642339126189183464312181400459328867584168455617503565191157"
   "6865461568238642288512687460344498356348113027914902760531720036933240317787"
   "8740518107396147180915879270363294258838666123108262284758481199506756470652"
   "1315531367328651283034207638696939582690379978817036561785681889534698957723"
   "5027088205460442351794897880942571717421045897126319166273632969414951787755"
   "9874672830266704609789918234706094696088154371800710845376080190363267021635"
   "9534751460199715045921613896191848798459137268947003074263735847160138597400"
   "3731305689924374414709872181655798849093974421846890694304057208270312532548"
   "4605082221240785115413101620330620957307211119505460116451027285704417087756"
   "6755652816360841460967461847052294355008494338135696639558907300784908289415"
   "5407764813623615434810848586310567292673544788163038104667431390951738079566"
   "2261646650198441553879977736911177250491753118761227575255309636989112991447"
   "84966708147184163"
},
#endif
{   
   0,
   NULL,
   NULL,
   NULL
}
};

static char error_code[128];

static int is_valid_idx(int n)
{
   int x;

   for (x = 0; sets[x].size; x++);
   if ((n < 0) || (n >= x)) {
      return 0;
   }
   return 1;
}


int dh_test(void)
{
    mp_int p, g, tmp;
    int x, res, primality;

    if (mp_init_multi(&p, &g, &tmp, NULL) != MP_OKAY) {
       crypt_error = "Out of memory in dh_test().";
       return CRYPT_ERROR;
    }

    for (x = 0; sets[x].size; x++) {
        /* see if g^((p-1)/2) == 1 mod p. */
        if (mp_read_radix(&g, sets[x].base, 10) != MP_OKAY) { goto error; }
        if (mp_read_radix(&p, sets[x].prime, 10) != MP_OKAY) { goto error; }

        /* ensure p is prime */
        if (is_prime(&p, &primality) == CRYPT_ERROR) { goto error; }
        if (primality == 0) { 
           sprintf(error_code, "%s modulus not prime.", sets[x].name);
           crypt_error = error_code;
           res = CRYPT_ERROR;
           goto done;
        }

        if (mp_sub_d(&p, 1, &tmp) != MP_OKAY) { goto error; }
        if (mp_div_2(&tmp, &tmp) != MP_OKAY) { goto error; }

        /* ensure (p-1)/2 is prime */
        if (is_prime(&p, &primality) == CRYPT_ERROR) { goto error; }
        if (primality == 0) {
           sprintf(error_code, "%s order not prime.", sets[x].name);
           crypt_error = error_code;
           res = CRYPT_ERROR;
           goto done;
        }

        /* now see if g^((p-1)/2) mod p is in fact 1 */
        if (mp_exptmod(&g, &tmp, &p, &tmp) != MP_OKAY) { goto error; }
        if (mp_cmp_d(&tmp, 1)) {
           sprintf(error_code, "%s failed test.", sets[x].name);
           crypt_error = error_code;
           res = CRYPT_ERROR;
           goto done;
        }
    }
    res = CRYPT_OK;
    goto done;
error:
    res = CRYPT_ERROR;
    crypt_error = "Out of memory in dh_test().";
done:
    mp_clear_multi(&tmp, &g, &p, NULL);
    return res;
}

void dh_sizes(int *low, int *high)
{
   int x;
   *low  = INT_MAX;
   *high = 0;
   for (x = 0; sets[x].size; x++) {
       if (*low > sets[x].size)  *low  = sets[x].size;
       if (*high < sets[x].size) *high = sets[x].size;
   }
}

int dh_get_size(dh_key *key)
{
    if (is_valid_idx(key->idx)) 
        return sets[key->idx].size;
    else
        return INT_MAX; /* large value that would cause dh_make_key() to fail */
}
  
int dh_make_key(int keysize, prng_state *prng, int wprng, dh_key *key)
{
   unsigned char buf[768];
   unsigned long x;
   mp_int p, g;
   int res;

   /* good prng? */
   if (prng_is_valid(wprng) == CRYPT_ERROR) {
      return CRYPT_ERROR;
   }

   /* find key size */
   for (x = 0; (keysize > sets[x].size) && (sets[x].size); x++);
   keysize = sets[x].size;

   if (sets[x].size == 0) {
      crypt_error = "Invalid keysize passed to dh_make_key().";
      return CRYPT_ERROR;
   }
   key->idx = x;

   /* make up random string */
   buf[0] = 0;
   if (prng_descriptor[wprng].read(buf+1, keysize, prng) != (unsigned long)keysize) {
      crypt_error = "Error reading PRNG in dh_make_key().";
      return CRYPT_ERROR;
   }

   /* init parameters */
   if (mp_init_multi(&g, &p, &key->x, &key->y, NULL) != MP_OKAY) {
      crypt_error = "Out of memory in dh_make_key().";
      return CRYPT_ERROR;
   }
   if (mp_read_radix(&g, sets[x].base, 10) != MP_OKAY) goto error2;
   if (mp_read_radix(&p, sets[x].prime, 10) != MP_OKAY) goto error2;

   /* load the x value */
   mp_read_raw(&key->x, buf, keysize+1);
   if (mp_exptmod(&g, &key->x, &p, &key->y) != MP_OKAY) goto error2;
   key->type = PK_PRIVATE;

   /* free up ram */
   res = CRYPT_OK;
   goto done2;
error2:
   res = CRYPT_ERROR;
   crypt_error = "Out of memory in dh_make_key().";
   mp_clear_multi(&key->x, &key->y, NULL);
done2:
   mp_clear_multi(&p, &g, NULL);
   zeromem(buf, sizeof(buf));
   return res;
}

void dh_free(dh_key *key)
{
   mp_clear_multi(&key->x, &key->y, NULL);
}

#define OUTPUT_BIGNUM(num, buf2, y, z)         \
{                                              \
      z = mp_raw_size(num);                    \
      STORE32L(z, buf2+y);                     \
      y += 4;                                  \
      mp_toraw(num, buf2+y);                   \
      y += z;                                  \
}


#define INPUT_BIGNUM(num, in, x, y)                              \
{                                                                \
     /* load value */                                            \
     LOAD32L(x, in+y);                                           \
     y += 4;                                                     \
                                                                 \
     /* sanity check... */                                       \
     if (x > 1024) {                                             \
        crypt_error = "Invalid size of data in dh_import().";    \
        goto error;                                              \
     }                                                           \
                                                                 \
     /* load it */                                               \
     if (mp_read_raw(num, (unsigned char *)in+y, x) != MP_OKAY) {\
        crypt_error = "Out of memory in dh_import().";           \
        goto error;                                              \
     }                                                           \
     y += x;                                                     \
}


int dh_export(unsigned char *out, unsigned long *outlen, int type, dh_key *key)
{
   unsigned char buf2[1536];
   unsigned long y, z;

   if (type == PK_PRIVATE && key->type != PK_PRIVATE) {
      crypt_error = "Invalid key type passed to dh_export().";
      return CRYPT_ERROR;
   }

   /* header */
   y = PACKET_SIZE;

   /* header */
   buf2[y++] = type;
   buf2[y++] = key->idx;

   /* export y */
   OUTPUT_BIGNUM(&key->y, buf2, y, z);

   if (type == PK_PRIVATE) { 
      /* export x */
      OUTPUT_BIGNUM(&key->x, buf2, y, z);
   }
   
   /* check for overflow */
   if (*outlen < y) {
      zeromem(buf2, sizeof(buf2));
      crypt_error = "Buffer overrun in dh_export().";
      return CRYPT_ERROR;
   }

   /* store header */
   packet_store_header(buf2, PACKET_SECT_DH, PACKET_SUB_KEY, y);

   /* output it */
   *outlen = y;
   memcpy(out, buf2, y);

   /* clear mem */
   zeromem(buf2, sizeof(buf2));
   return CRYPT_OK;
}

int dh_import(const unsigned char *in, dh_key *key)
{
   long x, y;

   /* check type byte */
   if (packet_valid_header((unsigned char *)in, PACKET_SECT_DH, PACKET_SUB_KEY) == CRYPT_ERROR) {
      return CRYPT_ERROR;
   }

   /* init */
   if (mp_init_multi(&key->x, &key->y, NULL) != MP_OKAY) { 
      crypt_error = "Out of memory in dh_import()."; 
      return CRYPT_ERROR; 
   }

   y = PACKET_SIZE;
   key->type = in[y++];
   key->idx  = in[y++];

   /* type check both values */
   if ((key->type != PK_PUBLIC) && (key->type != PK_PRIVATE))  {
      crypt_error = "Invalid type of key as input for dh_import().";
      goto error;
   }

   /* is the key idx valid? */
   if (!is_valid_idx(key->idx)) {
      crypt_error = "Invalid key idx found in input for dh_import().";
      goto error;
   }

   /* load public value g^x mod p*/
   INPUT_BIGNUM(&key->y, in, x, y);

   if (key->type == PK_PRIVATE) {
      INPUT_BIGNUM(&key->x, in, x, y);
   }
   return CRYPT_OK;
error:
   mp_clear_multi(&key->y, &key->x, NULL);
   return CRYPT_ERROR;
}

int dh_shared_secret(dh_key *private_key, dh_key *public_key, 
                     unsigned char *out, unsigned long *outlen)
{
   mp_int tmp, p;
   unsigned long x;
   int res;

   /* types valid? */
   if (private_key->type != PK_PRIVATE) {
      crypt_error = "Invalid key type passed to dh_shared_secret().";
      return CRYPT_ERROR;
   }

   /* same idx? */
   if (private_key->idx != public_key->idx) {
      crypt_error = "Key types don't match in dh_shared_secret().";
      return CRYPT_ERROR;
   }

   /* compute y^x mod p */
   if (mp_init_multi(&tmp, &p, NULL) != MP_OKAY) { 
      crypt_error = "Out of memory in dh_shared_secret()."; 
      return CRYPT_ERROR; 
   }

   if (mp_read_radix(&p, sets[private_key->idx].prime, 10) != MP_OKAY) { goto error; }
   if (mp_exptmod(&public_key->y, &private_key->x, &p, &tmp) != MP_OKAY) { goto error; }

   /* enough space for output? */
   x = mp_raw_size(&tmp);
   if (*outlen < x) {
      res = CRYPT_ERROR;
      crypt_error = "Buffer overrun in dh_shared_secret().";
      goto done;
   }
   mp_toraw(&tmp, out);
   *outlen = x;
   res = CRYPT_OK;
   goto done;
error:
   res = CRYPT_ERROR;
   crypt_error = "Out of memory in dh_shared_secret().";
done:
   mp_clear_multi(&p, &tmp, NULL);
   return res;
}

#include "dh_sys.c"

#endif

static const char *ID_TAG = "dh.c";


 
