#include <cassert>

#include <qcombobox.h>
#include <qlineedit.h>
#include <QMainWindow>
#include <qtimer.h>
#include <qspinbox.h>
#include <qstatusbar.h>
#include <qpushbutton.h>

#include <tagcoll/utils/set.h>
#include <ept/debtags/debtags.h>

// NUtil
#include <helpers.h>

// NPlugin
#include <iprovider.h>

// NPlugin
#include "relatedplugin.h"
#include "relatedinput.h"
#include "relatedfeedbackwidget.h"

// NUtil
#include "debtagshelper.h"

namespace NPlugin
{

const QString RelatedPlugin::PLUGIN_NAME = "RelatedPlugin";


RelatedPlugin::RelatedPlugin(const DebtagsPluginContainer& container)
	: _container(container)
{
	_pMainWindow = 0;
	_pProvider = 0;
	_pRelatedInput = 0;
	_pRelatedFeedbackWidget = 0;
	_isInactive = true;
}

RelatedPlugin::~RelatedPlugin()
{
	delete _pRelatedInput;
	delete _pRelatedFeedbackWidget;
}


/////////////////////////////////////////////////////
// Plugin Interface
/////////////////////////////////////////////////////

QString RelatedPlugin::title() const
{ 
	return tr("Similar Plugin"); 
}

QString RelatedPlugin::briefDescription() const
{
	return tr("Plugin for searching packages similar to another package.");
}

QString RelatedPlugin::description() const
{
	return tr("Plugin for searching packages similar to another package.");
}

void RelatedPlugin::init(IProvider* pProvider)
{
	_pProvider = pProvider;
	_pMainWindow = pProvider->mainWindow();
	
	_pRelatedInput = new RelatedInput(_pMainWindow, "RelatedInput");
	const set<string>& packages = pProvider->packages();
	for (set<string>::const_iterator it = packages.begin(); it != packages.end(); ++it)
		_pRelatedInput->_pPackageInput->addItem(toQString(*it));
	_pRelatedInput->_pPackageInput->setMinimumWidth(100);
	_pRelatedInput->_pPackageInput->setEditText("");
	connect(_pRelatedInput->_pPackageInput, SIGNAL(activated(const QString&)), SLOT(evaluateSearch()));
	connect(_pRelatedInput->_pClearButton, SIGNAL(clicked()), SLOT(onClearSearch()));
	
	_pRelatedFeedbackWidget = new RelatedFeedbackWidget(_pMainWindow, "RelatedFeedbackWidget");
	_pRelatedFeedbackWidget->setClearButton(
		pProvider->createClearButton(_pRelatedFeedbackWidget, "AptClearButton"), 0);
	connect(_pRelatedFeedbackWidget->_pClearButton, SIGNAL(clicked()), SLOT(onClearSearch()));

	_pRelatedFeedbackWidget->setShown(false);
	connect(
		_pRelatedInput->_pMaximumDistanceInput, SIGNAL(valueChanged(int)), 
		SLOT(evaluateSearch())
	);
	connect(
		_pRelatedInput->_pPackageInput, SIGNAL(textChanged(const QString&)), 
		SLOT(onInputTextChanged(const QString&))
	);
	if (_container.collection()==0)
		setWidgetsEnabled(false);
}

/////////////////////////////////////////////////////
// SearchPlugin Interface
/////////////////////////////////////////////////////

QString RelatedPlugin::inputWidgetTitle() const	
{
	return tr("Similar"); 
}

QWidget* RelatedPlugin::inputWidget() const
{ 
	return _pRelatedInput;
}

QWidget* RelatedPlugin::shortInputAndFeedbackWidget() const
{
	return _pRelatedFeedbackWidget;
}

void RelatedPlugin::clearSearch()
{
	_pRelatedInput->_pPackageInput->setEditText("");
}

const std::set<string>& RelatedPlugin::searchResult() const
{
	return _searchResult;
}

/////////////////////////////////////////////////////
// ScorePlugin Interface
/////////////////////////////////////////////////////
	
bool RelatedPlugin::offersScore() const 
{
	if (isInactive())
		return false;
 	string package = toString(_pRelatedInput->_pPackageInput->currentText());
	return !_pProvider->debtags().getTagsOfItem(NUtil::getPackageByName(package, _pProvider->apt())).empty();
}

map<string, float> RelatedPlugin::getScore(const set<string>& packages) const
{
	// the map that will be returned
	map<string, float> result;
	assert(!isInactive());
 	string package = toString(_pRelatedInput->_pPackageInput->currentText());
	int maxdist = _pRelatedInput->_pMaximumDistanceInput->value();
	std::set<Tag> ts = _pProvider->debtags().getTagsOfItem(NUtil::getPackageByName(package, _pProvider->apt()));
	for (set<string>::const_iterator it = packages.begin(); it != packages.end(); ++it)
	{
		int dist = tagcoll::utils::set_distance(ts,  _pProvider->debtags().getTagsOfItem(NUtil::getPackageByName(*it, _pProvider->apt())));
		result[*it] = float(maxdist - dist + 1) / (maxdist+1);
	}
	return result;
}


/////////////////////////////////////////////////////
// helper functions
/////////////////////////////////////////////////////

void RelatedPlugin::onInputTextChanged(const QString& text)
{
	if (text == "")
		evaluateSearch();
}

void RelatedPlugin::evaluateSearch()
{
	_pProvider->reportBusy(this, tr("Searching for similar packages"));
	_searchResult.clear();
	_pRelatedFeedbackWidget->_pSimilarSearchTextView->setText(
		_pRelatedInput->_pPackageInput->currentText()
	);
 	string package = toString(_pRelatedInput->_pPackageInput->currentText());
	_isInactive = package.empty();	// if no input was given the search is inactive
	if ( !_isInactive )
	{
		{
			int maxdist = _pRelatedInput->_pMaximumDistanceInput->value();
			
			Package packageEntity = NUtil::getPackageByName(package, _pProvider->apt());
			std::set<Tag> tags = _pProvider->debtags().getTagsOfItem(packageEntity);
			std::set<Package> result = _container.collection()->getRelatedItems(tags, maxdist);
			for (std::set<Package>::const_iterator it = result.begin(); it != result.end(); ++it)
			{
				_searchResult.insert(*it);
			}
		}
	}
	// show the feedback widget only if a search was entered
	_pRelatedFeedbackWidget->setShown(!_isInactive);
	_pProvider->reportReady(this);
	emit searchChanged(this);
}

void RelatedPlugin::debtagsDataChanged()
{
	if (_container.collection()==0)
		setWidgetsEnabled(false);
	else
		setWidgetsEnabled(true);
	clearSearch();
}

void RelatedPlugin::setWidgetsEnabled(bool enabled)
{
	if (_pRelatedInput)
		_pRelatedInput->setEnabled(enabled);
}
 
}	// namespace NPlugin

#undef emit
#include <ept/debtags/debtags.tcc>
