
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk.h 1441 2006-11-02 19:32:15Z mschwerin $
 *
 */

#ifndef HAVE_OTK_H
#define HAVE_OTK_H
#ifdef __cplusplus
extern "C" {
#endif

#include "event.h"
#include "odk.h"

/*
 * ***************************************************************************
 * Description:     alignments for text and buttons
 * ***************************************************************************
 */
#define OTK_ALIGN_LEFT              (0x001)
#define OTK_ALIGN_CENTER            (0x002)
#define OTK_ALIGN_RIGHT             (0x004)
#define OTK_ALIGN_TOP               (0x008)
#define OTK_ALIGN_VCENTER           (0x010)
#define OTK_ALIGN_BOTTOM            (0x020)

/*
 * ***************************************************************************
 * Description:     how can a widget be selected
 * ***************************************************************************
 */
#define OTK_SELECTABLE_NONE         (0x000)
#define OTK_SELECTABLE_KEY          (0x001)
#define OTK_SELECTABLE_MOUSE        (0x010)

/*
 * ***********************************************************************************
 * opaque data types
 * ***********************************************************************************
 */
typedef struct otk_s otk_t;
typedef struct otk_widget_s otk_widget_t;

/*
 * ***********************************************************************************
 * user contributed callbacks
 * ***********************************************************************************
 */
typedef void (*otk_cb_t) (void *user_data);

typedef int (*otk_int_get_cb_t) (void *user_data);
typedef int (*otk_int_set_cb_t) (void *user_data, int position);

typedef void (*otk_label_uc_t) (void *user_data, otk_widget_t * widget);
typedef void (*otk_editbox_cb_t) (void *user_data, char *text);

/*
 * ***********************************************************************************
 * Editbox-Widget 
 * ***********************************************************************************
 */
otk_widget_t *otk_editbox_new (otk_t * otk, int x, int y, int w, int h,
                               int max_length, otk_editbox_cb_t cb,
                               void *cb_data);
void otk_editbox_set_text (otk_widget_t * editbox, const char *text);
char *otk_editbox_get_text (otk_widget_t * editbox);

/*
 * ***********************************************************************************
 * Button-Widget 
 * ***********************************************************************************
 */
otk_widget_t *otk_text_button_new (otk_t * otk, int x, int y, int w, int h,
                                   const char *text,
                                   otk_cb_t cb, void *cb_data);
otk_widget_t *otk_bitmap_button_new (otk_t * otk, int x, int y, int w, int h,
                                     uint8_t * bitmap,
                                     otk_cb_t cb, void *cb_data);
otk_widget_t *otk_image_button_new (otk_t * otk, int x, int y, int w, int h,
                                    const char *mrl,
                                    const char *focus_mrl,
                                    const char *text_replace,
                                    const char *text_over,
                                    otk_cb_t cb, void *cb_data);
void otk_button_set_text (otk_widget_t * widget, const char *text);

/*
 * ***********************************************************************************
 * Slider-Widget 
 * ***********************************************************************************
 */

#define OTK_SLIDER_VERTICAL         (0)
#define OTK_SLIDER_HORIZONTAL       (1)

otk_widget_t *otk_slider_new (otk_t * otk, int x, int y, int w, int h,
                              int marker_w, int marker_h, bool simple_marker,
                              int direction, const char *value_format,
                              bool with_buttons, bool with_reset_button, 
                              int default_value, int min_value, int max_value,
                              int min_step, otk_int_get_cb_t get_value_cb,
                              void *get_value_cb_data,
                              otk_int_set_cb_t set_value_cb,
                              void *set_value_cb_data);

/*
 * ***********************************************************************************
 * Checkbox-Widget 
 * ***********************************************************************************
 */
otk_widget_t *otk_checkbox_new (otk_t * otk, int x, int y, int w,
                                bool is_checked, otk_cb_t cb,
                                void *cb_data);

/*
 * ***********************************************************************************
 * ListEntry-Widget
 * ***********************************************************************************
 */
otk_widget_t *otk_listentry_new (otk_widget_t * list, const char *text,
                                 otk_cb_t activate_cb,
                                 void *activate_cb_data,
                                 otk_cb_t select_cb,
                                 void *select_cb_data,
                                 otk_cb_t remove_cb,
                                 void *remove_cb_data);

/*
 * ***********************************************************************************
 * List-Widget
 * ***********************************************************************************
 */
#define OTK_LIST_NO_SELECTION           (0)
#define OTK_LIST_SINGLE_SELECTION       (1)
#define OTK_LIST_MULTIPLE_SELECTION     (2)

otk_widget_t * otk_list_new (otk_t * otk, int x, int y, int w, int h, 
                             int entry_height, int entry_spacing, 
                             bool show_border, bool show_scrollbar, 
                             int allow_select, void *list_cb_data);
void otk_clear_list (otk_widget_t * list);
int otk_list_get_pos (otk_widget_t * list);
void otk_list_set_pos (otk_widget_t * list, int pos);
int otk_list_get_focus (otk_widget_t * list);
void otk_list_set_focus (otk_widget_t * list, int pos);
int otk_list_get_length (otk_widget_t * list);

int *otk_list_get_selected_pos (otk_widget_t * list, int *num_selected);
void **otk_list_get_selected (otk_widget_t * list, int *num_selected);
void otk_list_set_selected (otk_widget_t * list, int pos, bool selected);
void otk_list_clear_selection (otk_widget_t * list);
int otk_list_get_selected_count (otk_widget_t * list);
void otk_list_select_all (otk_widget_t * list);

/*
 * ***********************************************************************************
 * Selector-Widget
 * ***********************************************************************************
 * This is a list, that only shows one entry.
 * ***********************************************************************************
 * The callback can be used to inform the application, that the visible entry
 * has changed.
 * ***********************************************************************************
 */
otk_widget_t *otk_selector_new (otk_t * otk, int x, int y, int w,
                                otk_int_set_cb_t cb, void *cb_data);

/*
 * ***********************************************************************************
 * Label-Widget
 * ***********************************************************************************
 */
otk_widget_t *otk_label_new (otk_t * otk, int x, int y, int w,
                             int alignment, const char *text);

void otk_label_set_text (otk_widget_t * widget, const char *text);
void otk_label_set_upcall (otk_widget_t * widget, otk_label_uc_t upcall,
                           void *data);

/*
 * ***********************************************************************************
 * Scrollbar-Widget
 * ***********************************************************************************
 */
otk_widget_t *otk_scrollbar_new (otk_t * otk, int x, int y, int w, int h,
                                 otk_int_set_cb_t cb_click,
                                 otk_cb_t cb_up,
                                 otk_cb_t cb_down, void *cb_data);
void otk_scrollbar_set (otk_widget_t * scrollbar_p, int position, int length);

/*
 * ***********************************************************************************
 * Border-Widget
 * ***********************************************************************************
 */
otk_widget_t *otk_border_new (otk_t * otk, int x, int y, int w, int h);

/*
 * ***********************************************************************************
 * Window-Widget
 * ***********************************************************************************
 */
otk_widget_t *otk_window_new (otk_t * otk, int x, int y, int w, int h,
                              bool show_border, bool fill);
void otk_window_set_current (otk_t * otk, otk_widget_t * window);
void otk_window_keep (otk_widget_t * window, bool keep);
void otk_window_destroy (otk_widget_t * window);

/*
 * ***********************************************************************************
 * Global Functions
 * ***********************************************************************************
 */
void otk_draw (otk_t * otk);
void otk_clear (otk_t * otk);

void otk_widget_set_focus (otk_widget_t * widget);
void otk_widget_unset_focus (otk_widget_t * widget);

void otk_widget_set_focus_callbacks (otk_widget_t * widget,
                                     otk_cb_t enter_cb, void *enter_cb_data,
                                     otk_cb_t leave_cb, void *leave_cb_data);

/* on 1, update this widget continously (every 100 ms) */
void otk_widget_set_update (otk_widget_t * widget, bool update);
void otk_widget_set_font (otk_widget_t * widget, const char *font,
                          int fontsize);
void otk_widget_set_alignment (otk_widget_t * widget, int alignment);
void otk_widget_set_enabled (otk_widget_t * widget, bool enabled);
void otk_widget_set_visible (otk_widget_t * widget, bool visible);
void otk_widget_set_selectable (otk_widget_t * widget, int selectable);

otk_t *otk_init (odk_t * odk);
void otk_free (otk_t * otk);

/*
 * ***********************************************************************************
 * Eventhandlers
 * ***********************************************************************************
 */
void otk_add_event_handler (otk_t * otk,
                            void (*cb) (void *data, oxine_event_t * ev),
                            void *data);
void otk_send_event (otk_t * otk, oxine_event_t * event);

#ifdef __cplusplus
}
#endif
#endif /* HAVE_OTK_H */
