
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_private.h 1574 2006-11-19 20:48:44Z mschwerin $
 *
 * ***************************************************************************
 * This file contains private types and methods used inside the odk system.
 * This file should not be included outside the odk system.
 * ***************************************************************************
 */

#ifndef HAVE_ODK_PRIVATE_H
#define HAVE_ODK_PRIVATE_H
#ifdef __cplusplus
extern "C" {
#endif

#include "odk.h"
#include "playlist.h"
#include "odk_plugin.h"

#ifdef HAVE_IMAGEMAGICK
#define HAVE_OSD_IMAGE
#endif

/* 
 * ***************************************************************************
 * Description:     Number of colors (defined by the xine engine).
 * ***************************************************************************
 */
#define NUM_COLORS 256

/* 
 * ***************************************************************************
 * Description:     Size of the image cache.
 * ***************************************************************************
 */
#define NUM_IMAGE_CACHE_ENTRIES 10

/* 
 * ***************************************************************************
 * Description:     A palette containing max. 256 colors
 * ***************************************************************************
 */
typedef struct {
    int num_colors;
    uint32_t colors[NUM_COLORS];
    uint8_t transparency[NUM_COLORS];
} odk_palette_t;

/* 
 * ***************************************************************************
 * Description:     A palette containing max. 256 colors
 * ***************************************************************************
 */
typedef struct {
    event_handler_t event_handler;
    void *event_handler_data;
    odk_event_handler_priority_t priority;
} odk_event_handler_t;

/* 
 * ***************************************************************************
 * Description:     The OSD information.
 * ***************************************************************************
 */
typedef struct {
    xine_osd_t *x_osd;

    odk_palette_t palette;

    int x_off;
    int y_off;

    int width;
    int height;

    double vscale;
    double hscale;
} odk_osd_t;

#ifdef HAVE_OSD_IMAGE

/* 
 * ***************************************************************************
 * Description: The OSD image object.
 * ***************************************************************************
 *
 * We support two methods to display an image. 
 *
 * 1) The image is stored in a pixel array (data). The colors are saved in the 
 *    standard palette. The image is drawn onto the standard OSD drawing
 *    surface. (odk_osd_draw_image)
 *
 * 2) The image is drawn onto a separate OSD drawing surface (x_osd). The
 *    colors are stored in that OSD. (odk_osd_show_image)
 *
 * No matter what method is used, the image is saved in the image cache.
 * Rescaling is automatically done when adapting the OSD size.
 *
 * ***************************************************************************
 */
struct odk_osd_image_s {
    char *mrl;

    uint8_t *data;
    xine_osd_t *x_osd;

    bool is_visible;

    int x;
    int y;

    int w;
    int h;

    int alignment;

    bool border_draw;
    uint32_t border_color;
};

/* 
 * ***************************************************************************
 * Description:     The image cache.
 * ***************************************************************************
 */
typedef struct {
    /* The time this entry was last used. */
    time_t timestamp;
    /* The image. */
    odk_osd_image_t *image;
} image_cache_entry_t;

typedef struct {
    /* The number of entries in the cache. */
    int fill;
    /* The cache entries. */
    image_cache_entry_t entries[NUM_IMAGE_CACHE_ENTRIES];
} odk_image_cache_t;
#endif

/*
 * ***************************************************************************
 * Description:     The ODK class structure.
 * ***************************************************************************
 */
struct odk_s {
    xine_t *xine;

    /* The onscreen display */
    odk_osd_t osd;

    /* The image cache */
#ifdef HAVE_OSD_IMAGE
    odk_image_cache_t image_cache;
#endif

    /* The output window. */
    odk_window_t *win;

    /* The event handlers. */
    l_list_t *event_handlers;

    /* The input plugins for joystick and LIRC support. */
#ifdef HAVE_LIRC
    odk_input_t *lirc;
#endif
#ifdef HAVE_JOYSTICK
    odk_input_t *joystick;
#endif

    /* Configuration */
    int visual_anim_style;
    int aspect_ratio;
    int encode_mrls;

    /* Some status variables. */
    bool is_seeking_in_stream;
    bool is_osd_visible;
    bool is_current_font_freetype;
    bool is_unscaled_osd_available;
    bool forward_events_to_xine;

    /* Info about current stream */
    odk_mode_t current_mode;
    char *current_mrl;
    char *current_subtitle_mrl;
    char *current_title;
    playlist_t *current_alternatives;
    char *current_background_mrl;

    /* Post plugin and background/ animation streams for streams that don't
     * have video. */
    xine_post_t *novideo_post;

    xine_stream_t *background_stream;
    const char *background_stream_mrl;

    xine_stream_t *animation_stream;
    const char *animation_stream_mrl;

    /* Subtitle stream */
    xine_stream_t *subtitle_stream;

    /* A mutex we use to allow only one instance of mutexed_play to be run at
     * a time. */
#ifdef DEBUG_THREADS
    pthread_mutexattr_t play_mutex_attr;
#endif
    pthread_mutex_t play_mutex;
};

/* Post plugins */
void odk_post_init (odk_t * odk);
void odk_post_audio_rewire (odk_t * odk);
void odk_post_audio_unwire (odk_t * odk);

/* OSD */
void odk_osd_init (odk_t * odk);
void odk_osd_free (odk_t * odk);

void odk_get_frame_size (odk_t * odk, int *w, int *h);

void odk_osd_adapt_size (odk_t * odk, xine_format_change_data_t *fcd);

/* Event handling */
void odk_event_init (odk_t * odk);
void odk_event_free (odk_t * odk);

#ifdef __cplusplus
}
#endif
#endif /* HAVE_ODK_PRIVATE_H */
