
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: disc.c 1541 2006-11-13 16:25:16Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <stddef.h>
#include <stdio.h>
#include <mntent.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef HAVE_LINUX_CDROM_H
#include <linux/cdrom.h>
#endif

#include "disc.h"
#include "disc_iso9660.h"
#include "environment.h"
#include "filelist.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "meta_info.h"
#include "mutex.h"
#include "oxine.h"

#ifndef HAVE_LINUX_CDROM_H
#undef HAVE_DISC_POLLING
#endif

extern oxine_t *oxine;

typedef struct {
    device_type_t type;
    char file[128];
} cdrom_files_t;

static const cdrom_files_t cdrom_files[] = {
    {DEVICE_TYPE_VIDEO_CD, "vcd/info.vcd"},
    {DEVICE_TYPE_VIDEO_CD, "svcd/info.svd"},
    {DEVICE_TYPE_VIDEO_DVD, "video_ts"},
    {DEVICE_TYPE_VIDEO_DVD, "VIDEO_TS"},
    {DEVICE_TYPE_VIDEO_DVD, "audio_ts"},
    {DEVICE_TYPE_VIDEO_DVD, "AUDIO_TS"},
    {0, ""}
};


static void
device_update_current_mrl (device_entry_t * entry)
{
    ho_free (entry->current_mrl);
    entry->current_mrl = NULL;

    switch (entry->current_type) {
    case DEVICE_TYPE_AUDIO_CD:
        {
            entry->current_mrl = ho_strdup_printf ("cdda:/%s", entry->device);
        }
        break;
    case DEVICE_TYPE_VIDEO_CD:
        {
            entry->current_mrl = ho_strdup_printf ("vcd://%s", entry->device);
        }
        break;
    case DEVICE_TYPE_VIDEO_DVD:
        {
            entry->current_mrl = ho_strdup_printf ("dvd://%s", entry->device);
        }
        break;
    case DEVICE_TYPE_CDROM:
    default:
        if (entry->mountpoint) {
            entry->current_mrl = ho_strdup (entry->mountpoint);
        }
        break;
    }
}


static void
device_update_current_type (device_entry_t * entry)
{
    entry->current_type = DEVICE_TYPE_UNKNOWN;

    if (!entry->is_type_cd)
        return;

#ifdef HAVE_LINUX_CDROM_H
    int fd = open (entry->device, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
        error (_("Could not open '%s': %s!"), entry->device,
               strerror (errno));
        return;
    }

    int drive_status = ioctl (fd, CDROM_DRIVE_STATUS);
    switch (drive_status) {
    case CDS_NO_INFO:
    case CDS_NO_DISC:
    case CDS_TRAY_OPEN:
    case CDS_DRIVE_NOT_READY:
        entry->current_type = DEVICE_TYPE_NONE;
        break;
    default:
        break;
    }

    int disc_status = ioctl (fd, CDROM_DISC_STATUS);
    switch (disc_status) {
    case CDS_AUDIO:
        entry->current_type = DEVICE_TYPE_AUDIO_CD;
        break;
    case CDS_DATA_1:
    case CDS_DATA_2:
        entry->current_type = DEVICE_TYPE_CDROM;
        break;
    case CDS_XA_2_1:
    case CDS_XA_2_2:
    case CDS_MIXED:
    case CDS_NO_INFO:
    default:
        break;
    }
    close (fd);

    if ((entry->current_type == DEVICE_TYPE_CDROM)
        && entry->mountpoint && disc_mount (entry->device)) {
        const cdrom_files_t *files = cdrom_files;
        while (files->type) {
            char filename[1024];
            snprintf (filename, 1024, "%s/%s", entry->mountpoint,
                      files->file);
            if (access (filename, F_OK) == 0) {
                entry->current_type = files->type;
                break;
            }
            files++;
        }
        disc_umount (entry->device);
    }
#endif
}


device_type_t
disc_get_type (const char *device)
{
    /* First we try to find the device in the list of devices we already know
     * about. */
    device_entry_t *device_entry = devicelist_first ();
    while (device_entry) {
        if (strcmp (device_entry->device, device) == 0) {
            device_update_current_type (device_entry);
            return device_entry->current_type;
        }
        device_entry = devicelist_next (device_entry);
    }

    /* Next we have a look to see if the device is a softlink. If it is we
     * resolve it and recursively call ourself again. */
    device_type_t type = DEVICE_TYPE_UNKNOWN;
    char *target = resolve_softlink (device);
    if (strcmp (target, device) != 0) {
        type = disc_get_type (target);
    }
    ho_free (target);

    return type;
}


static char *
_disc_get_title (device_entry_t * entry)
{
    char *title = NULL;

    char *disc_title = NULL;
    char *disc_type = NULL;

    switch (entry->current_type) {
    case DEVICE_TYPE_AUDIO_CD:
        disc_type = _("Audio CD");
        {
            char mrl[1024];
            snprintf (mrl, 1024, "cdda:/%s/1", entry->device);

            char *album = meta_info_get_from_mrl (mrl, META_INFO_ALBUM);
            char *artist = meta_info_get_from_mrl (mrl, META_INFO_ARTIST);

            if (album && artist) {
                disc_title = ho_strdup_printf ("%s - %s", artist, album);
                ho_free (album);
                ho_free (artist);
            }
        }
        break;
    case DEVICE_TYPE_VIDEO_CD:
        disc_type = _("Video CD");
        disc_title = meta_info_get_from_mrl (entry->current_mrl,
                                             META_INFO_TITLE);
        break;
    case DEVICE_TYPE_VIDEO_DVD:
        disc_type = _("Video DVD");
        disc_title = meta_info_get_from_mrl (entry->current_mrl,
                                             META_INFO_TITLE);
        break;
    case DEVICE_TYPE_CDROM:
        disc_type = _("CD-ROM");
        disc_title = disc_get_title_iso9660 (entry->device);
        break;
    case DEVICE_TYPE_NONE:
        disc_title = ho_strdup (_("No disc"));
        break;
    default:
        break;
    }

    if (!disc_type)
        disc_type = _("Removable Disc");
    if (!disc_title)
        disc_title = ho_strdup (_("Unknown Disc"));

    title = ho_strdup_printf ("[%s: %s]", disc_type, disc_title);

    ho_free (disc_title);

    return title;
}


char *
disc_get_title (const char *device)
{
    /* First we try to find the device in the list of devices we already know
     * about. */
    device_entry_t *device_entry = devicelist_first ();
    while (device_entry) {
        if (strcmp (device_entry->device, device) == 0) {
            device_update_current_type (device_entry);
            device_update_current_mrl (device_entry);
            return _disc_get_title (device_entry);
        }
        device_entry = devicelist_next (device_entry);
    }

    /* Next we have a look to see if the device is a softlink. If it is we
     * resolve it and recursively call ourself again. */
    char *title = NULL;
    char *target = resolve_softlink (device);
    if (strcmp (target, device) != 0) {
        title = disc_get_title (target);
    }
    ho_free (target);

    if (!title) {
        char *disc_type = _("Removable Disc");
        char *disc_title = _("Unknown Disc");
        title = ho_strdup_printf ("[%s: %s]", disc_type, disc_title);
    }

    return title;
}


bool
disc_is_mounted (const char *device)
{
    FILE *f = fopen ("/proc/mounts", "r");
    if (!f) {
        error (_("Could not open '%s': %s!"),
               "/proc/mounts", strerror (errno));
        return false;
    }

    bool mounted = false;
    char line[256];
    while (fgets (line, 255, f) != NULL) {
        if (strstr (line, device) != NULL) {
            mounted = true;
            break;
        }
    }
    fclose (f);

    return mounted;
}


bool
disc_is_present (const char *device)
{
#ifdef HAVE_LINUX_CDROM_H
    int fd = open (device, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
        error (_("Could not open '%s': %s!"), device, strerror (errno));
        return false;
    }

    bool present = false;
    int drive_status = ioctl (fd, CDROM_DRIVE_STATUS);
    switch (drive_status) {
    case CDS_NO_INFO:
        debug ("drive reports: no info");
        break;
    case CDS_NO_DISC:
        debug ("drive reports: no disc");
        break;
    case CDS_TRAY_OPEN:
        debug ("drive reports: tray open");
        break;
    case CDS_DRIVE_NOT_READY:
        debug ("drive reports: drive not ready");
        break;
    case CDS_DISC_OK:
        debug ("drive reports: disc ok");
        present = true;
        break;
    default:
        debug ("drive reports: unknown value");
        break;
    }
    close (fd);

    return present;
#else
    return false;
#endif
}


bool
disc_mount (const char *device)
{
    if (disc_is_mounted (device))
        return true;

#ifdef BIN_MOUNT
    char command[256];
    snprintf (command, 256, "%s %s", BIN_MOUNT, device);
    debug (command);

    if (system (command)) {
        error (_("Could not mount '%s': %s!"), device, _("Unknown"));
        return false;
    }

    return true;
#else
    return false;
#endif
}


bool
disc_umount (const char *device)
{
#ifdef BIN_UMOUNT
    char command[256];
    snprintf (command, 256, "%s %s", BIN_UMOUNT, device);
    debug (command);

    if (system (command)) {
        error (_("Could not unmount '%s': %s!"), device, _("Unknown"));
        return false;
    }

    return true;
#else
    return false;
#endif
}


bool
disc_eject (const char *device)
{
#if defined BIN_EJECT
    char command[256];
    snprintf (command, 256, "%s %s", BIN_EJECT, device);
    debug (command);

    if (system (command)) {
        error (_("Could not eject '%s': %s!"), device, _("Unknown"));
        return false;
    }

    return true;
#elif defined HAVE_LINUX_CDROM_H
    int fd = open (device, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
        error (_("Could not eject '%s': %s!"), device, strerror (errno));
        return false;
    }

    if (ioctl (fd, CDROMEJECT) < 0) {
        error (_("Could not eject '%s': %s!"), device, strerror (errno));
        close (fd);
        return false;
    }

    close (fd);
    return true;
#else
    return false;
#endif
}


static void
disc_autoscan (playlist_t * playlist, filelist_t * filelist, const char *mrl)
{
    // We only handle audio CDs here.
    assert (strncasecmp (mrl, "cdda", 4) == 0);

    int num_mrls;
    char **autoplay_mrls =
        xine_get_autoplay_mrls (oxine->xine, "CD", &num_mrls);

    int use_cddb = 1;
    if (autoplay_mrls) {
        int i;
        for (i = 0; i < num_mrls; i++) {
            char *track_title = NULL;
            char *track_mrl = ho_strdup_printf ("%s/%d", mrl, i + 1);

            if (use_cddb) {
                char *title = meta_info_get_from_mrl (track_mrl,
                                                      XINE_META_INFO_TITLE);

                if (title) {
                    track_title =
                        ho_strdup_printf ("%02d - %s", i + 1, title);
                    ho_free (title);
                    use_cddb = 1;
                } else {
                    // If getting the title from CDDB fails once we stop trying.
                    use_cddb = 0;
                }
            }

            if (!track_title)
                track_title =
                    ho_strdup_printf ("%s - %s %02d", _("Audio CD"),
                                      _("Title"), i + 1);

            if (playlist)
                playlist_add (playlist, track_title, track_mrl);
            if (filelist)
                filelist_add (filelist, track_title, track_mrl,
                              FILE_TYPE_REGULAR);

            ho_free (track_mrl);
            ho_free (track_title);
        }
    }
}


void
disc_autoscan_load (playlist_t * playlist, const char *mrl)
{
    disc_autoscan (playlist, NULL, mrl);
}


void
disc_autoscan_read (filelist_t * filelist)
{
    disc_autoscan (NULL, filelist, filelist->mrl);
}


#ifdef HAVE_DISC_POLLING
static struct {
    pthread_t thread;
    int continue_running;
} disc_thread_info;


static bool
device_add (device_entry_t * entry)
{
    /* we update the device type and the device MRL */
    device_update_current_type (entry);
    device_update_current_mrl (entry);

    if (!entry->current_mrl)
        return false;

    if (entry->current_type == DEVICE_TYPE_NONE)
        return false;

    if (filelist_contains (oxine->removable_discs, entry->current_mrl))
        return false;

    char *disc_title = _disc_get_title (entry);
    char *disc_mrl = entry->current_mrl;

    fileitem_t *item = NULL;

    switch (entry->current_type) {
    case DEVICE_TYPE_AUDIO_CD:
        debug ("an audio CD was inserted at %s", entry->device);
        item = filelist_add (oxine->removable_discs, disc_title,
                             disc_mrl, FILE_TYPE_AUTODISC);
        filelist_expand (item);
        break;
    case DEVICE_TYPE_VIDEO_DVD:
        debug ("a video DVD was inserted at %s", entry->device);
        item = filelist_add (oxine->removable_discs, disc_title,
                             disc_mrl, FILE_TYPE_REGULAR);
        break;
    case DEVICE_TYPE_VIDEO_CD:
        debug ("a VCD was inserted at %s", entry->device);
        item = filelist_add (oxine->removable_discs, disc_title,
                             disc_mrl, FILE_TYPE_REGULAR);
        break;
    case DEVICE_TYPE_CDROM:
        debug ("a CDROM was inserted at %s", entry->device);
        item = filelist_add (oxine->removable_discs, disc_title,
                             disc_mrl, FILE_TYPE_MOUNTPOINT);
        break;
    default:
        debug ("found new device at %s", entry->device);
        item = filelist_add (oxine->removable_discs, disc_title,
                             disc_mrl, FILE_TYPE_MOUNTPOINT);
        break;
    }

    item->user_data = entry;
    ho_free (disc_title);

    debug ("added disc %s", entry->device);

    return true;
}


static void
sublist_clear (filelist_t * filelist)
{
    if (!filelist)
        return;

    fileitem_t *item = filelist_first (filelist);
    while (item) {
        sublist_clear (item->sublist);
        item = filelist_next (filelist, item);
    }

    filelist_clear (filelist);
}


static bool
device_del (device_entry_t * entry)
{
    fileitem_t *item = filelist_first (oxine->removable_discs);
    while (item) {
        if (item->user_data == entry) {
            /* This is a bit of a nuissance! If we're currently displaying the
             * sublist of the item we're about to remove that sublist cannot
             * be freed as the reference counter is still > 0. But as the disc
             * is no longer in the drive we definitely do not want the user to
             * see it's contents anymore. So we have to clear it before
             * removing this item. */
            sublist_clear (item->sublist);
            filelist_remove (oxine->removable_discs, item);
            debug ("removed disc %s", entry->device);
            return true;
        }
        item = filelist_next (oxine->removable_discs, item);
    }

    return false;
}


static void
device_poll (device_entry_t * entry)
{
    bool repaint = false;

    /* We assume that for a removable drive to be attached the device file has
     * to exist and has to be readable. */
    bool is_disc_in = (access (entry->device, R_OK) == 0);

    /* If this device is a CDROM/ DVD drive we also have to see if there is a
     * disc in the drive. */
    if (is_disc_in && entry->is_type_cd) {
        int fd = open (entry->device, O_RDONLY | O_NONBLOCK);
        is_disc_in = (fd && (ioctl (fd, CDROM_DRIVE_STATUS) == CDS_DISC_OK));

        if (fd) {
            close (fd);
        } else {
            error (_("Could not open '%s': %s!"),
                   entry->device, strerror (errno));
            /* If we fail to open the device, we quit neither removing nor
             * adding this disc. */
            return;
        }
    }

    /* We want to make sure that we're not adding or removing anything while
     * another thread is updating the GUI using this data. */
    mutex_lock (&oxine->removable_discs->mutex);

    if (!is_disc_in) {
        if (entry->is_disc_in && device_del (entry)) {
            entry->is_disc_in = false;
            repaint = true;
        }
    }

    else {
        if (!entry->is_disc_in && device_add (entry)) {
            entry->is_disc_in = true;
            repaint = true;
        }
    }

    mutex_unlock (&oxine->removable_discs->mutex);

    /* If anything changed, we update the GUI. */
    if (repaint) {
        current_menu_cb (oxine);
    }
}


static void *
disc_thread (void *data)
{
    while (disc_thread_info.continue_running) {
        device_entry_t *entry = devicelist_first ();
        while (entry) {
            device_poll (entry);
            entry = devicelist_next (entry);
        }

        sleep (1);
    }

    pthread_exit (NULL);
    return NULL;
}


void
disc_start_polling (oxine_t * oxine)
{
    disc_thread_info.continue_running = 1;

    if (pthread_create (&disc_thread_info.thread, NULL,
                        disc_thread, NULL) != 0) {
        error (_("Could not create disc polling thread: %s!"),
               strerror (errno));
    } else {
        info (_("Successfully started disc polling thread."));
    }
}


void
disc_stop_polling (oxine_t * oxine)
{
    disc_thread_info.continue_running = 0;

    void *ret;
    pthread_join (disc_thread_info.thread, &ret);
    info (_("Successfully stopped disc polling thread."));
}
#endif
