///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/VariantComboBoxPropertyUI.h>
#include <core/undo/UndoManager.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(VariantComboBoxPropertyUI, PropertyParameterUI)

/******************************************************************************
* The constructor.
******************************************************************************/
VariantComboBoxPropertyUI::VariantComboBoxPropertyUI(PropertiesEditor* parentEditor, const char* propertyName) : 
	PropertyParameterUI(parentEditor, propertyName), _comboBox(NULL)
{
	// Create UI widget.
	_comboBox = new QComboBox();
	connect(_comboBox, SIGNAL(activated(int)), this, SLOT(updatePropertyValue()));	
}

/******************************************************************************
* Destructor, that releases all GUI controls.
******************************************************************************/
VariantComboBoxPropertyUI::~VariantComboBoxPropertyUI()
{
	// Release GUI controls. 
	delete comboBox(); 
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to. 
******************************************************************************/
void VariantComboBoxPropertyUI::resetUI()
{
	PropertyParameterUI::resetUI();	
	
	if(comboBox()) 
		comboBox()->setEnabled(editObject() != NULL && isEnabled());
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to. 
******************************************************************************/
void VariantComboBoxPropertyUI::updateUI()
{
	PropertyParameterUI::updateUI();	
	
	if(comboBox() && editObject()) {
		QVariant val = editObject()->property(propertyName());
		OVITO_ASSERT_MSG(val.isValid(), "VariantComboBoxPropertyUI::updateUI()", QString("The object class %1 does not define a property with the name %2.").arg(editObject()->metaObject()->className(), QString(propertyName())).toLocal8Bit().constData());
		if(!val.isValid()) {
			throw Exception(tr("The object class %1 does not define a property with the name %2.").arg(editObject()->metaObject()->className(), QString(propertyName())));
		}
		comboBox()->setCurrentIndex(comboBox()->findData(val));
		if(comboBox()->isEditable())
			comboBox()->setEditText(val.toString());
	}
}

/******************************************************************************
* Sets the enabled state of the UI.
******************************************************************************/
void VariantComboBoxPropertyUI::setEnabled(bool enabled)
{
	if(enabled == isEnabled()) return;
	PropertyParameterUI::setEnabled(enabled);
	if(comboBox()) comboBox()->setEnabled(editObject() != NULL && isEnabled());
}

/******************************************************************************
* Takes the value entered by the user and stores it in the property field 
* this property UI is bound to.
******************************************************************************/
void VariantComboBoxPropertyUI::updatePropertyValue()
{
	if(comboBox() && editObject() && comboBox()->currentIndex() >= 0) {
		
		UNDO_MANAGER.beginCompoundOperation(tr("Change Parameter"));			
		 
		QVariant newValue;
		if(comboBox()->isEditable())
			newValue = comboBox()->currentText();
		else
			newValue = comboBox()->itemData(comboBox()->currentIndex());
			
		if(!editObject()->setProperty(propertyName(), newValue)) {
			OVITO_ASSERT_MSG(false, "VariantComboBoxPropertyUI::updatePropertyValue()", QString("The value of property %1 of object class %2 could not be set.").arg(QString(propertyName()), editObject()->metaObject()->className()).toLocal8Bit().constData());
		}
		
		UNDO_MANAGER.endCompoundOperation();
	}
}

};

