///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file DataSet.h
 * \brief Contains definition of the Core::DataSet class.
 */

#ifndef __OVITO_DATASET_H
#define __OVITO_DATASET_H

#include <core/Core.h>
#include <core/viewport/ViewportConfiguration.h>
#include <core/reference/RefTarget.h>
#include <core/scene/animation/AnimationSettings.h>
#include <core/scene/SceneRoot.h>

namespace Core {

class SelectionSet;		// defined in SelectionSet.h
class RenderSettings;	// defined in RenderSettings.h

/**
 * \brief This class combines everything that belongs to a scene.
 *
 * A DataSet contains all data that belongs to the current scene edited by the user.
 * It can be saved to and loaded from a file.
 *
 * \author Alexander Stukowski
 * \sa DataSetManager
 */
class CORE_DLLEXPORT DataSet : public RefTarget
{
public:

	/// \brief Constructs an empty dataset.
	/// \param isLoading Indicates whether the object is being loaded from a file.
	///                  This parameter is only used by the object serialization system.
	DataSet(bool isLoading = false);

	/// \brief Returns a reference to the viewport configuration associated with this dataset.
	/// \return The internal object that contains the configuration of the viewports. This object is saved
	///         to the scene file and is used to restore the original viewport configuration
	///         when the scene file is loaded.
	/// \note Changing the viewport configuration object does not have any effect because
	///       It is automatically regenerated just before the dataset is saved to a file.
	ViewportConfiguration* viewportConfig() const { return _viewportConfig; }

	/// \brief Returns the animation settings.
	/// \return The internal object that stores the animation settings for the scene.
	/// \sa AnimManager
	AnimationSettings* animationSettings() const { return _animSettings; }

	/// \brief Returns this dataset's root scene node.
	/// \return The root node of the scene tree.
	/// \sa setSceneRoot()
	SceneRoot* sceneRoot() const { return _sceneRoot; }

	/// \brief Sets the dataset's root scene node.
	/// \param newScene The new scene tree. It will completely replace the old
	///                 scene object tree.
	/// \undoable
	/// \sa sceneRoot()
	void setSceneRoot(const SceneRoot::SmartPtr& newScene) { _sceneRoot = newScene; }

	/// \brief Returns the selection set.
	/// \return The internal selection set used to store the set of selected scene nodes.
	SelectionSet* selection() const { return _selection; }

	/// \brief Returns the general rendering settings for this scene.
	/// \return The internal object that stores the rendering settings.
	RenderSettings* renderSettings() const { return _renderSettings; }

	/// \brief Returns the state of the dirty flag that indicates whether the dataset has
	///        been changed and needs to be saved to disk.
	/// \return \c true if this dataset has been changed since the last save point.
	///
	/// The dirty flag will automatically be set when some subobject of the DataSet
	/// is being changed.
	///
	/// \sa setDirty()
	bool hasBeenChanged() const { return _hasBeenChanged; }

	/// \brief Marks the dataset as dirty or resets the dirty flag.
	/// \param dirty Speicifies whether the dirty flag should be set or reset.
	/// \sa hasBeenChanged()
	void setDirty(bool dirty = true) { _hasBeenChanged = dirty; }

	/// \brief Returns the path where this dataset is stored on disk.
	/// \return The location where the dataset is stored or will be stored on disk.
	/// \sa setFilePath()
	const QString& filePath() const { return _filePath; }

	/// \brief Sets the path where this dataset is stored.
	/// \param path The new path (should be absolute) where the dataset will be stored.
	/// \sa fielPath()
	void setFilePath(const QString& path) { _filePath = path; }

	/// \brief Rescales the key times for the whole scene.
	/// \param oldAnimationInterval The old animation interval that will be mapped to the new animation interval.
	/// \param newAnimationInterval The new animation interval.
	///
	/// This method calls Controller::rescaleTime() for all controllers in the scene.
	/// For keyed controllers this will rescale the key times of all keys from the
	/// old animation interval to the new interval using linear interpolation.
	///
	/// Please note that keys that lie outside the old animation interval will also be scaled
	/// according to a linear extrapolation.
	///
	/// \undoable
	/// \sa Controller::rescaleTime()
	void rescaleTime(const TimeInterval& oldAnimationInterval, const TimeInterval& newAnimationInterval);

	/// \brief Deletes all nodes from the scene.
	///
	/// \undoable
	/// \sa DataSetManager::fileReset()
	void clearScene();

protected:

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream);

	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);

	/// From RefMaker.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

	/// Is called by the system when the custom attributes of this RefMaker have changed in some way.
	virtual void onCustomAttributesChanged() {
		// Set dirty flag when the custom attributes of the data set have changed.
		setDirty();
		RefTarget::onCustomAttributesChanged();
	}

private:

	/// Contains the configuration of the viewports associated with this data set.
	ReferenceField<ViewportConfiguration> _viewportConfig;

	/// Contains the animation settings.
	ReferenceField<AnimationSettings> _animSettings;

	/// The scene node tree.
	ReferenceField<SceneRoot> _sceneRoot;

	/// Holds the current node selection set.
	ReferenceField<SelectionSet> _selection;

	/// The general settings for rendering of the scene.
	ReferenceField<RenderSettings> _renderSettings;

	/// The dirty flag.
	bool _hasBeenChanged;

	/// The file path where this DataSet is stored.
	QString _filePath;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(DataSet)
	DECLARE_REFERENCE_FIELD(_viewportConfig)
	DECLARE_REFERENCE_FIELD(_animSettings)
	DECLARE_REFERENCE_FIELD(_sceneRoot)
	DECLARE_REFERENCE_FIELD(_selection)
	DECLARE_REFERENCE_FIELD(_renderSettings)
};

};

#endif // __OVITO_DATASET_H
