///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include "BezierPolygon.h"
#include "BezierCurve.h"

namespace Core {

/******************************************************************************
* Creates a bezier polygon from a bezier curve.
******************************************************************************/
void BezierPolygon::setCurve(const BezierCurve& curve, int subdivisionSteps)
{
	OVITO_ASSERT(subdivisionSteps >= 0);
	_isClosed = curve.isClosed();
	FloatType delta = 1.0 / (subdivisionSteps + 1);
	_vertices.clear();
	int numSegments = curve.segmentCount();
	for(int segment = 0; segment < numSegments; segment++) {
		_vertices.push_back(PolygonVertex(PolygonVertex::BezierVertex, curve.vertices()[segment].pos()));
		if(curve.isCurveSegment(segment)) {
			for(int s = 1; s <= subdivisionSteps; s++) {
				Point3 p = curve.interpolateSegment(segment, delta * s);
				_vertices.push_back(PolygonVertex(PolygonVertex::InterpolationVertex, p));
			}
		}
	}
	if(curve.isClosed() == false) {
		_vertices.push_back(PolygonVertex(PolygonVertex::BezierVertex, curve.vertices().back().pos()));
	}
}

/******************************************************************************
* Saves the polygon to the given stream.
******************************************************************************/
void BezierPolygon::saveToStream(SaveStream& stream) const
{
	stream.beginChunk(0x01);

	stream << isClosed();
	stream.writeSizeT(vertexCount());
	for(QVector<PolygonVertex>::const_iterator vertex = vertices().begin(); vertex != vertices().end(); ++vertex) {
		stream.writeEnum(vertex->vertexType());
		stream << vertex->pos();
	}

	stream.endChunk();
}

/******************************************************************************
* Loads the polygon from the given stream.
******************************************************************************/
void BezierPolygon::loadFromStream(LoadStream& stream)
{
	stream.expectChunk(0x01);

	stream >> _isClosed;
	size_t nvertices;
	stream.readSizeT(nvertices);
	_vertices.resize(nvertices);
	for(QVector<PolygonVertex>::iterator vertex = vertices().begin(); vertex != vertices().end(); ++vertex) {
		stream.readEnum(vertex->_vertexType);
		stream >> vertex->_pos;
	}

	stream.closeChunk();
}

};	// End of namespace Core
