///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_STDACTIONS_H
#define __OVITO_STDACTIONS_H

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/actions/Action.h>
#include <core/actions/ViewportModeAction.h>

#include "XFormSystemChooser.h"

namespace Core {

/******************************************************************************
* Implements event handlers for the standard file actions.
******************************************************************************/
class FileActionsHandler : public QObject 
{
	Q_OBJECT
public:

	/// Constructs the action objects and registers them with the ActionManager. 
	FileActionsHandler();

private:

	/// Creates and registers a single command action.
	ActionProxy* addCommandAction(const QString& id, const QString& title, const char* iconPath = NULL, const QKeySequence& shortcut = QKeySequence());
	
public Q_SLOTS:

	/// Handles ACTION_EXIT command.
	void OnExit(); 	
	/// Handles ACTION_FILE_NEW command.
	void OnFileNew();
	/// Handles ACTION_FILE_RESET command.
	void OnFileReset();
	/// Handles ACTION_FILE_OPEN command.
	void OnFileOpen();
	/// Handles ACTION_FILE_SAVE command.
	void OnFileSave();
	/// Handles ACTION_FILE_SAVEAS command.
	void OnFileSaveAs();
	/// Handles ACTION_FILE_IMPORT command.
	void OnFileImport();
	/// Handles ACTION_FILE_EXPORT command.
	void OnFileExport();
	/// Handles ACTION_HELP_ABOUT command.
	void OnHelpAbout(); 
	/// Handles ACTION_HELP_SHOW_ONLINE_HELP command.
	void OnHelpShowOnlineHelp(); 
}; 

/******************************************************************************
* Implements event handlers for the standard viewport actions.
******************************************************************************/
class ViewportActionsHandler : public QObject 
{
	Q_OBJECT
public:

	/// Constructs the action objects and registers them with the ActionManager. 
	ViewportActionsHandler();

private:

	/// Creates and registers a single command action.
	ActionProxy* addCommandAction(const QString& id, const QString& title, const char* iconPath = NULL);
	
	/// Creates and registers a single viewport input mode action.
	ViewportModeAction* addViewportModeAction(const QString& id, ViewportInputHandler* handler, const QString& title, const char* iconPath = NULL); 
	
public Q_SLOTS:

	/// Handles ACTION_VIEWPORT_MAXIMIZE command.
	void OnViewportMaximize(); 
	/// Handles ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS commands.
	void OnViewportZoomSceneExtents(); 
	/// Handles ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS commands.
	void OnViewportZoomSelectionExtents(); 
	/// Handles ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS_ALL commands.
	void OnViewportZoomSceneExtentsAll(); 
	/// Handles ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS_ALL commands.
	void OnViewportZoomSelectionExtentsAll(); 
}; 

/******************************************************************************
* Implements event handlers for the standard editing actions.
******************************************************************************/
class EditingActionsHandler : public QObject 
{
	Q_OBJECT
public:

	/// Constructs the action objects and registers them with the ActionManager. 
	EditingActionsHandler();

private:

	/// Creates and registers a single command action.
	ActionProxy* addCommandAction(const QString& id, const QString& title, const char* iconPath = NULL, const QKeySequence& shortcut = QKeySequence());

	/// Creates and registers a single xform input mode action.
	ViewportModeAction* addXFormModeAction(const QString& id, ViewportInputHandler* handler, const QString& title, const char* iconPath = NULL); 
	
public Q_SLOTS:

	/// Handles ACTION_EDIT_UNDO command.
	void OnEditUndo(); 
	/// Handles ACTION_EDIT_REDO command.
	void OnEditRedo(); 
	/// Handles ACTION_EDIT_DELETE command.
	void OnEditDelete(); 

}; 

/******************************************************************************
* This viewport mode plays the animation while it is active.
******************************************************************************/
class AnimationPlaybackViewportMode : public ViewportInputHandler
{
	Q_OBJECT
	
public:

	/// Constructor.
	AnimationPlaybackViewportMode() {
	}

	/// Returns the activation behaviour of this input handler.
	virtual InputHandlerType handlerActivationType() { 
		return ViewportInputHandler::TEMPORARY; 
	}
		
protected:

	/// This is called by the system after the input handler has
	/// become the active handler.
	virtual void onActivated() {		 
		int timerSpeed = 1000;
		if(ANIM_MANAGER.playbackSpeed() > 1) timerSpeed /= ANIM_MANAGER.playbackSpeed(); 
		else if(ANIM_MANAGER.playbackSpeed() < -1) timerSpeed *= -ANIM_MANAGER.playbackSpeed(); 
		QTimer::singleShot(timerSpeed / ANIM_MANAGER.framesPerSecond(), this, SLOT(OnTimer())); 
	}
				
protected Q_SLOTS:
	
	/// Is called periodicaly by the timer.
	void OnTimer();
};

/******************************************************************************
* Implements event handlers for the standard animation actions.
******************************************************************************/
class AnimationActionsHandler : public QObject 
{
	Q_OBJECT
public:

	/// Constructs the action objects and registers them with the ActionManager. 
	AnimationActionsHandler();

private:

	/// Creates and registers a single command action.
	ActionProxy* addCommandAction(const QString& id, const QString& title, const char* iconPath = NULL);
	
	/// The viewport mode for animation playback.
	AnimationPlaybackViewportMode::SmartPtr animationPlaybackMode;
	
public Q_SLOTS:

	/// Handles ACTION_GOTO_START_OF_ANIMATION command.
	void OnGotoStartOfAnimation(); 
	/// Handles ACTION_GOTO_END_OF_ANIMATION command.
	void OnGotoEndOfAnimation(); 
	/// Handles ACTION_GOTO_PREVIOUS_FRAME command.
	void OnGotoPreviousFrame(); 
	/// Handles ACTION_GOTO_NEXT_FRAME command.
	void OnGotoNextFrame(); 
	/// Handles ACTION_START_ANIMATION_PLAYBACK command.
	void OnStartPlayback(); 
	/// Handles ACTION_STOP_ANIMATION_PLAYBACK command.
	void OnStopPlayback(); 
	/// Handles ACTION_ANIMATION_SETTINGS command.
	void OnAnimationSettings(); 
}; 


/******************************************************************************
* Implements event handlers for the rendering actions.
******************************************************************************/
class RenderingActionsHandler : public QObject 
{
	Q_OBJECT
public:

	/// Constructs the action objects and registers them with the ActionManager. 
	RenderingActionsHandler();

private:

	/// Creates and registers a single command action.
	ActionProxy* addCommandAction(const QString& id, const QString& title, const char* iconPath = NULL, const QKeySequence& shortcut = QKeySequence());

public Q_SLOTS:

	/// Handles ACTION_RENDER_PREVIEW command.
	void OnRenderPreview();

	/// Handles ACTION_RENDER_ACTIVE_VIEWPORT command.
	void OnRenderActiveViewport();
	
	/// Handles ACTION_SELECT_RENDERER_DIALOG command.
	void OnSelectRendererDialog();
}; 

/******************************************************************************
* Implements event handlers for the options menu actions.
******************************************************************************/
class OptionsActionsHandler : public QObject 
{
	Q_OBJECT
public:

	/// Constructs the action objects and registers them with the ActionManager. 
	OptionsActionsHandler();

private:

	/// Creates and registers a single command action.
	ActionProxy* addCommandAction(const QString& id, const QString& title, const char* iconPath = NULL, const QKeySequence& shortcut = QKeySequence());

public Q_SLOTS:

	/// Handles ACTION_SETTINGS_DIALOG command.
	void OnSettingsDialog();
}; 


};

#endif // __OVITO_STDACTIONS_H
