///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file Box2.h
 * \brief Contains the definition of the Base::Box_2 template class.
 */

#ifndef __OVITO_BOX2_H
#define __OVITO_BOX2_H

#include <base/Base.h>
#include "Vector2.h"
#include "Point2.h"

namespace Base {

/**
 * \brief A bounding box in 2d space.
 *
 * This class stores an axis-aligned box in 2d.
 * It is defined by minimum and maximum coordinates in X and Y direction.
 *
 * \author Alexander Stukowski
 * \sa Box_3
 */
template<typename T>
class Box_2
{
public:
	/// The coordinates of the lower left corner.
	Point_2<T> minc;
	/// The coordinates of the upper right corner.
	Point_2<T> maxc;

	/////////////////////////////// Constructors /////////////////////////////////

	/// \brief Creates an empty box.
	Box_2() : minc(numeric_limits<T>::max()), maxc(-numeric_limits<T>::max()) {}

	/// \brief Initializes the box with the minimum and maximum coordinates.
	/// \param minCorner A point that specifies the corner with minimum coordinates of the box.
	/// \param maxCorner A point that specifies the corner with maximum coordinates of the box.
	Box_2(const Point_2<T>& minCorner, const Point_2<T>& maxCorner) : minc(minCorner), maxc(maxCorner) {
		OVITO_ASSERT_MSG(minc.X <= maxc.X, "Box_3 constructor", "X component of the minimum corner point must not be larger than the maximum corner point.");
		OVITO_ASSERT_MSG(minc.Y <= maxc.Y, "Box_3 constructor", "Y component of the minimum corner point must not be larger than the maximum corner point.");
	}

	/// \brief Initializes the box with the given coordinates.
	Box_2(T xmin, T ymin, T xmax, T ymax) : minc(Point_2<T>(xmin, ymin)), maxc(Point_2<T>(xmax, ymax)) {
		OVITO_ASSERT(minc.X <= maxc.X);
		OVITO_ASSERT(minc.Y <= maxc.Y);
	}

	///////////////////////////////// Attributes /////////////////////////////////

	/// \brief Checks whether this is an empty box.
	///
	/// The box is considered empty when one of the maximum corner coodinates is less
	/// then the minimum corner coordinate.
	/// \return true if this box is empty; false otherwise.
	bool isEmpty() const {
        return (minc.X > maxc.X) || (minc.Y > maxc.Y);
	}

	/// \brief Resets the box to the empty state.
	void setEmpty() {
		minc = Point_2<T>( numeric_limits<T>::max());
		maxc = Point_2<T>(-numeric_limits<T>::max());
	}

	/// \brief Returns the position of one of the four corners of the box corner.
	/// \param i The index of the corner (0 - 3).
	/// \return The coordinate of the i-th corner of the box.
	Point_2<T> operator[](size_t i) const {
		switch(i) {
			case 0: return Point_2<T>(minc.X, minc.Y);
			case 1: return Point_2<T>(maxc.X, minc.Y);
			case 2: return Point_2<T>(maxc.X, maxc.Y);
			case 3: return Point_2<T>(minc.X, maxc.Y);
			default:
				OVITO_ASSERT_MSG(false, "Box2::operator[]", "Corner index out of range.");
				throw std::invalid_argument("Corner index out of range.");
				return ORIGIN;
		}
	}

	/// \brief Computes the width of the box.
	T width() const { return maxc.X - minc.X; }

	/// \brief Computes the height of the box.
	T height() const { return maxc.Y - minc.Y; }

	/// \brief Computes the center of the box.
	/// \return The center of the box.
	Point_2<T> center() const {
		return Point_2<T>((minc.X + maxc.X) / 2, (minc.Y + maxc.Y) / 2);
	}

	/// \brief Computes the size of the box.
	/// \return The difference between the maximum and minimum corner.
	/// \sa width(), height()
	Vector_2<T> size() const {
		return maxc - minc;
	}

	/// \brief Returns the size of the box in the given dimension.
	/// \param dimension The index of the dimension (0 or 1).
	/// \sa size(), width(), height()
	T size(size_t dimension) const {
		return maxc[dimension] - minc[dimension];
	}

	/////////////////////////////// Classification ///////////////////////////////

	/// \brief Checks whether a point is inside the box.
	/// \param p The point to test.
	/// \return true if the given point is inside or on the edge of the bounding box; false if it is completely outside the box.
	bool contains(const Point_2<T>& p) const {
		if(p.X < minc.X || p.X > maxc.X) return false;
		if(p.Y < minc.Y || p.Y > maxc.Y) return false;
		return true;
	}

	/// \brief Classifies the given point with respect to the box.
	///
	/// Returns -1 if the point is outside of the box.
	/// Returns 0 if the point is on the boundary of the box within the given tolerance.
	/// Returns +1 if the point is inside of the box.
	int classifyPoint(const Point_2<T>& p, FloatType epsilon = FLOATTYPE_EPSILON) const {
		if(p.X > maxc.X + epsilon || p.Y > maxc.Y + epsilon) return -1;
		if(p.X < minc.X - epsilon || p.Y < minc.Y - epsilon) return -1;
		if(p.X < maxc.X - epsilon && p.X > minc.X + epsilon &&
		   p.Y < maxc.Y - epsilon && p.Y > minc.Y + epsilon) return 1;
		return 0;
	}

	/// \brief Checks whether another box is contained in this box.
	/// \return true if the given box is completely inside the bounding box.
	bool containsBox(const Box_2<T>& b) const {
		return (b.minc.X >= minc.X && b.maxc.X <= maxc.X) &&
			(b.minc.Y >= minc.Y && b.maxc.Y <= maxc.Y);
	}

	/// \brief Checks wehther the intersection of two boxes is not empty.
	/// \return true if the given box is not completely outside of this box.
	bool intersects(const Box_2<T>& b) const {
		if(maxc.X <= b.minc.X || minc.X >= b.maxc.X) return false;
		if(maxc.Y <= b.minc.Y || minc.Y >= b.maxc.Y) return false;
		if(isEmpty() || b.isEmpty()) return false;
		return true;
	}

    //////////////////////////////// Modification ////////////////////////////////

	/// \brief Enlarges the box to include the given point.
	/// \sa addPoints(), addBox()
	void addPoint(const Point_2<T>& p) {
		minc.X = min(minc.X, p.X); maxc.X = max(maxc.X, p.X);
		minc.Y = min(minc.Y, p.Y); maxc.Y = max(maxc.Y, p.Y);
	}

	/// \brief Enlarges the box to include the given point.
	/// \sa addPoints(), addBox()
	void addPoint(T x, T y) {
		minc.X = min(minc.X, x); maxc.X = max(maxc.X, x);
		minc.Y = min(minc.Y, y); maxc.Y = max(maxc.Y, y);
	}

	/// \brief Enlarges the box to include the given point.
	/// \sa addPoint()
	Box_2& operator+=(const Point_2<T>& p) {
		addPoint(p);
		return *this;
	}

	/// \brief Enlarges the box to include the given points.
	/// \param points Pointer to the first element of an array of points.
	/// \param count The number of points in the array.
	/// \sa addPoint()
	void addPoints(const Point_2<T>* points, size_t count) {
		for(; count != 0; count--, points++) {
			minc.X = min(minc.X, points->X); maxc.X = max(maxc.X, points->X);
			minc.Y = min(minc.Y, points->Y); maxc.Y = max(maxc.Y, points->Y);
		}
	}

	/// \brief Enlarges this box to include the given box.
	/// \sa addPoint()
	void addBox(const Box_2& b) {
		minc.X = min(minc.X, b.minc.X); maxc.X = max(maxc.X, b.maxc.X);
		minc.Y = min(minc.Y, b.minc.Y); maxc.Y = max(maxc.Y, b.maxc.Y);
	}

	/// \brief Enlarges the box to include the given x coordindate.
    void includeX(T x) {
        minc.X = min(minc.X, x); maxc.X = max(maxc.X, x);
	}

	/// \brief Enlarges the box to include the given y coordindate.
    void includeY(T y) {
        minc.Y = min(minc.Y, y); maxc.Y = max(maxc.Y, y);
	}

    ////////////////////////////////// Utilities /////////////////////////////////

	/// \brief Returns a string representation of this box.
	QString toString() const {
		return "[Min: " + minc.toString() + " Max: " + maxc.toString() + "]";
	}
};

/// \brief Writes the box to a logging stream.
template<typename T>
inline LoggerObject& operator<<(LoggerObject& log, const Box_2<T>& b)
{
	log.nospace() << '[' << b.minc << "] - [" << b.maxc << ']';
	return log;
}

/// \brief Writes a 2d box to a binary output stream.
template<typename T>
inline SaveStream& operator<<(SaveStream& stream, const Box_2<T>& b)
{
	return stream << b.minc << b.maxc;
}

/// \brief Reads a 2d box from a binary input stream.
template<typename T>
inline LoadStream& operator>>(LoadStream& stream, Box_2<T>& b)
{
	return stream >> b.minc >> b.maxc;
}

/**
 * \fn typedef Box2
 * \brief Template class instance of the Box_2 class used for floating-point calculations based on Point2.
 */
typedef Box_2<FloatType>	Box2;

/**
 * \fn typedef Box2I
 * \brief Template class instance of the Box_2 class used for integer calculations based on Point2I.
 */
typedef Box_2<int>			Box2I;

};	// End of namespace Base

Q_DECLARE_METATYPE(Base::Box2)
Q_DECLARE_METATYPE(Base::Box2I)
Q_DECLARE_TYPEINFO(Base::Box2, Q_MOVABLE_TYPE);
Q_DECLARE_TYPEINFO(Base::Box2I, Q_MOVABLE_TYPE);

#endif // __OVITO_BOX2_H
