///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file OnTheFlyNeighborList.h
 * \brief Contains the definition of the AtomViz::OnTheFlyNeighborList class.
 */

#ifndef __ONTHEFLY_NEIGHBOR_LIST_H
#define __ONTHEFLY_NEIGHBOR_LIST_H

#include <core/Core.h>
#include <boost/multi_array.hpp>
#include <atomviz/AtomViz.h>

namespace AtomViz {

class AtomsObject;		// defined in AtomsObject.h

/**
 * \brief This class can calculate the list of neighbors
 *        for each of the atoms in an AtomsObject.
 *        The neighbor list is not stored in memory but calculated
 *        on demand for each atom.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT OnTheFlyNeighborList
{
private:

	// A internal atom structure.
	struct NeighborListAtom {
		/// The next atom in the linked list used for binning.
		NeighborListAtom* nextInBin;
		/// The index of the atom in the original AtomsObject.
		int index;
		/// The wrapped position of the atom.
		Point3 pos;
	};

public:

	/// \brief Constructor.
	OnTheFlyNeighborList(AtomsObject* atoms, FloatType cutoffRadius);

	/// \brief Tests whether two atoms are closer to each other than the
	///        nearest-neighbor cutoff radius.
	bool areNeighbors(int atom1, int atom2) const;

	/// \brief This iterator class lists all neighbors of a given atom.
	class ATOMVIZ_DLLEXPORT iterator {
	public:
		/// \brief Constructor
		iterator(const OnTheFlyNeighborList& neighborList, int atomIndex);

		bool atEnd() const { return dir[0] > 1; }
		int next();
		int current() { return neighborindex; }
		const Vector3& delta() const { return _delta; }
		FloatType distanceSquared() const { return distsq; }

	private:
		const OnTheFlyNeighborList& _list;
		Point3 center;
		int centerindex;
		int dir[3];
		int centerbin[3];
		int currentbin[3];
		NeighborListAtom* binatom;
		int neighborindex;
		Vector3 pbcshift;
		Vector3 _delta;
		FloatType distsq;
	};

private:

	/// The neighbor criterion.
	FloatType _cutoffRadius;

	/// The neighbor criterion.
	FloatType _cutoffRadiusSquared;

	// Simulation cell properties.
	AffineTransformation simCell;
	AffineTransformation simCellInverse;
	array<bool,3> pbc;

	/// Number of bins in each spatial direction.
	int binDim[3];

	/// The internal list of atoma.
	QVector<NeighborListAtom> atoms;

	/// An 3d array of cubic bins. Each bin is a linked list of atoms.
	typedef multi_array<NeighborListAtom*, 3> BinsArray;
	BinsArray bins;
};

};	// End of namespace AtomViz

#endif // __ONTHEFLY_NEIGHBOR_LIST_H
