///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/scene/ObjectNode.h>
#include <core/gui/properties/StringPropertyUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <atomviz/utils/muparser/muParser.h>

#include "SelectExpressionModifier.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(SelectExpressionModifier, SelectionModifierBase)
DEFINE_PROPERTY_FIELD(SelectExpressionModifier, "Expression", _expression)
SET_PROPERTY_FIELD_LABEL(SelectExpressionModifier, _expression, "Boolean expression")

/******************************************************************************
* Determines the available variable names.
******************************************************************************/
QStringList SelectExpressionModifier::getVariableNames()
{
	QStringList variableNames;
	Q_FOREACH(DataChannel* channel, input()->dataChannels()) {

		// Channels of custom data type are not supported by this modifier.
		if(channel->type() != qMetaTypeId<int>() && channel->type() != qMetaTypeId<FloatType>()) continue;

		// Alter the data channel name to make it a valid variable name for the parser.
		QString variableName = channel->name();
		variableName.remove(QRegExp("[^A-Za-z\\d_]"));
		if(channel->componentNames().empty()) {
			OVITO_ASSERT(channel->componentCount() == 1);
			variableNames << variableName;
		}
		else {
			Q_FOREACH(QString componentName, channel->componentNames()) {
				componentName.remove(QRegExp("[^A-Za-z\\d_]"));
				variableNames << (variableName + "." + componentName);
			}
		}
	}

	if(input()->getStandardDataChannel(DataChannel::AtomIndexChannel) == NULL)
		variableNames << "AtomIndex";

	return variableNames;
}

/**
 * This helper class is needed to enable multi-threaded evaluation of math expressions
 * for all atoms. Each instance of this class is assigned a chunk of atoms that it processes.
 */
class SelExpressionEvaluationKernel
{
private:
	struct ExpressionVariable {
		double value;
		const char* dataPointer;
		size_t stride;
		bool isFloat;
	};

public:
	/// Initializes the expressions parsers.
	bool initialize(const QString& expression, const QStringList& variableNames, AtomsObject* input, int timestep) {
		variables.resize(variableNames.size());
		bool usesTimeInExpression = false;

		// Compile the expression string.
		try {
			// Configure parser to accept '.' in variable names.
			parser.DefineNameChars("0123456789_abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ.");

			// Add the atan2() function.
			parser.DefineFun("atan2", atan2, false);

			// Let the muParser process the math expression.
			parser.SetExpr(expression.toStdString());

			// Register variables
			for(int v=0; v<variableNames.size(); v++)
				parser.DefineVar(variableNames[v].toStdString(), &variables[v].value);

			// If the current animation time is used in the math expression then we have to
			// reduce the validity interval to the current time only.
			mu::varmap_type usedVariables = parser.GetUsedVar();
			if(usedVariables.find("t") != usedVariables.end())
				usesTimeInExpression = true;

			// Add constants.
			parser.DefineConst("pi", 3.1415926535897932);
			parser.DefineConst("N", input->atomsCount());
			parser.DefineConst("t", timestep);
		}
		catch(mu::Parser::exception_type& ex) {
			throw Exception(QString("%1").arg(QString::fromStdString(ex.GetMsg())));
		}

		// Setup data pointers to input channels.
		size_t vindex = 0;
		Q_FOREACH(DataChannel* channel, input->dataChannels()) {
			if(channel->type() == qMetaTypeId<FloatType>()) {
				for(size_t k=0; k<channel->componentCount(); k++) {
					OVITO_ASSERT(vindex < variableNames.size());
					variables[vindex].dataPointer = reinterpret_cast<const char*>(channel->constDataFloat() + k);
					variables[vindex].stride = channel->perAtomSize();
					variables[vindex].isFloat = true;
					vindex++;
				}
			}
			else if(channel->type() == qMetaTypeId<int>()) {
				for(size_t k=0; k<channel->componentCount(); k++) {
					OVITO_ASSERT(vindex < variableNames.size());
					variables[vindex].dataPointer = reinterpret_cast<const char*>(channel->constDataInt() + k);
					variables[vindex].stride = channel->perAtomSize();
					variables[vindex].isFloat = false;
					vindex++;
				}
			}
			else OVITO_ASSERT(false);
		}

		// Add the special AtomIndex variable if there is no dedicated data channel.
		if(input->getStandardDataChannel(DataChannel::AtomIndexChannel) == NULL) {
			variables[vindex].dataPointer = NULL;
			variables[vindex].stride = 0;
			variables[vindex].isFloat = false;
			vindex++;
		}

		OVITO_ASSERT(vindex == variableNames.size());

		return usesTimeInExpression;
	}

	void run(int startIndex, int endIndex, DataChannel* outputChannel) {
		try {
			// Position pointers.
			for(vector<ExpressionVariable>::iterator v = variables.begin(); v != variables.end(); ++v)
				v->dataPointer += v->stride * startIndex;

			nSelected = 0;
			for(int i = startIndex; i < endIndex; i++) {

				// Update variable values for the current atom.
				for(vector<ExpressionVariable>::iterator v = variables.begin(); v != variables.end(); ++v) {
					if(v->isFloat)
						v->value = *reinterpret_cast<const FloatType*>(v->dataPointer);
					else if(v->dataPointer)
						v->value = *reinterpret_cast<const int*>(v->dataPointer);
					else
						v->value = i;
					v->dataPointer += v->stride;
				}

				// Evaluate expression for the current atom.
				double value = parser.Eval();

				// Store computed value in output channel.
				if(value) {
					outputChannel->setInt(i, 1);
					nSelected++;
				}
				else outputChannel->setInt(i, 0);
			}
		}
		catch(const mu::Parser::exception_type& ex) {
			errorMsg = QString::fromStdString(ex.GetMsg());
		}
	}

	QString errorMsg;
	int nSelected;
private:
	mu::Parser parser;
	vector<ExpressionVariable> variables;
};


/******************************************************************************
* This modifies the input object.
******************************************************************************/
EvaluationStatus SelectExpressionModifier::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	// Get list of available input variables.
	_variableNames = getVariableNames();

	// If the user has not yet entered an expression let him know which
	// data channels can be used in the expression.
	if(expression().isEmpty()) {
		return EvaluationStatus(EvaluationStatus::EVALUATION_WARNING, tr("Please enter a boolean expression."));
	}

	//QTime timer;
	//timer.start();

	QString statusMessage = tr("%n input atoms", 0, input()->atomsCount());

	// The number of selected atoms.
	size_t nSelected = 0;

	// Create and initialize the worker threads.
	QVector<SelExpressionEvaluationKernel> workers(max(QThread::idealThreadCount(), 1));
	for(QVector<SelExpressionEvaluationKernel>::iterator worker = workers.begin(); worker != workers.end(); ++worker) {
		if(worker->initialize(expression(), _variableNames, input(), ANIM_MANAGER.timeToFrame(time)))
			validityInterval.intersect(TimeInterval(time));
	}

	// Get the deep copy of the selection channel.
	DataChannel* selChannel = outputStandardChannel(DataChannel::SelectionChannel);
	selChannel->setVisible(selectionShown());
	// This call is necessary to deep copy the memory array of the output channel before accessing it from multiple threads.
	selChannel->data();

	// Spawn worker threads.
	QFutureSynchronizer<void> synchronizer;
	int chunkSize = max((int)input()->atomsCount() / workers.size(), 1);
	for(int i = 0; i < workers.size(); i++) {
		// Setup data range.
		int startIndex = i * chunkSize;
		int endIndex = min((i+1) * chunkSize, (int)input()->atomsCount());
		if(i == workers.size() - 1) endIndex = input()->atomsCount();
		if(endIndex <= startIndex) continue;

		synchronizer.addFuture(QtConcurrent::run(&workers[i], &SelExpressionEvaluationKernel::run, startIndex, endIndex, selChannel));
	}
	synchronizer.waitForFinished();

	// Check for errors.
	for(int i = 0; i < workers.size(); i++) {
		if(workers[i].errorMsg.isEmpty() == false)
			throw Exception(workers[i].errorMsg);

		nSelected += workers[i].nSelected;
	}

	//MsgLogger() << "Selection took " << timer.elapsed() << "msec." << endl;

	statusMessage += tr("\n%n atoms selected\n", 0, nSelected);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, QString(), statusMessage);
}

IMPLEMENT_PLUGIN_CLASS(SelectExpressionModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void SelectExpressionModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	QWidget* rollout = createRollout(tr("Select By Expression"), rolloutParams, "atomviz.modifiers.select_by_expression");

    // Create the rollout contents.
	QVBoxLayout* layout = new QVBoxLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);

	layout->addWidget(new QLabel(tr("Boolean expression:")));
	StringPropertyUI* expressionUI = new StringPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SelectExpressionModifier, _expression));
	layout->addWidget(expressionUI->textBox());

	layout->addSpacing(2);
	BooleanPropertyUI* showSelUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SelectionModifierBase, _selectionShown));
	layout->addWidget(showSelUI->checkBox());

	// Status label.
	layout->addSpacing(12);
	layout->addWidget(statusLabel());

	QWidget* variablesRollout = createRollout(tr("Variables"), rolloutParams.after(rollout), "atomviz.modifiers.select_by_expression");
    QVBoxLayout* variablesLayout = new QVBoxLayout(variablesRollout);
    variablesLayout->setContentsMargins(4,4,4,4);
	variableNamesList = new QLabel();
	variableNamesList->setWordWrap(true);
	variableNamesList->setTextInteractionFlags(Qt::TextSelectableByMouse | Qt::TextSelectableByKeyboard | Qt::LinksAccessibleByMouse | Qt::LinksAccessibleByKeyboard);
	variablesLayout->addWidget(variableNamesList);
}


/******************************************************************************
* This method is called when a reference target changes.
******************************************************************************/
bool SelectExpressionModifierEditor::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(source == editObject() && msg->type() == REFTARGET_CHANGED) {
		updateEditorFields();
	}
	/*
	else if(msg->sender() == editObject() && msg->type() == MODIFIER_EVALUATION_MESSAGE_UPDATE) {
		updateEditorFields();
	}
	*/
	return AtomsObjectModifierEditorBase::onRefTargetMessage(source, msg);
}

/******************************************************************************
* Updates the enabled/disabled status of the editor's controls.
******************************************************************************/
void SelectExpressionModifierEditor::updateEditorFields()
{
	SelectExpressionModifier* mod = static_object_cast<SelectExpressionModifier>(editObject());
	if(!mod) return;

	QString labelText(tr("The following variables can be used in the boolean expressions:<ul>"));
	Q_FOREACH(QString s, mod->lastVariableNames()) {
		labelText.append(QString("<li>%1</li>").arg(s));
	}
	labelText.append(QString("<li>N (number of atoms)</li>"));
	labelText.append(QString("<li>t (current time frame)</li>"));
	labelText.append("</ul><p></p>");
	variableNamesList->setText(labelText);
}


};	// End of namespace AtomViz
