///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __CREATE_EXPRESSION_CHANNEL_MODIFIER_H
#define __CREATE_EXPRESSION_CHANNEL_MODIFIER_H

#include <core/Core.h>
#include <core/gui/properties/StringPropertyUI.h>
#include <core/gui/properties/IntegerPropertyUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/VariantComboBoxPropertyUI.h>

#include <atomviz/AtomViz.h>
#include "../AtomsObjectModifierBase.h"

namespace AtomViz {

/**
 * \brief Creates a data channel that is filled with values computed by an
 *        user-defined math expression.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT CreateExpressionChannelModifier : public AtomsObjectModifierBase
{
public:

	/// \brief Constructs a new instance of this class.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	CreateExpressionChannelModifier(bool isLoading = false) :
		AtomsObjectModifierBase(isLoading), _dataChannelId(DataChannel::UserDataChannel), _dataChannelVisibility(true),
		_dataChannelDataType(qMetaTypeId<FloatType>()), _dataChannelName(tr("Custom channel 1")), _expressions(QStringList("0")),
		_onlySelectedAtoms(false)
	{
		INIT_PROPERTY_FIELD(CreateExpressionChannelModifier, _expressions);
		INIT_PROPERTY_FIELD(CreateExpressionChannelModifier, _dataChannelId);
		INIT_PROPERTY_FIELD(CreateExpressionChannelModifier, _dataChannelName);
		INIT_PROPERTY_FIELD(CreateExpressionChannelModifier, _dataChannelDataType);
		INIT_PROPERTY_FIELD(CreateExpressionChannelModifier, _dataChannelVisibility);
		INIT_PROPERTY_FIELD(CreateExpressionChannelModifier, _onlySelectedAtoms);
	}

	//////////////////////////// from base classes ////////////////////////////

	/// \brief Asks the modifier for its validity interval at the given time.
	///
	/// This modifier has no animatable parameters, so it's validity interval is infinite.
	virtual TimeInterval modifierValidity(TimeTicks time) { return TimeForever; }

	/////////////////////////// specific methods ///////////////////////////////

	/// \brief Sets the math expressions that are used to calculate the values of the new data channel's components.
	/// \param expressions The mathematical formulas, one for each component of the data channel to create.
	/// \undoable
	/// \sa expressions()
	void setExpressions(const QStringList& expressions) { _expressions = expressions; }

	/// \brief Returns the math expressions that are used to calculate the values of the new data channel's component.
	/// \return The math formulas.
	/// \sa setExpressions()
	const QStringList& expressions() const { return _expressions; }

	/// \brief Sets the math expression that is used to calculate the values of one of the new data channel's components.
	/// \param index The data channel component for which the expression should be set.
	/// \param expression The math formula.
	/// \undoable
	/// \sa expression()
	void setExpression(const QString& expression, int index = 0) {
		if(index < 0 || index >= expressions().size())
			throw Exception("Data channel component index is out of range.");
		QStringList copy = _expressions;
		copy[index] = expression;
		_expressions = copy;
	}

	/// \brief Returns the math expression that is used to calculate the values of one of the new data channel's components.
	/// \param index The data channel component for which the expression should be returned.
	/// \return The math formula used to calculates the channel's values.
	/// \undoable
	/// \sa setExpression()
	const QString& expression(int index = 0) const {
		if(index < 0 || index >= expressions().size())
			throw Exception("Data channel component index is out of range.");
		return expressions()[index];
	}

	/// \brief Returns the identifier of the data channel being created by this modifier.
	/// \return The identifier.
	/// \sa setDataChannelId()
	DataChannel::DataChannelIdentifier dataChannelId() const { return _dataChannelId; }

	/// \brief Sets the identifier of the data channel being created by this modifier.
	/// \param newId The new identifier. If this is one of the standard data channels then the
	///              other channel parameters will be set to the defaults according to the standard channel identifier.
	/// \undoable
	/// \sa dataChannelId()
	void setDataChannelId(DataChannel::DataChannelIdentifier newId);

	/// \brief Returns the name of the data channel being created by this modifier.
	/// \return The name of the new data channel.
	/// \sa setDataChannelName()
	const QString& dataChannelName() const { return _dataChannelName; }

	/// \brief Sets the name of the data channel being created by this modifier.
	/// \param newName The name of the new data channel.
	/// \undoable
	/// \sa dataChannelName()
	void setDataChannelName(const QString& newName) { _dataChannelName = newName; }

	/// \brief Returns the data type of the data channel being created.
	/// \return The id of the Qt data type of the new data channel.
	/// \sa setDataChannelDataType()
	int dataChannelDataType() const { return _dataChannelDataType; }

	/// \brief Sets the data type of the data channel being created.
	/// \param newDataType The id of the Qt data type of the new data channel.
	/// \undoable
	/// \sa dataChannelDataType()
	void setDataChannelDataType(int newDataType) { _dataChannelDataType = newDataType; }

	/// \brief Returns the number of vector components of the data channel to create.
	/// \return The number of vector components.
	/// \sa setDataChannelComponentCount()
	int dataChannelComponentCount() const { return ((const QStringList&)_expressions).size(); }

	/// \brief Sets the number of vector components of the data channel to create.
	/// \param newComponentCount The number of vector components.
	/// \undoable
	/// \sa dataChannelComponentCount()
	void setDataChannelComponentCount(int newComponentCount);

	/// \brief Returns the visibility flag of the data channel to create.
	/// \return \c true if the new data channel is visible; \c false if it will be hidden.
	/// \sa setDataChannelVisibility()
	bool dataChannelVisibility() const { return _dataChannelVisibility; }

	/// \brief Sets the visibility flag of the data channel to create.
	/// \param visible The visibility flag.
	/// \undoable
	/// \sa dataChannelVisibility()
	void setDataChannelVisibility(bool visible) { _dataChannelVisibility = visible; }

	/// \brief Returns whether the math expression is only evaluated for selected atoms.
	/// \return \c true if the expression is only evaluated for selected atoms; \c false if it is calculated for all atoms.
	/// \sa setOnlySelectedAtoms()
	bool onlySelectedAtoms() const { return _onlySelectedAtoms; }

	/// \brief Sets whether the math expression is only evaluated for selected atoms.
	/// \param enable Specifies the restriction to selected atoms.
	/// \undoable
	/// \sa onlySelectedAtoms()
	void setOnlySelectedAtoms(bool enable) { _onlySelectedAtoms = enable; }

	/// \brief Returns the list of variables during the last evaluation.
	const QStringList& lastVariableNames() const { return _variableNames; }

public:

	Q_PROPERTY(QStringList expressions READ expressions WRITE setExpressions)
	Q_PROPERTY(AtomViz::DataChannel::DataChannelIdentifier dataChannelId READ dataChannelId WRITE setDataChannelId)
	Q_PROPERTY(QString dataChannelName READ dataChannelName WRITE setDataChannelName)
	Q_PROPERTY(int dataChannelDataType READ dataChannelDataType WRITE setDataChannelDataType)
	Q_PROPERTY(int dataChannelComponentCount READ dataChannelComponentCount WRITE setDataChannelComponentCount)
	Q_PROPERTY(bool dataChannelVisibility READ dataChannelVisibility WRITE setDataChannelVisibility)
	Q_PROPERTY(bool onlySelectedAtoms READ onlySelectedAtoms WRITE setOnlySelectedAtoms)

protected:

	/// Modifies the atoms object.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

	/// \brief Determines the available variable names.
	/// \return A list of variable names that can be used in the math expression.
	///
	/// The variable names are derived form the data channels names of the input object.
	QStringList getVariableNames();

	/// The math expressions that are used to calculate the values of the new data channel' components.
	PropertyField<QStringList> _expressions;

	/// The identifier of the data channel to create.
	PropertyField<DataChannel::DataChannelIdentifier, int> _dataChannelId;

	/// The name of the data channel to create.
	PropertyField<QString> _dataChannelName;

	/// The data type of the data channel to create.
	PropertyField<int> _dataChannelDataType;

	/// Controls the visibility of the newly created data channel.
	PropertyField<bool> _dataChannelVisibility;

	/// Controls whether the math expression is evaluated only for selected atoms.
	PropertyField<bool> _onlySelectedAtoms;

	/// The list of variables during the last evaluation.
	QStringList _variableNames;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(CreateExpressionChannelModifier)
	DECLARE_PROPERTY_FIELD(_expressions)
	DECLARE_PROPERTY_FIELD(_dataChannelId)
	DECLARE_PROPERTY_FIELD(_dataChannelName)
	DECLARE_PROPERTY_FIELD(_dataChannelDataType)
	DECLARE_PROPERTY_FIELD(_dataChannelVisibility)
	DECLARE_PROPERTY_FIELD(_onlySelectedAtoms)
};

/**
 * \brief A properties editor for the CreateExpressionChannelModifier class.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT CreateExpressionChannelModifierEditor : public AtomsObjectModifierEditorBase
{
public:

	/// Sets the object being edited in this editor.
	virtual void setEditObject(RefTarget* newObject) {
		AtomsObjectModifierEditorBase::setEditObject(newObject);
		updateEditorFields();
	}

protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

protected Q_SLOTS:

	/// Is called when the user has typed in an expression.
	void onExpressionEditingFinished();

private:

	/// Updates the enabled/disabled status of the editor's controls.
	void updateEditorFields();

	StringPropertyUI* dataChannelNameUI;
	VariantComboBoxPropertyUI* dataChannelTypeUI;
	IntegerPropertyUI* numComponentsUI;

	QList<QLineEdit*> expressionBoxes;
	QList<QLabel*> expressionBoxLabels;

	QVBoxLayout* expressionsLayout;

	QLabel* variableNamesList;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CreateExpressionChannelModifierEditor)
};

};	// End of namespace AtomViz

#endif // __CREATE_EXPRESSION_CHANNEL_MODIFIER_H
