///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <boost/iterator/counting_iterator.hpp>

#include "CNAModifier.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CommonNeighborAnalysisModifier, AtomsObjectAnalyzerBase)
DEFINE_VECTOR_REFERENCE_FIELD(CommonNeighborAnalysisModifier, AtomType, "AtomTypes", _atomTypesList)
DEFINE_REFERENCE_FIELD(CommonNeighborAnalysisModifier, AtomTypeDataChannel, "CNATypeChannel", _cnaChannel)
SET_PROPERTY_FIELD_LABEL(CommonNeighborAnalysisModifier, _atomTypesList, "Atom Types")

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
CommonNeighborAnalysisModifier::CommonNeighborAnalysisModifier(bool isLoading)
	: AtomsObjectAnalyzerBase(isLoading)
{
	INIT_PROPERTY_FIELD(CommonNeighborAnalysisModifier, _atomTypesList);
	INIT_PROPERTY_FIELD(CommonNeighborAnalysisModifier, _cnaChannel);

	if(!isLoading) {

		// Create the internal atom types.
		intrusive_ptr<AtomType> fccType(new AtomType());
		fccType->setName(tr("FCC"));
		fccType->setColor(Color(0.4f, 1.0f, 0.4f));
		_atomTypesList.push_back(fccType);

		intrusive_ptr<AtomType> hcpType(new AtomType());
		hcpType->setName(tr("HCP"));
		hcpType->setColor(Color(1.0f, 0.4f, 0.4f));
		_atomTypesList.push_back(hcpType);

		intrusive_ptr<AtomType> bccType(new AtomType());
		bccType->setName(tr("BCC"));
		bccType->setColor(Color(0.4f, 0.4f, 1.0f));
		_atomTypesList.push_back(bccType);

		intrusive_ptr<AtomType> icosahedralType(new AtomType());
		icosahedralType->setName(tr("Icosahedral"));
		icosahedralType->setColor(Color(1.0f, 1.0f, 0.7f));
		_atomTypesList.push_back(icosahedralType);

		intrusive_ptr<AtomType> noneType(new AtomType());
		noneType->setName(tr("Other"));
		noneType->setColor(Color(1.0f, 1.0f, 1.0f));
		_atomTypesList.push_back(noneType);

		intrusive_ptr<AtomType> cgnType(new AtomType());
		cgnType->setName(tr("cg-N"));
		cgnType->setColor(Color(1.0f, 1.0f, 0.0f));
		_atomTypesList.push_back(cgnType);

		intrusive_ptr<AtomType> diamondType(new AtomType());
		diamondType->setName(tr("Diamond"));
		diamondType->setColor(Color(1.0f, 0.4f, 0.0f));
		_atomTypesList.push_back(diamondType);

		intrusive_ptr<AtomType> hexDiamondType(new AtomType());
		hexDiamondType->setName(tr("Hex-Diamond"));
		hexDiamondType->setColor(Color(0.7f, 0.0f, 1.0f));
		_atomTypesList.push_back(hexDiamondType);

		intrusive_ptr<AtomType> diamondStackingFaultType(new AtomType());
		diamondStackingFaultType->setName(tr("Diamond stacking fault"));
		diamondStackingFaultType->setColor(Color(0.2f, 1.0f, 1.0f));
		_atomTypesList.push_back(diamondStackingFaultType);

		intrusive_ptr<AtomType> bccTwinType(new AtomType());
		bccTwinType->setName(tr("BCC Twin"));
		bccTwinType->setColor(Color(1.0f, 0.4f, 0.4f));
		_atomTypesList.push_back(bccTwinType);

		// Create data channel for the computation results.
		_cnaChannel = new AtomTypeDataChannel(DataChannel::CNATypeChannel);
	}
}

/******************************************************************************
* Applies the previously calculated analysis results to the atoms object.
******************************************************************************/
EvaluationStatus CommonNeighborAnalysisModifier::applyResult(TimeTicks time, TimeInterval& validityInterval)
{
	// Check if it is still valid.
	if(input()->atomsCount() != _cnaChannel->size())
		throw Exception(tr("Number of atoms of input object has changed. Analysis results became invalid."));

	// Create a copy of the CNA atom type channel that stores the calculated atom types.
	CloneHelper cloneHelper;
	AtomTypeDataChannel::SmartPtr cnaAtomTypeChannel = cloneHelper.cloneObject(_cnaChannel, true);

	// Remove old atoms types.
	while(!cnaAtomTypeChannel->atomTypes().empty())
		cnaAtomTypeChannel->removeAtomType(cnaAtomTypeChannel->atomTypes().front());

	// Add analysis atom types.
	Vector3 atomTypeColors[NumCNAAtomTypes];
	int index = 0;
	Q_FOREACH(AtomType* atype, atomTypes()) {
		// Add only a copy of the atom type to the AtomsObject and keep the original.
		cnaAtomTypeChannel->insertAtomType(cloneHelper.cloneObject(atype, true).get());
		atype->colorController()->getValue(time, atomTypeColors[index++], validityInterval);
	}

	// Put the new channel into the output object.
	output()->insertDataChannel(cnaAtomTypeChannel);

	int typeCounters[NumCNAAtomTypes];
	for(size_t i = 0; i < NumCNAAtomTypes; i++)
		typeCounters[i] = 0;
	const int* in = _cnaChannel->constDataInt();
	Vector3* colorIter = outputStandardChannel(DataChannel::ColorChannel)->dataVector3();
	for(size_t i = _cnaChannel->size(); i != 0; --i, ++in) {
		OVITO_ASSERT(*in < NumCNAAtomTypes && *in >= 0);
		*colorIter++ = atomTypeColors[*in];
		typeCounters[*in]++;
	}


	QString statusMessage;
	statusMessage += tr("%n FCC atoms\n", 0, typeCounters[FCC]);
	statusMessage += tr("%n HCP atoms\n", 0, typeCounters[HCP]);
	statusMessage += tr("%n BCC atoms\n", 0, typeCounters[BCC]);
	statusMessage += tr("%n diamond atoms\n", 0, typeCounters[DIAMOND]);
	statusMessage += tr("%n other atoms", 0, typeCounters[OTHER]);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, QString(), statusMessage);
}

/******************************************************************************
* This is the actual analysis method.
******************************************************************************/
EvaluationStatus CommonNeighborAnalysisModifier::doAnalysis(TimeTicks time, bool suppressDialogs)
{
	if(calculate(input(), suppressDialogs))
		return EvaluationStatus();
	else
		return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR, tr("Calculation has been canceled by the user."));
}

/******************************************************************************
* Performs the analysis.
* Throws an exception on error.
* Returns false when the operation has been canceled by the user.
******************************************************************************/
bool CommonNeighborAnalysisModifier::calculate(AtomsObject* atomsObject, bool suppressDialogs)
{
	ProgressIndicator progress(tr("Performing common neighbor analysis (on %n processor(s))", NULL, QThread::idealThreadCount()), atomsObject->atomsCount(), suppressDialogs);

	// Build the nearest-neighbor list.
	if(!buildNeighborList()) {
		_cnaChannel->setSize(0);
		return false;
	}
	if(!nearestNeighborList() || !nearestNeighborList()->isValid()) return false;
	const NearestNeighborList& neighborList = *nearestNeighborList();

	// Reserve output array memory.
	_cnaChannel->setSize(atomsObject->atomsCount());

	// Measure computation time.
	QTime timer;
	timer.start();

	// Execute CNA analysis code for each atom in a parallel fashion.
	Kernel kernel(neighborList, _cnaChannel);
	boost::counting_iterator<int> firstAtom(0);
	boost::counting_iterator<int> lastAtom(atomsObject->atomsCount());
	QFuture<void> future = QtConcurrent::map(firstAtom, lastAtom, kernel);
	progress.waitForFuture(future);

	// Release neighbor list memory.
	nearestNeighborList()->clear();

	// Throw away results obtained so far if the user cancels the calculation.
	if(future.isCanceled()) {
		_cnaChannel->setSize(0);
		return false;
	}
	else {
		VerboseLogger() << "Common neighbor analysis took" << (timer.elapsed()/1000) << "sec." << endl;
	}

	return true;
}

/// A pair of atoms that are near-neighbors.
typedef pair<NearestNeighborList::NeighborListAtom*, NearestNeighborList::NeighborListAtom*> Bond;

void findCommonNeighbors(const NearestNeighborList::NeighborListAtom* atom1, const NearestNeighborList::NeighborListAtom* atom2, NearestNeighborList::NeighborList& commonNeighbors)
{
	for(int ni1 = atom1->neighbors.size(); ni1--; ) {
		NearestNeighborList::NeighborListAtom* n1 = atom1->neighbors[ni1];
		for(int ni2 = atom2->neighbors.size(); ni2--; ) {
			if(n1 == atom2->neighbors[ni2]) {
				commonNeighbors.append(n1);
			}
		}
	}
}

void findNeighborBonds(const NearestNeighborList::NeighborListAtom* atom1, const NearestNeighborList::NeighborListAtom* atom2, const NearestNeighborList::NeighborList& commonNeighbors, vector<Bond>& neighborBonds)
{
	for(int ni1 = commonNeighbors.size(); ni1--; ) {
		NearestNeighborList::NeighborListAtom* n1 = commonNeighbors[ni1];
		for(int ni2 = ni1; ni2--; ) {
			NearestNeighborList::NeighborListAtom* n2 = commonNeighbors[ni2];
			if(n1->hasNeighbor(n2))
				neighborBonds.push_back(Bond(n1, n2));
		}
	}
}

int getAdjacentBonds(const NearestNeighborList::NeighborListAtom* atom, vector<Bond>& bondsToProcess, vector<const NearestNeighborList::NeighborListAtom*>& atomsToProcess, vector<const NearestNeighborList::NeighborListAtom*>& atomsProcessed)
{
    atomsProcessed.push_back(atom);
    int adjacentBonds = 0;
	for(vector<Bond>::iterator bond = bondsToProcess.begin(); bond != bondsToProcess.end(); ) {
		if(atom == bond->first || atom == bond->second) {
            ++adjacentBonds;

    		if(find(atomsProcessed.begin(), atomsProcessed.end(), bond->first) == atomsProcessed.end()
    			&& find(atomsToProcess.begin(), atomsToProcess.end(), bond->first) == atomsToProcess.end())
    			atomsToProcess.push_back(bond->first);

    		if(find(atomsProcessed.begin(), atomsProcessed.end(), bond->second) == atomsProcessed.end()
    			&& find(atomsToProcess.begin(), atomsToProcess.end(), bond->second) == atomsToProcess.end())
    			atomsToProcess.push_back(bond->second);

            bond = bondsToProcess.erase(bond);
		}
		else ++bond;
	}
	return adjacentBonds;
}

int calcMaxChainLength(vector<Bond>& bondsToProcess)
{
    // Group the common bonds into clusters.
	int maxChainLength = 0;
	while(!bondsToProcess.empty()) {
        // Make a new cluster starting with the first remaining bond to be processed.
		Bond newBond = bondsToProcess.back();
		bondsToProcess.pop_back();
        vector<const NearestNeighborList::NeighborListAtom*> atomsToProcess, atomsProcessed;
		atomsToProcess.push_back(newBond.first);
		atomsToProcess.push_back(newBond.second);
		int clusterSize = 1;
        while(!atomsToProcess.empty()) {
			const NearestNeighborList::NeighborListAtom* nextAtom = atomsToProcess.back();
			atomsToProcess.pop_back();
			clusterSize += getAdjacentBonds(nextAtom, bondsToProcess, atomsToProcess, atomsProcessed);
		}
        if(clusterSize > maxChainLength)
        	maxChainLength = clusterSize;
	}
	return maxChainLength;
}

/******************************************************************************
* Determines the crystal structure type of a single atom.
******************************************************************************/
void CommonNeighborAnalysisModifier::Kernel::operator()(int atomIndex)
{
	int& result = atomTypeIndices.dataInt()[atomIndex];
	result = OTHER; // Start with assuming that the CNA type is unknown.

	const NearestNeighborList::NeighborListAtom& atom1 = nnlist.neighborLists()[atomIndex];

	if(atom1.neighbors.size() == 12) { // Detect FCC and HCP atoms each having 12 NN.
		size_t n421 = 0;
		size_t n422 = 0;
		size_t n552 = 0;
		for(int ni2 = atom1.neighbors.size(); ni2--; ) {
			const NearestNeighborList::NeighborListAtom* atom2 = atom1.neighbors[ni2];

			// Determine number of neighbors the two pair atoms have in common.
			NearestNeighborList::NeighborList commonNeighbors;
			findCommonNeighbors(&atom1, atom2, commonNeighbors);
			if(commonNeighbors.size() != 4 && commonNeighbors.size() != 5) return;

			// Determine the number of bonds among the shared neighbors.
			vector<Bond> neighborBonds;
			findNeighborBonds(&atom1, atom2, commonNeighbors, neighborBonds);
			if(neighborBonds.size() != 2 && neighborBonds.size() != 5) return;
			int numNeighborBonds = neighborBonds.size();

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds);
			if(commonNeighbors.size() == 4 && numNeighborBonds == 2) {
				if(maxChainLength == 1) n421++;
				else if(maxChainLength == 2) n422++;
				else return;
			}
			else if(commonNeighbors.size() == 5 && numNeighborBonds == 5 && maxChainLength == 2) n552++;
			else return;
		}
		if(n421 == 12) result = FCC;
		else if(n421 == 6 && n422 == 6) result = HCP;
		else if(n552 == 12) result = ICOSAHEDRAL;
	}
	else if(atom1.neighbors.size() == 14) { // Detect BCC atoms having 14 NN (in 1st and 2nd shell).
		size_t n444 = 0;
		size_t n555 = 0;
		size_t n666 = 0;
		for(int ni2 = atom1.neighbors.size(); ni2--; ) {
			const NearestNeighborList::NeighborListAtom* atom2 = atom1.neighbors[ni2];

			// Determine number of neighbors the two pair atoms have in common.
			NearestNeighborList::NeighborList commonNeighbors;
			findCommonNeighbors(&atom1, atom2, commonNeighbors);
			if(commonNeighbors.size() != 4 && commonNeighbors.size() != 5 && commonNeighbors.size() != 6) return;

			// Determine the number of bonds among the shared neighbors.
			vector<Bond> neighborBonds;
			findNeighborBonds(&atom1, atom2, commonNeighbors, neighborBonds);
			if(neighborBonds.size() != 4 && neighborBonds.size() != 5 && neighborBonds.size() != 6) return;
			int numNeighborBonds = neighborBonds.size();

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds);
			if(commonNeighbors.size() == 4 && numNeighborBonds == 4 && maxChainLength == 4) n444++;
			else if(commonNeighbors.size() == 6 && numNeighborBonds == 6 && maxChainLength == 6) n666++;
			else if(commonNeighbors.size() == 5 && numNeighborBonds == 5 && maxChainLength == 5) n555++;
			else return;
		}
		if(n666 == 8 && n444 == 6) result = BCC;
		else if(n666 == 6 && n444 == 4 && n555 == 4) result = BCC_TWIN;
	}
	else if(atom1.neighbors.size() == 9) { // Detect cg-N atoms having 9 NN.
		size_t n211 = 0;
		size_t n421 = 0;
		for(int ni2 = atom1.neighbors.size(); ni2--; ) {
			const NearestNeighborList::NeighborListAtom* atom2 = atom1.neighbors[ni2];
			// Determine number of neighbors the two pair atoms have in common.
			NearestNeighborList::NeighborList commonNeighbors;
			findCommonNeighbors(&atom1, atom2, commonNeighbors);
			if(commonNeighbors.size() != 2 && commonNeighbors.size() != 4) return;

			// Determine the number of bonds among the shared neighbors.
			vector<Bond> neighborBonds;
			findNeighborBonds(&atom1, atom2, commonNeighbors, neighborBonds);
			if(neighborBonds.size() != 1 && neighborBonds.size() != 2) return;
			int numNeighborBonds = neighborBonds.size();

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds);
			if(commonNeighbors.size() == 2 && numNeighborBonds == 1 && maxChainLength == 1) n211++;
			else if(commonNeighbors.size() == 4 && numNeighborBonds == 2 && maxChainLength == 1) n421++;
			else return;
		}
		if(n211 == 6 && n421 == 3) result = CG_N;
	}
	else if(atom1.neighbors.size() == 16) { // Detect diamond atoms having 16 NN.
		size_t n543 = 0;
		size_t n555 = 0;
		size_t n663 = 0;
		size_t n677 = 0;
		size_t n699 = 0;
		size_t n799 = 0;
		for(int ni2 = atom1.neighbors.size(); ni2--; ) {
			const NearestNeighborList::NeighborListAtom* atom2 = atom1.neighbors[ni2];

			// Determine number of neighbors the two pair atoms have in common.
			NearestNeighborList::NeighborList commonNeighbors;
			findCommonNeighbors(&atom1, atom2, commonNeighbors);
			if(commonNeighbors.size() < 5 || commonNeighbors.size() > 7) return;

			// Determine the number of bonds among the shared neighbors.
			vector<Bond> neighborBonds;
			findNeighborBonds(&atom1, atom2, commonNeighbors, neighborBonds);
			if(neighborBonds.size() < 4 || neighborBonds.size() > 9) return;
			int numNeighborBonds = neighborBonds.size();

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds);
			if(commonNeighbors.size() == 5 && numNeighborBonds == 4 && maxChainLength == 3) n543++;
			else if(commonNeighbors.size() == 6 && numNeighborBonds == 6 && maxChainLength == 3) n663++;
			else if(commonNeighbors.size() == 6 && numNeighborBonds == 7 && maxChainLength == 7) n677++;
			else if(commonNeighbors.size() == 5 && numNeighborBonds == 5 && maxChainLength == 5) n555++;
			else if(commonNeighbors.size() == 6 && numNeighborBonds == 9 && maxChainLength == 9) n699++;
			else if(commonNeighbors.size() == 7 && numNeighborBonds == 9 && maxChainLength == 9) n799++;
			else return;
		}
		if(n543 == 12 && n663 == 4) result = DIAMOND;
		else if(n555 == 6 && n677 == 3 && n699 == 1 && n543 == 3 && n799 == 3) result = DIAMOND_STACKINGFAULT;
	}
	else if(atom1.neighbors.size() == 17) { // Detect hexagonal diamond atoms having 17 NN.
		size_t n677 = 0;
		size_t n555 = 0;
		size_t n81212 = 0;
		size_t n699 = 0;
		size_t n543 = 0;
		size_t n799 = 0;
		size_t n663 = 0;
		for(int ni2 = atom1.neighbors.size(); ni2--; ) {
			const NearestNeighborList::NeighborListAtom* atom2 = atom1.neighbors[ni2];

			// Determine number of neighbors the two pair atoms have in common.
			NearestNeighborList::NeighborList commonNeighbors;
			findCommonNeighbors(&atom1, atom2, commonNeighbors);
			if(commonNeighbors.size() < 5 || commonNeighbors.size() > 8) return;

			// Determine the number of bonds among the shared neighbors.
			vector<Bond> neighborBonds;
			findNeighborBonds(&atom1, atom2, commonNeighbors, neighborBonds);
			if(neighborBonds.size() < 4 || neighborBonds.size() > 12) return;
			int numNeighborBonds = neighborBonds.size();

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds);
			if(commonNeighbors.size() == 5 && numNeighborBonds == 4 && maxChainLength == 3) n543++;
			else if(commonNeighbors.size() == 6 && numNeighborBonds == 7 && maxChainLength == 7) n677++;
			else if(commonNeighbors.size() == 6 && numNeighborBonds == 6 && maxChainLength == 3) n663++;
			else if(commonNeighbors.size() == 5 && numNeighborBonds == 5 && maxChainLength == 5) n555++;
			else if(commonNeighbors.size() == 8 && numNeighborBonds == 12 && maxChainLength == 12) n81212++;
			else if(commonNeighbors.size() == 6 && numNeighborBonds == 9 && maxChainLength == 9) n699++;
			else if(commonNeighbors.size() == 7 && numNeighborBonds == 9 && maxChainLength == 9) n799++;
			else return;
		}
		if(n677 == 6 && n555 == 6 && n81212 == 3 && n699 == 2) result = HEX_DIAMOND;
		else if(n699 == 1 && n677 == 3 && n543 == 9 && n799 == 3 && n663 == 1) result = DIAMOND_STACKINGFAULT;
	}
}

IMPLEMENT_PLUGIN_CLASS(CommonNeighborAnalysisModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void CommonNeighborAnalysisModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Common neighbor analysis"), rolloutParams, "atomviz.modifiers.common_neighbor_analysis");

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(0);

	BooleanPropertyUI* autoUpdateUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange));
	layout1->addWidget(autoUpdateUI->checkBox());

	BooleanPropertyUI* saveResultsUI = new BooleanPropertyUI(this, "storeResultsWithScene", tr("Save results in scene file"));
	layout1->addWidget(saveResultsUI->checkBox());

	QPushButton* recalcButton = new QPushButton(tr("Calculate"), rollout);
	layout1->addSpacing(6);
	layout1->addWidget(recalcButton);
	connect(recalcButton, SIGNAL(clicked(bool)), this, SLOT(onRecalculate()));

	// Status label.
	layout1->addSpacing(10);
	layout1->addWidget(statusLabel());

	// Derive a custom class from the list parameter UI to
	// give the items a color.
	class CustomRefTargetListParameterUI : public RefTargetListParameterUI {
	public:
		CustomRefTargetListParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField)
			: RefTargetListParameterUI(parentEditor, refField) {}
	protected:
		virtual QVariant getItemData(RefTarget* target, const QModelIndex& index, int role) {
			if(role == Qt::DecorationRole && target != NULL) {
				return (QColor)static_object_cast<AtomType>(target)->color();
			}
			else return RefTargetListParameterUI::getItemData(target, index, role);
		}
		/// Do not open sub-editor for selected atom type.
		virtual void openSubEditor() {}
	};

	atomsTypesPUI = new CustomRefTargetListParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(CommonNeighborAnalysisModifier, _atomTypesList));
	layout1->addSpacing(10);
	layout1->addWidget(new QLabel(tr("Type colors:")));
	layout1->addWidget(atomsTypesPUI->listWidget());
	connect(atomsTypesPUI->listWidget(), SIGNAL(doubleClicked(const QModelIndex&)), this, SLOT(onDoubleClickAtomType(const QModelIndex&)));

	// Open a sub-editor for the NearestNeighborList sub-object.
	SubObjectParameterUI* subEditorUI = new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _nearestNeighborList), rolloutParams.before(rollout));
}

/******************************************************************************
* Is called when the user presses the Recalculate button.
******************************************************************************/
void CommonNeighborAnalysisModifierEditor::onRecalculate()
{
	if(!editObject()) return;
	CommonNeighborAnalysisModifier* modifier = static_object_cast<CommonNeighborAnalysisModifier>(editObject());
	try {
		modifier->performAnalysis(ANIM_MANAGER.time());
	}
	catch(Exception& ex) {
		ex.prependGeneralMessage(tr("Failed to perform common neighbor analysis (CNA)."));
		ex.showError();
	}
}

/******************************************************************************
* Is called when the user has double-clicked on one of the CNA
* atom types in the list widget.
******************************************************************************/
void CommonNeighborAnalysisModifierEditor::onDoubleClickAtomType(const QModelIndex& index)
{
	// Let the user select a color for the atom type.
	AtomType* atype = static_object_cast<AtomType>(atomsTypesPUI->selectedObject());
	if(!atype || !atype->colorController()) return;

	QColor oldColor = Color(atype->colorController()->getCurrentValue());
	QColor newColor = QColorDialog::getColor(oldColor, container());
	if(!newColor.isValid() || newColor == oldColor) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Change color"));
	atype->colorController()->setCurrentValue(Color(newColor));
	UNDO_MANAGER.endCompoundOperation();
}

};	// End of namespace AtomViz
