///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//
//  Parts of the Ackland analysis routine have been adopted from the
//  Ackland routine in LAMMPS written by Gerolf Ziegenhain.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <boost/iterator/counting_iterator.hpp>

#include "AcklandAnalysisModifier.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(AcklandAnalysisModifier, AtomsObjectAnalyzerBase)
DEFINE_VECTOR_REFERENCE_FIELD(AcklandAnalysisModifier, AtomType, "AtomTypes", _atomTypesList)
DEFINE_REFERENCE_FIELD(AcklandAnalysisModifier, AtomTypeDataChannel, "CNATypeChannel", _cnaChannel)
SET_PROPERTY_FIELD_LABEL(AcklandAnalysisModifier, _atomTypesList, "Atom Types")

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
AcklandAnalysisModifier::AcklandAnalysisModifier(bool isLoading)
	: AtomsObjectAnalyzerBase(isLoading)
{
	INIT_PROPERTY_FIELD(AcklandAnalysisModifier, _atomTypesList);
	INIT_PROPERTY_FIELD(AcklandAnalysisModifier, _cnaChannel);

	if(!isLoading) {

		// Create the internal atom types.
		intrusive_ptr<AtomType> fccType(new AtomType());
		fccType->setName(tr("FCC"));
		fccType->setColor(Color(0.4f, 1.0f, 0.4f));
		_atomTypesList.push_back(fccType);

		intrusive_ptr<AtomType> hcpType(new AtomType());
		hcpType->setName(tr("HCP"));
		hcpType->setColor(Color(1.0f, 0.4f, 0.4f));
		_atomTypesList.push_back(hcpType);

		intrusive_ptr<AtomType> bccType(new AtomType());
		bccType->setName(tr("BCC"));
		bccType->setColor(Color(0.4f, 0.4f, 1.0f));
		_atomTypesList.push_back(bccType);

		intrusive_ptr<AtomType> icosahedralType(new AtomType());
		icosahedralType->setName(tr("Icosahedral"));
		icosahedralType->setColor(Color(1.0f, 1.0f, 0.7f));
		_atomTypesList.push_back(icosahedralType);

		intrusive_ptr<AtomType> noneType(new AtomType());
		noneType->setName(tr("Other"));
		noneType->setColor(Color(1.0f, 1.0f, 1.0f));
		_atomTypesList.push_back(noneType);

		// Create data channel for the computation results.
		_cnaChannel = new AtomTypeDataChannel(DataChannel::CNATypeChannel);
	}
}

/******************************************************************************
* Applies the previously calculated analysis results to the atoms object.
******************************************************************************/
EvaluationStatus AcklandAnalysisModifier::applyResult(TimeTicks time, TimeInterval& validityInterval)
{
	// Check if it is still valid.
	if(input()->atomsCount() != _cnaChannel->size())
		throw Exception(tr("Number of atoms of input object has changed. Analysis results became invalid."));

	// Create a copy of the CNA atom type channel that stores the calculated atom types.
	CloneHelper cloneHelper;
	AtomTypeDataChannel::SmartPtr cnaAtomTypeChannel = cloneHelper.cloneObject(_cnaChannel, true);

	// Remove old atoms types.
	while(!cnaAtomTypeChannel->atomTypes().empty())
		cnaAtomTypeChannel->removeAtomType(cnaAtomTypeChannel->atomTypes().front());

	// Add analysis atom types.
	Vector3 atomTypeColors[NumCNAAtomTypes];
	int index = 0;
	Q_FOREACH(AtomType* atype, atomTypes()) {
		// Add only a copy of the atom type to the AtomsObject and keep the original.
		cnaAtomTypeChannel->insertAtomType(cloneHelper.cloneObject(atype, true).get());
		atype->colorController()->getValue(time, atomTypeColors[index++], validityInterval);
	}

	// Put the new channel into the output object.
	output()->insertDataChannel(cnaAtomTypeChannel);

	int typeCounters[NumCNAAtomTypes];
	for(size_t i = 0; i < NumCNAAtomTypes; i++)
		typeCounters[i] = 0;
	const int* in = _cnaChannel->constDataInt();
	Vector3* colorIter = outputStandardChannel(DataChannel::ColorChannel)->dataVector3();
	for(size_t i = _cnaChannel->size(); i != 0; --i, ++in) {
		OVITO_ASSERT(*in < NumCNAAtomTypes && *in >= 0);
		*colorIter++ = atomTypeColors[*in];
		typeCounters[*in]++;
	}

	QString statusMessage;
	statusMessage += tr("%n FCC atoms\n", 0, typeCounters[FCC]);
	statusMessage += tr("%n HCP atoms\n", 0, typeCounters[HCP]);
	statusMessage += tr("%n BCC atoms\n", 0, typeCounters[BCC]);
	statusMessage += tr("%n other atoms", 0, typeCounters[OTHER]);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, QString(), statusMessage);
}

/******************************************************************************
* This is the actual analysis method.
******************************************************************************/
EvaluationStatus AcklandAnalysisModifier::doAnalysis(TimeTicks time, bool suppressDialogs)
{
	if(calculate(input(), suppressDialogs))
		return EvaluationStatus();
	else
		return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR, tr("Calculation has been canceled by the user."));
}

/******************************************************************************
* Performs the analysis.
* Throws an exception on error.
* Returns false when the operation has been canceled by the user.
******************************************************************************/
bool AcklandAnalysisModifier::calculate(AtomsObject* atomsObject, bool suppressDialogs)
{
	ProgressIndicator progress(tr("Performing Ackland analysis (on %n processor(s))", NULL, QThread::idealThreadCount()), atomsObject->atomsCount(), suppressDialogs);

	// Prepare the neighbor list.
	OnTheFlyNeighborList neighborList(atomsObject, nearestNeighborList()->nearestNeighborCutoff());

	// Reserve output array memory.
	_cnaChannel->setSize(atomsObject->atomsCount());

	// Measure computation time.
	QTime timer;
	timer.start();

	// Execute analysis code for each atom in a parallel fashion.
	Kernel kernel(neighborList, _cnaChannel);
	boost::counting_iterator<int> firstAtom(0);
	boost::counting_iterator<int> lastAtom(atomsObject->atomsCount());
	QFuture<void> future = QtConcurrent::map(firstAtom, lastAtom, kernel);
	progress.waitForFuture(future);

	// Release neighbor list memory.
	nearestNeighborList()->clear();

	// Throw away results obtained so far if the user cancels the calculation.
	if(future.isCanceled()) {
		_cnaChannel->setSize(0);
		return false;
	}
	else {
		VerboseLogger() << "Ackland analysis took" << (timer.elapsed()/1000) << "sec." << endl;
	}

	return true;
}

/******************************************************************************
* find k smallest values in array of length n.
* routine sorts auxiliary array at same time
******************************************************************************/
void select2(int k, int n, FloatType *arr, int *iarr)
{
#define SWAP(a,b)   tmp = a; a = b; b = tmp;
#define ISWAP(a,b) itmp = a; a = b; b = itmp

	int i, ir, j, l, mid, ia, itmp;
	FloatType a, tmp;

	arr--;
	iarr--;
	l = 1;
	ir = n;
	for (;;) {
		if (ir <= l + 1) {
			if (ir == l + 1 && arr[ir] < arr[l]) {
				SWAP(arr[l],arr[ir]);
				ISWAP(iarr[l],iarr[ir]);
			}
			return;
		} else {
			mid = (l + ir) >> 1;
			SWAP(arr[mid],arr[l+1]);
			ISWAP(iarr[mid],iarr[l+1]);
			if (arr[l] > arr[ir]) {
				SWAP(arr[l],arr[ir]);
				ISWAP(iarr[l],iarr[ir]);
			}
			if (arr[l + 1] > arr[ir]) {
				SWAP(arr[l+1],arr[ir]);
				ISWAP(iarr[l+1],iarr[ir]);
			}
			if (arr[l] > arr[l + 1]) {
				SWAP(arr[l],arr[l+1]);
				ISWAP(iarr[l],iarr[l+1]);
			}
			i = l + 1;
			j = ir;
			a = arr[l + 1];
			ia = iarr[l + 1];
			for (;;) {
				do i++; while (arr[i] < a);
				do j--; while (arr[j] > a);
				if(j < i) break;
				SWAP(arr[i],arr[j]);
				ISWAP(iarr[i],iarr[j]);
			}
			arr[l + 1] = arr[j];
			arr[j] = a;
			iarr[l + 1] = iarr[j];
			iarr[j] = ia;
			if (j >= k)
				ir = j - 1;
			if (j <= k)
				l = i;
		}
	}
}

/******************************************************************************
* Determines the crystal structure type of a single atom.
******************************************************************************/
void AcklandAnalysisModifier::Kernel::operator()(int atomIndex)
{
	int& result = atomTypeIndices.dataInt()[atomIndex];
	result = OTHER; // Start with assuming that the CNA type is unknown.

	QVarLengthArray<FloatType, 64> distsq;
	QVarLengthArray<int, 64> nearest;
	QVarLengthArray<Vector3, 64> delta;

	for(OnTheFlyNeighborList::iterator neighborIter(nnlist, atomIndex); !neighborIter.atEnd(); neighborIter.next()) {
		nearest.append(nearest.size());
		distsq.append(LengthSquared(neighborIter.delta()));
		delta.append(neighborIter.delta());
	}
	int n = distsq.size();
	if(n < 6) return;

	select2(6, n, distsq.data(), nearest.data());

	// Mean squared separation
	FloatType r0_sq = 0.0;
	for(int j = 0; j < 6; j++)
		r0_sq += distsq[j];
	r0_sq /= 6.0;

	// n0 near neighbors with: distsq<1.45*r0_sq
	// n1 near neighbors with: distsq<1.55*r0_sq

	QVarLengthArray<int, 64> nearest_n0(n);
	QVarLengthArray<int, 64> nearest_n1(n);

	double n0_dist_sq = 1.45 * r0_sq, n1_dist_sq = 1.55 * r0_sq;
	int n0 = 0, n1 = 0;
	for(int j = 0; j < n; j++) {
		if(distsq[j] < n1_dist_sq) {
			nearest_n1[n1++] = nearest[j];
			if(distsq[j] < n0_dist_sq) {
				nearest_n0[n0++] = nearest[j];
			}
		}
	}

	// Evaluate all angles <(r_ij,rik) for all n0 particles with: distsq<1.45*r0_sq
	FloatType bond_angle;
	FloatType norm_j, norm_k;
	int chi[8] = {0, 0, 0, 0, 0, 0, 0, 0};
	FloatType x_ij, y_ij, z_ij, x_ik, y_ik, z_ik;
	for(int j = 0; j < n0; j++) {
		x_ij = -delta[nearest_n0[j]].X;
		y_ij = -delta[nearest_n0[j]].Y;
		z_ij = -delta[nearest_n0[j]].Z;
		norm_j = sqrt (x_ij*x_ij + y_ij*y_ij + z_ij*z_ij);
		if(norm_j <= 0.) continue;
		for(int k = j+1; k < n0; k++) {
			x_ik = -delta[nearest_n0[k]].X;
			y_ik = -delta[nearest_n0[k]].Y;
			z_ik = -delta[nearest_n0[k]].Z;
			norm_k = sqrt(x_ik*x_ik + y_ik*y_ik + z_ik*z_ik);
			if(norm_k <= 0.0)
				continue;

			bond_angle = (x_ij*x_ik + y_ij*y_ik + z_ij*z_ik) / (norm_j*norm_k);

			// Build histogram for identifying the relevant peaks.
			if(-1. <= bond_angle && bond_angle < -0.945) { chi[0]++; }
			else if(-0.945 <= bond_angle && bond_angle < -0.915) { chi[1]++; }
			else if(-0.915 <= bond_angle && bond_angle < -0.755) { chi[2]++; }
			else if(-0.755 <= bond_angle && bond_angle < -0.195) { chi[3]++; }
			else if(-0.195 <= bond_angle && bond_angle < 0.195) { chi[4]++; }
			else if(0.195 <= bond_angle && bond_angle < 0.245) { chi[5]++; }
			else if(0.245 <= bond_angle && bond_angle < 0.795) { chi[6]++; }
			else if(0.795 <= bond_angle && bond_angle < 1.) { chi[7]++; }
		}
	}

	// Calculate deviations from the different lattice structures.
	FloatType delta_bcc = 0.35*chi[4]/(FloatType)(chi[5]+chi[6]-chi[4]),
		delta_cp = abs(1.-(FloatType)chi[6]/24.),
		delta_fcc = 0.61*(abs((FloatType)(chi[0]+chi[1]-6.))+(FloatType)chi[2])/6.,
		delta_hcp = (abs((FloatType)chi[0]-3.)+abs((FloatType)chi[0]+(FloatType)chi[1]+(FloatType)chi[2]+(FloatType)chi[3]-9.))/12.;

	// Identification of the local structure according to the reference
	if(chi[0] == 7)       { delta_bcc = 0.; }
	else if(chi[0] == 6)  { delta_fcc = 0.; }
	else if(chi[0] <= 3)  { delta_hcp = 0.; }

	if(chi[7] > 0.)
		result = OTHER;
	else if(chi[4] < 3.) {
		if(n1 > 13 || n1 < 11)
			result = OTHER;
		else
			result = ICOSAHEDRAL;
	}
	else if(delta_bcc <= delta_cp) {
		if(n1 < 11)
			result = OTHER;
		else
			result = BCC;
	}
	else if(n1 > 12 || n1 < 11)
		result = OTHER;
	else if(delta_fcc < delta_hcp)
		result = FCC;
	else
		result = HCP;
}

IMPLEMENT_PLUGIN_CLASS(AcklandAnalysisModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void AcklandAnalysisModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Ackland analysis"), rolloutParams, "atomviz.modifiers.ackland_analysis");

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(0);

	BooleanPropertyUI* autoUpdateUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange));
	layout1->addWidget(autoUpdateUI->checkBox());

	BooleanPropertyUI* saveResultsUI = new BooleanPropertyUI(this, "storeResultsWithScene", tr("Save results in scene file"));
	layout1->addWidget(saveResultsUI->checkBox());

	QPushButton* recalcButton = new QPushButton(tr("Calculate"), rollout);
	layout1->addSpacing(6);
	layout1->addWidget(recalcButton);
	connect(recalcButton, SIGNAL(clicked(bool)), this, SLOT(onRecalculate()));

	// Status label.
	layout1->addSpacing(10);
	layout1->addWidget(statusLabel());

	// Derive a custom class from the list parameter UI to
	// give the items a color.
	class CustomRefTargetListParameterUI : public RefTargetListParameterUI {
	public:
		CustomRefTargetListParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField)
			: RefTargetListParameterUI(parentEditor, refField) {}
	protected:
		virtual QVariant getItemData(RefTarget* target, const QModelIndex& index, int role) {
			if(role == Qt::DecorationRole && target != NULL) {
				return (QColor)static_object_cast<AtomType>(target)->color();
			}
			else return RefTargetListParameterUI::getItemData(target, index, role);
		}
		/// Do not open sub-editor for selected atom type.
		virtual void openSubEditor() {}
	};

	atomsTypesPUI = new CustomRefTargetListParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AcklandAnalysisModifier, _atomTypesList));
	layout1->addSpacing(10);
	layout1->addWidget(new QLabel(tr("Type colors:")));
	layout1->addWidget(atomsTypesPUI->listWidget());
	connect(atomsTypesPUI->listWidget(), SIGNAL(doubleClicked(const QModelIndex&)), this, SLOT(onDoubleClickAtomType(const QModelIndex&)));

	// Open a sub-editor for the NearestNeighborList sub-object.
	SubObjectParameterUI* subEditorUI = new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _nearestNeighborList), rolloutParams.before(rollout));
}

/******************************************************************************
* Is called when the user presses the Recalculate button.
******************************************************************************/
void AcklandAnalysisModifierEditor::onRecalculate()
{
	if(!editObject()) return;
	AcklandAnalysisModifier* modifier = static_object_cast<AcklandAnalysisModifier>(editObject());
	try {
		modifier->performAnalysis(ANIM_MANAGER.time());
	}
	catch(Exception& ex) {
		ex.prependGeneralMessage(tr("Failed to perform Ackland analysis."));
		ex.showError();
	}
}

/******************************************************************************
* Is called when the user has double-clicked on one of the CNA
* atom types in the list widget.
******************************************************************************/
void AcklandAnalysisModifierEditor::onDoubleClickAtomType(const QModelIndex& index)
{
	// Let the user select a color for the atom type.
	AtomType* atype = static_object_cast<AtomType>(atomsTypesPUI->selectedObject());
	if(!atype || !atype->colorController()) return;

	QColor oldColor = Color(atype->colorController()->getCurrentValue());
	QColor newColor = QColorDialog::getColor(oldColor, container());
	if(!newColor.isValid() || newColor == oldColor) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Change Color"));
	atype->colorController()->setCurrentValue(Color(newColor));
	UNDO_MANAGER.endCompoundOperation();
}

};	// End of namespace AtomViz
