///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/ViewportPanel.h>
#include <core/data/units/ParameterUnit.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/ApplicationManager.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/gui/properties/IntegerPropertyUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/utilities/ProgressIndicator.h>

#include "AmbientLightingModifier.h"
#include <atomviz/atoms/AtomsRenderer.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(AmbientLightingModifier, AtomsObjectAnalyzerBase)
DEFINE_REFERENCE_FIELD(AmbientLightingModifier, FloatController, "Intensity", _intensity)
DEFINE_PROPERTY_FIELD(AmbientLightingModifier, "SamplingLevel", _samplingLevel)
DEFINE_PROPERTY_FIELD(AmbientLightingModifier, "BufferResolution", _bufferResolution)
SET_PROPERTY_FIELD_LABEL(AmbientLightingModifier, _intensity, "Intensity")
SET_PROPERTY_FIELD_LABEL(AmbientLightingModifier, _samplingLevel, "Sampling level")
SET_PROPERTY_FIELD_LABEL(AmbientLightingModifier, _bufferResolution, "Buffer resolution")
SET_PROPERTY_FIELD_UNITS(AmbientLightingModifier, _intensity, PercentParameterUnit)

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
AmbientLightingModifier::AmbientLightingModifier(bool isLoading)
	: AtomsObjectAnalyzerBase(isLoading)
{
	INIT_PROPERTY_FIELD(AmbientLightingModifier, _intensity);
	INIT_PROPERTY_FIELD(AmbientLightingModifier, _samplingLevel);
	INIT_PROPERTY_FIELD(AmbientLightingModifier, _bufferResolution);
	if(!isLoading) {
		_intensity = CONTROLLER_MANAGER.createDefaultController<FloatController>();

		setIntensity(0.7f);
		setSamplingLevel(3);
		setBufferResolution(4);
	}
}

/******************************************************************************
* Asks the modifier for its validity interval at the given time.
******************************************************************************/
TimeInterval AmbientLightingModifier::modifierValidity(TimeTicks time)
{
	TimeInterval interval = TimeForever;
	_intensity->validityInterval(time, interval);
	return interval;
}

/******************************************************************************
* Applies the previously calculated analysis results to the atoms object.
******************************************************************************/
EvaluationStatus AmbientLightingModifier::applyResult(TimeTicks time, TimeInterval& validityInterval)
{
	// Check if it is still valid.
	if(input()->atomsCount() != ambientArray.size())
		throw Exception(tr("Number of atoms of input object has changed. Lighting state became invalid."));

	FloatType intensity = 1.0;
	if(intensityController()) intensityController()->getValue(time, intensity, validityInterval);
	if(intensity < 0.0)
		throw Exception(tr("Invalid intensity value."));
	if(intensity > 1.0) intensity = 1.0;

	// Get the color channel copy.
	DataChannel* colorChannel = outputStandardChannel(DataChannel::ColorChannel);

	// Dim the atom colors by their respective ambient value.
	QVector<unsigned int>::const_iterator ambientp = ambientArray.begin();
	Vector3* c = colorChannel->dataVector3();
	if(inputStandardChannel(DataChannel::ColorChannel) != NULL) {
		for(size_t i=colorChannel->size(); i != 0; --i, ++c) {
			FloatType val = (FloatType)*ambientp++ / maxAmbient;
			val = min((FloatType)1.0, (FloatType)1.0 - intensity + val);
			c->X *= val;
			c->Y *= val;
			c->Z *= val;
		}
	}
	else {
		QVector<Color> oldColors = input()->getAtomColors(time, validityInterval);
		QVector<Color>::const_iterator oldc = oldColors.constBegin();
		for(size_t i=colorChannel->size(); i != 0; --i, ++oldc, ++c) {
			FloatType val = (FloatType)*ambientp++ / maxAmbient;
			val = min((FloatType)1.0, (FloatType)1.0 - intensity + val);
			c->X = oldc->r * val;
			c->Y = oldc->g * val;
			c->Z = oldc->b * val;
		}
	}

	return EvaluationStatus();
}

static Matrix4 sphericalSampling1[] = {
	AffineTransformation::rotation(Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(0,1,0), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,0,0), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(0,-1,0), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 180.0f / 180.0f * FLOATTYPE_PI)),
};

static Matrix4 sphericalSampling2[] = {
	AffineTransformation::rotation(Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(0,1,0), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,0,0), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(0,-1,0), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,1,0), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,1,0), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,-1,0), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,-1,0), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,1,0), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,1,0), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,-1,0), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,-1,0), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 180.0f / 180.0f * FLOATTYPE_PI)),
};

static Matrix4 sphericalSampling3[] = {
	AffineTransformation::rotation(Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 60.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 120.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 180.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 240.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 300.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,1,0), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,1,0), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,-1,0), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,-1,0), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,1,0), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,1,0), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,-1,0), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,-1,0), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 180.0f / 180.0f * FLOATTYPE_PI)),
};

static Matrix4 sphericalSampling4[] = {
	AffineTransformation::rotation(Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 180.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 225.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 270.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 315.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 20.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 65.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 110.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 155.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 200.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 245.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 290.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 335.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 20.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 65.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 110.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 155.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 200.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 245.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 290.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 335.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,1,0), 30.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,1,0), 30.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,-1,0), 30.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,-1,0), 30.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,1,0), 150.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,1,0), 150.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,-1,0), 150.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(-1,-1,0), 150.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 180.0f / 180.0f * FLOATTYPE_PI)),
};

static Matrix4 sphericalSampling5[] = {
	AffineTransformation::rotation(Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 30.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 60.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 120.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 150.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 180.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 210.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 240.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 270.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 300.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 90.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 330.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 20.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 65.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 110.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 155.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 200.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 245.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 290.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 60.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 335.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 20.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 65.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 110.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 155.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 200.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 245.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 290.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 120.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 335.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 30.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 30.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 30.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 30.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 30.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 180.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 30.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 225.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 30.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 270.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 30.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 315.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 150.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 0.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 150.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 45.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 150.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 90.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 150.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 135.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 150.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 180.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 150.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 225.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 150.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 270.0f / 180.0f * FLOATTYPE_PI)),
	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 150.0f / 180.0f * FLOATTYPE_PI) * Rotation(Vector3(0,0,1), 315.0f / 180.0f * FLOATTYPE_PI)),

	AffineTransformation::rotation(Rotation(Vector3(1,0,0), 180.0f / 180.0f * FLOATTYPE_PI)),
};

/******************************************************************************
* Calculates the ambient lighting.
******************************************************************************/
EvaluationStatus AmbientLightingModifier::doAnalysis(TimeTicks time, bool suppressDialogs)
{
	// Validty interval.
	TimeInterval validityInterval = TimeForever;

	// Release old data
	ambientArray.clear();

	bool useOnscreenRendering = false;

	// ATI Mobility Radeon X1400 driver's support of the frame buffer objects extension is buggy.
	// Have to do on-screen rendering on this platform.
	if(QByteArray((const char*)glGetString(GL_RENDERER)).contains("ATI Mobility Radeon X1400"))
		useOnscreenRendering = true;

	// Check for the required OpenGL extensions to render the atoms.
	if(APPLICATION_MANAGER.guiMode() == false || VIEWPORT_MANAGER.activeViewport() == NULL)
		throw Exception(tr("Cannot perform ambient lighting calculation in non-interactive mode."));
	const OpenGLExtensions& ogl = *(VIEWPORT_MANAGER.activeViewport());
	if(!ogl.hasFrameBufferExtension() && !QGLPixelBuffer::hasOpenGLPbuffers()) {
		MsgLogger() << tr("Warning: The OpenGL driver does not support the EXT_framebuffer_object or the ARB_pixel_buffer_object extension. Falling back to on-screen rendering for ambient lighting.") << endl;
		useOnscreenRendering = true;
	}

	// Get the atom positions.
	DataChannel* posChannel = expectStandardChannel(DataChannel::PositionChannel);
	DataChannel* radiusChannel = inputStandardChannel(DataChannel::RadiusChannel);
	if(!posChannel->size()) throw Exception(tr("There are no atoms."));
	size_t numAtoms = posChannel->size();

	// We only have 16 million colors available.
	if(numAtoms > 0xFFFFFF)
		throw Exception(tr("Number of input atoms (%1) exceeds upper limit of %2 atoms for lighting calculation.").arg(numAtoms).arg(0xFFFFFF));

	/// The size of the off-screen render buffer in pixels.
	int renderBufferSize;
	int bufferResolution = this->bufferResolution();

	if(bufferResolution <= 0) renderBufferSize = 64;
	else if(bufferResolution >= 5) renderBufferSize = 2048;
	else renderBufferSize = 64 << bufferResolution;

	int samplingLevel = this->samplingLevel();

	const Matrix4* samplingPoints;
	size_t numSamplingPoints;

	switch(samplingLevel) {
	case 0: samplingPoints = sphericalSampling1; numSamplingPoints = sizeof(sphericalSampling1)/sizeof(sphericalSampling1[0]); break;
	case 1: samplingPoints = sphericalSampling2; numSamplingPoints = sizeof(sphericalSampling2)/sizeof(sphericalSampling2[0]); break;
	case 2: samplingPoints = sphericalSampling3; numSamplingPoints = sizeof(sphericalSampling3)/sizeof(sphericalSampling3[0]); break;
	case 3: samplingPoints = sphericalSampling4; numSamplingPoints = sizeof(sphericalSampling4)/sizeof(sphericalSampling4[0]); break;
	case 4: samplingPoints = sphericalSampling5; numSamplingPoints = sizeof(sphericalSampling5)/sizeof(sphericalSampling5[0]); break;
	default: samplingPoints = sphericalSampling1; numSamplingPoints = sizeof(sphericalSampling1)/sizeof(sphericalSampling1[0]); break;
	}

	ProgressIndicator progress(tr("Computing ambient lighting."), numSamplingPoints, useOnscreenRendering || suppressDialogs);

	Window3D* renderingWindow = NULL;
	Window3DContainer* renderingContainer;
	if(useOnscreenRendering) {
		// Create the rendering window for on-screen rendering.

		class OnscreenRenderingWindow : public Window3D {
		public:
			OnscreenRenderingWindow(Window3DContainer* container) : Window3D(container) {}
		protected:
			virtual void renderWindow() {}
		};

		renderingContainer = new Window3DContainer();
		renderingContainer->resize(renderBufferSize, renderBufferSize);
		renderingContainer->move(0,0);
		renderingWindow = new OnscreenRenderingWindow(renderingContainer);
		renderingWindow->setGeometry(renderingContainer->rect());
		renderingContainer->show();
	}
	else renderingWindow = VIEWPORT_MANAGER.activeViewport();

	{

	// Prepare atom renderer
	AtomsRenderer atomRenderer;
	atomRenderer.prepare(renderingWindow, true);
	atomRenderer.beginAtoms(numAtoms);
	const Point3* posIter = posChannel->constDataPoint3();
	QVector<FloatType> radiusArray = input()->getAtomRadii(time, validityInterval);
	QVector<FloatType>::const_iterator radiusIter = radiusArray.begin();
	for(size_t i=0; i<numAtoms; i++, ++posIter) {
		atomRenderer.specifyAtom(*posIter,
			(GLubyte)(i & 255), (GLubyte)((i>>8) & 255), (GLubyte)((i>>16) & 255), *radiusIter++);
	}
	atomRenderer.endAtoms();

	// Allocate ambient array.
	ambientArray.fill(0, numAtoms);

	GLuint frameBuffer;
	GLuint renderBuffer;
	GLuint depthRenderBuffer;
	scoped_ptr<QGLPixelBuffer> pbuffer;

	if(useOnscreenRendering) {
		renderingWindow->makeCurrent();
	}
	else if(renderingWindow->hasFrameBufferExtension()) {
		// Use OpenGL frame buffer for off-screen rendering.

	    // Create the framebuffer object - make sure to have a current
	    // context before creating it.
		renderingWindow->makeCurrent();

		// Create a frame-buffer object and a render-buffer object for color and depth channel.
		VerboseLogger() << logdate << "Allocating OpenGL off-screen frame buffer:" << renderBufferSize << "x" << renderBufferSize << "pixels" << endl;
		CHECK_OPENGL(renderingWindow->glGenFramebuffersEXT(1, &frameBuffer));
		CHECK_OPENGL(renderingWindow->glGenRenderbuffersEXT(1, &renderBuffer));
		CHECK_OPENGL(renderingWindow->glGenRenderbuffersEXT(1, &depthRenderBuffer));

		// Initialize the render-buffers.
		VerboseLogger() << logdate << "Binding OpenGL framebuffer." << endl;
		CHECK_OPENGL(renderingWindow->glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, depthRenderBuffer));
		CHECK_OPENGL(renderingWindow->glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT24, renderBufferSize, renderBufferSize));
		CHECK_OPENGL(renderingWindow->glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, renderBuffer));
		CHECK_OPENGL(renderingWindow->glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_RGBA8, renderBufferSize, renderBufferSize));

		// Activate the off-screen frame-buffer and associate the render-buffer objects.
		CHECK_OPENGL(renderingWindow->glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, frameBuffer));
		CHECK_OPENGL(renderingWindow->glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_RENDERBUFFER_EXT, renderBuffer));
		CHECK_OPENGL(renderingWindow->glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, depthRenderBuffer));

		// Check for errors...
		GLenum status = renderingWindow->glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
		if(status != GL_FRAMEBUFFER_COMPLETE_EXT)
			throw Exception(tr("The OpenGL framebuffer object is not valid."));

		CHECK_OPENGL(glReadBuffer(GL_COLOR_ATTACHMENT0_EXT));
	}
	else {
		// Use OpenGL pixel buffer for off-screen rendering.

		VerboseLogger() << logdate << "Allocating OpenGL off-screen pixel buffer:" << renderBufferSize << "x" << renderBufferSize << "pixels" << endl;
		pbuffer.reset(new QGLPixelBuffer(renderBufferSize, renderBufferSize, QGLFormat(QGL::DepthBuffer | QGL::Rgba | QGL::DirectRendering | QGL::NoAccumBuffer | QGL::NoStencilBuffer)));

		if(!pbuffer->makeCurrent())
			throw Exception(tr("The OpenGL pixel buffer could not be activated."));
	}

	// Prepare OpenGL rendering state.
	CHECK_OPENGL(glPushAttrib(GL_LIGHTING_BIT|GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_ENABLE_BIT|GL_VIEWPORT_BIT|GL_TRANSFORM_BIT));
	CHECK_OPENGL(glDisable(GL_SCISSOR_TEST));
	CHECK_OPENGL(glEnable(GL_DEPTH_TEST));
	CHECK_OPENGL(glDepthFunc(GL_LESS));
	CHECK_OPENGL(glDepthMask(GL_TRUE));
	CHECK_OPENGL(glViewport(0, 0, renderBufferSize, renderBufferSize));
	CHECK_OPENGL(glClearColor(1.0f, 1.0f, 1.0f, 1.0f));

	// Setup view and projection matrix.
	CHECK_OPENGL(glMatrixMode(GL_MODELVIEW));
	CHECK_OPENGL(glPushMatrix());
	CHECK_OPENGL(glMatrixMode(GL_PROJECTION));
	CHECK_OPENGL(glPushMatrix());
	const Box3& bb = atomRenderer.boundingBox();
	Vector3 center = bb.center() - ORIGIN;
	FloatType diameter = Length(bb.size());
    FloatType znear = -diameter / 2.0;
	FloatType zfar  = +diameter / 2.0;
	FloatType zoom  =  diameter / 2.0;
	Matrix4 projMatrix = Matrix4::ortho(-zoom, zoom, -zoom, zoom, znear, zfar);
	CHECK_OPENGL(glLoadMatrix(projMatrix.constData()));
	CHECK_OPENGL(glMatrixMode(GL_MODELVIEW));

	// Allocate user-memory pixel buffer.
	scoped_array<quint32> pixelData(new quint32[renderBufferSize * renderBufferSize]);

	for(size_t samplingIndex=0; samplingIndex<numSamplingPoints; samplingIndex++) {
		if(progress.isCanceled()) break;
		progress.setValue(samplingIndex);

		CHECK_OPENGL(glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT));

		glLoadMatrix(samplingPoints[samplingIndex].constData());
		glTranslatef(-(GLfloat)center.X, -(GLfloat)center.Y, -(GLfloat)center.Z);

		atomRenderer.renderOffscreen(false, projMatrix, QSize(renderBufferSize, renderBufferSize));

		// Flush the buffer to force drawing of all objects thus far.
		CHECK_OPENGL(glFlush());
		if(useOnscreenRendering)
			renderingWindow->swapBuffers();

		if(useOnscreenRendering || renderingWindow->hasFrameBufferExtension()) {
			CHECK_OPENGL(glReadPixels(0, 0, renderBufferSize, renderBufferSize, GL_RGBA, GL_UNSIGNED_BYTE, pixelData.get()));

			//QImage image((const uchar*)pixelData.get(), renderBufferSize, renderBufferSize, QImage::Format_ARGB32);
			//image.save(QString("lighting_%1.png").arg(samplingIndex));

			const quint32* p = pixelData.get();
			for(size_t i=renderBufferSize*renderBufferSize; i--; ++p) {
				quint32 index = (*p) & 0xFFFFFF;
				OVITO_ASSERT(index < numAtoms || index == 0xFFFFFF);
				if(index != 0xFFFFFF) ambientArray[index]++;
			}
		}
	}
	progress.setValue(numSamplingPoints);

	CHECK_OPENGL(glMatrixMode(GL_PROJECTION));
	CHECK_OPENGL(glPopMatrix());
	CHECK_OPENGL(glMatrixMode(GL_MODELVIEW));
	CHECK_OPENGL(glPopMatrix());

	// Restore old settings.
	CHECK_OPENGL(glPopAttrib());

	if(!useOnscreenRendering && renderingWindow->hasFrameBufferExtension()) {
		// Make the normal window the target again.
		renderingWindow->glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

		// Clean up
		renderingWindow->glDeleteFramebuffersEXT(1, &frameBuffer);
		renderingWindow->glDeleteRenderbuffersEXT(1, &renderBuffer);
		renderingWindow->glDeleteRenderbuffersEXT(1, &depthRenderBuffer);
	}

	}

	if(useOnscreenRendering)
		delete renderingContainer;

	// Determine the maximum ambient value.
	maxAmbient = 0;

	QVector<unsigned int>::const_iterator ambientp = ambientArray.begin();
	for(; ambientp != ambientArray.end(); ++ambientp)
		if(*ambientp > maxAmbient) maxAmbient = *ambientp;

	VerboseLogger() << logdate << "Ambient lighting finished." << endl;

	intersectAnalysisValidityInterval(validityInterval);
	return EvaluationStatus();
}

/******************************************************************************
* Saves the class' contents to the given stream.
******************************************************************************/
void AmbientLightingModifier::saveToStream(ObjectSaveStream& stream)
{
	AtomsObjectAnalyzerBase::saveToStream(stream);

	stream.beginChunk(0x661AC93);
	stream << maxAmbient;
	stream << ambientArray;
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream.
******************************************************************************/
void AmbientLightingModifier::loadFromStream(ObjectLoadStream& stream)
{
	AtomsObjectAnalyzerBase::loadFromStream(stream);

	stream.expectChunk(0x661AC93);
	stream >> maxAmbient;
	stream >> ambientArray;
	stream.closeChunk();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr AmbientLightingModifier::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	AmbientLightingModifier::SmartPtr clone = static_object_cast<AmbientLightingModifier>(AtomsObjectAnalyzerBase::clone(deepCopy, cloneHelper));

	clone->maxAmbient = this->maxAmbient;
	clone->ambientArray = this->ambientArray;

	return clone;
}


IMPLEMENT_PLUGIN_CLASS(AmbientLightingModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor
******************************************************************************/
void AmbientLightingModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Ambient Lighting"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(0);

	QGridLayout* layout2 = new QGridLayout();
	layout2->setContentsMargins(0,0,0,0);
	layout2->setSpacing(0);
	layout2->setColumnStretch(1, 1);
	layout1->addLayout(layout2);

	// Intensity parameter.
	FloatControllerUI* intensityPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(AmbientLightingModifier, _intensity));
	layout2->addWidget(intensityPUI->label(), 0, 0);
	layout2->addWidget(intensityPUI->textBox(), 0, 1);
	layout2->addWidget(intensityPUI->spinner(), 0, 2);
	intensityPUI->setMinValue(0);
	intensityPUI->setMaxValue(1);

	// Sampling level parameter.
	IntegerPropertyUI* samplingLevelPUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AmbientLightingModifier, _samplingLevel));
	layout2->addWidget(samplingLevelPUI->label(), 1, 0);
	layout2->addWidget(samplingLevelPUI->textBox(), 1, 1);
	layout2->addWidget(samplingLevelPUI->spinner(), 1, 2);
	samplingLevelPUI->setMinValue(0);
	samplingLevelPUI->setMaxValue(4);

	// Buffer resolution parameter.
	IntegerPropertyUI* bufferResPUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AmbientLightingModifier, _bufferResolution));
	layout2->addWidget(bufferResPUI->label(), 2, 0);
	layout2->addWidget(bufferResPUI->textBox(), 2, 1);
	layout2->addWidget(bufferResPUI->spinner(), 2, 2);
	bufferResPUI->setMinValue(0);
	bufferResPUI->setMaxValue(5);

	BooleanPropertyUI* autoUpdateUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange));
	layout1->addWidget(autoUpdateUI->checkBox());

	QPushButton* recalcButton = new QPushButton(tr("Update"), rollout);
	layout1->addSpacing(6);
	layout1->addWidget(recalcButton);
	connect(recalcButton, SIGNAL(clicked(bool)), this, SLOT(onRecalculate()));

	// Status label.
	layout1->addSpacing(10);
	layout1->addWidget(statusLabel());
}

/******************************************************************************
* Is called when the user presses the Recalculate button.
******************************************************************************/
void AmbientLightingModifierEditor::onRecalculate()
{
	AmbientLightingModifier* modifier = static_cast<AmbientLightingModifier*>(editObject());
	try {
		modifier->performAnalysis(ANIM_MANAGER.time());
	}
	catch(Exception& ex) {
		ex.prependGeneralMessage(tr("Failed to calculate ambient lighting."));
		ex.showError();
	}
}

};	// End of namespace AtomViz
