///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/Vector3PropertyUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/BooleanGroupBoxPropertyUI.h>
#include <core/gui/properties/ColorPropertyUI.h>
#include <core/data/units/ParameterUnit.h>
#include <core/viewport/ViewportManager.h>

#include "SimulationCell.h"
#include "AtomsObject.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(SimulationCell, RefTarget)
DEFINE_PROPERTY_FIELD(SimulationCell, "CellVector1", _cellVector1)
DEFINE_PROPERTY_FIELD(SimulationCell, "CellVector2", _cellVector2)
DEFINE_PROPERTY_FIELD(SimulationCell, "CellVector3", _cellVector3)
DEFINE_PROPERTY_FIELD(SimulationCell, "CellTranslation", _cellOrigin)
DEFINE_PROPERTY_FIELD(SimulationCell, "PeriodicX", _pbcX)
DEFINE_PROPERTY_FIELD(SimulationCell, "PeriodicY", _pbcY)
DEFINE_PROPERTY_FIELD(SimulationCell, "PeriodicZ", _pbcZ)
DEFINE_PROPERTY_FIELD(SimulationCell, "RenderSimulationCell", _renderSimulationCell)
DEFINE_PROPERTY_FIELD(SimulationCell, "SimulationCellLineWidth", _simulationCellLineWidth)
DEFINE_PROPERTY_FIELD(SimulationCell, "SimulationCellRenderingColor", _simulationCellColor)
SET_PROPERTY_FIELD_LABEL(SimulationCell, _cellVector1, "Cell vector 1")
SET_PROPERTY_FIELD_LABEL(SimulationCell, _cellVector2, "Cell vector 2")
SET_PROPERTY_FIELD_LABEL(SimulationCell, _cellVector3, "Cell vector 3")
SET_PROPERTY_FIELD_LABEL(SimulationCell, _cellOrigin, "Cell origin")
SET_PROPERTY_FIELD_LABEL(SimulationCell, _pbcX, "Periodic boundary conditions (X)")
SET_PROPERTY_FIELD_LABEL(SimulationCell, _pbcY, "Periodic boundary conditions (Y)")
SET_PROPERTY_FIELD_LABEL(SimulationCell, _pbcZ, "Periodic boundary conditions (Z)")
SET_PROPERTY_FIELD_LABEL(SimulationCell, _simulationCellLineWidth, "Line width")
SET_PROPERTY_FIELD_LABEL(SimulationCell, _renderSimulationCell, "Render simulation cell")
SET_PROPERTY_FIELD_LABEL(SimulationCell, _simulationCellColor, "Line color")
SET_PROPERTY_FIELD_UNITS(SimulationCell, _cellVector1, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(SimulationCell, _cellVector2, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(SimulationCell, _cellVector3, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(SimulationCell, _cellOrigin, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(SimulationCell, _simulationCellLineWidth, WorldParameterUnit)

/******************************************************************************
* Creates the storage for the internal parameters.
******************************************************************************/
void SimulationCell::init(bool isLoading)
{
	INIT_PROPERTY_FIELD(SimulationCell, _cellVector1);
	INIT_PROPERTY_FIELD(SimulationCell, _cellVector2);
	INIT_PROPERTY_FIELD(SimulationCell, _cellVector3);
	INIT_PROPERTY_FIELD(SimulationCell, _cellOrigin);
	INIT_PROPERTY_FIELD(SimulationCell, _pbcX);
	INIT_PROPERTY_FIELD(SimulationCell, _pbcY);
	INIT_PROPERTY_FIELD(SimulationCell, _pbcZ);
	INIT_PROPERTY_FIELD(SimulationCell, _renderSimulationCell);
	INIT_PROPERTY_FIELD(SimulationCell, _simulationCellLineWidth);
	INIT_PROPERTY_FIELD(SimulationCell, _simulationCellColor);
	_renderSimulationCell = true;
	_simulationCellLineWidth = 0.4;
	_simulationCellColor = Color(0.5, 0.5, 1.0);
}

/******************************************************************************
* Renders the simulation cell into the viewport.
******************************************************************************/
void SimulationCell::render(TimeTicks time, Viewport* vp, ObjectNode* contextNode)
{
	AffineTransformation simCell = cellMatrix();

	Point3 origin = ORIGIN + simCell.getTranslation();

	TriMesh mesh;
	mesh.setVertexCount(8);
	mesh.setVertex(0, origin);
	mesh.setVertex(1, origin + simCell.column(0));
	mesh.setVertex(2, origin + simCell.column(0) + simCell.column(1));
	mesh.setVertex(3, origin + simCell.column(1));
	mesh.setVertex(4, origin + simCell.column(2));
	mesh.setVertex(5, origin + simCell.column(0) + simCell.column(2));
	mesh.setVertex(6, origin + simCell.column(0) + simCell.column(1) + simCell.column(2));
	mesh.setVertex(7, origin + simCell.column(1) + simCell.column(2));

	// Build faces.
	mesh.setFaceCount(12);
	if(simCell.determinant() >= 0.0) {
		mesh.face(0).setVertices(0, 1, 5);
		mesh.face(0).setEdgeVisibility(true, true, false);
		mesh.face(1).setVertices(0, 5, 4);
		mesh.face(1).setEdgeVisibility(false, true, true);
		mesh.face(2).setVertices(1, 2, 6);
		mesh.face(2).setEdgeVisibility(true, true, false);
		mesh.face(3).setVertices(1, 6, 5);
		mesh.face(3).setEdgeVisibility(false, true, true);
		mesh.face(4).setVertices(2, 3, 7);
		mesh.face(4).setEdgeVisibility(true, true, false);
		mesh.face(5).setVertices(2, 7, 6);
		mesh.face(5).setEdgeVisibility(false, true, true);
		mesh.face(6).setVertices(3, 0, 4);
		mesh.face(6).setEdgeVisibility(true, true, false);
		mesh.face(7).setVertices(3, 4, 7);
		mesh.face(7).setEdgeVisibility(false, true, true);
		mesh.face(8).setVertices(4, 5, 6);
		mesh.face(8).setEdgeVisibility(true, true, false);
		mesh.face(9).setVertices(4, 6, 7);
		mesh.face(9).setEdgeVisibility(false, true, true);
		mesh.face(10).setVertices(0, 3, 2);
		mesh.face(10).setEdgeVisibility(true, true, false);
		mesh.face(11).setVertices(0, 2, 1);
		mesh.face(11).setEdgeVisibility(false, true, true);
	}
	else {
		// Flip faces on negative volume.
		mesh.face(0).setVertices(0, 5, 1);
		mesh.face(0).setEdgeVisibility(false, true, true);
		mesh.face(1).setVertices(0, 4, 5);
		mesh.face(1).setEdgeVisibility(true, true, false);
		mesh.face(2).setVertices(1, 6, 2);
		mesh.face(2).setEdgeVisibility(false, true, true);
		mesh.face(3).setVertices(1, 5, 6);
		mesh.face(3).setEdgeVisibility(true, true, false);
		mesh.face(4).setVertices(2, 7, 3);
		mesh.face(4).setEdgeVisibility(false, true, true);
		mesh.face(5).setVertices(2, 6, 7);
		mesh.face(5).setEdgeVisibility(true, true, false);
		mesh.face(6).setVertices(3, 4, 0);
		mesh.face(6).setEdgeVisibility(false, true, true);
		mesh.face(7).setVertices(3, 7, 4);
		mesh.face(7).setEdgeVisibility(true, true, false);
		mesh.face(8).setVertices(4, 6, 5);
		mesh.face(8).setEdgeVisibility(false, true, true);
		mesh.face(9).setVertices(4, 7, 6);
		mesh.face(9).setEdgeVisibility(true, true, false);
		mesh.face(10).setVertices(0, 2, 3);
		mesh.face(10).setEdgeVisibility(false, true, true);
		mesh.face(11).setVertices(0, 1, 2);
		mesh.face(11).setEdgeVisibility(true, true, false);
	}

	mesh.invalidateVertices();
	mesh.invalidateFaces();

	if(vp->isPicking()) {
		vp->renderMeshShaded(mesh);
	}
	else {
		if(contextNode->isSelected())
			vp->setRenderingColor(Viewport::getVPColor(Viewport::COLOR_SELECTION));
		else
			vp->setRenderingColor(contextNode->displayColor());

		vp->renderMeshWireframe(mesh);
	}
}

/******************************************************************************
* Renders a cylinder connecting two 3d points using OpenGL.
******************************************************************************/
static void renderCylinderLine(const Point3& a, const Point3& b, FloatType width)
{
	glPushMatrix();
	AffineTransformation tm;
	tm.column(2) = b - a;
	if(tm.column(2) == NULL_VECTOR) return;
	tm.column(1) = CrossProduct(tm.column(2), Vector3(0,0,1));
	if(tm.column(1) == NULL_VECTOR) tm.column(1) = CrossProduct(tm.column(2), Vector3(1,0,0));
	tm.column(0) = CrossProduct(tm.column(2), tm.column(1));
	tm.column(0) = Normalize(tm.column(0));
	tm.column(1) = Normalize(tm.column(1));
	tm.column(2) = Normalize(tm.column(2));
	tm.column(3) = b - ORIGIN;
	glMultMatrix(Matrix4(tm).constData());
	glScalef(-1,-1,-1);
	GLUquadricObj* quadric = gluNewQuadric();
	gluQuadricNormals(quadric, GLU_SMOOTH);
	gluCylinder(quadric, width, width, Distance(a, b), 16, 1);
	gluDeleteQuadric(quadric);
	glPopMatrix();
}

/******************************************************************************
* Renders a sphere using OpenGL.
******************************************************************************/
static void renderCornerSphere(const Point3& a, FloatType radius)
{
	glPushMatrix();
	glTranslate(a.X, a.Y, a.Z);
	GLUquadricObj* quadric = gluNewQuadric();
	gluQuadricNormals(quadric, GLU_SMOOTH);
	gluSphere(quadric, radius, 16, 16);
	gluDeleteQuadric(quadric);
	glPopMatrix();
}

/******************************************************************************
* Renders the simulation cell in high-quality mode to an offscreen buffer.
******************************************************************************/
void SimulationCell::renderHQ(TimeTicks time, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext)
{
	// Check if rendering of the cell is enabled.
	if(!renderSimulationCell()) return;

	// Render simulation box.
	FloatType lineWidth = simulationCellLineWidth();
	if(lineWidth > 0) {
		Color color = simulationCellRenderingColor();

		// Setup material
		float ambient[4] = { 0.1f, 0.1f, 0.1f, 1.0f };
		glMaterialfv(GL_FRONT, GL_AMBIENT, ambient);
		float diffuse[4] = { (float)color.r, (float)color.g, (float)color.b, 1.0f };
		glMaterialfv(GL_FRONT, GL_DIFFUSE, diffuse);
		float specular[4] = { 0.1f, 0.1f, 0.1f, 1.0f };
		glMaterialfv(GL_FRONT, GL_SPECULAR, specular);
		float emission[4] = { 0, 0, 0, 1 };
		glMaterialfv(GL_FRONT, GL_EMISSION, emission);
		glMaterialf(GL_FRONT, GL_SHININESS, 0);
		glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, 0);

		AffineTransformation simBox = cellMatrix();
		Point3 corners[8];
		corners[0] = ORIGIN + simBox.getTranslation();
		corners[1] = corners[0] + simBox.column(0);
		corners[2] = corners[0] + simBox.column(0) + simBox.column(1);
		corners[3] = corners[0] + simBox.column(1);
		corners[4] = corners[0] + simBox.column(2);
		corners[5] = corners[1] + simBox.column(2);
		corners[6] = corners[2] + simBox.column(2);
		corners[7] = corners[3] + simBox.column(2);

		lineWidth *= 0.5;

		renderCylinderLine(corners[0], corners[1], lineWidth);
		renderCylinderLine(corners[1], corners[2], lineWidth);
		renderCylinderLine(corners[2], corners[3], lineWidth);
		renderCylinderLine(corners[3], corners[0], lineWidth);

		renderCylinderLine(corners[4], corners[5], lineWidth);
		renderCylinderLine(corners[5], corners[6], lineWidth);
		renderCylinderLine(corners[6], corners[7], lineWidth);
		renderCylinderLine(corners[7], corners[4], lineWidth);

		renderCylinderLine(corners[0], corners[4], lineWidth);
		renderCylinderLine(corners[1], corners[5], lineWidth);
		renderCylinderLine(corners[2], corners[6], lineWidth);
		renderCylinderLine(corners[3], corners[7], lineWidth);

		for(size_t i=0; i<8; i++)
			renderCornerSphere(corners[i], lineWidth);
	}
}

IMPLEMENT_PLUGIN_CLASS(SimulationCellEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void SimulationCellEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Simulation cell"), rolloutParams, "atomviz.objects.simulation_cell");

	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(0,0,0,0);
	layout1->setSpacing(0);
	QTabWidget* tabWidget = new QTabWidget(rollout);
	layout1->addWidget(tabWidget);
	QWidget* generalTab = new QWidget(tabWidget);
	tabWidget->addTab(generalTab, tr("General"));

	layout1 = new QVBoxLayout(generalTab);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(8);

	{
		QGroupBox* pbcGroupBox = new QGroupBox(tr("Periodic boundary conditions"), generalTab);
		layout1->addWidget(pbcGroupBox);

		QGridLayout* layout2 = new QGridLayout(pbcGroupBox);
		layout2->setContentsMargins(4,4,4,4);
		layout2->setSpacing(2);

		BooleanPropertyUI* pbcxPUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _pbcX));
		pbcxPUI->checkBox()->setText("X");
		layout2->addWidget(pbcxPUI->checkBox(), 0, 0);

		BooleanPropertyUI* pbcyPUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _pbcY));
		pbcyPUI->checkBox()->setText("Y");
		layout2->addWidget(pbcyPUI->checkBox(), 0, 1);

		BooleanPropertyUI* pbczPUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _pbcZ));
		pbczPUI->checkBox()->setText("Z");
		layout2->addWidget(pbczPUI->checkBox(), 0, 2);
	}

	{
		QGroupBox* sizeGroupBox = new QGroupBox(tr("Size"), generalTab);
		layout1->addWidget(sizeGroupBox);

		QGridLayout* layout2 = new QGridLayout(sizeGroupBox);
		layout2->setContentsMargins(4,4,4,4);
		layout2->setSpacing(0);
		layout2->setColumnStretch(1, 1);

		QSignalMapper* signalMapperValueChanged = new QSignalMapper(this);
		QSignalMapper* signalMapperDragStart = new QSignalMapper(this);
		QSignalMapper* signalMapperDragStop = new QSignalMapper(this);
		QSignalMapper* signalMapperDragAbort = new QSignalMapper(this);
		for(int i=0; i<3; i++) {
			QLineEdit* textBox = new QLineEdit(generalTab);
			simCellSizeSpinners[i] = new SpinnerWidget(generalTab, textBox);
			simCellSizeSpinners[i]->setUnit(UNITS_MANAGER.getWorldParameterUnit());
			simCellSizeSpinners[i]->setMinValue(0.0);
			layout2->addWidget(textBox, i, 1);
			layout2->addWidget(simCellSizeSpinners[i], i, 2);

			connect(simCellSizeSpinners[i], SIGNAL(spinnerValueChanged()), signalMapperValueChanged, SLOT(map()));
			connect(simCellSizeSpinners[i], SIGNAL(spinnerDragStart()), signalMapperDragStart, SLOT(map()));
			connect(simCellSizeSpinners[i], SIGNAL(spinnerDragStop()), signalMapperDragStop, SLOT(map()));
			connect(simCellSizeSpinners[i], SIGNAL(spinnerDragAbort()), signalMapperDragAbort, SLOT(map()));

			signalMapperValueChanged->setMapping(simCellSizeSpinners[i], i);
			signalMapperDragStart->setMapping(simCellSizeSpinners[i], i);
			signalMapperDragStop->setMapping(simCellSizeSpinners[i], i);
			signalMapperDragAbort->setMapping(simCellSizeSpinners[i], i);
		}
		connect(signalMapperValueChanged, SIGNAL(mapped(int)), this, SLOT(onSizeSpinnerValueChanged(int)));
		connect(signalMapperDragStart, SIGNAL(mapped(int)), this, SLOT(onSizeSpinnerDragStart(int)));
		connect(signalMapperDragStop, SIGNAL(mapped(int)), this, SLOT(onSizeSpinnerDragStop(int)));
		connect(signalMapperDragAbort, SIGNAL(mapped(int)), this, SLOT(onSizeSpinnerDragAbort(int)));
		layout2->addWidget(new QLabel(tr("Width (X):")), 0, 0);
		layout2->addWidget(new QLabel(tr("Length (Y):")), 1, 0);
		layout2->addWidget(new QLabel(tr("Height (Z):")), 2, 0);

		connect(this, SIGNAL(contentsChanged(RefTarget*)), this, SLOT(updateSimulationBoxSize()));
	}

	{
		QGroupBox* renderingGroupBox = (new BooleanGroupBoxPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _renderSimulationCell)))->groupBox();
		layout1->addWidget(renderingGroupBox);

		QGridLayout* layout2 = new QGridLayout(renderingGroupBox);
		layout2->setContentsMargins(4,4,4,4);
		layout2->setSpacing(0);
		layout2->setColumnStretch(1, 1);

		// Simulation cell line width parameter.
		lineWidthUI = new FloatPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _simulationCellLineWidth));
		layout2->addWidget(lineWidthUI->label(), 0, 0);
		layout2->addWidget(lineWidthUI->textBox(), 0, 1);
		layout2->addWidget(lineWidthUI->spinner(), 0, 2);
		lineWidthUI->setMinValue(0);

		// Background color parameter.
		ColorPropertyUI* cellColorPUI = new ColorPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _simulationCellColor));
		layout2->addWidget(cellColorPUI->label(), 1, 0);
		layout2->addWidget(cellColorPUI->colorPicker(), 1, 1, 1, 2);
	}
	layout1->addStretch(1);

	QWidget* geometryTab = new QWidget(tabWidget);
	tabWidget->addTab(geometryTab, tr("Geometry"));

	layout1 = new QVBoxLayout(geometryTab);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(0);

	{	// First cell vector.
		layout1->addWidget(new QLabel(tr("Cell vector 1:"), rollout));
		QGridLayout* layout2 = new QGridLayout();
		layout2->setContentsMargins(0,0,0,0);
		layout2->setSpacing(0);
		layout2->setColumnStretch(0, 1);
		layout1->addLayout(layout2);
		for(size_t i=0; i<3; i++) {
			Vector3PropertyUI* vPUI = new Vector3PropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _cellVector1), i);
			layout2->addWidget(vPUI->textBox(), i, 0);
			layout2->addWidget(vPUI->spinner(), i, 1);
		}
	}

	{	// Second cell vector.
		layout1->addWidget(new QLabel(tr("Cell vector 2:"), rollout));
		QGridLayout* layout2 = new QGridLayout();
		layout2->setContentsMargins(0,0,0,0);
		layout2->setSpacing(0);
		layout2->setColumnStretch(0, 1);
		layout1->addLayout(layout2);
		for(size_t i=0; i<3; i++) {
			Vector3PropertyUI* vPUI = new Vector3PropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _cellVector2), i);
			layout2->addWidget(vPUI->textBox(), i, 0);
			layout2->addWidget(vPUI->spinner(), i, 1);
		}
	}

	{	// Third cell vector.
		layout1->addWidget(new QLabel(tr("Cell vector 3:"), rollout));
		QGridLayout* layout2 = new QGridLayout();
		layout2->setContentsMargins(0,0,0,0);
		layout2->setSpacing(0);
		layout2->setColumnStretch(0, 1);
		layout1->addLayout(layout2);
		for(size_t i=0; i<3; i++) {
			Vector3PropertyUI* vPUI = new Vector3PropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _cellVector3), i);
			layout2->addWidget(vPUI->textBox(), i, 0);
			layout2->addWidget(vPUI->spinner(), i, 1);
		}
	}

	layout1->addSpacing(6);

	{	// Cell origin.
		layout1->addWidget(new QLabel(tr("Cell origin:"), rollout));
		QGridLayout* layout2 = new QGridLayout();
		layout2->setContentsMargins(0,0,0,0);
		layout2->setSpacing(0);
		layout2->setColumnStretch(0, 1);
		layout1->addLayout(layout2);
		for(size_t i=0; i<3; i++) {
			Vector3PropertyUI* vPUI = new Vector3PropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SimulationCell, _cellOrigin), i);
			layout2->addWidget(vPUI->textBox(), i, 0);
			layout2->addWidget(vPUI->spinner(), i, 1);
		}
	}

	layout1->addStretch(1);
}

/******************************************************************************
* After the user has changed a spinner value, this method changes the
* simulation cell geometry.
******************************************************************************/
void SimulationCellEditor::changeSimulationBoxSize(int dim)
{
	OVITO_ASSERT(dim >=0 && dim < 3);

	SimulationCell* cell = static_object_cast<SimulationCell>(editObject());
	if(!cell) return;

	AffineTransformation cellTM = cell->cellMatrix();
	FloatType newSize = simCellSizeSpinners[dim]->floatValue();
	cellTM.column(3)[dim] -= 0.5 * (newSize - cellTM(dim, dim));
	cellTM(dim, dim) = newSize;
	cell->setCellMatrix(cellTM);
}

/******************************************************************************
* After the simulation cell size has changed, updates the UI controls.
******************************************************************************/
void SimulationCellEditor::updateSimulationBoxSize()
{
	SimulationCell* cell = static_object_cast<SimulationCell>(editObject());
	if(!cell) return;

	AffineTransformation cellTM = cell->cellMatrix();
	for(size_t dim=0; dim<3; dim++) {
		if(simCellSizeSpinners[dim]->isDragging() == false)
			simCellSizeSpinners[dim]->setFloatValue(cellTM(dim,dim));
	}

	lineWidthUI->setEnabled(cell->renderSimulationCell());
}

/******************************************************************************
* Is called when a spinner's value has changed.
******************************************************************************/
void SimulationCellEditor::onSizeSpinnerValueChanged(int dim)
{
	ViewportSuspender noVPUpdate;
	if(!UNDO_MANAGER.isRecording()) {
		UNDO_MANAGER.beginCompoundOperation(tr("Change simulation cell size"));
		changeSimulationBoxSize(dim);
		UNDO_MANAGER.endCompoundOperation();
	}
	else {
		UNDO_MANAGER.currentCompoundOperation()->clear();
		changeSimulationBoxSize(dim);
	}
}

/******************************************************************************
* Is called when the user begins dragging a spinner interactively.
******************************************************************************/
void SimulationCellEditor::onSizeSpinnerDragStart(int dim)
{
	OVITO_ASSERT(!UNDO_MANAGER.isRecording());
	UNDO_MANAGER.beginCompoundOperation(tr("Change simulation cell size"));
}

/******************************************************************************
* Is called when the user stops dragging a spinner interactively.
******************************************************************************/
void SimulationCellEditor::onSizeSpinnerDragStop(int dim)
{
	OVITO_ASSERT(UNDO_MANAGER.isRecording());
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Is called when the user aborts dragging a spinner interactively.
******************************************************************************/
void SimulationCellEditor::onSizeSpinnerDragAbort(int dim)
{
	OVITO_ASSERT(UNDO_MANAGER.isRecording());
	UNDO_MANAGER.currentCompoundOperation()->clear();
	UNDO_MANAGER.endCompoundOperation();
}


};	// End of namespace AtomViz
