///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ATOMS_RENDERER_H
#define __ATOMS_RENDERER_H

#include <atomviz/AtomViz.h>

#include <core/Core.h>
#include <core/viewport/Window3D.h>
#include <core/viewport/OpenGLShader.h>

namespace AtomViz {

/**
 * \brief This helper class is used to render the atoms stored in an AtomsObject
 *        via OpenGL.
 * It uses the best/fastest rendering method available on the installed
 * graphics hardware.
 * This class manages an internal buffer, so the atoms data can be specified
 * once and then rendered many times.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AtomsRenderer
{
public:

	/// Default constructor.
	AtomsRenderer();

	/// Destructor.
	~AtomsRenderer();

	/// Returns true if the atoms renderer has been initialized.
	/// Other wise Prepare has to be called.
	bool isInitialized() const { return container != NULL; }

	/// Initializes the internal buffers of this helper class if they have not been initialized yet.
	/// It will be associated with the given rendering window and may subsequently
	/// only be used with rendering windows that belong to the same Window3DContainer.
	void prepare(Window3D* win, bool flatShading = false, bool useImposters = true);

	/// Return true if the  internal rendering buffer is filled. Otherwise
	/// The buffers have to be filled using BeginAtoms()/SpecifyAtom()/EndAtoms() before
	/// they can be rendered using Render().
	bool isFilled() const { return hasBeenFilled; }

	/// Allocates a memory buffer for the specified number of atoms.
	/// After all atoms have been specified, EndAtoms() must be called.
	void beginAtoms(GLuint numAtoms);

	/// Stores a single atom in the internal buffer. This function may only be called
	/// N times between a call to BeginAtoms() and EndAtoms();
	void specifyAtom(const Point3& pos, GLubyte r, GLubyte g, GLubyte b, FloatType radius);

	/// This method must be called after BeginAtoms() has been called and all atom
	/// data has been written to the supplied buffer.
	void endAtoms();

	/// Renders the buffered atoms in the given rendering window.
	void render(Window3D* win);

	/// Render the buffered atoms to an offscreen buffer.
	/// This function is used by the ambient lighting modifier.
	void renderOffscreen(bool isPerspective, const Matrix4& projMatrix, QSize windowSize);

	/// Returns the local bounding box of all atoms stored in the buffer.
	const Box3& boundingBox() const { return _boundingBox; }

	/// Calculates the lighting for the atoms using an ambient occlusion algorithm.
	bool calculateAmbientOcclusion(Window3D* glcontext);

private:

	/// This structure holds all properties of a single atom
	/// that are relevant for OpenGL rendering.
	struct OpenGLAtom {
		float x, y, z;		// The position of the atom.
		GLubyte r, g, b, a;	// The color of the atom.
		float radius;		// The radius of the atom.
	};

	/// The rendering window this class is associated with.
	QPointer<Window3D> container;

	/// This array is used if no vertex buffer objects are available.
	QVector<OpenGLAtom> internalArray;

	/// Enables or disables the use of imposters for fast rendering.
	bool useImposters;

	/// Enables or disables the shading of atoms.
	bool flatShading;

	/// Indicates whether atomic coordinates have transfered.
	bool hasBeenFilled;

	/// The number of atoms in the buffer.
	GLuint numAtoms;

	/// A pointer to the atom in the internal array that will be set
	/// on the next call to SpecifyAtom().
	OpenGLAtom* currentAtom;

	/// If all atoms in the buffer have the same radius then this field contains
	/// the uniform rdaius. If there are at least two atoms with differnt radii then
	/// this field is zero.
	FloatType uniformRadius;

	/// The radius of the biggest atom.
	FloatType maxRadius;

	/// The bounding box that has been computed for the atoms.
	Box3 _boundingBox;

	/// The identifiers of OpenGL textures that are used for billboard rendering of shaded and flat atoms.
	GLuint textureID[2];

	/// The identifier of the OpenGL vertex buffer object.
	GLuint vboVerticesID;

	/// The OpenGL shader programs that are used to render the atoms.
	OpenGLShader* flatImposterShader;
	OpenGLShader* shadedImposterShader;
	OpenGLShader* raytracedSphereShader;

	/// Specifies the number of atoms per chunk if the whole atoms array
	/// is rendered not at once but chunk by chunk. Some graphics cards
	/// can only render a limited amount of point sprites at once so the
	/// array has to be split up into chunks.
	GLuint chunkRenderSize;

	/// Renders the atoms stored in the internal atoms array
	/// using the GL_ARB_point_parameters OpenGL extension.
	void renderInternalArrayPointSprites(bool isPerspective, const Matrix4& projMatrix, float windowHeight);

	/// Renders the atoms stored in the internal atoms array
	/// using the custom vertex and fragment shaders.
	void renderInternalImpostersWithShader(bool isPerspective, const Matrix4& projMatrix, float windowHeight);

	/// Renders the atoms stored in the internal atoms array
	/// using the custom vertex and fragment shaders.
	void renderInternalRaytracedWithShaders(bool isPerspective);

	/// Renders the atoms stored in the internal atoms array
	/// without the GL_ARB_point_parameters OpenGL extension.
	void renderInternalArrayQuads();

	/// Creates and activates the texture used for billboard rendering of atoms.
	void initializeBillboardTextures(bool withAlpha = true);

	/// Release all cached OpenGL resources.
	void reset();

public:			// Global settings

	/// \brief Returns whether the use of the OpenGL point parameters extension is enabled for atom rendering.
	static bool arePointSpritesEnabled();

	/// \brief Sets whether the use of the OpenGL point parameters extension should be enabled for atom rendering.
	static void enablePointSprites(bool enable);

	/// \brief Returns whether the use of OpenGL shader programs is enabled for atom rendering.
	static bool areHWShadersEnabled();

	/// \brief Sets whether the use of OpenGL shader programs should be enabled for atom rendering.
	static void enableHWShaders(bool enable);

private:

	/// \brief Indicates that the render settings have been loaded from the application settings store.
	static bool _settingsLoaded;

	/// Enables the use of OpenGL shader programs if supported by the hardware.
	static bool _enableHWShaders;

	/// Enables the use of OpenGL point parameters extension if supported by the driver.
	static bool _enablePointExtension;

	/// \brief Loads the render settings from the application settings store.
	static void loadRenderSettings();
};

};	// End of namespace AtomViz

#endif // __ATOMS_RENDERER_H
