/* -*-c++-*- */
/**
* OsgAL - OpenSceneGraph Audio Library
* Copyright (C) 2004 VRlab, Ume University
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
*
* Example of streaming from a file. Only ogg files are supported at the
* moment (2003-02-15). The program takes one argument; the name of the file
* to stream data from.
*/

#include <openalpp/Source>
#include <openalpp/FileStream>

#include <iostream>


#define CRTDBG_MAP_ALLOC
#include <stdlib.h>
#ifdef WIN32
#include <crtdbg.h>
#endif



using namespace openalpp;

void WaitForPlaying(Source* source)
{
    while (source->getState() != openalpp::Playing)
    {
        usleep(1000);
        int checkCount = 0;
        if (checkCount++ > 1000)
        {
            throw openalpp::Error("Timed out while waiting for stream to play");
        }
    }
}

void WaitForStopped(Source* source)
{
    while (source->getState() != openalpp::Stopped)
    {
        usleep(1000);
        int checkCount = 0;
        if (checkCount++ > 1000)
        {
            throw openalpp::Error("Timed out while waiting for stream to stop");
        }
    }
}

void TestSeek()
{
    std::string numbers("one_ten_b.ogg");
    

    std::vector<float> seekPoints;
    seekPoints.push_back(7.0);
    seekPoints.push_back(5.0);
    seekPoints.push_back(2.0);
    seekPoints.push_back(3.0);

    try 
    {
        osg::ref_ptr<Source> source = new Source;

        source->setSound(new FileStream(numbers));
        source->setAmbient();

        for (size_t k=0; k<seekPoints.size(); ++k)
        {
            fprintf(stdout, "%.0f ", seekPoints[k]);
            source->seek(seekPoints[k]);
            if (k==0)
            {
                source->play();
            }
            usleep(500*1000);
        }
        fprintf(stdout, "\n");

        source->stop();

        WaitForStopped(source.get());

    }
    catch(openalpp::Error e) 
    {
        std::cerr << e << "\n";
    } 
    catch(...) 
    {
        std::cerr << "Unknown error!\n";
    }
    
}



int main(int argc,char **argv) {
#ifdef WIN32
    // MSVC memory leak detection
    _CrtSetDbgFlag ( _CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF ); // {,,msvcr71d.dll}_crtBreakAlloc
#endif
    std::string file1("test1.ogg");
    std::string file2("test2.ogg");
    std::string file3("test3.ogg");
    
    TestSeek();
    usleep(500000);
    try 
    {

        osg::ref_ptr<Source> source = new Source;

        source->setSound(new FileStream(file1));
        source->setAmbient();
        source->play();

        // Test transition to audio file when first finishes
        std::cout << "Waiting for first audio file to stop" << std::endl;

        WaitForStopped(source.get());
        
        std::cout << "Playing second audio file" << std::endl;

        // For now, the source has to be re-created for stable behavior
        source = new Source; 

        source->setSound(new FileStream(file3));
        source->setAmbient();
        source->play();

        // Test pause and resume
        std::cout << "Press return to pause audio file" << std::endl;
        std::cin.get();

        source->pause();

        std::cout << "Press return to resume audio file" << std::endl;
        std::cin.get();

        source->play();

        // Test switching between files mid-stream
        std::cout << "Press return to switch to next audio file" << std::endl;
        std::cin.get();


        for (int nSwitch = 0; nSwitch < 2; nSwitch++)
        {
            source = new Source;

            source->setSound(new FileStream(file2));
            source->setAmbient();
            source->play();

            std::cout << "Press return to switch to next audio file" << std::endl;
            std::cin.get();

            source->stop();
            WaitForStopped(source.get());

            source = new Source;

            source->setSound(new FileStream(file1));
            source->setAmbient();
            source->play();

            std::cout << "Press return to switch to next audio file" << std::endl;
            std::cin.get();

            source->stop();
            WaitForStopped(source.get());

        }
        
        std::cout << "Press return to exit" << std::endl;
        std::cin.get();
    } 
    catch(openalpp::Error e) 
    {
        std::cerr << e << "\n";
    } 
    catch(...) 
    {
        std::cerr << "Unknown error!\n";
    }


    return 0;
}
