///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @file ValueAccessor.h
///
/// When traversing a grid in a spatially coherent pattern (e.g., iterating
/// over neighboring voxels), request a @c ValueAccessor from the grid
/// (with Grid::getAccessor()) and use the accessor's @c getValue() and
/// @c setValue() methods.  These will typically be significantly faster
/// than accessing voxels directly in the grid's tree.
///
/// @par Example:
///
/// @code
/// FloatGrid grid;
/// FloatGrid::Accessor acc = grid.getAccessor();
/// // First access is slow:
/// acc.setValue(Coord(0, 0, 0), 100);
/// // Subsequent nearby accesses are fast, since the accessor now holds pointers
/// // to nodes that contain (0, 0, 0) along the path from the root of the grid's
/// // tree to the leaf:
/// acc.setValue(Coord(0, 0, 1), 100);
/// acc.getValue(Coord(0, 2, 0), 100);
/// // Slow, because the accessor must be repopulated:
/// acc.getValue(Coord(-1, -1, -1));
/// // Fast:
/// acc.getValue(Coord(-1, -1, -2));
/// acc.setValue(Coord(-1, -2, 0), -100);
/// @endcode

#ifndef OPENVDB_TREE_VALUEACCESSOR_HAS_BEEN_INCLUDED
#define OPENVDB_TREE_VALUEACCESSOR_HAS_BEEN_INCLUDED

#include <boost/mpl/front.hpp>
#include <boost/mpl/pop_front.hpp>
#include <boost/mpl/push_back.hpp>
#include <boost/mpl/size.hpp>
#include <boost/mpl/vector.hpp>
#include <boost/static_assert.hpp>
#include <boost/type_traits/is_const.hpp>
#include <tbb/null_mutex.h>
#include <tbb/spin_mutex.h>
#include <openvdb/version.h>
#include <openvdb/Types.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tree {

/// InvertedTree<RootNodeType, RootNodeType::LEVEL>::Type is a boost::mpl::vector
/// that lists the types of the nodes of the tree rooted at RootNodeType
/// in reverse order, from LeafNode to RootNode.  For example, for RootNodeType
/// RootNode<InternalNode<InternalNode<LeafNode> > >, InvertedTree::Type is
///     boost::mpl::vector<
///         LeafNode,
///         InternalNode<LeafNode>,
///         InternalNode<InternalNode<LeafNode> >,
///         RootNode<InternalNode<InternalNode<LeafNode> > > >.
template<typename HeadT, int HeadLevel>
struct InvertedTree {
    typedef typename InvertedTree<typename HeadT::ChildNodeType, HeadLevel-1>::Type SubtreeT;
    typedef typename boost::mpl::push_back<SubtreeT, HeadT>::type Type;
};
/// For level one nodes (either RootNode<LeafNode> or InternalNode<LeafNode>),
/// InvertedTree::Type is either boost::mpl::vector<LeafNode, RootNode<LeafNode> >
/// or boost::mpl::vector<LeafNode, InternalNode<LeafNode> >.
template<typename HeadT>
struct InvertedTree<HeadT, /*HeadLevel=*/1> {
    typedef typename boost::mpl::vector<typename HeadT::ChildNodeType, HeadT>::type Type;
};


////////////////////////////////////////


// An element of a compile-time linked list of node pointers, ordered from LeafNode to RootNode
template<typename TreeCacheT, typename NodeVecT, bool AtRoot>
class CacheItem
{
public:
    typedef typename boost::mpl::front<NodeVecT>::type NodeType;
    typedef typename NodeType::ValueType               ValueType;
    typedef typename NodeType::LeafNodeType            LeafNodeType;
    typedef std::numeric_limits<Int32>                 CoordLimits;

    CacheItem(TreeCacheT& parent):
        mParent(&parent),
        mHash(CoordLimits::max()),
        mNode(NULL),
        mNext(parent)
    {
    }

    //@{
    /// Copy another CacheItem's node pointers and hash keys, but not its parent pointer.
    CacheItem(TreeCacheT& parent, const CacheItem& other):
        mParent(&parent),
        mHash(other.mHash),
        mNode(other.mNode),
        mNext(parent, other.mNext)
    {
    }

    CacheItem& copy(TreeCacheT& parent, const CacheItem& other)
    {
        mParent = &parent;
        mHash = other.mHash;
        mNode = other.mNode;
        mNext.copy(parent, other.mNext);
        return *this;
    }
    //@}

    bool isCached(const Coord& xyz) const
    {
        return (this->isHashed(xyz) || mNext.isCached(xyz));
    }

    /// Cache the given node at this level.
    void insert(const Coord& xyz, const NodeType* node)
    {
        mHash = (node != NULL) ? xyz & ~(NodeType::DIM-1) : Coord::max();
        mNode = node;
    }
    /// Forward the given node to another level of the cache.
    template<typename OtherNodeType>
    void insert(const Coord& xyz, const OtherNodeType* node) { mNext.insert(xyz, node); }

    /// Erase the node at this level.
    void erase(const NodeType*) { mHash = Coord::max(); mNode = NULL; }
    /// Erase the node at another level of the cache.
    template<typename OtherNodeType>
    void erase(const OtherNodeType* node) { mNext.erase(node); }

    /// Erase the nodes at this and lower levels of the cache.
    void clear() { mHash = Coord::max(); mNode = NULL; mNext.clear(); }

    /// Return the cached node (if any) at this level.
    void getNode(const NodeType*& node) const { node = mNode; }
    void getNode(const NodeType*& node) { node = mNode; }
    void getNode(NodeType*& node)
    {
        // This combination of a static assertion and a const_cast might not be elegant,
        // but it is a lot simpler than specializing TreeCache for const Trees.
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        node = const_cast<NodeType*>(mNode);
    }
    /// Forward the request to another level of the cache.
    template<typename OtherNodeType>
    void getNode(OtherNodeType*& node) { mNext.getNode(node); }

    LeafNodeType* touchLeaf(const Coord& xyz)
    {
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        if (this->isHashed(xyz)) {
            assert(mNode);
            return const_cast<NodeType*>(mNode)->touchLeafAndCache(xyz, *mParent);
        }
        return mNext.touchLeaf(xyz);
    }

    LeafNodeType* probeLeaf(const Coord& xyz)
    {
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        if (this->isHashed(xyz)) {
            assert(mNode);
            return const_cast<NodeType*>(mNode)->probeLeafAndCache(xyz, *mParent);
        }
        return mNext.probeLeaf(xyz);
    }

    const ValueType& getValue(const Coord& xyz, bool& state, int& level)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            return mNode->getValue(xyz, state, level, *mParent);
        }
        return mNext.getValue(xyz, state, level);
    }
    template <typename ValueProbeType>
    const ValueType& probe(const Coord& xyz, ValueProbeType &p)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            return mNode->probe(xyz, p, *mParent);
        }
        return mNext.probe(xyz, p);
    }
    template <bool State, bool Level>
    const ValueType& probe(const Coord& xyz, bool& state, int& level)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            return mNode->probe<State, Level>(xyz, state, level, *mParent);
        }
        return mNext.probe<State, Level>(xyz, state, level);
    }

    // ====================> DEPRECATED METHODS <========================

    /// Return the value of the voxel at the given coordinates.
    const ValueType& getValue(const Coord& xyz)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            return mNode->getValueAndCache(xyz, *mParent);
        }
        return mNext.getValue(xyz);
    }

    const ValueType& newGetValue(const Coord& xyz)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            const ValueType& val = mNode->getValue(xyz);
            mNode->updateCache(xyz, *mParent);
            return val;
        }
        return mNext.newGetValue(xyz);
    }

    /// Return the active state of the voxel at the given coordinates.
    bool isValueOn(const Coord& xyz)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            return mNode->isValueOnAndCache(xyz, *mParent);
        }
        return mNext.isValueOn(xyz);
    }

    /// Return the active state and value of the voxel at the given coordinates.
    bool probeValue(const Coord& xyz, ValueType& value)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            return mNode->probeValueAndCache(xyz, value, *mParent);
        }
        return mNext.probeValue(xyz, value);
    }

     int getValueDepth(const Coord& xyz)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            return static_cast<int>(TreeCacheT::RootNodeT::LEVEL) -
                   static_cast<int>(mNode->getValueLevelAndCache(xyz, *mParent));
        } else {
            return mNext.getValueDepth(xyz);
        }
    }

    bool isVoxel(const Coord& xyz)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            return mNode->getValueLevelAndCache(xyz, *mParent)==0;
        } else {
            return mNext.isVoxel(xyz);
        }
    }

    // ==================================================================

    /// Set the value of the voxel at the given coordinates and mark the voxel as active.
    void setValue(const Coord& xyz, const ValueType& value)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
            const_cast<NodeType*>(mNode)->setValueAndCache(xyz, value, *mParent);
        } else {
            mNext.setValue(xyz, value);
        }
    }
    void setValueOnly(const Coord& xyz, const ValueType& value)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
            const_cast<NodeType*>(mNode)->setValueOnlyAndCache(xyz, value, *mParent);
        } else {
            mNext.setValueOnly(xyz, value);
        }
    }
    void setValueOn(const Coord& xyz, const ValueType& value) { this->setValue(xyz, value); }

    void newSetValue(const Coord& xyz, const ValueType& value)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
            const_cast<NodeType*>(mNode)->setValue(xyz, value);
            mNode->updateCache(xyz, *mParent);
        } else {
            mNext.setValue(xyz, value);
        }
    }

    /// Set the value of the voxel at the given coordinates to the sum of its current
    /// value and the given value, and mark the voxel as active.
    void setValueOnSum(const Coord& xyz, const ValueType& value)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
            const_cast<NodeType*>(mNode)->setValueOnSumAndCache(xyz, value, *mParent);
        } else {
            mNext.setValueOnSum(xyz, value);
        }
    }

    /// Set the value of the voxel at the given coordinates and mark the voxel as inactive.
    void setValueOff(const Coord& xyz, const ValueType& value)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
            const_cast<NodeType*>(mNode)->setValueOffAndCache(xyz, value, *mParent);
        } else {
            mNext.setValueOff(xyz, value);
        }
    }

    /// Set the active state of the voxel at the given coordinates.
    void setActiveState(const Coord& xyz, bool on)
    {
        if (this->isHashed(xyz)) {
            assert(mNode);
            BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
            const_cast<NodeType*>(mNode)->setActiveStateAndCache(xyz, on, *mParent);
        } else {
            mNext.setActiveState(xyz, on);
        }
    }

private:
    CacheItem(const CacheItem&);
    CacheItem& operator=(const CacheItem&);

    bool isHashed(const Coord& xyz) const
    {
        return (xyz[0] & ~Coord::ValueType(NodeType::DIM-1)) == mHash[0]
            && (xyz[1] & ~Coord::ValueType(NodeType::DIM-1)) == mHash[1]
            && (xyz[2] & ~Coord::ValueType(NodeType::DIM-1)) == mHash[2];
    }

    TreeCacheT* mParent;
    Coord mHash;
    const NodeType* mNode;
    typedef typename boost::mpl::pop_front<NodeVecT>::type RestT; // NodeVecT minus its first item
    CacheItem<TreeCacheT, RestT, /*AtRoot=*/boost::mpl::size<RestT>::value == 1> mNext;
};// end of CacheItem


/// The tail of a compile-time list of cached node pointers, ordered from LeafNode to RootNode
template<typename TreeCacheT, typename NodeVecT>
class CacheItem<TreeCacheT, NodeVecT, /*AtRoot=*/true>
{
public:
    typedef typename boost::mpl::front<NodeVecT>::type RootNodeType;
    typedef typename RootNodeType::ValueType           ValueType;
    typedef typename RootNodeType::LeafNodeType        LeafNodeType;

    CacheItem(TreeCacheT& parent): mParent(&parent), mRoot(NULL) {}
    CacheItem(TreeCacheT& parent, const CacheItem& other): mParent(&parent), mRoot(other.mRoot) {}

    CacheItem& copy(TreeCacheT& parent, const CacheItem& other)
    {
        mParent = &parent;
        mRoot = other.mRoot;
        return *this;
    }

    bool isCached(const Coord& xyz) const { return this->isHashed(xyz); }

    void insert(const Coord&, const RootNodeType* root) { mRoot = root; }

    void erase(const RootNodeType*) { mRoot = NULL; }

    void clear() { mRoot = NULL; }

    void getNode(RootNodeType*& node)
    {
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        node = const_cast<RootNodeType*>(mRoot);
    }
    void getNode(const RootNodeType*& node) const { node = mRoot; }

    LeafNodeType* touchLeaf(const Coord& xyz)
    {
        assert(mRoot);
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        return const_cast<RootNodeType*>(mRoot)->touchLeafAndCache(xyz, *mParent);
    }

    LeafNodeType* probeLeaf(const Coord& xyz)
    {
        assert(mRoot);
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        return const_cast<RootNodeType*>(mRoot)->probeLeafAndCache(xyz, *mParent);
    }

    // ====================> DEPRECATED METHODS <========================
    int getValueDepth(const Coord& xyz)
    {
        assert(mRoot);
        return mRoot->getValueDepthAndCache(xyz, *mParent);
    }
    bool isValueOn(const Coord& xyz)
    {
        assert(mRoot);
        return mRoot->isValueOnAndCache(xyz, *mParent);
    }

    bool probeValue(const Coord& xyz, ValueType& value)
    {
        assert(mRoot);
        return mRoot->probeValueAndCache(xyz, value, *mParent);
    }
    bool isVoxel(const Coord& xyz)
    {
        assert(mRoot);
        return mRoot->getValueDepthAndCache(xyz, *mParent)==static_cast<int>(RootNodeType::LEVEL);
    }
    const ValueType& getValue(const Coord& xyz)
    {
        assert(mRoot);
        return mRoot->getValueAndCache(xyz, *mParent);
    }

    const ValueType& newGetValue(const Coord& xyz)
    {
        assert(mRoot);
        const ValueType& val = mRoot->getValue(xyz);
        mRoot->updateCache(xyz, *mParent);
        return val;
    }
    //=======================================================================

    const ValueType& getValue(const Coord& xyz, bool& state, int& level)
    {
        assert(mRoot);
        return mRoot->getValue(xyz, state, level, *mParent);
    }

    template <bool State, bool Level>
    const ValueType& probe(const Coord& xyz, bool& state, int& level)
    {
        assert(mRoot);
        return mRoot->probe<State, Level>(xyz, state, level, *mParent);
    }

    template <typename ValueProbeType>
    const ValueType& probe(const Coord& xyz, ValueProbeType& p)
    {
        assert(mRoot);
        return mRoot->probe(xyz, p, *mParent);
    }

    void setValue(const Coord& xyz, const ValueType& value)
    {
        assert(mRoot);
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        const_cast<RootNodeType*>(mRoot)->setValueAndCache(xyz, value, *mParent);
    }
    void setValueOnly(const Coord& xyz, const ValueType& value)
    {
        assert(mRoot);
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        const_cast<RootNodeType*>(mRoot)->setValueOnlyAndCache(xyz, value, *mParent);
    }
    void setValueOn(const Coord& xyz, const ValueType& value) { this->setValue(xyz, value); }

    void newSetValue(const Coord& xyz, const ValueType& value)
    {
        assert(mRoot);
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        const_cast<RootNodeType*>(mRoot)->setValue(xyz, value);
        mRoot->updateCache(xyz, *mParent);
    }

    void setValueOnSum(const Coord& xyz, const ValueType& value)
    {
        assert(mRoot);
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        const_cast<RootNodeType*>(mRoot)->setValueOnSumAndCache(xyz, value, *mParent);
    }

    void setValueOff(const Coord& xyz, const ValueType& value)
    {
        assert(mRoot);
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        const_cast<RootNodeType*>(mRoot)->setValueOffAndCache(xyz, value, *mParent);
    }

    void setActiveState(const Coord& xyz, bool on)
    {
        assert(mRoot);
        BOOST_STATIC_ASSERT(!TreeCacheT::IsConstTree);
        const_cast<RootNodeType*>(mRoot)->setActiveStateAndCache(xyz, on, *mParent);
    }

private:
    CacheItem(const CacheItem&);
    CacheItem& operator=(const CacheItem&);

    bool isHashed(const Coord&) const { return false; }

    TreeCacheT* mParent;
    const RootNodeType* mRoot;
};// end of CacheItem specialized for RootNode


////////////////////////////////////////


/// @brief This base class for ValueAccessors manages registration of an accessor
/// with a tree so that the tree can automatically clear the accessor whenever
/// one of its nodes is deleted.
/// @internal A base class is needed because ValueAccessor is templated on both
/// a Tree type and a mutex type.  The various instantiations of the template
/// are distinct, unrelated types, so they can't easily be stored in a container
/// such as the Tree's CacheRegistry.  This base class, in contrast, is templated
/// only on the Tree type, so for any given Tree, only two distinct instantiations
/// are possible, ValueAccessorBase<Tree> and ValueAccessorBase<const Tree>.
template<typename TreeType>
class ValueAccessorBase
{
public:
    ValueAccessorBase(TreeType& tree): mTree(&tree) { tree.attachAccessor(*this); }
    virtual ~ValueAccessorBase() { if (mTree) mTree->releaseAccessor(*this); }

    TreeType* getTree() const { return mTree; }

    ValueAccessorBase(const ValueAccessorBase<TreeType>& other): mTree(other.mTree)
    {
        if (mTree) mTree->attachAccessor(*this);
    }

    ValueAccessorBase& operator=(const ValueAccessorBase<TreeType>& other)
    {
        if (&other != this) {
            if (mTree) mTree->releaseAccessor(*this);
            mTree = other.mTree;
            if (mTree) mTree->attachAccessor(*this);
        }
        return *this;
    }

    virtual void clear() = 0;

protected:
    // Allow trees to deregister themselves.
    template<typename> friend class Tree;

    virtual void release() { mTree = NULL; }

    TreeType* mTree;
};


////////////////////////////////////////


/// When traversing a grid in a spatially coherent pattern (e.g., iterating
/// over neighboring voxels), request a @c ValueAccessor from the grid
/// (with Grid::getAccessor()) and use the accessor's @c getValue() and
/// @c setValue() methods.  These will typically be significantly faster
/// than accessing voxels directly in the grid's tree.
/// @note If @c MutexType is a TBB-compatible mutex, then multiple threads
/// may safely access a single, shared accessor.  However, it is recommended that,
/// instead, each thread be assigned its own, non-mutex-protected accessor.
template<typename _TreeType, typename MutexType = tbb::null_mutex>
class ValueAccessor: public ValueAccessorBase<_TreeType>
{
public:
    typedef typename _TreeType::RootNodeType RootNodeT;
    typedef typename _TreeType::LeafNodeType LeafNodeT;
    typedef typename InvertedTree<RootNodeT, RootNodeT::LEVEL>::Type InvTreeT;
    typedef CacheItem<ValueAccessor, InvTreeT, /*AtRoot=*/false> CacheItemT;
    typedef ValueAccessorBase<_TreeType> BaseT;
    typedef typename MutexType::scoped_lock LockT;
    typedef _TreeType TreeType;
    typedef typename RootNodeT::ValueType     ValueType;
    static const bool IsConstTree = boost::is_const<TreeType>::value;

    ValueAccessor(TreeType& tree): BaseT(tree), mCache(*this)
    {
        mCache.insert(Coord(), &tree.getRootNode());
    }

    ValueAccessor(const ValueAccessor& other): BaseT(other), mCache(*this, other.mCache) {}

    ValueAccessor& operator=(const ValueAccessor& other)
    {
        if (&other != this) {
            this->BaseT::operator=(other);
            mCache.copy(*this, other.mCache);
        }
        return *this;
    }
    virtual ~ValueAccessor() {}

    /// Return @c true if nodes along the path to the given voxel have been cached.
    bool isCached(const Coord& xyz) const { LockT lock(mMutex); return mCache.isCached(xyz); }

    /// Return the value of the voxel at the given coordinates.
    const ValueType& getValue(const Coord& xyz) const
    {
        LockT lock(mMutex);
        return mCache.getValue(xyz);
    }

    /// Return the active state of the voxel at the given coordinates.
    bool isValueOn(const Coord& xyz) const { LockT lock(mMutex); return mCache.isValueOn(xyz); }

    /// Return the active state of the voxel as well as its value
    bool probeValue(const Coord& xyz, ValueType& value) const
    {
        LockT lock(mMutex);
        return mCache.probeValue(xyz,value);
    }

    /// Return the tree depth (0 = root) at which the value of voxel (x, y, z) resides,
    /// or -1 if (x, y, z) isn't explicitly represented in the tree (i.e., if it is
    /// implicitly a background voxel).
    int getValueDepth(const Coord& xyz) const
    {
        LockT lock(mMutex);
        return mCache.getValueDepth(xyz);
    }

    /// Return @c true if the value of voxel (x, y, z) resides at the leaf level
    /// of the tree, i.e., if it is not a tile value.
    bool isVoxel(const Coord& xyz) const { LockT lock(mMutex); return mCache.isVoxel(xyz); }

    //@{
    /// Set the value of the voxel at the given coordinates and mark the voxel as active.
    void setValue(const Coord& xyz, const ValueType& value)
    {
        LockT lock(mMutex);
        mCache.setValue(xyz, value);
    }
    void setValueOn(const Coord& xyz, const ValueType& value) { this->setValue(xyz, value); }
    //@}

    /// Set the value of the voxel at the given coordinate but preserves its active state.
    void setValueOnly(const Coord& xyz, const ValueType& value)
    {
        LockT lock(mMutex);
        mCache.setValueOnly(xyz, value);
    }

    /// Set the value of the voxel at the given coordinates and mark the voxel
    /// as active.  [Experimental]
    void newSetValue(const Coord& xyz, const ValueType& value)
    {
        LockT lock(mMutex);
        mCache.newSetValue(xyz, value);
    }

    /// Set the value of the voxel at the given coordinates and mark the voxel as inactive.
    void setValueOff(const Coord& xyz, const ValueType& value)
    {
        LockT lock(mMutex);
        mCache.setValueOff(xyz, value);
    }

    /// Set the value of the voxel at the given coordinates to the sum of its current
    /// value and the given value, and mark the voxel as active.
    void setValueOnSum(const Coord& xyz, const ValueType& value)
    {
        LockT lock(mMutex);
        mCache.setValueOnSum(xyz, value);
    }

    /// Set the active state of the voxel at the given coordinates without changing its value.
    void setActiveState(const Coord& xyz, bool on = true)
    {
        LockT lock(mMutex);
        mCache.setActiveState(xyz, on);
    }
    /// Mark the voxel at the given coordinates as active without changing its value.
    void setValueOn(const Coord& xyz) { this->setActiveState(xyz, true); }
    /// Mark the voxel at the given coordinates as inactive without changing its value.
    void setValueOff(const Coord& xyz) { this->setActiveState(xyz, false); }

    /// Return the cached node of type @a NodeType.  [Mainly for internal use]
    template<typename NodeType>
    NodeType* getNode()
    {
        LockT lock(mMutex);
        NodeType* node = NULL;
        mCache.getNode(node);
        return node;
    }

    /// Cache the given node, which should lie along the path from the root node to
    /// the node containing voxel (x, y, z).  [Mainly for internal use]
    template<typename NodeType>
    void insertNode(const Coord& xyz, NodeType& node)
    {
        LockT lock(mMutex);
        mCache.insert(xyz, &node);
    }

    /// If a node of the given type exists in the cache, remove it, so that
    /// isCached(xyz) returns @c false for any voxel (x, y, z) contained in
    /// that node.  [Mainly for internal use]
    template<typename NodeType>
    void eraseNode() { LockT lock(mMutex); NodeType* node = NULL; mCache.erase(node); }

    LeafNodeT* touchLeaf(const Coord& xyz)
    {
        LockT lock(mMutex);
        return mCache.touchLeaf(xyz);
    }

    LeafNodeT* probeLeaf(const Coord& xyz)
    {
        LockT lock(mMutex);
        return mCache.probeLeaf(xyz);
    }

    /// Remove all nodes from this cache, then reinsert the root node.
    virtual void clear()
    {
        LockT lock(mMutex);
        mCache.clear();
        if (this->mTree) mCache.insert(Coord(), &(this->mTree->getRootNode()));
    }

private:
    // Allow nodes to insert themselves into the cache.
    template<typename> friend class RootNode;
    template<typename, Index> friend class InternalNode;
    template<typename, Index> friend class LeafNode;
    // Allow trees to deregister themselves.
    template<typename> friend class Tree;

    /// Prevent this accessor from calling Tree::releaseCache() on a tree that
    /// no longer exists.  (Called by mTree when it is destroyed.)
    virtual void release()
    {
        LockT lock(mMutex);
        this->BaseT::release();
        mCache.clear();
    }

    /// Cache the given node, which should lie along the path from the root node to
    /// the node containing voxel (x, y, z).
    /// @note This operation is not mutex-protected and is intended to be called
    /// only by nodes and only in the context of a getValue() or setValue() call.
    template<typename NodeType>
    void insert(const Coord& xyz, NodeType* node) { mCache.insert(xyz, node); }

    mutable CacheItemT mCache;
    mutable MutexType  mMutex;

}; // class ValueAccessor


/// This accessor is thread-safe (at the cost of speed) for both reading and
/// writing to a tree.  That is, multiple threads may safely access a single,
/// shared ValueAccessorRW.  For better performance, however, it is recommended
/// that, instead, each thread be assigned its own (non-mutex protected) accessor.
template<typename TreeType>
struct ValueAccessorRW: public ValueAccessor<TreeType, tbb::spin_mutex>
{
    ValueAccessorRW(TreeType& tree): ValueAccessor<TreeType, tbb::spin_mutex>(tree) {}
};

} // namespace tree
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TREE_VALUEACCESSOR_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
