///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @author Ken Museth
/// @file Magnitude.h

#ifndef OPENVDB_TOOLS_MAGNITUDE_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_MAGNITUDE_HAS_BEEN_INCLUDED

#include <openvdb/Grid.h>
#include <openvdb/math/Math.h>
#include <tbb/parallel_reduce.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief Computes the magnitude of a vector-valued grid
template<typename InGridT>
class Magnitude
{
public:
    typedef InGridT                                                  InGridType;
    typedef typename InGridType::ValueType::value_type               Real;
    typedef typename InGridType::ConstAccessor                       Accessor;
    typedef typename InGridType::TreeType                            InTreeType;
    typedef typename InTreeType::LeafNodeType                        LeafType;
    typedef typename tree::IteratorRange<typename InTreeType::LeafCIter> RangeType;
    typedef typename InTreeType::template ValueConverter<Real>::Type OutTreeType;
    typedef Grid<OutTreeType>                                        OutGridType;

    Magnitude(const InGridType& grid):
        mInputGrid(&grid),
        mOutputGrid(OutGridType::create(mInputGrid->background().length())),
        mBackground(0),
        mAccessor(mInputGrid->getConstAccessor())
    {
    }

    /// Copy constructor called by tbb::parallel_reduce threads
    Magnitude(const Magnitude& other, tbb::split):
        mInputGrid(other.mInputGrid),
        mOutputGrid(OutGridType::create(mInputGrid->background().length())),
        mBackground(other.mBackground),
        mAccessor(mInputGrid->getConstAccessor()) // local accessor
    {
    }

    ~Magnitude() {}

    Real operator()(const Coord& xyz) const { return mAccessor.getValue(xyz).length(); }

    typename OutGridType::Ptr process(bool threaded = true)
    {
        RangeType range(mInputGrid->tree().cbeginLeaf());
        if (threaded) {
            tbb::parallel_reduce(range, *this);
        } else {
            (*this)(range);
        }
        return mOutputGrid;
    }

    void operator()(RangeType& range)
    {
        typename OutGridType::Accessor outAccessor = mOutputGrid->getAccessor();
        for ( ; range; ++range) {
            for (typename LeafType::ValueOnCIter v = range.iterator()->beginValueOn(); v; ++v) {
                const Coord xyz = v.getCoord();
                outAccessor.setValue(xyz, (*this)(xyz));
            }
        }
    }

    void join(const Magnitude& other) { mOutputGrid->merge(*other.mOutputGrid); }

protected:
    const InGridType*           mInputGrid;
    typename OutGridType::Ptr   mOutputGrid;
    const Real                  mBackground;
    Accessor                    mAccessor;
}; // end of Magnitude class

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_MAGNITUDE_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
